/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <vector>
#include <gtk/gtk.h>
#include <gtkmm/iconfactory.h>
#include <gtkmm/stock.h>
#include <gtkmm/clipboard.h>

#include "gcs-mainwindow.h"
#include "dialogs/gcs-about-window.h"
#include "dialogs/gcs-save-chooser.h"
#include "widgets/gcs-bookmarklist.h"
#include "core/log-stream.h"
#include "gcs-history.h"
#include "gcs-i18n.h"

namespace gcs
{

    void clipboard_owner_changed(GtkClipboard* clipboard, GdkEvent* event, gpointer data)
    {
        MainWindow::Instance().update_paste_status();
    }


    void MainWindow::init_actions(void)
    {
        // Register stock icons for GNOME Colorscheme
        Glib::RefPtr<Gtk::IconFactory> iconFactory = Gtk::IconFactory::create();
        iconFactory->add_default();

        // the icon for lightening a scheme
        try
        {
        Glib::RefPtr<Gdk::Pixbuf> refLighten = Gdk::Pixbuf::create_from_file(
                AGAVE_ICONDIR "/lighten.png");
        boost::shared_ptr<Gtk::StockID> pLightenID(new Gtk::StockID("gcs-lighten"));
        boost::shared_ptr<Gtk::IconSet> pLighten(new Gtk::IconSet(refLighten));

        // the icon for darkening a scheme
        Glib::RefPtr<Gdk::Pixbuf> refDarken = Gdk::Pixbuf::create_from_file(
                AGAVE_ICONDIR "/darken.png");
        boost::shared_ptr<Gtk::StockID> pDarkenID(new Gtk::StockID("gcs-darken"));
        boost::shared_ptr<Gtk::IconSet> pDarken(new Gtk::IconSet(refDarken));

        // the icon for increasing a scheme's saturation
        Glib::RefPtr<Gdk::Pixbuf> refSat = Gdk::Pixbuf::create_from_file(
                AGAVE_ICONDIR "/saturate.png");
        boost::shared_ptr<Gtk::StockID> pSatID(new Gtk::StockID("gcs-saturate"));
        boost::shared_ptr<Gtk::IconSet> pSat(new Gtk::IconSet(refSat));

        // the icon for desaturating a scheme
        Glib::RefPtr<Gdk::Pixbuf> refDesat = Gdk::Pixbuf::create_from_file(
                AGAVE_ICONDIR "/desaturate.png");
        boost::shared_ptr<Gtk::StockID> pDesatID(new Gtk::StockID("gcs-desaturate"));
        boost::shared_ptr<Gtk::IconSet> pDesat(new Gtk::IconSet(refDesat));

        // add all icons to the factory so that they can be used later on.
        iconFactory->add(*pLightenID, *pLighten);
        iconFactory->add(*pDarkenID, *pDarken);
        iconFactory->add(*pSatID, *pSat);
        iconFactory->add(*pDesatID, *pDesat);


        /* Create the main menu */
        m_refActionGroup = Gtk::ActionGroup::create();
        LOG("Created ActionGroup");

        /* The File Menu */
        m_refActionGroup->add(Gtk::Action::create("MenuFile", _("_File")));
        m_refActionGroup->add(Gtk::Action::create("New", Gtk::Stock::NEW),
                sigc::mem_fun(*this, &MainWindow::on_action_file_new) );
        m_refActionGroup->add(Gtk::Action::create("SaveScheme",
                    Gtk::Stock::SAVE, _("_Save Scheme..."),
                    _("Save current scheme as a GIMP Palette file")),
                sigc::mem_fun(*this, &MainWindow::on_action_file_save));
        m_refActionGroup->add(Gtk::Action::create("ExportFavorites",
                    Gtk::Stock::SAVE_AS, _("E_xport Favorites..."),
                    _("Export favorite colors as a GIMP Palette file")),
                sigc::mem_fun(*this, &MainWindow::on_action_file_export));
        m_refActionGroup->add(Gtk::Action::create("Quit", Gtk::Stock::QUIT),
                sigc::mem_fun(*this, &MainWindow::on_action_file_quit));   

        /* The Edit Menu */
        m_refActionGroup->add(Gtk::Action::create("MenuEdit", _("_Edit")));
        m_refActionGroup->add(Gtk::Action::create("Preferences", Gtk::Stock::PREFERENCES),
                sigc::mem_fun(*this, &MainWindow::on_action_edit_prefs));   

        /* The Help menu */
        m_refActionGroup->add(Gtk::Action::create("MenuHelp", _("_Help")));
        m_refActionGroup->add(Gtk::Action::create("Contents", Gtk::Stock::HELP),
                sigc::mem_fun(*this, &MainWindow::on_action_help_contents));   
        // Gtk::Stock::About requires gtkmm 2.6
        m_refActionGroup->add(Gtk::Action::create("About", Gtk::Stock::ABOUT),
                sigc::mem_fun(*this, &MainWindow::on_action_help_about));   


        // History Navigation
        m_refActionGroup->add(Gtk::Action::create("HistoryBack", Gtk::Stock::GO_BACK),
                Gtk::AccelKey("<alt>Left"),
                sigc::mem_fun(*this, &MainWindow::on_action_history_back));
        m_refActionGroup->get_action("HistoryBack")->set_sensitive(false);
        m_refActionGroup->get_action("HistoryBack")->property_is_important() = true;
        m_refActionGroup->add(Gtk::Action::create("HistoryFwd", Gtk::Stock::GO_FORWARD),
                Gtk::AccelKey("<alt>Right"),
                sigc::mem_fun(*this, &MainWindow::on_action_history_fwd));
        m_refActionGroup->get_action("HistoryFwd")->set_sensitive(false);

        // Bookmark Actions
        m_refActionGroup->add(Gtk::Action::create("MenuBookmarks", _("Fa_vorites")));
        m_refActionGroup->add(Gtk::Action::create("AddBookmark", Gtk::Stock::ADD,
                    _("Add to Favorites"), _("Add the current color to favorites")),
                Gtk::AccelKey("<control>D"),
                sigc::mem_fun(*this, &MainWindow::on_action_add_bookmark));
        m_refActionGroup->add(Gtk::Action::create("RemoveBookmark", Gtk::Stock::REMOVE,
                    _("Remove Selected"), _("Remove the selected color from your favorites")),
                Gtk::AccelKey("<control>X"),
                sigc::mem_fun(*this, &MainWindow::on_action_remove_bookmark));
        m_refActionGroup->get_action("RemoveBookmark")->set_sensitive(false);
        m_refActionGroup->add(Gtk::Action::create("ClearBookmarks", Gtk::Stock::CLEAR, 
                    _("_Clear All"), _("Clear the list of favorite colors")),
                    sigc::mem_fun(*this, &MainWindow::on_action_clear_bookmarks));
        m_refActionGroup->get_action("ClearBookmarks")->set_sensitive(false);

        // Actions for lightness and saturation
        m_refActionGroup->add(Gtk::Action::create("LightenScheme", *pLightenID,
                    _("_Lighten Scheme"), _("Increase the brightness")),
                Gtk::AccelKey("KP_Add"),
                sigc::mem_fun(*this, &MainWindow::on_action_lighten_scheme));
        m_refActionGroup->add(Gtk::Action::create("DarkenScheme", *pDarkenID,
                    _("_Darken Scheme"), _("Decrease the brightness")),
                Gtk::AccelKey("KP_Subtract"),
                sigc::mem_fun(*this, &MainWindow::on_action_darken_scheme));
        m_refActionGroup->add(Gtk::Action::create("SaturateScheme", *pSatID,
                    _("_Saturate Scheme"), _("Increase the saturation")),
                Gtk::AccelKey("<control>KP_Add"),
                sigc::mem_fun(*this, &MainWindow::on_action_saturate_scheme));
        m_refActionGroup->add(Gtk::Action::create("DesaturateScheme", *pDesatID,
                    _("D_esaturate Scheme"), _("Decrease the saturation")),
                Gtk::AccelKey("<control>KP_Subtract"),
                sigc::mem_fun(*this, &MainWindow::on_action_desaturate_scheme));

        m_refActionGroup->add(Gtk::Action::create("Randomize", Gtk::Stock::REFRESH,
                    _("_Random"), _("Generate a random color")),
                Gtk::AccelKey("<control>R"),
                sigc::mem_fun(*this, &MainWindow::on_action_randomize));
        m_refActionGroup->get_action("Randomize")->property_is_important() = true;

        // paste color
        m_refActionGroup->add(Gtk::Action::create("Paste", Gtk::Stock::PASTE),
                Gtk::AccelKey("<control>V"),
                sigc::mem_fun(*this, &MainWindow::on_action_paste));

        // We need to figure out when there is valid data in the clipboard so
        // that we can disable the menu item when there's nothing to paste.  We
        // do this by connecting to the "owner-change" signal of the clipboard
        // to check the status of the clipboard whenever it changes
        Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get();
        // For now, connect to the clipboard owner-change signal in gtk+ since
        // it's not wrapped by gtkmm yet
        g_signal_connect(refClipboard->gobj(), "owner-change",
                G_CALLBACK(clipboard_owner_changed), NULL);
        // update the paste status at startup so it's disabled if there's
        // nothing in the clipboard at startup
        update_paste_status();

        // Actions for context menu on colorswatches
        m_refActionGroup->add(Gtk::Action::create("UseColor", "_Use"),
                sigc::mem_fun(*this, &MainWindow::on_action_use_color));
        }

        catch (Glib::Error& ex)
        {
            std::cerr << __FILE__ << ": " << ex.what() << std::endl;
        }
    }


    /* File Menu Actions */
    void MainWindow::on_action_file_new(void)
    {
        LOG("file->new");
        static gint i = 1;
        std::vector<Glib::ustring> palettes;
        palettes.push_back(AGAVE_PALETTEDIR "/Tango-Palette.gpl");
        palettes.push_back(AGAVE_PALETTEDIR "/Web.gpl");
        palettes.push_back(AGAVE_PALETTEDIR "/Visibone.gpl");
        palettes.push_back(AGAVE_PALETTEDIR "/Ximian-Palette.gpl");
        m_PaletteView->set_from_file(palettes[i % palettes.size()]);
        i++;
    }


    void MainWindow::on_action_file_save(void)
    {
        // FIXME: add stuff
    }


    void MainWindow::on_action_file_export(void)
    {
        Glib::ustring filename;
        Dialogs::SaveChooser save_dialog("Export Favorites...");

        if (save_dialog.run() == Gtk::RESPONSE_OK)
        {
            filename = save_dialog.get_filename();
        }
        LOG("saving file to " << filename);
        m_pBookmarkList->save_to_disk(filename);
    }


    void MainWindow::on_action_file_quit(void)
    {
        quit();
    }


    /* Edit Menu Actions */
    void MainWindow::on_action_edit_prefs(void)
    {
    }


    /* Help Menu Actions */
    void MainWindow::on_action_help_contents(void)
    {
    }


    void MainWindow::on_action_help_about(void)
    {
        m_pAbout->show();
    }


    void MainWindow::on_action_history_back(void)
    {
        LOG("Going Back");
        ColorPtr c = Color::create(m_pHistory->go_back());
        set_color(c);
    }


    void MainWindow::on_action_history_fwd(void)
    {
        LOG("Going Forward");
        ColorPtr c = Color::create(m_pHistory->go_forward());
        set_color(c);
    }


    void MainWindow::on_action_lighten_scheme(void)
    {
        ColorPtr clr = Color::create(m_pColorButton->get_color());
        // bump value by 5%
        clr->set_value(static_cast<int>(0.05 * maxSvValue + clr->get_value()));
        set_color(clr);
    }


    void MainWindow::on_action_darken_scheme(void)
    {
        ColorPtr clr = Color::create(m_pColorButton->get_color());
        // decrease value by 5%
        clr->set_value(static_cast<int>(clr->get_value() - 0.05 * maxSvValue));
        set_color(clr);
    }


    void MainWindow::on_action_saturate_scheme(void)
    {
        ColorPtr clr = Color::create(m_pColorButton->get_color());
        // increase saturation by 5%
        clr->set_saturation(static_cast<int>(clr->get_saturation() + 0.05 * maxSvValue));
        set_color(clr);
    }


    void MainWindow::on_action_desaturate_scheme(void)
    {
        ColorPtr clr = Color::create(m_pColorButton->get_color());
        // decrease value by 5%
        clr->set_saturation(static_cast<int>(clr->get_saturation() - 0.05 * maxSvValue));
        set_color(clr);
    }


    void MainWindow::on_action_randomize(void)
    {
        gint r, g, b;
        r = rand_range(maxRgbValue);
        g = rand_range(maxRgbValue);
        b = rand_range(maxRgbValue);
        ColorPtr clr = Color::create(r, g, b);
        set_color(clr);
    }


    void MainWindow::on_action_paste(void)
    {
        Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get();
        refClipboard->request_contents("UTF8_STRING",  sigc::mem_fun(*this,
                    &MainWindow::on_clipboard_received) );
    }


    void MainWindow::on_clipboard_received(const Gtk::SelectionData& data)
    {
        Glib::ustring clipboard_data = data.get_text();
        if (!clipboard_data.empty())
        {
            ColorPtr c = Color::create();
            if (c->set(clipboard_data))
            {
                set_color(c);
            }
        }
    }


    void MainWindow::update_paste_status(void)
    {
        //Disable the paste button if there is nothing to paste.
        Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get();

        //Discover whether text is available in the clipboard
        refClipboard->request_targets(sigc::mem_fun(*this,
                    &MainWindow::on_clipboard_received_targets));
    }


    void MainWindow::on_clipboard_received_targets(const Glib::StringArrayHandle& targets_array)
    {
        // Get the list of available clipboard targets:
        std::list<std::string> targets = targets_array;

        const bool bPasteIsPossible = std::find(targets.begin(), targets.end(),
                "UTF8_STRING") != targets.end();

        // Enable/Disable the Paste button appropriately:
        m_refActionGroup->get_action("Paste")->set_sensitive(bPasteIsPossible);
    }


    gint MainWindow::rand_range(gint range)
    {
        return static_cast<gint>(
                static_cast<double>(rand()) / static_cast<double>(RAND_MAX) *
                static_cast<double>(range));
    }


    void MainWindow::on_action_use_color(void)
    {}


    void MainWindow::on_action_add_bookmark(void)
    {
        ColorPtr clr = Color::create(m_pColorButton->get_color());
        m_pBookmarkList->add(clr);
        update_bookmark_actions();
    }


    void MainWindow::on_action_remove_bookmark(void)
    {
        m_pBookmarkList->remove_selected();
        update_bookmark_actions();
    }


    void MainWindow::on_action_clear_bookmarks(void)
    {
        LOG("Clearing Bookmarks");
        m_pBookmarkList->clear();
        update_bookmark_actions();
    }


    void MainWindow::update_bookmark_actions(void)
    {
        m_refActionGroup->get_action("RemoveBookmark")->set_sensitive(!m_pBookmarkList->empty()
                && m_pBookmarkList->count_selected());
        m_refActionGroup->get_action("ClearBookmarks")->set_sensitive(!m_pBookmarkList->empty());
    }


} // namespace gcs
