/**********************************************************************
 ** ChatLine - contains methods and attributes for a chatline, which is
 **            an object that defines a particular communications line
 **
 ** Last reviewed:
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the
 **   License, or any later version.
 **
 **   This program is distributed in the hope that it will be useful, but
 **   WITHOUT ANY WARRANTY; without even the implied warranty of
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 **   General Public License for more details.
 **
 **   You should have received a copy of the GNU General Public License
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **
 **********************************************************************/

#ifndef CHATLINE_C
#define CHATLINE_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "chatline.h"
#include "lexer.h"
#include "builder.h"
#include "parse.h"
#include "utils.h"
#include "player.h"
#include "lineflags.h"
#include "objtype.h"
#include "newfuncts.h"
#include "global.h"
#include "configs.h"
#include "adminflags.h"

/***********************************************************************
 ** ChatLine (constructor) - loads chatline name and attributes
 **
 ** Parameters: chatline_name - this chatline name
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

ChatLine::ChatLine(char *chatline_name)
{
   Strings tmp_name;

   tmp_name.assign_word(chatline_name, 1);

   if (tmp_name.str_len() > MAXNAMELEN)
      tmp_name.truncate(MAXNAMELEN);
   set_name(tmp_name.str_show());

   modified = 0;
   obj_type = OBJ_TYPE_CHATLINE;
   line_flags = new_Flags(1);
}


/***********************************************************************
 ** ~ChatLine (destructor) - cleans up the chat line
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

ChatLine::~ChatLine(void)
{
   delete_Flags(line_flags);
}


/***********************************************************************
 ** set_lead_in_actor - sets the lead_in_actor string for this chatline
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int ChatLine::set_lead_in_actor(char *the_string)
{
   if (the_string == NULL)
      return -1;

   lead_in_actor = the_string;
   return 1;
}

/***********************************************************************
 ** get_lead_in_actor - gets the lead_in_actor string for this chatline
 **
 ** Parameters: None
 **
 ** Returns: pointer to the lead_in string
 **
 ***********************************************************************/

char *ChatLine::get_lead_in_actor(void)
{
   return lead_in_actor.str_show();
}



/***********************************************************************
 ** set_lead_in_crowd - sets the lead_in_crowd string for this chatline
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int ChatLine::set_lead_in_crowd(char *the_string)
{
   if (the_string == NULL)
      return -1;

   lead_in_crowd = the_string;
   return 1;
}

/***********************************************************************
 ** get_lead_in_crowd - gets the lead_in_crowd string for this chatline
 **
 ** Parameters: None
 **
 ** Returns: pointer to the lead_in_crowd string
 **
 ***********************************************************************/

char *ChatLine::get_lead_in_crowd(void)
{
   return lead_in_crowd.str_show();
}


/***********************************************************************
 ** set_text_color - sets the text_color string for this chatline
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int ChatLine::set_text_color(char *the_string)
{
   if (the_string == NULL)
      return -1;

   text_color = the_string;
   return 1;
}

/***********************************************************************
 ** get_text_color - gets the text_color string for this chatline
 **
 ** Parameters: None
 **
 ** Returns: pointer to the text_color string
 **
 ***********************************************************************/

char *ChatLine::get_text_color(void)
{
   return text_color.str_show();
}



/***********************************************************************
 ** set_border - sets the border string for this chatline
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int ChatLine::set_border(char *the_string)
{
   if (the_string == NULL)
      return -1;

   border = the_string;
   return 1;
}

/***********************************************************************
 ** get_border - gets the border string for this chatline
 **
 ** Parameters: None
 **
 ** Returns: pointer to the border string
 **
 ***********************************************************************/

char *ChatLine::get_border(void)
{
   return border.str_show();
}


/***********************************************************************
 ** execute_chat - executes the chat, displaying as appropriate
 **
 ** Parameters: the_parsed - parse information from user input
 **             the_player - who is executing the chat
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/

int ChatLine::execute_chat(Parse *the_parsed, MudObject *the_user)
{
   Strings    formatted;
   Strings    tmp_lead_in;
   Strings    holder;
   Strings    auth_chatlines;
   Flags      *tmp_lineflags;
   Flags      *tmp_adminflags;
   int        the_num = 0;
   Player     *the_player = NULL;
   Builder    *the_builder = NULL;
   Player     *player_list;
   Builder    *builder_list;

   tmp_lineflags = get_lineflags();

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
   {
     the_player = (Player *) the_user;

      auth_chatlines = the_player->get_chatlines();
      if (!auth_chatlines.find_in_str(get_name()))
      {
        the_player->send_plr(_("You are not authorized to use this chat line.\n"));
        return -1;
      }

   }
   else
   {
     the_builder = (Builder *) the_user;
   }

   /* modification added 3/12/01 JH */
   /* if there is nothing after chat, send a message to the player */
   if ((the_parsed->get_speech() == NULL)||
        (strlen(the_parsed->get_speech()) <= 0))
   {
      the_player->send_plr(_("Format: chat <phrase>\n"));
      return -1;
   }
    /* end modification */

   tmp_lead_in.format_for_actions(get_lead_in_actor(), the_user, NULL);
   if (get_border() != NULL)
      tmp_lead_in.str_cat(get_border());
   if (get_text_color() != NULL)
      tmp_lead_in.str_cat(get_text_color());

   formatted.format_for_comm(the_parsed->get_speech(), tmp_lead_in.cstr_len(),
                                                   the_config.max_line_len);

   if (the_player != NULL)
      the_player->send_plr("%s%s%s&*\n", tmp_lead_in.str_show(), 
			formatted.str_show(), (get_border() == NULL) ? "" : 
                                               get_border());
   else
      the_builder->send_bldr("%s%s%s&*\n", tmp_lead_in.str_show(), 
			formatted.str_show(), (get_border() == NULL) ? "" : 
                                               get_border());


   tmp_lead_in.format_for_actions(get_lead_in_crowd(), the_user, NULL);
   if (get_border() != NULL)
      tmp_lead_in.str_cat(get_border());
   if (get_text_color() != NULL)
      tmp_lead_in.str_cat(get_text_color());
   if (tmp_lineflags->get_flag(LINEFLAG_MESSAGEBEEP))
      tmp_lead_in.str_cat("\a");

   formatted.format_for_comm(the_parsed->get_speech(), tmp_lead_in.cstr_len(),
                                                   the_config.max_line_len);

   holder.sprintf("%s%s%s&*\n", tmp_lead_in.str_show(), 
			formatted.str_show(), (get_border() == NULL) ? "" : 
                                               get_border());

   /* now we have to figure out who this should go to */
   if (the_player != NULL)
   {
     player_list = mainstruct->get_first_player();
     while (player_list != NULL)
     {
        tmp_adminflags = player_list->get_admflags();
        auth_chatlines.truncate(0);
        auth_chatlines = player_list->get_chatlines();
        if ((player_list != the_player) && 
            ((tmp_lineflags->get_flag(LINEFLAG_SENDALL)) ||
             ((tmp_lineflags->get_flag(LINEFLAG_SENDBUILDER)) &&
	      (tmp_adminflags->get_flag(ADMINFLAG_BUILDER))) ||
             ((tmp_lineflags->get_flag(LINEFLAG_SENDADMIN)) &&
	      (tmp_adminflags->get_flag(ADMINFLAG_ADMINISTRATOR))) ||
             ((tmp_lineflags->get_flag(LINEFLAG_SENDBUILDER)) &&
	      (tmp_adminflags->get_flag(ADMINFLAG_BUILDER))) ||
             ((tmp_lineflags->get_flag(LINEFLAG_SENDHASCHAT)) &&
              (auth_chatlines.find_in_str(get_name())))))
	  player_list->send_plr("%s", holder.str_show());
        player_list = player_list->get_next_player();
     }
   }
   else
   {
     builder_list = mainstruct->get_first_builder();
     while (builder_list != NULL)
     {
        tmp_adminflags = builder_list->get_adminflags();
        if ((builder_list != the_builder) && 
            ((tmp_lineflags->get_flag(LINEFLAG_SENDALL)) ||
             ((tmp_lineflags->get_flag(LINEFLAG_SENDBUILDER)) &&
	      (tmp_adminflags->get_flag(ADMINFLAG_BUILDER))) ||
             ((tmp_lineflags->get_flag(LINEFLAG_SENDADMIN)) &&
	      (tmp_adminflags->get_flag(ADMINFLAG_ADMINISTRATOR))) ||
             ((tmp_lineflags->get_flag(LINEFLAG_SENDBUILDER)) &&
	      (tmp_adminflags->get_flag(ADMINFLAG_BUILDER)))))
	  builder_list->send_bldr("%s", holder.str_show());

        builder_list = builder_list->get_next_builder();
     }
   }
   return 1;
}


/***********************************************************************
 ** load_chatline - loads a chatline from a file into memory
 **
 ** Parameters: the_file - where we are getting the chatline from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int ChatLine::load_chatline(FILE *the_file, ErrLog *error_log, int is_builder)
{
   token_record *the_token;
   Flags        *tmp_lineflags;
   Strings      holder;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');

       if (the_token->token_type != T_NUMERICAL)
       {
	 error_log->invalid_header("chatlines", "load_chatline");
         return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "chatlines", _("lead_in_actor"), 
				    "load_chatline");
     return -1;
   }

   /* get the lead_in_actor of this chatline */
   the_token = get_token(the_file, '^');
   set_lead_in_actor(the_token->the_string);   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "chatlines", _("lead_in_crowd"), 
				    "load_chatline");
     return -1;
   }

   /* get the lead_in_crowd of this chatline */
   the_token = get_token(the_file, '^');
   set_lead_in_crowd(the_token->the_string);   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "chatlines", _("text_color"), 
				    "load_chatline");
     return -1;
   }

   /* get the text_color of this chatline */
   the_token = get_token(the_file, '^');
   set_text_color(the_token->the_string);   
 
   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "chatlines", _("border"), 
				    "load_chatline");
     return -1;
   }

   /* get the border of this chatline */
   the_token = get_token(the_file, '^');
   set_border(the_token->the_string);   
 
   /* read in the line flags */
   tmp_lineflags = get_lineflags();
   if (tmp_lineflags->read_flags(the_file, error_log) <= 0)
   {
     error_log->invalid_attr_format(get_name(), "chatlines", _("lineflags"), 
				    "load_chatline");
     return -1;
   }

   return 1;
}


/***********************************************************************
 ** describe - describes the chatline to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void ChatLine::describe(Builder *the_builder)
{
   the_builder->send_bldr(_("\n&+GChatLine: \t&+M%s&*\n"), get_name());
   the_builder->send_bldr(_("&+GLeadInActor: \t&*%s&*\n"), get_lead_in_actor());
   the_builder->send_bldr(_("&+GLeadInCrowd: \t&*%s&*\n"), get_lead_in_crowd());
   the_builder->send_bldr(_("&+GTextColor: \t%sThis Color&*\n"), 
                   (get_text_color() == NULL) ? "" : get_text_color());
   the_builder->send_bldr(_("&+GBorder: \t&*%s&*\n"), get_border());
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the chatline to a builder
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void ChatLine::describe(Player *the_player)
{
   the_player->send_plr(_("\n&+GChatLine: \t&+M%s&*\n"), get_name());
   the_player->send_plr(_("&+GLeadInActor: \t&*%s&*\n"), get_lead_in_actor());
   the_player->send_plr(_("&+GLeadInCrowd: \t&*%s&*\n"), get_lead_in_crowd());
   the_player->send_plr(_("&+GTextColor: \t%sThis Color&*\n"),
                   (get_text_color() == NULL) ? "" : get_text_color());
   the_player->send_plr(_("&+GBorder: \t&*%s&*\n"), get_border());
   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int ChatLine::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr(_("You can set the following attributes on a chatline.\n"
               "   leadinactor, leadincrowd, textcolor, border and lineflags"));
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("leadinactor"),
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
               send_bldr(_("You need to specify a LeadIn to set to.\n"));
         return -1;
      }

      set_lead_in_actor(the_parsed->get_speech());

      the_builder->send_bldr(_("LeadInActor on %s set to: %s\n"), get_name(), 
                                                     get_lead_in_actor());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("leadincrowd"),
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
               send_bldr(_("You need to specify a LeadInCrowd to set to.\n"));
         return -1;
      }

      set_lead_in_crowd(the_parsed->get_speech());

      the_builder->send_bldr(_("LeadInCrowd on %s set to: %s\n"), get_name(), 
                                                     get_lead_in_crowd());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("textcolor"),
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
               send_bldr(_("You need to specify a TextColor to set to.\n"));
         return -1;
      }

      set_text_color(the_parsed->get_speech());

      the_builder->send_bldr(_("TextColor on %s set to: %sText Color&*\n"), 
                                              get_name(), get_text_color());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("border"),
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
               send_bldr(_("You need to specify a Border to set to.\n"));
         return -1;
      }

      set_border(the_parsed->get_speech());

      the_builder->send_bldr(_("Border on %s set to: %s&*\n"), 
                                              get_name(), get_border());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("lineflags"),
                               strlen(the_parsed->get_target1())))
   {
      Flags *tmp_lineflags;
      int flagnum;
      Strings holder;

      tmp_lineflags = get_lineflags();

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("Set which line flag?\n"));
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);

      if ((flagnum =
       tmp_lineflags->get_by_name(holder.str_show(), lineflagnames)) == -1)
      {
         the_builder->send_bldr(_("That is not a line flag.\n"));
         return -1;
      }

      holder.assign_word(the_parsed->get_speech(), 2);
      if (holder.str_show() == NULL)
      {
         the_builder->send_bldr(_("Set that flag to what?\n"
                                "Valid choices are: On or Off\n"));
         return -1;
      }

      if (holder.str_n_cmp(_("On"), holder.str_len()))
         tmp_lineflags->set_flag(flagnum);
      else if (holder.str_n_cmp(_("Off"), holder.str_len()))
         tmp_lineflags->clr_flag(flagnum);
      else
      {
         the_builder->send_bldr(_("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n"));
         return -1;
      }
      the_builder->send_bldr(_("Flag &+M%s&* has been set to: %s\n"),
          lineflagnames[flagnum], (tmp_lineflags->get_flag(flagnum)) ?
			     _("&+GOn&*") : _("&+ROff&*"));
      return 1;
   }
   the_builder->send_bldr(_("The attribute '%s' is not a chatline attribute.\n"),
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the chatline to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

void ChatLine::write_object(FILE *the_file, int build_format)
{
   Flags *tmp_lineflags;

   fprintf(the_file, "\nchatline %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   fprintf(the_file, "^%s^\n", (get_lead_in_actor() == NULL) ? "" : 
                                                       get_lead_in_actor());
   fprintf(the_file, "^%s^\n", (get_lead_in_crowd() == NULL) ? "" : 
                                                       get_lead_in_crowd());
   fprintf(the_file, "^%s^\n", (get_text_color() == NULL) ? "" : get_text_color());
   fprintf(the_file, "^%s^\n", (get_border() == NULL) ? "" : get_border());
   tmp_lineflags = get_lineflags();
   tmp_lineflags->write_flag(the_file);

   return;
}


/***********************************************************************
 ** is_modified - has this chatline been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/

int ChatLine::is_modified(void)
{
   return modified;
}



/***********************************************************************
 ** set_modified - shall we set this?
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ChatLine::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** get_lineflags - gets the line flag structure
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the line flag
 **
 ***********************************************************************/

Flags *ChatLine::get_lineflags()
{
   return line_flags;
}


/***********************************************************************
 ** copy_object - copies the chatline to a chatline of a different name
 **
 ** Parameters: copy_from - copy attributes from this object
 **
 ** Returns:  1 if succeeded
 **           0 if failed
 **
 ***********************************************************************/
int ChatLine::copy_object(Entity *copy_obj)
{
   ChatLine *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_CHATLINE)
      return 0;

   copy_from = (ChatLine *) copy_obj;

   /******* set the chatline attributes *****/
   set_lead_in_actor(copy_from->get_lead_in_actor());
   set_lead_in_crowd(copy_from->get_lead_in_crowd());
   set_text_color(copy_from->get_text_color());
   set_border(copy_from->get_border());

   line_flags->copy_flags(copy_from->get_lineflags());

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int ChatLine::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int ChatLine::get_mem_size_dynamic()
{
   int  size = 0;

   size += get_mem_size_entity();
   size += lead_in_actor.get_mem_size_dynamic();
   size += lead_in_crowd.get_mem_size_dynamic();
   size += text_color.get_mem_size_dynamic();
   size += border.get_mem_size_dynamic();
   size += line_flags->get_mem_size();

   return size;
}

#endif


