/**********************************************************************
 ** Input class: takes a string from the connection and parses it. Calls
 **              the command based on the verb list
 **
 **   
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Ed Boraas
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef INPUT_C
#define INPUT_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#ifdef WIN32
#include "../win32/winconnection.h"
#else
#include "connection.h"
#endif
#include "mudobject.h"
#include "location.h"
#include "individual.h"
#include "player.h"
#include "builder.h"
#include "verb.h"
#include "verb_list.h"
#include "parse.h"
#include "input.h"
#include "lexer.h"
#include "btree_l.h"
#include "area_dbase.h"
#include "btree_a.h"
#include "logs.h"
#include "errlog.h"
#include "object_list.h"
#ifdef WIN32
#include "../win32/winport.h"
#else
#include "port.h"
#endif
#include "mud.h"
#include "global.h"
#include "commands.h"
#include "bcommands.h"
#include "jcommands.h"
#include "objtype.h"
#include "flags.h"
#include "comflags.h"
#include "newfuncts.h"
#include "action.h"
#include "chatline.h"

/***********************************************************************
 ** handle_input - gets the user input and parses it
 **
 ** Parameters: input_str - the string from the user
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/
   
int Input::handle_input(Strings *user_input, MudObject *the_user)
{
   Verb_List *the_verbs;    /* holds the list of verbs */
   Strings   the_verb;      /* the verb that the player entered */
   Parse     *the_parsed;   /* holds the parse structure */
   Player    *the_player = NULL;
   Builder   *the_builder = NULL;
   char      *tmp_str;
   int       done = 0;
   long      timespan;
   Strings   holder;

#if defined( AIME_WIN32 )
   FILETIME time_start, time_end;
#else
   struct timeval time_start, time_end;
#endif // AIME_WIN32


   if (user_input->str_show() == NULL)
      return -1;

   /* get the verb list */
   if (the_user->get_type() == OBJ_TYPE_PLAYER)
   {
      the_verbs = mainstruct->get_player_verb_list();
      the_player = (Player *) the_user;
   }
   else if (the_user->get_type() == OBJ_TYPE_BUILDER)
   {
      the_verbs = mainstruct->get_builder_verb_list();
      the_builder = (Builder *) the_user;
   }
   else
      return -1;

   /* copy the input string into the class attribute */
   input_str.str_copy(user_input->str_show());
   
   tmp_str = user_input->str_show();
   while (!done)
   {
      the_command = NULL;

      /* if we don't manage to get the first word, raise an error */
      if (the_verb.assign_word(tmp_str, 1) <= 0)
         return -1;

      /* see if we can find the command, if so, execute it */
      if ((the_command = the_verbs->get_verb(the_verb.str_show())) == NULL)
      {
         if (the_player != NULL)
            the_player->send_plr("I don't understand the command '%s'\n", 
                                                     the_verb.str_show());
         else if (the_builder != NULL)
         {
           the_builder->send_bldr("I don't understand the command '%s'\n", 
                                                      the_verb.str_show());
         }
         return -1;
      }

      /* check the permissions for this command */
      if (the_player != NULL)
      {
         Flags *tmp_flags;

         tmp_flags = the_player->get_comflags();
         if (the_command->get_type() == OBJ_TYPE_ACTION)
         {
            if (!tmp_flags->get_flag(COMFLAG_ACTION))
            {
               the_player->
                send_plr("You don't have permissions to use that command.\n");
	       return -1;
            }
         } 
         else if (the_command->get_type() == OBJ_TYPE_SPELL)
	 {
            holder = "cast ";
            holder.str_cat(user_input->str_show());
            user_input->str_copy(holder.str_show());
            return handle_input(user_input, the_user);
	 }
         else if (the_command->get_type() == OBJ_TYPE_SKILL)
	 {
            holder = "perform ";
            holder.str_cat(user_input->str_show());
            user_input->str_copy(holder.str_show());
            return handle_input(user_input, the_user);            
	 }
         else if (the_command->get_type() == OBJ_TYPE_VERB) 
         {
            if (!tmp_flags->get_flag(((Verb *) the_command)->
                                                      get_verb_comflag()))
            {
               the_player->
                send_plr("You don't have permissions to use that command.\n");
	       return -1;
            }
         }
         else if (the_command->get_type() == OBJ_TYPE_CHATLINE)
	 {

	 }
         else
            return -1;

      } else if (the_builder != NULL)
      {
         Flags *tmp_flags;

         if (the_command->get_type() == OBJ_TYPE_VERB)
	 {
            tmp_flags = the_builder->get_comflags();
            if (!tmp_flags->get_flag(((Verb *) the_command)->
                                                        get_verb_comflag()))
            {
              the_builder->
               send_bldr("You don't have permissions to use that command.\n");
	       return -1;
            }
         }
         else if (the_command->get_type() == OBJ_TYPE_CHATLINE)
	 {

	 }
         else
            return -1;
      }     
                    
      /* create a parse structure, parsing the rest of the user input */
      the_parsed = new_Parse(tmp_str, the_command);


#if defined( AIME_WIN32 )
      GetSystemTimeAsFileTime( &time_start );
#else
	   gettimeofday(&time_start, NULL);
#endif // AIME_WIN32

      /* call the command associated with the verb, passing in parse struct */
      if (the_builder != NULL)
         call_command(the_command, the_parsed, the_builder);
      else if (the_player != NULL)
      {
         call_command(the_command, the_parsed, the_player);
      }


#if defined( AIME_WIN32 )
      GetSystemTimeAsFileTime( &time_end );

      if( time_start.dwHighDateTime != time_end.dwHighDateTime )
      {
         timespan = ( long )( ( time_end.dwHighDateTime
                                - time_start.dwHighDateTime )
                              * 4294.967296 );
      }
      else
      {
         timespan = 0;
      }

      timespan += ( time_end.dwLowDateTime
                    - time_start.dwLowDateTime ) / 1000000;

#else // AIME_WIN32
      gettimeofday(&time_start, NULL);

      if (time_start.tv_sec != time_end.tv_sec)
      {
         timespan = ((1000000 - time_start.tv_usec) + 
                    ((time_end.tv_sec - (time_start.tv_sec+1)) * 1000000) +
                    (time_end.tv_usec));
      }
      else
         timespan = time_end.tv_usec - time_start.tv_usec;
#endif // AIME_WIN32


      if (the_command->get_type() == OBJ_TYPE_VERB)
         the_verbs->add_new_execute((Verb *) the_command, timespan);

      /* we are done with the parse struct, remove it */
      if (the_parsed != NULL)
         delete_Parse(the_parsed);

      /* see if there is more input in this string */
      while ((*tmp_str) && (*tmp_str != '\n') && (*tmp_str != '\r') && 
                                                    (*tmp_str != '\0'))
      {
        tmp_str++;
      }

      if (*tmp_str != '\0')
         tmp_str++;

      if ((*tmp_str == '\r') || (*tmp_str == '\n'))
         tmp_str++;
   
      if (!*tmp_str)
         done = 1;
   }

   return 1;   
}


/***********************************************************************
 ** call_command - calls the command the user specified
 **
 ** Parameters: input_str - the string from the user
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/
   
int Input::call_command(Entity *the_verb, Parse *the_parsed, Player *the_player)
{
   Cmd_Func the_command;  /* holds the function name to call */

#ifdef DEBUG_INPUT
   printf("calling player command\n");
#endif

   if (the_verb->get_type() == OBJ_TYPE_VERB)
   {
      the_command = ((Verb *) the_verb)->get_command();
      the_command(the_parsed, (MudObject *) the_player);  
   }
   else if (the_verb->get_type() == OBJ_TYPE_ACTION)
   {
      ((Action *)the_verb)->execute_action(the_parsed, the_player);
   }
   else if (the_verb->get_type() == OBJ_TYPE_CHATLINE)
   {
      ((ChatLine *)the_verb)->execute_chat(the_parsed, the_player);
   }
   return 1;
}


/***********************************************************************
 ** call_command - calls the command the user specified
 **
 ** Parameters: input_str - the string from the user
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/
   
int Input::call_command(Entity *the_verb, Parse *the_parsed, Builder *the_bldr)
{
   Cmd_Func the_command;  /* holds the function name to call */

#ifdef DEBUG_INPUT
   printf("calling builder command\n");
#endif

   if (the_verb->get_type() == OBJ_TYPE_VERB)
   {
      the_command = ((Verb *) the_verb)->get_command();
      the_command(the_parsed, (MudObject *) the_bldr);  
   }
   else if (the_verb->get_type() == OBJ_TYPE_ACTION)
   {

   }
   else if (the_verb->get_type() == OBJ_TYPE_CHATLINE)
   {
      ((ChatLine *)the_verb)->execute_chat(the_parsed, the_bldr);
   }

   return 1;
}

#endif





