/**********************************************************************
 ** JCommands: This is a listing of all the commands that both a player
 **           and a builder can call. The functions here must pass back void
 **           and must accept a Parse pointer and a Player pointer, like
 **           this: void chatcom(Parse *this_parse, MudObject *this_user)
 **
 **    
 ** Reviewed through: 
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Kelly Gerke
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef JCOMMANDS_C
#define JCOMMANDS_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "global.h"
#include "mudobject.h"
#include "player.h"
#include "builder.h"
#include "pager.h"
#include "objtype.h"
#include "indflags.h"
#include "boat.h"
#include "adminflags.h"
#include "shop.h"
#include "locflags.h"
#include "spellflags.h"
#include "skillflags.h"
#include "bullflags.h"
#include "gameflags.h"
#include "action.h"
#include "spell.h"
#include "skill.h"
#include "book.h"
#include "key.h"
#include "food.h"
#include "mobile.h"
#include "wearable.h"
#include "text.h"
#include "door.h"
#include "areaflags.h"
#include "actflags.h"
#include "newfuncts.h"
#include "inp_funct.h"
#include "jcommands.h"
#include "comflags.h"
#include "mask.h"
#include "chatline.h"
#include "verb_list.h"
#include "lineflags.h"

/***********************************************************************
 ** infocom - shows an information message
 **
 ***********************************************************************/
void infocom(Parse *the_parsed, MudObject *the_user) {
   Player *the_player = NULL;
   Builder *the_builder = NULL;
   Strings the_file;
   Pager   *the_pager;
   Strings the_input;

   switch (the_user->get_type()) {
   case OBJ_TYPE_PLAYER: the_player = (Player *) the_user; break;
   case OBJ_TYPE_BUILDER: the_builder = (Builder *) the_user; break;
   default:
        return;
   }
 
   if (the_parsed->get_target1() == NULL)
   {   if (the_player)
          the_player->send_plr("Request info on what?\n");
       else
          the_builder->send_bldr("Request info on what?\n");
       return;
   }

   the_input = the_parsed->get_target1();
   the_input.lowercase();
   the_file.sprintf("%s/%s%s%s", the_config.basedir.str_show(), 
                          INFODIR, the_input.str_show(), INFOEXTENTION);

   if (the_player != NULL)
      the_pager = new_Pager(the_file.str_show(), the_player->get_pager_lines());
   else
      the_pager = new_Pager(the_file.str_show(), the_builder->get_pager_lines());

   if (the_pager && the_pager->file_available())
   { 
       Inp_Handler *tmp_handler;
       int results;

       if (the_player)
       {
          tmp_handler = the_player->get_input_handler();
          results = the_pager->paged_read(the_player->get_connection());
       }
       else
       {
          tmp_handler = the_builder->get_input_handler();
          results = the_pager->paged_read(the_builder->get_connection());
       }

       if (results)
       {
          delete_Pager(the_pager);
       }
       else
       {
          tmp_handler->push_input_handler(pager_funct, 
             "&+c<&+G******************&+Bpress return to continue, "
             "q to quit&+G**********************&+c>&*", HANDLER_DATA_PAGER);
          tmp_handler->set_data((void *) the_pager);
       }
       return;

   }    
   else if (the_pager)
   {   if (the_player)
          the_player->send_plr("Could not find any information on %s.\n", 
                                                           the_input.str_show());
       else
          the_builder->send_bldr("Could not find any information on %s.\n",
                                                           the_input.str_show());
       delete_Pager(the_pager);
   }  
}


/***********************************************************************
 ** helpcom - shows a help file
 **
 ***********************************************************************/
void helpcom(Parse *the_parsed, MudObject *the_user) {
   Player *the_player = NULL;
   Builder *the_builder = NULL;
   Strings the_file;
   Pager   *the_pager;
   Strings the_input;

   switch (the_user->get_type()) {
   case OBJ_TYPE_PLAYER: the_player = (Player *) the_user; break;
   case OBJ_TYPE_BUILDER: the_builder = (Builder *) the_user; break;
   default:
        return;
   }
 
   if ((the_parsed->get_target1() == NULL) ||
       (strlen(the_parsed->get_target1()) <= 0))
   {
       if (the_player)
       {
          the_input = "topics";
       }
       else
       {
          the_builder->send_bldr("Request help on what?\n");
          return;
       }
   }
   else
      the_input = the_parsed->get_target1();

   the_input.lowercase();
   the_file.sprintf("%s/%s%s%s", the_config.basedir.str_show(), 
                                 HELPDIR, the_input.str_show(), HELPEXTENTION);

   if (the_player != NULL)
      the_pager = new_Pager(the_file.str_show(), the_player->get_pager_lines());
   else
      the_pager = new_Pager(the_file.str_show(), the_builder->get_pager_lines());

   if (the_pager && the_pager->file_available())
   { 
       Inp_Handler *tmp_handler;
       int results;

       if (the_player)
       {
          tmp_handler = the_player->get_input_handler();
          results = the_pager->paged_read(the_player->get_connection());
       }
       else
       {
          tmp_handler = the_builder->get_input_handler();
          results = the_pager->paged_read(the_builder->get_connection());
       }

       if (results)
       {
          delete_Pager(the_pager);
       }
       else
       {
          tmp_handler->push_input_handler(pager_funct, 
             "&+c<&+G******************&+Bpress return to continue, "
             "q to quit&+G**********************&+c>&*", HANDLER_DATA_PAGER);
          tmp_handler->set_data((void *) the_pager);
       }
       return;

   }    
   else if (the_pager)
   {   if (the_player)
          the_player->send_plr("No help exists on the topic '%s'.\n",
                                                  the_input.str_show());
       else
          the_builder->send_bldr("No help exists on the topic '%s'.\n", 
                                                  the_input.str_show());
       delete_Pager(the_pager);
   }  
}


/***********************************************************************
 **  quitcom - quits the game
 **
 ***********************************************************************/

void quitcom(Parse *the_parsed, MudObject *the_user) {
   Player *the_player = NULL;
   Builder *the_bldr = NULL;
   Strings holder;
   Location *the_loc;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else if (the_user->get_type() == OBJ_TYPE_BUILDER)
      the_bldr = (Builder *) the_user;
   else
      return;

   if ((the_player != NULL) && (the_player->is_asleep()))
   {
      the_player->send_plr("You can't quit while sleeping.\n");
      return;
   }

   if (the_player != NULL)
   {
      Flags       *tmp_indflags;

      tmp_indflags = the_player->get_indflags();
      if (tmp_indflags->get_flag(INDFLAG_FIGHTING))
      {
         the_player->send_plr("You can't quit while in a fight.\n");
         return;
      }

      if (tmp_indflags->get_flag(INDFLAG_INBOAT)) 
      {
         Boat *the_boat;

         the_boat = (Boat *) the_player->get_contained_by();
         the_boat->exit_boat(the_player);
      }
      
      mainstruct->save_player(the_player);
      the_player->send_plr("Successfully saved.\n");

      the_player->send_plr("&+WCome back soon!&*\n\n\n");
      holder.sprintf("%s quits from the game.\n", the_player->get_title());
      the_loc = the_player->get_loc();
      the_loc->send_location(holder.str_show(), the_player);
      the_player->set_quitting();
   }
   else
   {
      the_bldr->send_bldr("&+WQuitting from the builder port.&*\n\n\n");
      the_bldr->set_quitting();
   }
   the_parsed = NULL;
}


/***********************************************************************
 **  describecom - lists the attributes of an object
 **
 ***********************************************************************/

void describecom(Parse *the_parsed, MudObject *the_user) {
   Builder       *the_builder = NULL;
   Player        *the_player = NULL;
   Entity        *the_object;
   char          *the_area;


   if (the_user->get_type() == OBJ_TYPE_BUILDER)
      the_builder = (Builder *) the_user;
   else
      the_player = (Player *) the_user;

   if ((the_player != NULL) && (the_parsed->get_target1() == NULL))
   {
      the_player->send_plr("Format: describe <objectname>@<area>\n");
      return;
   }

   if ((the_parsed->get_target1() != NULL) &&
       (!STRNCASECMP(the_parsed->get_target1(), "area", 
                                     strlen(the_parsed->get_target1()))))
   {
      Area_Dbase *tmp_area;
      Flags       *admflags;

      if (the_player != NULL)
      {
         the_player->send_plr("Only on the builder port.\n");
         return;
      }

      admflags = the_builder->get_adminflags();
      if (!admflags->get_flag(ADMINFLAG_SETAREAS))
      {
         the_builder->send_bldr("You don't have permissions to view"
                                                       " area data.\n");
         return;
      }

      if ((tmp_area = the_builder->get_area()) == NULL)
      {
         the_builder->send_bldr("You need to load an area first.\n");
         return;
      }

      tmp_area->describe(the_builder);
      return;
   }


   if ((the_parsed->get_target1() != NULL) &&
       (!STRNCASECMP(the_parsed->get_target1(), "areaflags", 
                                     strlen(the_parsed->get_target1()))))
   {
      Area_Dbase *tmp_area;
      Flags      *admflags;
      Flags      *tmp_areaflags;

      if (the_player != NULL)
      {
         the_player->send_plr("Only on the builder port.\n");
         return;
      }

      admflags = the_builder->get_adminflags();
      if (!admflags->get_flag(ADMINFLAG_SETAREAS))
      {
         the_builder->send_bldr("You don't have permissions to view"
                                                       " area data.\n");
         return;
      }

      if ((tmp_area = the_builder->get_area()) == NULL)
      {
         the_builder->send_bldr("You need to load an area first.\n");
         return;
      }

      tmp_areaflags = tmp_area->get_areaflags();
      tmp_areaflags->display_flags(the_builder, tmp_area->get_areaname(), 
                              areaflagnames, "AreaFlags", null_frozen);
      return;
   }

   /* if they have not set a modifying object yet or they did not specify
      an existing object */

   if (the_builder != NULL)
   {
      if ((the_object = the_builder->get_modify()) == NULL)
      {
         the_builder->send_bldr("You have not set an object to modify yet.\n");
         return;
      }
   }
   else
   {
      the_area = the_parsed->get_target1();
      while ((*the_area) && (*the_area != '@'))
         the_area++;

      if ((!(*the_area)) || (!(*(the_area+1))))
      {
         Player *a_player;

         if ((a_player = mainstruct->get_player(the_parsed->get_target1()))
                                                              == NULL)
	 {
            the_player->send_plr("Could not find player '%s'\n", 
                                         the_parsed->get_target1());
            return;
         }
 
         a_player->describe(the_player);
         return;
      }

      if (!STRCASECMP((the_area+1), "actions"))
      {
         Verb_List *tmp_verbs;

         tmp_verbs = mainstruct->get_player_verb_list();
         *the_area = '\0';
         if ((the_object = tmp_verbs->get_verb(the_parsed->get_target1()))
                                                                 == NULL)
	 {
            the_player->send_plr("Could not find the action '%s'.\n", 
                                     the_parsed->get_target1());
            return;
         }
      }
      else if (!STRCASECMP((the_area+1), "levels"))
      {
         the_player->send_plr("Use '&+cstats levels&*' for level "
                                     "information or the builder port.\n");
         return;
      }

      else if (!STRCASECMP((the_area+1), "abilities"))
      {
         *the_area = '\0';
         if ((the_object = mainstruct->get_ability(the_parsed->get_target1()))
                                                                 == NULL)
	 {
            the_player->send_plr("Could not find the ability '%s'.\n", 
                                     the_parsed->get_target1());
            return;
         }
      }

      else if ((the_object = mainstruct->get_object(the_parsed->get_target1()))
                                                                  == NULL)
      {
         if ((the_object = mainstruct->
                           get_special(the_parsed->get_target1())) == NULL)
	 {
            the_player->send_plr("Can't find object specified.\n"
               "Be sure to specify in this format: <objectname>@<area>\n");
            return;
	 }
      }
   }

   /* handle desc for actions */
   if (the_object->get_type() == OBJ_TYPE_ACTION)
   {
      Action *the_action;

      the_action = (Action *) the_object;
      if ((the_builder != NULL) &&
          (the_parsed->get_target1() != NULL) && 
          (!STRNCASECMP(the_parsed->get_target1(), "actflags", 
                                        strlen(the_parsed->get_target1()))))
      {
         Flags *tmp_actflags;


         tmp_actflags = the_action->get_actflags();

         tmp_actflags->display_flags(the_builder, the_action->get_name(), 
                            actflagnames, "Indflags", null_frozen);
         return;
      }
      else
      {
         if (the_builder != NULL)
            the_action->describe(the_builder);
         else
            the_action->describe(the_player);

         return;
      }
   }

   if ((the_builder != NULL) && (the_parsed->get_target1() != NULL))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), "shop", 
                                           strlen(the_parsed->get_target1())))
      {
         Shop *the_shop;
         Mobile *the_mobile;

         if (the_object->get_type() != OBJ_TYPE_MOBILE)
	 {
            the_builder->send_bldr("Only mobiles can contain shops.\n");
            return;
         }

         the_mobile = (Mobile *) the_object;

         if ((the_shop = the_mobile->get_shop()) == NULL)
	 {
            the_builder->send_bldr("No shop is assigned to this mobile.\n");
            return;
         }

         the_shop->display_shop(the_builder);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "indflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_indflags;
         Mobile *the_mobile;

         if (the_object->get_type() != OBJ_TYPE_MOBILE)
	 {
            the_builder->send_bldr("Only mobiles will have indflags.\n");
            return;
         }

         the_mobile = (Mobile *) the_object;
         tmp_indflags = the_mobile->get_indflags();

         tmp_indflags->display_flags(the_builder, the_mobile->get_name(), 
                              indflagnames, "Indflags", indflagfrozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "locflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_locflags;
         Location *the_loc;

         if (the_object->get_type() != OBJ_TYPE_LOCATION)
	 {
            the_builder->send_bldr("Only locations will have locflags.\n");
            return;
         }

         the_loc = (Location *) the_object;
         tmp_locflags = the_loc->get_locflags();

         tmp_locflags->display_flags(the_builder, the_loc->get_name(), 
                                  locflagnames, "LocFlags", null_frozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "spellflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_spellflags;
         Spell *the_spell;

         if (the_object->get_type() != OBJ_TYPE_SPELL)
	 {
            the_builder->send_bldr("Only spells will have spellflags.\n");
            return;
         }

         the_spell = (Spell *) the_object;
         tmp_spellflags = the_spell->get_spellflags();

         tmp_spellflags->display_flags(the_builder, the_spell->get_name(), 
                               spellflagnames, "SpellFlags", null_frozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "skillflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_skillflags;
         Skill *the_skill;

         if (the_object->get_type() != OBJ_TYPE_SKILL)
	 {
            the_builder->send_bldr("Only skills will have skillflags.\n");
            return;
         }

         the_skill = (Skill *) the_object;
         tmp_skillflags = the_skill->get_skillflags();

         tmp_skillflags->display_flags(the_builder, the_skill->get_name(), 
                               skillflagnames, "SkillFlags", null_frozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "itemflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_itemflags;
         Item *the_item;

         if ((!the_object->is_a_mudobject()) || 
             (!((MudObject *)the_object)->is_an_item()))
	 {
            the_builder->send_bldr("This object does not have itemflags.\n");
            return;
         }

         the_item = (Item *) the_object;
         tmp_itemflags = the_item->get_itemflags();

         tmp_itemflags->display_flags(the_builder, the_item->get_name(), 
                                  itemflagnames, "Itemflags", itemflagfrozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "comflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_flags;
         Mask *the_mask;

         if (the_object->get_type() != OBJ_TYPE_MASK)
	 {
            the_builder->send_bldr("This object does not have comflags.\n");
            return;
         }

         the_mask = (Mask *) the_object;
         tmp_flags = the_mask->get_comflags();

         tmp_flags->display_flags(the_builder, the_mask->get_name(), 
                                  comflagnames, "Comflags", null_frozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "lineflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_flags;
         ChatLine *the_chatline;

         if (the_object->get_type() != OBJ_TYPE_CHATLINE)
	 {
            the_builder->send_bldr("This object does not have lineflags.\n");
            return;
         }

         the_chatline = (ChatLine *) the_object;
         tmp_flags = the_chatline->get_lineflags();

         tmp_flags->display_flags(the_builder, the_chatline->get_name(), 
                                  lineflagnames, "Lineflags", null_frozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "adminflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_flags;
         Mask *the_mask;

         if (the_object->get_type() != OBJ_TYPE_MASK)
	 {
            the_builder->send_bldr("This object does not have adminflags.\n");
            return;
         }

         the_mask = (Mask *) the_object;
         tmp_flags = the_mask->get_admflags();

         tmp_flags->display_flags(the_builder, the_mask->get_name(), 
                                  admflagnames, "Adminflags", null_frozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "bullflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_bullflags;
         Bulletin *the_bull;

         if (the_object->get_type() != OBJ_TYPE_BULLETIN)
	 {
            the_builder->send_bldr("This object does not have itemflags.\n");
            return;
         }

         the_bull = (Bulletin *) the_object;
         tmp_bullflags = the_bull->get_bullflags();

         tmp_bullflags->display_flags(the_builder, the_bull->get_name(), 
                                  bullflagnames, "Bullflags", null_frozen);
         return;
      }
      if (!STRNCASECMP(the_parsed->get_target1(), "wearflags", 
                                           strlen(the_parsed->get_target1())))
      {
         Flags *tmp_wearflags;
         Wearable *the_wearable;

         if (the_object->get_type() != OBJ_TYPE_WEARABLE)
	 {
            the_builder->send_bldr("Only wearables will have wearflags.\n");
            return;
         }

         the_wearable = (Wearable *) the_object;
         tmp_wearflags = the_wearable->get_wearflags();

         tmp_wearflags->display_flags(the_builder, the_wearable->get_name(), 
                                  wearflagnames, "Wearflags", null_frozen);
         return;
      }
   }


   if (the_builder != NULL)
      the_object->describe(the_builder);
   else
      the_object->describe(the_player);

}



/***********************************************************************
 **  userscom - gives the users on the game
 **
 ***********************************************************************/

void userscom(Parse *the_parsed, MudObject *the_user) {
   Player    *the_list;
   Builder   *bldr_list;
   int       count = 0;    /* the number of players online */
   Player    *the_player = NULL;
   Builder   *the_builder = NULL;
   Flags     *gameflags;
   Strings   displayline;
   struct    tm *time_struct;
   char      buff[30];
   char		 locbuff[25];
   time_t    idle_time;
   int       cur_lvl;
   int       for_game = 1;
   char      *single;
   char      *plural;
   Entity    *the_modified;


   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else if (the_user->get_type() == OBJ_TYPE_BUILDER)
      the_builder = (Builder *) the_user;
   else
      return;


   if (the_parsed->get_target1() != NULL)
   {
      if (!STRNCASECMP("game", the_parsed->get_target1(), 
                                       strlen(the_parsed->get_target1())))
      {
         for_game = 1;
         single = "player";
         plural = "players";
      }
      else if (!STRNCASECMP("builder", the_parsed->get_target1(), 
                                       strlen(the_parsed->get_target1())))
      {
         if (the_player != NULL)
	 {
            if (!((the_player->
                         get_admflags())->get_flag(ADMINFLAG_SEEBUILDER)))
	    {
               the_player->send_plr(
                    "You don't have permission to see the builder port.\n");
               return;
	    }
	 }
         for_game = 0;
         single = "builders";
         plural = "builders";
      }
      else
      {
         if (the_player != NULL)
            the_player->send_plr("The '%s' port does not exist.\n", 
                                           the_parsed->get_target1());
         else
            the_builder->send_bldr("The '%s' port does not exist.\n", 
                                           the_parsed->get_target1());

         return;
      }     
   }
   else
   {
      if (the_player != NULL)
      {
        for_game = 1;
        single = "player";
        plural = "players";
      }
      else
      {
        for_game = 0;
        single = "builder";
        plural = "builders";

      }
   }

   if (for_game)
   {
      displayline = "&+W";
      if (STRCASECMP(the_config.users_level_chain.str_show(), "none"))
         displayline.str_cat(" Level\t");

      /* display the header */

      if(the_player != NULL)
      {
         if((the_player->get_admflags())->get_flag(ADMINFLAG_SEELOCATIONS))
         {
            the_player->send_plr("&+WGame Users List&*\n\n");
            the_player->send_plr("%s   Name\t\t\t    Sex\t\tIdle\t\tLocation&*\n", 
                                                  displayline.str_show());
            the_player->send_plr(
            "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");
         }
         else
         {
            the_player->send_plr("&+WGame Users List&*\n\n");
            the_player->send_plr("%s   Name\t\t\t    Sex\t\tIdle&*\n", 
                                                  displayline.str_show());
            the_player->send_plr(
            "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");
         }
      }
      else
      {
         if((the_builder->get_adminflags())->get_flag(ADMINFLAG_SEELOCATIONS))
         {
            the_builder->send_bldr("&+WGame Users List&*\n\n");
            the_builder->send_bldr("%s   Name\t\t\t    Sex\t\tIdle\t\tLocation&*\n", 
                                                  displayline.str_show());
            the_builder->send_bldr(
            "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");
         }
         else
         {
            the_builder->send_bldr("&+WGame Users List&*\n\n");
            the_builder->send_bldr("%s   Name\t\t\t    Sex\t\tIdle&*\n", 
                                                  displayline.str_show());
            the_builder->send_bldr(
            "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");
         }
      }

      the_list = mainstruct->get_first_player();
      while (the_list != NULL)
      {
         gameflags = the_list->get_gameflags();
         if (gameflags->get_flag(GAMEFLAG_ONGAME))
         {
            displayline.truncate(0);
            if (STRCASECMP(the_config.users_level_chain.str_show(), "none"))
            {
               cur_lvl = 
                the_list->check_levels(the_config.users_level_chain.str_show());
               displayline.sprintf("&+g[&+G%5d&+g]\t", cur_lvl);
	    }

            idle_time = the_list->get_idle();
            time_struct = (struct tm *) gmtime((const time_t *) &idle_time);

            /* prepare idle string */
            BZERO(buff, 30);
            strftime (buff, 10, "%H:%M:%S", time_struct);

			/* prepare location string */
			strncpy(locbuff, the_list->get_location(), 22);

            /* ## AMERIST 07-19-00 ##
               the ADMINFLAG_SEELOCATIONS is checked if it is ON then locbuff is 
               displayed to the user who is using the 'users' command in bright green. */
            if(the_player != NULL)
			{
                if((the_player->get_admflags())->get_flag(ADMINFLAG_SEELOCATIONS))
                {              
                    the_player->send_plr("%s&+W   %-25s&+C %s\t\t%s\t&+G%s&*\n",
                        (displayline.str_show() == NULL) ? "" : displayline.str_show(),
                        the_list->get_title(), gameflags->get_flag(GAMEFLAG_MALE) ? "M" 
                                                                  : "F", buff, locbuff);
                }
                else
                {
                    the_player->send_plr("%s&+W   %-25s&+C %s\t\t%s&*\n",
                        (displayline.str_show() == NULL) ? "" : displayline.str_show(),
                        the_list->get_title(), gameflags->get_flag(GAMEFLAG_MALE) ? "M" 
                                                                           : "F", buff);
                }
            }
            else
            {
                if((the_builder->get_adminflags())->get_flag(ADMINFLAG_SEELOCATIONS))
                {
                    the_builder->send_bldr("%s&+W   %-25s&+C %s\t\t%s\t&+G%s&*\n",
                        (displayline.str_show() == NULL) ? "" : displayline.str_show(),
                        the_list->get_title(), gameflags->get_flag(GAMEFLAG_MALE) ? "M" 
                                                                  : "F", buff, locbuff);
                }
                else
                {
                    the_builder->send_bldr("%s&+W   %-25s&+C %s\t\t%s&*\n",
                        (displayline.str_show() == NULL) ? "" : displayline.str_show(),
                        the_list->get_title(), gameflags->get_flag(GAMEFLAG_MALE) ? "M" 
                                                                           : "F", buff);
                }
            }
            

            count++;
         }
         the_list = the_list->get_next_player();
      }
   }
   else
   {

      if (the_player != NULL)
      {
         the_player->send_plr("&+WBuilder Port Users List&*\n\n");
         the_player->
                  send_plr("&+WName\t\tArea Loaded\tModifying\t   Idle&*\n");
         the_player->send_plr(
         "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");
      }
      else
      {
         the_builder->send_bldr("&+WBuilder Port Users List&*\n\n");
         the_builder->
                  send_bldr("&+WName\t\tArea Loaded\tModifying\t   Idle&*\n");
         the_builder->send_bldr(
         "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");
      }

      bldr_list = mainstruct->get_first_builder();
      while (bldr_list != NULL)
      {
         idle_time = bldr_list->get_idle();
         time_struct = (struct tm *) gmtime((const time_t *) &idle_time);

         /* prepare idle string */
         BZERO(buff, 30);       
         strftime (buff, 10, "%H:%M:%S", time_struct);
      
         the_modified = bldr_list->get_modify();

         if (the_player != NULL)
	 {
            the_player->send_plr("&+G%-15s &+y%-13s&*   &+g%-17s&+W %s&*\n", 
                bldr_list->get_title(), (bldr_list->get_areaname() == NULL) ? 
                "none" : bldr_list->get_areaname(), (the_modified == NULL) ? 
                "none" : the_modified->get_name(), buff);
	 }
         else
	 {
            the_builder->send_bldr("&+G%-15s &+y%-13s&*   &+g%-17s&+W %s&*\n", 
                bldr_list->get_title(), (bldr_list->get_areaname() == NULL) ? 
                "none" : bldr_list->get_areaname(), (the_modified == NULL) ? 
                "none" : the_modified->get_name(), buff);
	 }
         count++;
         bldr_list = bldr_list->get_next_builder();
      }
   }
   if (the_player != NULL)
   {
      the_player->send_plr(
         "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");

      the_player->send_plr("There %s &+W%d&* %s online.\n", 
                              ((count == 1) ? "is" : "are"), count,
                              ((count == 1) ? single : plural));
   }
   else
   {
      the_builder->send_bldr(
         "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-&*\n");

      the_builder->send_bldr("There %s &+W%d&* %s online.\n", 
                              ((count == 1) ? "is" : "are"), count,
                              ((count == 1) ? single : plural));
   }
   the_parsed = NULL;
}


#endif

