/**********************************************************************
 ** Access_List class: a list that either denies or allows access to the
 **                    mud or a feature on the mud.
 **
 ** Last reviewed: v.51
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef ACCESS_LIST_C
#define ACCESS_LIST_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "access_list.h"
#include "player.h"
#include "timespan.h"
#include "global.h"
#include "errno.h"
#include "memchk.h"

/***********************************************************************
 ** Access_List (constructor) - Opens the access list
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Access_List::Access_List(char *filename)
{
   if (open_list(filename) <= 0)
   {
      printf(_("Error - could not open the access list %s\n"), filename);
      return;
   }
}

/***********************************************************************
 ** ~Access_List (destructor) - closes the access list file
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Access_List::~Access_List()
{ 
   xfclose(the_list, "fp_access_list");
}


/***********************************************************************
 ** open_list - opens a list for appending to 
 **
 ** Parameters: filename - the name of the list file to open 
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Access_List::open_list(char *filename)
{
   Strings full_filename;

   if (filename == NULL)
      return -1;

   /* load the filename into the object */
   strncpy(listname, filename, MAXFILENAMELEN);
   full_filename.sprintf("%s/%s", the_config.basedir.str_show(), listname);
   
   /* open the filename for business */
   if ((the_list = xfopen(full_filename.str_show(), "a+", "fp_access_list")) 
                                                                       == NULL)
   {
      perror("open_list");
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** open_list - opens a list for appending to 
 **
 ** Parameters: filename - the name of the list file to open 
 **
 ** Returns: num lines displayed if success, -1 if error
 **
 ***********************************************************************/

int Access_List::display_list(Player *the_player)
{
   int count = 0;
   char temp_str[MAXHOSTNAMELEN+1];
   Timespan the_time;

   /* If the listfile is not there, return */ 
   if (the_list == NULL)
   {
      return -1;
   }

   if (fseek(the_list, 0, SEEK_SET) != 0)
   {
      perror("display_list fseek");
      return -1;
   }

   if (feof(the_list))
   {
      return 0;
   }

   while (!feof(the_list))
   {
      fgets(temp_str, MAXHOSTNAMELEN, the_list);
      if (feof(the_list))
         break;

      the_player->send_plr("\t%s", temp_str);
      count++;
   }
   return count;
}

/***********************************************************************
 ** add_entry - adds an entry to the file
 **
 ** Parameters: the_entry - the entry string to add 
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Access_List::add_entry(char *the_entry)
{
   char     *tmp_ptr;
   Timespan the_time;

   if ((the_list == NULL) || (the_entry == NULL))
   {
      return -1;
   }

   if (fseek(the_list, 0, SEEK_END) != 0)
   {
      perror("add_entry fseek");
      return -1;
   }

   if (fputs(the_entry, the_list) == EOF)
   {
      perror("add_entry fputs");
      return -1;
   }
   tmp_ptr = the_entry;
   while ((*tmp_ptr) && (*tmp_ptr != 0) && (*tmp_ptr != ' ') && 
                                              (*tmp_ptr != '\n'))
      tmp_ptr++;

   if (*tmp_ptr != '\n')
   {
      if (fputs("\n", the_list) == EOF)
      {
         perror("add_entry fputs");
         return -1;
      }
   }
   fflush(the_list);
   return 1;
}


/***********************************************************************
 ** del_entry - deletes an entry from the file
 **
 ** Parameters: the_entry - the string to delete 
 **
 ** Returns: 1 if success, 0 if not found, -1 if error
 **
 ***********************************************************************/

int Access_List::del_entry(char *the_entry)
{
   char *tmp_ptr;
   FILE *temp_file = NULL;
   char temp_str[MAXHOSTNAMELEN+1];
   Timespan the_time;
   Strings full_filename;
   int found = 0;

   /* If the listfile is not there, return */ 
   if ((the_list == NULL) || (the_entry == NULL))
   {
      return -1;
   }

   temp_file = tmpfile();
   if (fseek(the_list, 0, SEEK_SET) != 0)
   {
      perror("del_entry fseek");
      return -1;
   }

   if (feof(the_list))
   {
      return 0;
   }

   /* how to delete an entry, we do it by first reading in and writing
      the old file to a temp file */
   while (!feof(the_list))
   {
      fgets(temp_str, MAXHOSTNAMELEN, the_list);
      if (feof(the_list))
         break;

      tmp_ptr = &temp_str[0];
      while ((*tmp_ptr) && (*tmp_ptr != '\0') && (*tmp_ptr != '\n'))
         tmp_ptr++;
  
      if (*tmp_ptr == '\n')
         *tmp_ptr = 0;

      if (STRCASECMP(temp_str, the_entry))
      {
         fprintf(temp_file, "%s\n", temp_str);
      }
      else
         found = 1;
   }

   if (!found)
   {
      xfclose(temp_file, "temp_access_list");
      return 0;
   }

   /* now we close the list file and re-open it as write, to erase it
      so we can rewrite it */
   xfclose(the_list, "fp_access_list");

   full_filename.sprintf("%s/%s", the_config.basedir.str_show(), listname);
   
   /* open the filename for business */
   if ((the_list = xfopen(full_filename.str_show(), "w", 
                                                "fp_access_list")) == NULL)
   {
      perror("del_entry");
      return -1;
   }

   /* go to the beginning of the temp file to read it back */
   if (fseek(temp_file, 0, SEEK_SET) != 0)
   {
      perror("del_entry fseek");
      return -1;
   }
   
   /* now copy it back over */
   while (!feof(temp_file))
   {
      fgets(temp_str, MAXHOSTNAMELEN, temp_file);
      if (feof(temp_file))
         break;
 
      fprintf(the_list, "%s", temp_str);
   }

   xfclose(the_list, "fp_access_list");

   /* finally, reopen our old file */
   if ((the_list = xfopen(full_filename.str_show(), "a+", 
                                                "fp_access_list")) == NULL)
   {
      perror("del_entry");
      return -1;
   }

   xfclose(temp_file, "temp_access_list");
   return 1;
}


/***********************************************************************
 ** find_in_list - determines if an entry is in the list
 **
 ** Parameters: the_host - the ip string or host string
 **
 ** Returns:  1 if denied
 **           0 if allowed
 **          -1 if an error found
 **
 ***********************************************************************/

int Access_List::find_in_list(char *the_host)
{
   char     temp_str[MAXHOSTNAMELEN+1];
   char     *temp_ptr;
   char     *start_ptr;
   int      end_any = 0;
   int      start_any = 0;
   char     *track_ptr;


   /* If the logfile is not there, return */ 
   if (the_list == NULL)
   {
      return 0;
   }

   if (fseek(the_list, 0, SEEK_SET) != 0)
   {
      perror("find_in_list fseek");
      return 0;
   }

   if (feof(the_list))
   {
      return 0;
   }

   /* loop through all entries in the access list looking for a match */
   while (!feof(the_list))
   {
      if (fgets(temp_str, MAXHOSTNAMELEN, the_list) == NULL)
      {
         return 0;
      }

      // If the first char is a space, newline, or something else, skip it
      temp_ptr = &temp_str[0];
      if ((*temp_ptr == '\n') || (*temp_ptr == '\r') || (*temp_ptr == ' ') ||
	  (*temp_ptr == '\0'))
	return 0;

      /* remove all trailing newlines and spaces and detect * chars */
      start_ptr = temp_ptr;
      while ((*temp_ptr) && (*temp_ptr != ' ') && (*temp_ptr != '\n'))
      {
         if (*temp_ptr == '*')
	 {
            if (temp_ptr == &temp_str[0])
	    {
               start_ptr++;
               start_any = 1;
	    }
            else
	    {
               end_any = 1;
               break;
	    }
	 }
         temp_ptr++;
      }
      if (*temp_ptr)
         *temp_ptr = '\0';

      if (start_any)
      {
         if (!(track_ptr = strstr(the_host, start_ptr)))
            continue;

         if (end_any)
            return 1;
         else
	 {
            if (*(track_ptr + (strlen(start_ptr) + 1)) != '\0')
	       continue;
         }
         return 1;
      }
      else if (end_any)
      {
         if (!strncmp(start_ptr, the_host, strlen(start_ptr)))
            return 1;
         continue;
      }
      else
      {
         if (!strcmp(start_ptr, the_host))
            return 1;
         continue;
      }
   }

   return 0;
}



#endif









