// ---------------------------------------------------------------------------
// - Aodcalls.cpp                                                            -
// - aleph object database - command calls implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Aod.hpp"
#include "Aodcalls.hpp"
#include "OutputFile.hpp"

namespace aleph {

  // open a collection file

  Object* aod_open (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for 1 argument
    if (argc == 1) {
      String fname = argv->getstring (0);
      Collection* result = aod->open (fname);
      robj->post (result);
      delete argv;
      return result;
    }
    delete argv;
    throw Exception ("aod-error", "invalid open arguments");
  }

  // save a collection in a file

  Object* aod_save (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for 1 argument
    if (argc == 1) {
      String fname = argv->getstring (0);
      aod->save (fname);
      delete argv;
      return nilp;
    }
    delete argv;
    throw Exception ("aod-error", "invalid save arguments");
  }

  // print some collection info

  Object* aod_info (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for 0 argument
    if (argc == 0) {
      aod->info (robj->getos ());
      delete argv;
      return nilp;
    }
    // check for one argument
    if (argc == 1) {
      Object* obj = argv->get (0);
      Output* os  = dynamic_cast <Output*> (obj);
      if (os == nilp) 
	throw Exception ("type-error", "invalid output stream object",
			 Object::repr (obj));
      try {
	aod->info (os);
	delete argv;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    delete argv;
    throw Exception ("argument-error", "too many arguments with info");
  }

  // print the collection table info

  Object* aod_tinfo (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for 0 argument
    if (argc == 0) {
      aod->tinfo (robj->getos ());
      delete argv;
      return nilp;
    }
    // check for one argument
    if (argc == 1) {
      Object* obj = argv->get (0);
      Output* os  = dynamic_cast <Output*> (obj);
      if (os == nilp) 
	throw Exception ("type-error", "invalid output stream object",
			 Object::repr (obj));
      try {
	aod->tinfo (os);
	delete argv;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    delete argv;
    throw Exception ("argument-error", "too many arguments with table info");
  }

  // create a new table - by name eventually - make it the default

  Object* aod_create (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for 0 argument
    if (argc == 0) {
      try {
	Table* tbl = aod->create ();
	delete argv;
	robj->post (tbl);
	return tbl;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    // check for one argument
    if (argc == 1) {
      try {
	String name = argv->getstring (0);
	Table* tbl = aod->create ();
	tbl->setname (name);
	robj->post (tbl);
	delete argv;
	return tbl;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    delete argv;
    throw Exception ("argument-error", "too many arguments with create");
  }

  // import data into the table

  Object* aod_import (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for one argument
    if (argc == 1) {
      try {
	String fname = argv->getstring (0);
	aod->import (fname);
	delete argv;
	return nilp;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    delete argv;
    throw Exception ("argument-error", "too many arguments with import");
  }

  // export a table into a file

  Object* aod_export (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for one argument
    if (argc == 1) {
      try {
	String fname = argv->getstring (0);
	OutputFile os (fname);
	aod->tblrpt (&os, 0, 0, false);
	delete argv;
	return nilp;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    delete argv;
    throw Exception ("argument-error", "too many arguments with export");
  }

  // select a table - and make it the default one

  Object* aod_select (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // check for 0 argument
    if (argc == 0) {
      try {
	aod->select (0);
	delete argv;
	return nilp;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    // check for one argument
    if (argc == 1) {
      try {
	long index = argv->getint (0);
	aod->select (index);
	delete argv;
	return nilp;
      } catch (...) {
	delete argv;
	throw;
      }
    }
    delete argv;
    throw Exception ("argument-error", "too many arguments with info");
  }

  // insert elements to the current table

  Object* aod_insert (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // try to add the element
    try {
      aod->insert (argv);
      delete argv;
      return nilp;
    } catch (...) {
      delete argv;
      throw;
    }
  }

  // print a report of the current table

  Object* aod_tblrpt (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();
    // get the command object
    Aod* aod = dynamic_cast <Aod*> (robj);
    // format the table
    try {
      // check for 0 argument
      if (argc == 0) {
	aod->tblrpt (aod->getos (), 0, 0, true);
	delete argv;
	return nilp;
      }
      // check for 1 argument
      if (argc == 1) {
	long max = argv->getint (0);
	aod->tblrpt (aod->getos (), max, 0, true);
	delete argv;
	return nilp;
      } 
      // check for 2 arguments
      if (argc == 2) {
	long max   = argv->getint (0);
	long start = argv->getint (1);
	aod->tblrpt (aod->getos (), max, start, true);
	delete argv;
	return nilp;
      }
      throw Exception ("argument-error", 
		       "too many arguments with table-report");
    } catch (...) {
      delete argv;
      throw;
    }
  }
}
