// ---------------------------------------------------------------------------
// - Cell.hpp                                                                -
// - aleph:odb library - cell class definition                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_CELL_HPP
#define  ALEPH_CELL_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The Cell class is the foundation block of the aleph object database. 
  /// A cell is a simple binding between a name and a literal object. For this
  /// reason a cell is a serializable object. 
  /// @author amaury darsch

  class Cell : public Serial {
  private:
    /// the cell quark
    long d_quark;
    /// the cell object
    Literal* p_cobj;
    /// the const flag
    bool d_const;

  public:
    /// create an empty cell
    Cell (void);

    /// create a cell with a literal
    /// @param cobj the cell object
    Cell (Literal* cobj);

    /// create a cell with a name and an object
    /// @param name the cell name
    /// @param cobj the cell object
    Cell (const String& name, Literal* cobj);
    
    /// copy construct this cell
    /// @param that the cell to copy
    Cell (const Cell& that);

    /// destroy this cell
    ~Cell (void);

    /// @return the object name
    String repr (void) const;

    /// check that the cell match the quark
    /// @param quark the quark to match
    bool operator == (const long quark) const;

    /// @return the cell serial id
    t_byte serialid (void) const;

    /// serialize this cell
    /// @param os the output stream
    void wrstream (Output& os) const;

    /// deserialize this cell
    /// @param is the input stream
    void rdstream (Input& os);

    /// @return the cell name
    String getname (void) const;

    /// set the cell name
    /// @param name the cell name to set
    void setname (const String& name);

    /// @return the cell value
    Literal* get (void) const;

    /// set the cell value
    void set (Literal* cobj);

    /// generate a new cell
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);
 
    /// set an object to this cell
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, Object* object);

    /// apply this cell with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);

  private:
    // make the assignment operator private
    Cell& operator = (const Cell&);
  };
}

#endif
