// ---------------------------------------------------------------------------
// - Url.cpp                                                                 -
// - aleph:www library - url class implementation                            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Url.hpp"
#include "Regex.hpp"
#include "Vector.hpp"
#include "Buffer.hpp"
#include "Integer.hpp"
#include "Exception.hpp"

namespace aleph {
  // the url supported quarks
  static const long QUARK_PARSE       = String::intern ("parse");
  static const long QUARK_GETPORT     = String::intern ("get-port");
  static const long QUARK_GETHOST     = String::intern ("get-host");
  static const long QUARK_GETPATH     = String::intern ("get-path");
  static const long QUARK_GETQUERY    = String::intern ("get-query");
  static const long QUARK_GETSCHEME   = String::intern ("get-scheme");
  static const long QUARK_GETFRAGMENT = String::intern ("get-fragment");

  // this procedure map an hexadecimal character to its value
  static inline t_byte htob (const char c) {
    if ((c >= '0') && (c <= '9')) return (c - '0');
    if ((c >= 'a') && (c <= 'f')) return (c - 'a' + 10);
    if ((c >= 'A') && (c <= 'F')) return (c - 'A' + 10);
    throw Exception ("url-error", "invalid escape character", c);
  }

  // this procedure fix a url name by converting the escape characters
  static String utohs (const String& name) {
    long len = name.length ();
    if (len == 0) return name;
    // create a buffer and loop
    Buffer buf;
    for (long i = 0; i < len; i++) {
      char c = name[i];
      if (c == '+') {
	buf.add (' ');
	continue;
      }
      if (c == '%') {
	t_byte val = 0;
	if (++i < len) val  = htob (name[i]) << 4;
	if (++i < len) val += htob (name[i]);
	if (i >= len) 
	  throw Exception ("url-error", "unterminated escape sequence");
	c = (char) val;
      }
      buf.add (c);
    }
    return buf.tostring ();
  }

  // create a default url

  Url::Url (void) {
    reset ();
  }

  // create a url by name

  Url::Url (const String& url) {
    reset ();
    parse (url);
  }

  // return the class name

  String Url::repr (void) const {
    return "Url";
  }

  // reset an url information

  void Url::reset (void) {
    wrlock ();
    d_scheme   = "http";
    d_host     = "";
    d_port     = 80;
    d_path     = "/";
    d_query    = "";
    d_fragment = "";
    unlock ();
  }

  // parse an url and update the url data structure

  void Url::parse (const String& url) {
    // check for nothing first
    long len = url.length ();
    if (len == 0) {
      reset ();
      return;
    }
    // ok let's deal with it
    wrlock ();
    reset  ();
    try {
      // the reminding string
      String rest;
      // extract the scheme
      Regex rsch ("($l$a*):(/+$N*)");
      if (rsch == url) {
	d_scheme = rsch.getstr (0);
	rest     = rsch.getstr (1);
      } else {
	rest = url;
      }
      // now extract the host - can be up to a component separator
      Regex rhst ("//(<$a+-._>+)(<:/?#>$N*)");
      if (rhst == rest) {
	d_host = rhst.getstr (0);
	rest   = rhst.getstr (1);
      } else {
	Regex rhst ("//(<$a+-._>+)");
	if (rhst == rest) {
	  d_host = rhst.getstr (0);
	  rest   = "";
	}
      }
      // now extract the port
      Regex rprt (":($d+)(</?#>$N*)");
      if (rprt == rest) {
	d_port = rprt.getint (0);
	rest   = rprt.getstr (1);
      } else {
	Regex rprt (":($d+)");
	if (rprt == rest) {
	  d_port = rprt.getint (0);
	  rest   = "";
	}
      }
      // now extract the path
      Regex rpth ("(/<$a+-_.%/>*)(<?#>$N*)");
      if (rpth == rest) {
	d_path = utohs (rpth.getstr (0));
	rest   = rpth.getstr (1);
      } else {
	Regex rpth ("(/<$a+-_.%/>*)");
	if (rpth == rest) {
	  d_path = utohs (rpth.getstr (0));
	  rest   = "";
	}
      }
      // now extract the query
      Regex rqry ("$?(<$a+-_.%/=>*)(<#>$N*)");
      if (rqry == rest) {
	d_query = rqry.getstr (0);
	rest    = rqry.getstr (1);
      } else {
	Regex rqry ("$?(<$a+-_.%/=>*)");
	if (rqry == rest) {
	  d_query = rqry.getstr (0);
	  rest    = "";
	}
      }
      // now extract the fragment
      Regex rfrg ("$#($N*)");
      if (rfrg == rest) {
	d_fragment = utohs (rfrg.getstr (0));
	rest = "";
      }
      if (rest.length () != 0) 
	throw Exception ("url-error", "incorrect url format", url);
    } catch (...) {
      reset  ();
      unlock ();
      throw;
    }
  }

  // return this url scheme
  
  String Url::getscheme (void) const {
    rdlock ();
    String result = d_scheme;
    unlock ();
    return result;
  }

  // return this url host
  
  String Url::gethost (void) const {
    rdlock ();
    String result = d_host;
    unlock ();
    return result;
  }

  // return this url port
  
  long Url::getport (void) const {
    rdlock ();
    long result = d_port;
    unlock ();
    return result;
  }

  // return this url path
  
  String Url::getpath (void) const {
    rdlock ();
    String result = d_path;
    unlock ();
    return result;
  }

  // return this url query
  
  String Url::getquery (void) const {
    rdlock ();
    String result = d_query;
    unlock ();
    return result;
  }

  // return this url fragment
  
  String Url::getfragment (void) const {
    rdlock ();
    String result = d_fragment;
    unlock ();
    return result;
  }

  // create a new url object in a generic way

  Object* Url::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new Url;

    // check for one argument
    if (argc == 1) {
      String url = argv->getstring (0);
      return new Url (url);
    }
    
    // wrong arguments
    throw Exception ("argument-error", "invalid arguments with url object");
  }

  // apply this url object with a set of arguments and a quark

  Object* Url::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETPORT)     return new Integer (getport     ());
      if (quark == QUARK_GETHOST)     return new String  (gethost     ());
      if (quark == QUARK_GETPATH)     return new String  (getpath     ());
      if (quark == QUARK_GETQUERY)    return new String  (getquery    ());
      if (quark == QUARK_GETSCHEME)   return new String  (getscheme   ());
      if (quark == QUARK_GETFRAGMENT) return new String  (getfragment ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_PARSE) {
	parse (argv->getstring (0));
	return nilp;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
