// ---------------------------------------------------------------------------
// - XHtmlPage.cpp                                                           -
// - aleph:www library - xhtml page class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Runnable.hpp"
#include "XHtmlPage.hpp"
#include "Exception.hpp"

namespace aleph {

  // the xhtml page supported quarks
  static const long QUARK_SETLANG = String::intern ("set-language");
  static const long QUARK_GETLANG = String::intern ("get-language");

  // the base http header
  static const char* HTTPTYP = "Content-type: text/html\n";
  // the page document type
  static const char* DOCTYPE = 
    "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n"
    "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\"\n"
    "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">";
  static const char* CONTENT = "<meta http-equiv=\"Content-Type\""
                               "content=\"text/html; charset=iso-8859-1\"/>\n";

  // this procedure return the html setup with the xml language
  static String get_html (const String& lang) {
    String html = "<html xmlns=\"http://www.w3.org/1999/xhtml\" "; 
    html = html + "xml:lang=" + '"' + lang + '"' + ">";
    return html;
  }

  // create a new html page

  XHtmlPage::XHtmlPage (void) : HtmlPage (CONTENT) {
    // set default lang to english
    d_lang = "en";
    // add the page type to the http header
    addhttp (HTTPTYP);
  }

  // return the class name

  String XHtmlPage::repr (void) const {
    return "XHtmlPage";
  }

  // set the xml language

  void XHtmlPage::setlang (const String& lang) {
    wrlock ();
    d_lang = lang;
    unlock ();
  }

  // return the xml language

  String XHtmlPage::getlang (void) const {
    rdlock ();
    String result = d_lang;
    unlock ();
    return result;
  }

  // add a page style

  void XHtmlPage::addstyle (const String& style) {
    String line = "<style type=\"text/css\">\n";
    line = line + " @import url(" + style + ");\n";
    line = line + "</style>\n";
    addhead (line);
  }

  // add a meta markup by name and content

  void XHtmlPage::addmeta (const String& name, const String& content) {
    String line = "<meta name=";
    line = line + '"' + name + '"';
    line = line + ' ' + "content=";
    line = line + '"' + content + "\"\\>\n";
    addhead (line);
  }

  // write the page content

  void XHtmlPage::writepage (Output& os) const {
    rdlock ();
    try {
      // add the dtd and html markup
      os << DOCTYPE     << eolc;
      os << get_html (d_lang) << eolc;
      writehead (os);
      writebody (os);
      os << "</html>" << eolc;
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();    
  }

  // return a buffer with the xhtml page

  Buffer* XHtmlPage::getbuffer (void) const {
    rdlock ();
    Buffer* result = new Buffer;
    try {
      // add the dtd and html markup
      result->add (String (DOCTYPE));
      result->add (eolc);
      result->add (get_html (d_lang));
      result->add (eolc);
      // add the head
      result->add (String ("<head>\n"));
      result->add (d_head);
      result->add (String ("</head>\n"));
      // add the body
      result->add (String ("<body>\n"));
      result->add (d_body);
      result->add (String ("</body>\n"));
      // close the html markup
      result->add (String ("</html>\n"));
    } catch (...) {
      delete result;
      unlock ();
      throw;
    }
    unlock ();
    return result;
  }

  // create a new xhtml page in a generic way

  Object* XHtmlPage::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new XHtmlPage;
    // wrong arguments
    throw Exception ("argument-error", 
		     "too many arguments with xhtml page object");
  }
  
  // apply this xhtml page object with a set of arguments and a quark
  
  Object* XHtmlPage::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETLANG) return new String (getlang ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETLANG) {
	String lang = argv->getstring (0);
	setlang (lang);
	return nilp;
      }
    }

    // call the html page method
    return HtmlPage::apply (robj, nset, quark, argv);
  }
}
