// ---------------------------------------------------------------------------
// - HashTable.hpp                                                           -
// - standard object library - hash table object class definition            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_HASHTABLE_HPP
#define  ALEPH_HASHTABLE_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The HashTable class is another container class which maps an object
  /// with a name. The hash table is dynamic and get resized automatically
  /// when needed. When an object is added, the object reference count is
  /// increased. When the object is retreived, the reference count is not
  /// touched. The lookup method throw an exception if the name is not found.
  /// The get method returns nilp if the object is not found.
  /// @author amaury darsch

  class HashTable : public virtual Object {
  private:
    /// the hash table size
    long d_size;
    /// number of elements
    long d_count;
    /// threshold before resizing
    long d_thrs;
    /// array of buckets
    struct s_bucket** p_table;

  public:
    /// create a hash table with a default size
    HashTable (void);
  
    /// create a hash table with an initial size
    HashTable (const long size);
  
    /// destroy the hash table. 
    ~HashTable (void);

    /// return the class name
    String repr (void) const;

    /// make this hash table a shared object
    void mksho (void);

    /// set or create a new object in the table.
    /// @param name the object name to set
    /// @param object the object to set or add
    void add (const String& name, Object* object);

    /// get an object by name. If the object is not found nil is returned.
    /// this means that this function always succeds. You can use lookup 
    /// instead to get an exception if the name is not found. You can also use
    /// the exists method to check for a name but exists and get is very slow.
    /// @param name the object name to search
    Object* get (const String& name) const;

    /// find an object by name or throw an exception
    /// @param name the object name to find
    Object* lookup (const String& name) const;

    /// remove an object in the hash table. 
    /// @param name the object name to remove
    void remove (const String& name);
  
    /// @return true if the name exists in this hash table
    bool exists (const String& name) const;

    /// clear the hash table
    void clear (void);

    /// @return a vector of objects in this hash table
    Vector* getvector (void) const;

  private:
    // make the copy constructor private
    HashTable (const HashTable&);
    // make the assignment operator private
    HashTable& operator = (const HashTable&);    
    // resize this hash table
    void resize (const long size);
  };
}

#endif
