// ---------------------------------------------------------------------------
// - Relatif.hpp                                                             -
// - standard object library - relatif big number class definition           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_RELATIF_HPP
#define ALEPH_RELATIF_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

namespace aleph {

  /// The Relatif class is a big number integer implementation. The class
  /// behaves like the Relatif class except that it can handle any integer
  /// size. The class is a also a literal object. The name 'relatif' comes
  /// from the french 'entier relatif'.
  /// @author amaury darsch

  class Relatif : public Literal {
  private:
    /// the bytes size
    long d_size;
    /// the relatif array
    t_byte* p_byte;
    /// the relatif sign
    bool d_sign;

  public:
    /// create a default relatif
    Relatif (void);

    /// create a relatif with a native integer
    /// @param value the value to convert
    Relatif (const t_long value);

    /// create a relatif with a string
    /// @param value the value to convert
    Relatif (const String& value);

    /// create a relatif with a size, an array of bytes and a sign
    /// @param size the array size
    /// @param byte the byte array
    /// @param sign the relatif sign
    Relatif (const long size, t_byte* byte, const bool sign);

    /// copy construct this relatif
    /// @param that the relatif to copy
    Relatif (const Relatif& that);

    /// destroy this relatif
    ~Relatif (void);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this relatif
    String toliteral (void) const;

    /// @return a string representation of this relatif
    String tostring (void) const;

    /// @return a clone of this relatif
    Object* clone (void) const;

    /// @return the relatif serial code
    t_byte serialid (void) const;

    /// serialize this relatif to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a relatif from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// assign a relatif to this one
    /// @param that the relatif to assign
    Relatif& operator = (const Relatif& that);

    /// compare two relatifs
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const Relatif& value) const;

    /// compare two relatifs
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator != (const Relatif& value) const;

    /// compare two relatifs
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator < (const Relatif& value) const;

    /// compare two relatifs
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator <= (const Relatif& value) const;

    /// compare two relatifs
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator > (const Relatif& value) const;

    /// compare two relatifs
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator >= (const Relatif& value) const;

    /// change the sign of a relatif
    /// @param x the argument to change
    friend Relatif operator - (const Relatif& x);

    /// negate this relatif
    /// @param x the argument to negate
    friend Relatif operator ! (const Relatif& x);

    /// or two relatifs together
    /// @param x the first argument to or
    /// @param y the second argument to or
    friend Relatif operator | (const Relatif& x, const Relatif& y);

    /// and two relatifs together
    /// @param x the first argument to and
    /// @param y the second argument to and
    friend Relatif operator & (const Relatif& x, const Relatif& y);

    /// xor two relatifs together
    /// @param x the first argument to xor
    /// @param y the second argument to xor
    friend Relatif operator ^ (const Relatif& x, const Relatif& y);

    /// add two relatifs together
    /// @param x the first argument to add
    /// @param y the second argument to add
    friend Relatif operator + (const Relatif& x, const Relatif& y);

    /// substract two relatifs together
    /// @param x the first argument to substract
    /// @param y the second argument to substract
    friend Relatif operator - (const Relatif& x, const Relatif& y);

    /// multiply two relatifs together
    /// @param x the first argument to multiply
    /// @param y the second argument to multiply
    friend Relatif operator * (const Relatif& x, const Relatif& y);

    /// divide two relatifs together
    /// @param x the first argument to divide
    /// @param y the second argument to divide
    friend Relatif operator / (const Relatif& x, const Relatif& y);

    /// get the modulo of two relatifs together
    /// @param x the first argument to divide
    /// @param y the second argument to divide
    friend Relatif operator % (const Relatif& x, const Relatif& y);

    /// increment a relatif by one
    Relatif& operator ++ (void);

    /// decrement a relatif by one
    Relatif& operator -- (void);

    /// add a relatif to this one
    /// @param x the argument to add
    Relatif& operator += (const Relatif& x);

    /// substract a relatif to this one
    /// @param x the argument to substract
    Relatif& operator -= (const Relatif& x);

    /// multiply a relatif with this one
    /// @param x the argument to multiply
    Relatif& operator *= (const Relatif& x);

    /// divide a relatif with this one
    /// @param x the argument to divide
    Relatif& operator /= (const Relatif& x);

    /// shift left a relatif by a certain amount
    /// @param val the amount of shift
    Relatif operator << (const long val);

    /// shift right a relatif by a certain amount
    /// @param val the amount of shift
    Relatif operator >> (const long val);

    /// @return the absolute value of this number
    Relatif abs (void) const;

    /// @return a native integer representation
    t_long tointeger (void) const;

    /// @return true if the relatif is odd
    bool isodd (void) const;

    /// generate a new relatif
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// operate this realtif with another object
    /// @param robj   the current runnable
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (Runnable* robj, t_oper type, Object* object);

    /// set an object to this relatif
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, Object* object);

    /// apply this realtif with a set of arguments and a quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // normalize a relatif number
    void normalize (void);
    // compare two absolute relatifs
    bool geq (const Relatif& value) const;
    bool gth (const Relatif& value) const;
  };
}

#endif
