/*
 *   ALSA driver for VIA VT8233 (South Bridge)
 *
 *	Copyright (c) 2000 Jaroslav Kysela <perex@suse.cz>,
 *			   Tjeerd.Mulder@fujitsu-siemens.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */      

#define SND_MAIN_OBJECT_FILE

#include "../include/driver.h"
#include "../include/pcm.h"
#include "../include/mixer.h"
#include "../include/info.h"
#include "../include/ac97_codec.h"
#include "../include/initval.h"

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("\
Driver: VIA VT8233\n\
PCI: 0x1106=0x3059\n\
");
MODULE_LICENSE("GPL");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_pbk_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
int snd_cap_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for VIA 8233 bridge.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for VIA 8233 bridge.");
MODULE_AUTHOR("Jaroslav Kysela <perex@suse.cz>");
MODULE_PARM(snd_pbk_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_pbk_frame_size, "Playback frame size in kB.");
MODULE_PARM(snd_cap_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_cap_frame_size, "Capture frame size in kB.");

/*
 *  Direct registers
 */

#ifndef PCI_DEVICE_ID_VIA_8233_5
#define PCI_DEVICE_ID_VIA_8233_5	0x3059
#endif

#define VIAREG(via, x) ((via)->port + VIA_REG_##x)

/* offsets */
#define VIA_REG_OFFSET_STATUS		0x00	/* byte - channel status */
#define   VIA_REG_STAT_ACTIVE		0x80	/* RO */
#define   VIA_REG_STAT_TRIGGER_QUEUED	0x08	/* RO */
#define   VIA_REG_STAT_STOPPED		0x04	/* RWC */
#define   VIA_REG_STAT_EOL		0x02	/* RWC */
#define   VIA_REG_STAT_FLAG		0x01	/* RWC */
#define VIA_REG_OFFSET_CONTROL		0x01	/* byte - channel control */
#define   VIA_REG_CTRL_START		0x80	/* WO */
#define   VIA_REG_CTRL_TERMINATE	0x40	/* WO */
#define   VIA_REG_CTRL_AUTOSTART	0x20
#define   VIA_REG_CTRL_PAUSE		0x08	/* RW */
#define   VIA_REG_CTRL_INT_STOP		0x04		
#define   VIA_REG_CTRL_INT_EOL		0x02
#define   VIA_REG_CTRL_INT_FLAG		0x01
#define   VIA_REG_CTRL_INT (VIA_REG_CTRL_INT_FLAG | VIA_REG_CTRL_INT_EOL | VIA_REG_CTRL_AUTOSTART)
#define VIA_REG_OFFSET_TABLE_PTR	0x04	/* dword - channel table pointer */
#define VIA_REG_OFFSET_CURR_PTR		0x04	/* dword - channel current pointer */
#define VIA_REG_OFFSET_TYPE		0x08	/* long - stop index, channel type, sample rate */
#define   VIA_REG_TYPE_16BIT		0x00200000	/* RW */
#define   VIA_REG_TYPE_STEREO		0x00100000	/* RW */
#define VIA_REG_OFFSET_CURR_COUNT	0x0c	/* dword - channel current count (24 bit) */
#define VIA_REG_OFFSET_CURR_INDEX	0x0f	/* byte - channel current index */

#define VIA_NUM_OF_DMA_CHANNELS	2
/* playback block */
#define VIA_REG_PLAYBACK_STATUS		0x00	/* byte - channel status */
#define VIA_REG_PLAYBACK_CONTROL	0x01	/* byte - channel control */
#define VIA_REG_PLAYBACK_VOLUME_L	0x02	/* byte */
#define VIA_REG_PLAYBACK_VOLUME_R	0x03	/* byte */
#define VIA_REG_PLAYBACK_TABLE_PTR	0x04	/* dword - channel table pointer */
#define VIA_REG_PLAYBACK_CURR_PTR	0x04	/* dword - channel current pointer */
#define VIA_REG_PLAYBACK_TYPE		0x08    /* long - stop index, channel type, sample rate */ /* byte - channel type */
#define VIA_REG_PLAYBACK_CURR_COUNT	0x0c	/* dword - channel current count (24 bit) */
#define VIA_REG_PLAYBACK_CURR_INDEX	0x0f	/* byte - channel current index */
/* capture block */
#define VIA_REG_CAPTURE_STATUS		0x60	/* byte - channel status */
#define VIA_REG_CAPTURE_CONTROL		0x61	/* byte - channel control */
#define VIA_REG_CAPTURE_FIFO		0x62	/* byte - bit 6 = fifo  enable */
#define   VIA_REG_CAPTURE_FIFO_ENABLE	0x40
#define VIA_REG_CAPTURE_CHANNEL		0x63	/* byte - input select */
#define   VIA_REG_CAPTURE_CHANNEL_MIC	0x4
#define   VIA_REG_CAPTURE_CHANNEL_LINE	0
#define VIA_REG_CAPTURE_TABLE_PTR	0x64	/* dword - channel table pointer */
#define VIA_REG_CAPTURE_CURR_PTR	0x64	/* dword - channel current pointer */
#define VIA_REG_CAPTURE_TYPE		0x68	/* byte - channel type */
#define VIA_REG_CAPTURE_CURR_COUNT	0x6c	/* dword - channel current count (24 bit) */
#define VIA_REG_CAPTURE_CURR_INDEX	0x6f	/* byte - channel current index */

/* AC'97 */
#define VIA_REG_AC97			0x80	/* dword */
#define   VIA_REG_AC97_CODEC_ID_MASK	(3<<30)
#define   VIA_REG_AC97_CODEC_ID_SHIFT	30
#define   VIA_REG_AC97_SECONDARY_VALID	(1<<27)
#define   VIA_REG_AC97_PRIMARY_VALID	(1<<25)
#define   VIA_REG_AC97_BUSY		(1<<24)
#define   VIA_REG_AC97_READ		(1<<23)
#define   VIA_REG_AC97_CMD_SHIFT	16
#define   VIA_REG_AC97_CMD_MASK		0x7e
#define   VIA_REG_AC97_DATA_SHIFT	0
#define   VIA_REG_AC97_DATA_MASK	0xffff
#define VIA_REG_SGD_SHADOW		0x84	/* dword */

/*
 *  
 */

typedef struct {
	unsigned int reg_offset;
	unsigned int *table;
	unsigned int rates;
        snd_pcm_subchn_t *subchn;
        unsigned long physbuf;
        unsigned int size;
        unsigned int fragsize;
	unsigned int position;
} viadev_t;

typedef struct snd_stru_via8233 via8233_t;

struct snd_stru_via8233 {
	snd_dma_t * dma_pbk;	/* playback */
	snd_dma_t * dma_cap;	/* capture */
	snd_irq_t * irqptr;

	unsigned long port;
	unsigned char revision;

	struct pci_dev *pci;
	snd_card_t *card;

	snd_pcm_t *pcm;
	viadev_t playback;
	viadev_t capture;

	ac97_t *ac97;
	unsigned short ac97_ext_id;
	snd_kmixer_t *mixer;

	spinlock_t reg_lock;
	snd_info_entry_t *proc_entry;

	void *tables;
};

typedef struct snd_via8233_card {
	struct pci_dev *pci;
	snd_irq_t *irqptr;
	snd_dma_t *dma_pbk;	/* playback */
	snd_dma_t *dma_cap;	/* capture */
	snd_card_t *card;
	via8233_t *codec;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
} via8233_card_t;

static via8233_card_t *snd_via8233_cards[SND_CARDS] = SND_DEFAULT_PTR;

/*
 *  Basic I/O
 */

static inline unsigned int snd_via8233_codec_xread(via8233_t *codec)
{
	return (unsigned int)inw(VIAREG(codec, AC97)) |
	       ((unsigned int)inw(VIAREG(codec, AC97) + 2) << 16);
}
 
static inline void snd_via8233_codec_xwrite(via8233_t *codec, unsigned int val)
{
	outw((unsigned short)val, VIAREG(codec, AC97));
	outw((unsigned short)(val >> 16), VIAREG(codec, AC97) + 2);
}
 
static int snd_via8233_codec_ready(via8233_t *codec, int secondary, int sched)
{
	signed long end_time;
	unsigned int val;

	end_time = jiffies + 3 * (HZ >> 2);
	do {
		if (!((val = snd_via8233_codec_xread(codec)) & VIA_REG_AC97_BUSY))
			return val & 0xffff;
		if (sched) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(1);
		}
	} while (end_time - (signed long)jiffies >= 0);
	snd_printk("snd_via8233_codec_ready: codec %i is not ready [0x%x]\n", secondary, snd_via8233_codec_xread(codec));
	return -EIO;
}
 
static int snd_via8233_codec_valid(via8233_t *codec, int secondary)
{
	signed long end_time;
	int val;
	int stat = !secondary ? VIA_REG_AC97_PRIMARY_VALID :
				VIA_REG_AC97_SECONDARY_VALID;
	
	end_time = jiffies + 3 * (HZ >> 2);
	do {
		if ((val = snd_via8233_codec_xread(codec)) & stat)
			return 0;
	} while (end_time - (signed long)jiffies >= 0);
	snd_printk("snd_via8233_codec_valid: codec %i is not valid [0x%x]\n", secondary, snd_via8233_codec_xread(codec));
	return -EIO;
}
 
static void snd_via8233_codec_write(void *private_data,
				    unsigned short reg,
				    unsigned short val)
{
	via8233_t *codec = (via8233_t *)private_data;
	unsigned int xval;
	
	snd_via8233_codec_ready(codec, 0, 0);
	xval = 0 << VIA_REG_AC97_CODEC_ID_SHIFT;
	xval |= VIA_REG_AC97_PRIMARY_VALID;
	xval |= reg << VIA_REG_AC97_CMD_SHIFT;
	xval |= val << VIA_REG_AC97_DATA_SHIFT;
	snd_via8233_codec_xwrite(codec, xval);
}

static unsigned short snd_via8233_codec_read(void *private_data,
					      unsigned short reg)
{
	via8233_t *codec = (via8233_t *)private_data;
	unsigned int xval;

	if (snd_via8233_codec_ready(codec, 0, 0) < 0)
		return ~0;
	xval = 0 << VIA_REG_AC97_CODEC_ID_SHIFT;
	xval |= VIA_REG_AC97_PRIMARY_VALID;
	xval |= VIA_REG_AC97_READ;
	xval |= reg << VIA_REG_AC97_CMD_SHIFT;
	snd_via8233_codec_xwrite(codec, xval);
	if (snd_via8233_codec_ready(codec, 0, 0) < 0)
		return ~0;
	if (snd_via8233_codec_valid(codec, 0) < 0)
		return ~0;
	return snd_via8233_codec_xread(codec) & 0xffff;
}

#if 0
static void snd_via8233_channel_print(via8233_t *codec, viadev_t *viadev)
{
	unsigned long port = codec->port + viadev->reg_offset;

	printk("[0x%lx] status = 0x%x, control = 0x%x, type = 0x%x, ptr = 0x%x, count = 0x%x\n",
			port,
			inb(port + VIA_REG_OFFSET_STATUS),
			inb(port + VIA_REG_OFFSET_CONTROL),
			inl(port + VIA_REG_OFFSET_TYPE),
			inl(port + VIA_REG_OFFSET_CURR_PTR),
			inl(port + VIA_REG_OFFSET_CURR_COUNT));
}
#endif

static void snd_via8233_channel_reset(via8233_t *codec, viadev_t *viadev)
{
	unsigned long port = codec->port + viadev->reg_offset;

	outb(VIA_REG_CTRL_TERMINATE , port + VIA_REG_OFFSET_CONTROL);
	udelay(50);
	/* disable interrupts */
	outb(0, port + VIA_REG_OFFSET_CONTROL);
	/* clear interrupts */
	outb(0x3, port + VIA_REG_OFFSET_STATUS);
}

static int snd_via8233_trigger(via8233_t *codec, viadev_t *viadev, int cmd)
{
	unsigned char val = 0;
	unsigned long port = codec->port + viadev->reg_offset;
	
	switch (cmd) {
	case SND_PCM_TRIGGER_GO:
		val = VIA_REG_CTRL_INT | VIA_REG_CTRL_START;
		break;
	case SND_PCM_TRIGGER_STOP:
		val = VIA_REG_CTRL_TERMINATE;
		break;
	case SND_PCM_TRIGGER_PAUSE_PUSH:
		val = VIA_REG_CTRL_INT | VIA_REG_CTRL_PAUSE;
		break;
	case SND_PCM_TRIGGER_PAUSE_RELEASE:
		val = VIA_REG_CTRL_INT;
		break;
	default:
		return -EINVAL;
	}
	outb(val, port + VIA_REG_OFFSET_CONTROL);
	if (cmd == SND_PCM_TRIGGER_STOP)
		snd_via8233_channel_reset(codec, viadev);
	return 0;
}

static void snd_via8233_setup_fragments(via8233_t *codec, viadev_t *viadev,
					snd_pcm_runtime_t *runtime) 
{
	int idx, frags=1;
	unsigned int *table = viadev->table;
	unsigned long port = codec->port + viadev->reg_offset;

	snd_via8233_channel_reset(codec, viadev);
	outl(virt_to_bus(viadev->table), port + VIA_REG_OFFSET_TABLE_PTR);

	outl(((runtime->format.format == SND_PCM_SFMT_S16_LE ? VIA_REG_TYPE_16BIT : 0) |
	      (runtime->format.voices > 1 ? VIA_REG_TYPE_STEREO : 0)) |
			0xff000000,	/* STOP index never reached */
			port + VIA_REG_OFFSET_TYPE);

	if (viadev->size == viadev->fragsize) {
		table[0] = viadev->physbuf;
		table[1] = 0xc0000000 | /* EOL + flag */
			   viadev->fragsize;
	} else {
		frags = viadev->size / viadev->fragsize;
		for (idx = 0; idx < frags - 1; idx++) {
			table[(idx << 1) + 0] = (viadev->physbuf + (idx * viadev->fragsize));
			table[(idx << 1) + 1] = 0x40000000 |	/* flag */
						viadev->fragsize;
		}
		table[((frags-1) << 1) + 0] = (viadev->physbuf + ((frags-1) * viadev->fragsize));
		table[((frags-1) << 1) + 1] = 0x80000000 |	/* EOL */
					      viadev->fragsize;
	}
}

/*
 *  Interrupt handler
 */

static inline void snd_via8233_update(via8233_t *codec, viadev_t *viadev)
{
	spin_lock(&codec->reg_lock);
	viadev->position = inb(VIAREG(codec, OFFSET_CURR_INDEX) + viadev->reg_offset) * viadev->fragsize;
	viadev->position %= viadev->size;
	spin_unlock(&codec->reg_lock);
	snd_pcm_transfer_done(viadev->subchn);
	outb(VIA_REG_STAT_FLAG | VIA_REG_STAT_EOL, VIAREG(codec, OFFSET_STATUS) + viadev->reg_offset);
}

static void snd_via8233_card_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	via8233_card_t *scard = (via8233_card_t *) dev_id;
	via8233_t *codec;

	if (scard == NULL || (codec = scard->codec) == NULL)
		return;
	if (inb(VIAREG(codec, PLAYBACK_STATUS)) & (VIA_REG_STAT_EOL|VIA_REG_STAT_FLAG))
		snd_via8233_update(codec, &codec->playback);
	if (inb(VIAREG(codec, CAPTURE_STATUS)) & (VIA_REG_STAT_EOL|VIA_REG_STAT_FLAG))
		snd_via8233_update(codec, &codec->capture);
}

/*
 *  PCM part
 */

static int snd_via8233_playback_ioctl(void *private_data,
				      snd_pcm_subchn_t * subchn,
				      unsigned int cmd,
				      unsigned long *arg)
{
	int result;
	// via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);

	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	if (cmd == SND_PCM_IOCTL1_PARAMS) {
		snd_pcm_runtime_t *runtime = subchn->runtime;

		if (runtime->frags > 32)
			runtime->frags = 32;
	}
	return 0;
}

static int snd_via8233_capture_ioctl(void *private_data,
				      snd_pcm_subchn_t * subchn,
				      unsigned int cmd,
				      unsigned long *arg)
{
	int result;
	// via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);

	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	if (cmd == SND_PCM_IOCTL1_PARAMS) {
		snd_pcm_runtime_t *runtime = subchn->runtime;

		if (runtime->frags > 32)
			runtime->frags = 32;
	}
	return 0;
}

static int snd_via8233_playback_trigger(void *private_data,
					snd_pcm_subchn_t * pcm,
					int cmd)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);

	return snd_via8233_trigger(codec, &codec->playback, cmd);
}

static int snd_via8233_capture_trigger(void *private_data,
				       snd_pcm_subchn_t * subchn,
				       int cmd)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);

	return snd_via8233_trigger(codec, &codec->capture, cmd);
}

static int snd_via8233_playback_prepare(void *private_data,
					 snd_pcm_subchn_t * subchn)
{
	unsigned long flags, tmp;
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;

	codec->playback.physbuf = virt_to_bus(runtime->dma_area->buf);
	codec->playback.size = snd_pcm_lib_transfer_size(subchn);
	codec->playback.fragsize = snd_pcm_lib_transfer_fragment(subchn);
	codec->playback.position = 0;
	/* disable DAC power */
	snd_ac97_write_bitmask_lock(codec->ac97, AC97_POWERDOWN, ~0x0200, 0x0200);
	snd_ac97_write_lock(codec->ac97, AC97_PCM_FRONT_DAC_RATE, runtime->format.rate);
	/* enable DAC power */
	snd_ac97_write_bitmask_lock(codec->ac97, AC97_POWERDOWN, ~0x0200, 0x0000);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_via8233_setup_fragments(codec, &codec->playback, runtime);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	/* I don't understand this stuff but its from the documentation and this way it works */
	outb(0 , VIAREG(codec, PLAYBACK_VOLUME_L));
	outb(0 , VIAREG(codec, PLAYBACK_VOLUME_R));
	tmp = inl(VIAREG(codec, PLAYBACK_TYPE)) & ~0xfffff;
	outl(tmp | (0xffff * (unsigned)runtime->format.rate)/(unsigned)(48000/16), VIAREG(codec, PLAYBACK_TYPE));
	return 0;
}

static int snd_via8233_capture_prepare(void *private_data,
				       snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;

	codec->capture.physbuf = virt_to_bus(runtime->dma_area->buf);
	codec->capture.size = snd_pcm_lib_transfer_size(subchn);
	codec->capture.fragsize = snd_pcm_lib_transfer_fragment(subchn);
	codec->capture.position = 0;
	/* disable ADC power */
	snd_ac97_write_bitmask_lock(codec->ac97, AC97_POWERDOWN, ~0x0100, 0x0100);
	snd_ac97_write_lock(codec->ac97, AC97_PCM_LR_ADC_RATE, runtime->format.rate);
	/* enable ADC power */
	snd_ac97_write_bitmask_lock(codec->ac97, AC97_POWERDOWN, ~0x0100, 0x0000);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_via8233_setup_fragments(codec, &codec->capture, runtime);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	outb(VIA_REG_CAPTURE_CHANNEL_LINE, VIAREG(codec, CAPTURE_CHANNEL));
	outb(VIA_REG_CAPTURE_FIFO_ENABLE, VIAREG(codec, CAPTURE_FIFO));
	return 0;
}

static unsigned int snd_via8233_playback_pointer(void *private_data,
						  snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);
	unsigned int result;

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (inb(VIAREG(codec, PLAYBACK_STATUS)) & VIA_REG_STAT_ACTIVE) {
		result = codec->playback.position +
			 (codec->playback.fragsize -
			 (inl(VIAREG(codec, PLAYBACK_CURR_COUNT)) & 0xffffff));
	} else {
		result = 0;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

static unsigned int snd_via8233_capture_pointer(void *private_data,
						snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);
	unsigned int result;

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (inb(VIAREG(codec, CAPTURE_STATUS)) & VIA_REG_STAT_ACTIVE) {
		result = codec->capture.position +
			 (codec->capture.fragsize -
			 (inl(VIAREG(codec, CAPTURE_CURR_COUNT)) & 0xffffff));
	} else {
		result = 0;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

static snd_pcm_hardware_t snd_via8233_playback =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_BLOCK_TRANSFER |
	SND_PCM_CHNINFO_MMAP_VALID |
	SND_PCM_CHNINFO_PAUSE,	/* flags */
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* hardware formats */
	0,			/* overwritten */
	8000,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	32,			/* min. fragment size */
	128 * 1024,		/* max. fragment size */
	31,			/* fragment align */
	0,			/* FIFO size (unknown) */
	16,			/* transfer block size */
	snd_via8233_playback_ioctl,
	snd_via8233_playback_prepare,
	snd_via8233_playback_trigger,
	snd_via8233_playback_pointer
};

static snd_pcm_hardware_t snd_via8233_capture =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_BLOCK_TRANSFER |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* hardware formats */
	0,			/* overwritten */
	8000,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	32,			/* min. fragment size */
	128 * 1024,		/* max. fragment size */
	31,			/* fragment align */
	0,			/* FIFO size (unknown) */
	16,			/* transfer block size */
	snd_via8233_capture_ioctl,
	snd_via8233_capture_prepare,
	snd_via8233_capture_trigger,
	snd_via8233_capture_pointer
};

static int snd_via8233_playback_open(void *private_data,
			             snd_pcm_subchn_t * subchn)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);
	snd_pcm_hardware_t *hw;
	int err;

	hw = (snd_pcm_hardware_t *)snd_kmalloc(sizeof(snd_pcm_hardware_t), GFP_KERNEL);
	if (hw == NULL)
		return -ENOMEM;
	if ((err = snd_pcm_dma_alloc(subchn, codec->dma_pbk,
				     "VIA 8233 - DAC")) < 0) {
		snd_kfree(hw);
		return err;
	}
	codec->playback.subchn = subchn;
	memcpy(hw, &snd_via8233_playback, sizeof(*hw));
	hw->rates = codec->playback.rates;
	if (!(hw->rates & SND_PCM_RATE_8000))
		hw->min_rate = 48000;
	subchn->runtime->hw = hw;
	subchn->runtime->hw_free = (snd_kfree_type)_snd_kfree;
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->ac97->me_playback);
	return 0;
}

static int snd_via8233_capture_open(void *private_data,
				    snd_pcm_subchn_t * subchn)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);
	snd_pcm_hardware_t *hw;
	int err;

	hw = (snd_pcm_hardware_t *)snd_kmalloc(sizeof(snd_pcm_hardware_t), GFP_KERNEL);
	if (hw == NULL)
		return -ENOMEM;
	if ((err = snd_pcm_dma_alloc(subchn, codec->dma_cap,
				     "VIA 8233 - ADC")) < 0) {
		snd_kfree(hw);
		return err;
	}
	codec->capture.subchn = subchn;
	memcpy(hw, &snd_via8233_capture, sizeof(*hw));
	hw->rates = codec->capture.rates;
	if (!(hw->rates & SND_PCM_RATE_8000))
		hw->min_rate = 48000;
	subchn->runtime->hw = hw;
	subchn->runtime->hw_free = (snd_kfree_type)_snd_kfree;
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->ac97->me_capture);
	return 0;
}

static int snd_via8233_playback_close(void *private_data,
				      snd_pcm_subchn_t * subchn)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);

	snd_via8233_channel_reset(codec, &codec->playback);	/* just to be sure */
	codec->playback.subchn = NULL;
	snd_pcm_dma_free(subchn);
	/* disable DAC power */
	snd_ac97_write_bitmask_lock(codec->ac97, AC97_POWERDOWN, ~0x0200, 0x0200);
	return 0;
}

static int snd_via8233_capture_close(void *private_data,
				     snd_pcm_subchn_t * subchn)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, -ENXIO);

	snd_via8233_channel_reset(codec, &codec->capture);	/* just to be sure */
	codec->capture.subchn = NULL;
	snd_pcm_dma_free(subchn);
	/* disable ADC power */
	snd_ac97_write_bitmask_lock(codec->ac97, AC97_POWERDOWN, ~0x0100, 0x0100);
	return 0;
}

static void snd_via8233_pcm_free(void *private_data)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, );
	codec->pcm = NULL;
}

int snd_via8233_pcm(via8233_t * codec, int device, snd_pcm_t ** rpcm)
{
	snd_pcm_t *pcm;
	int err;

	*rpcm = NULL;
	err = snd_pcm_new(codec->card, "VIA 8233", device, 1, 1, &pcm);
	if (err < 0)
		return err;

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_via8233_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_via8233_playback_close;

	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_via8233_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_via8233_capture_close;

	pcm->private_data = codec;
	pcm->private_free = snd_via8233_pcm_free;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE |
			  SND_PCM_INFO_DUPLEX;
	strcpy(pcm->name, "VIA 8233");
	*rpcm = codec->pcm = pcm;
	return 0;
}

/*
 *  Mixer part
 */

static void snd_via8233_codec_init(void *private_data, ac97_t *ac97)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, );

	if (codec->ac97_ext_id & 0x0001)	/* VRA */
		snd_ac97_write_lock(ac97, AC97_EXTENDED_STATUS, 0x0009);	
	/* disable DAC & ADC power */
	snd_ac97_write_bitmask_lock(ac97, AC97_POWERDOWN, ~0x0300, 0x0300);
	/* disable center DAC/surround DAC/LFE DAC/MIC ADC */
	snd_ac97_write_bitmask_lock(ac97, AC97_EXTENDED_STATUS, (short)~0xe800, 0xe800);
}

static void snd_via8233_mixer_free_ac97(void *private_data)
{
	via8233_t *codec = snd_magic_cast(via8233_t, private_data, );
	codec->mixer = NULL;
}

int snd_via8233_mixer(via8233_t * codec, int device, int pcm_count, int *pcm_devs, snd_kmixer_t ** rmixer)
{
	snd_kmixer_t *mixer;
	ac97_t ac97;
	int err;

	*rmixer = NULL;
	memset(&ac97, 0, sizeof(ac97));
	ac97.write = snd_via8233_codec_write;
	ac97.read = snd_via8233_codec_read;
	ac97.init = snd_via8233_codec_init;
	ac97.private_data = codec;
	ac97.private_free = snd_via8233_mixer_free_ac97;
	if ((err = snd_ac97_mixer(codec->card, device, &ac97, pcm_count, pcm_devs, &mixer)) < 0) {
		return err;
	}
	codec->ac97 = snd_magic_cast(ac97_t, mixer->private_data, -ENXIO);
	*rmixer = codec->mixer = mixer;
	return 0;
}

/*
 *
 */

static int snd_via8233_test_rate(via8233_t *codec, int reg, int rate)
{
	unsigned short val;

	snd_via8233_codec_write(codec, reg, rate);
	val = snd_via8233_codec_read(codec, reg);
	return val == rate;
}

static void snd_via8233_determine_rates(via8233_t *codec, int reg, unsigned int *r_result)
{
	unsigned int result = 0;

	/* test a non-standard frequency */
	if (snd_via8233_test_rate(codec, reg, 11000))
		result |= SND_PCM_RATE_CONTINUOUS;
	/* let's try to obtain standard frequencies */
	if (snd_via8233_test_rate(codec, reg, 8000))
		result |= SND_PCM_RATE_8000;
	if (snd_via8233_test_rate(codec, reg, 11025))
		result |= SND_PCM_RATE_11025;
	if (snd_via8233_test_rate(codec, reg, 16000))
		result |= SND_PCM_RATE_16000;
	if (snd_via8233_test_rate(codec, reg, 22050))
		result |= SND_PCM_RATE_22050;
	if (snd_via8233_test_rate(codec, reg, 32000))
		result |= SND_PCM_RATE_32000;
	if (snd_via8233_test_rate(codec, reg, 44100))
		result |= SND_PCM_RATE_44100;
	if (snd_via8233_test_rate(codec, reg, 48000))
		result |= SND_PCM_RATE_48000;
	*r_result = result;
}

static int snd_via8233_chip_init(via8233_t *codec)
{
	unsigned short ext_id;

	/* deassert ACLink reset */
	pci_write_config_byte(codec->pci, 0x41, 0x40);
	udelay(100);
	/* deassert ACLink reset, force SYNC (warm AC'97 reset) */
	pci_write_config_byte(codec->pci, 0x41, 0x60);
	udelay(2);
	/* ACLink on, deassert ACLink reset, VSR, SGD data out */
	pci_write_config_byte(codec->pci, 0x41, 0xcc);

	snd_via8233_codec_ready(codec, 0, 1);
	snd_via8233_codec_write(codec, AC97_POWERDOWN, 0x0000);
	snd_via8233_codec_ready(codec, 0, 1);

#if 0
	{
	unsigned char cmdb;

	pci_read_config_byte(codec->pci, 0x40, &cmdb);
	printk("PCI[0x40] = 0x%x\n", cmdb);
	pci_read_config_byte(codec->pci, 0x42, &cmdb);
	printk("PCI[0x42] = 0x%x\n", cmdb);
	pci_read_config_byte(codec->pci, 0x43, &cmdb);
	printk("PCI[0x43] = 0x%x\n", cmdb);
	pci_read_config_byte(codec->pci, 0x44, &cmdb);
	printk("PCI[0x44] = 0x%x\n", cmdb);
	pci_read_config_byte(codec->pci, 0x48, &cmdb);
	printk("PCI[0x48] = 0x%x\n", cmdb);
	}
#endif

	codec->ac97_ext_id = 
	ext_id = snd_via8233_codec_read(codec, AC97_EXTENDED_ID);
	if (ext_id == 0xffff) {
		snd_printk("snd_via8233_chip_init: invalid AC'97 codec\n");
		return -EIO;
	}
	if (ext_id & 0x0001) {
		snd_via8233_codec_write(codec, AC97_EXTENDED_STATUS, 0x0009);
		snd_via8233_determine_rates(codec, AC97_PCM_FRONT_DAC_RATE, &codec->playback.rates);
		snd_via8233_determine_rates(codec, AC97_PCM_LR_ADC_RATE, &codec->capture.rates);
	} else {
		codec->playback.rates = SND_PCM_RATE_48000;
		codec->capture.rates = SND_PCM_RATE_48000;
	}
	/* disable interrupts */
	snd_via8233_channel_reset(codec, &codec->playback);
	snd_via8233_channel_reset(codec, &codec->capture);
	return 0;
}

static int snd_via8233_free(via8233_t * codec)
{
	/* disable interrupts */
	snd_via8233_channel_reset(codec, &codec->playback);
	snd_via8233_channel_reset(codec, &codec->capture);
	/* --- */
	synchronize_irq();
	if (codec->tables)
		snd_kfree(codec->tables);
	snd_magic_kfree(codec);
	return 0;
}

static int __init snd_via8233_create(snd_card_t * card,
				     struct pci_dev *pci,
				     snd_dma_t * dma_pbk,
				     snd_dma_t * dma_cap,
				     snd_irq_t * irqptr,
				     via8233_t ** r_via)
{
	via8233_t *codec;
	unsigned short cmdw;
	int err;
        static snd_device_ops_t ops = {
		(snd_dev_free_t *)snd_via8233_free,
		NULL,
		NULL
        };

	if ((codec = snd_magic_kcalloc(via8233_t, 0, GFP_KERNEL)) == NULL)
		return -ENOMEM;

	spin_lock_init(&codec->reg_lock);
	codec->card = card;
	codec->pci = pci;
	codec->dma_pbk = dma_pbk;
	codec->dma_cap = dma_cap;
	codec->irqptr = irqptr;
	codec->port = pci_resource_start(pci, 0);
	pci_read_config_word(pci, PCI_COMMAND, &cmdw);
	if ((cmdw & PCI_COMMAND_IO) != PCI_COMMAND_IO) {
		cmdw |= PCI_COMMAND_IO;
		pci_write_config_word(pci, PCI_COMMAND, cmdw);
	}
#if 0
	{
	unsigned char cmdb;
	pci_set_master(pci);
	pci_read_config_byte(pci, PCI_LATENCY_TIMER, &cmdb);
	if (cmdb < 64)
		cmdb = 64;
	pci_write_config_byte(pci, PCI_LATENCY_TIMER, cmdb);
	}
#endif
	pci_read_config_byte(pci, PCI_REVISION_ID, &codec->revision);
	synchronize_irq();

	/* initialize offsets */
	codec->playback.reg_offset = VIA_REG_PLAYBACK_STATUS;
	codec->capture.reg_offset = VIA_REG_CAPTURE_STATUS;

	/* allocate buffer descriptor lists */
	/* the start of each lists must be aligned to 8 bytes */
	codec->tables = (unsigned int *)snd_kcalloc(8 + sizeof(unsigned int) * 32 * 2 * VIA_NUM_OF_DMA_CHANNELS, GFP_KERNEL);
	if (codec->tables == NULL) {
		snd_via8233_free(codec);
		return -ENOMEM;
	}
	codec->playback.table = (unsigned int *)(((unsigned long)codec->tables + 7) & ~7);
	codec->capture.table = codec->playback.table + 32 * 2;

	if ((err = snd_via8233_chip_init(codec)) < 0) {
		snd_via8233_free(codec);
		return err;
	}

	if ((err = snd_device_new(card, SND_DEV_LOWLEVEL, codec, 0, &ops, NULL)) < 0) {
		snd_via8233_free(codec);
		return err;
	}

	*r_via = codec;
	return 0;
}

static void snd_via8233_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_via8233_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int __init snd_via8233_detect(snd_card_t * card, via8233_card_t *scard,
				      unsigned short vendor, unsigned short device)
{
	if ((scard->pci = pci_find_device(vendor, device, scard->pci)) == NULL)
		return -ENODEV;
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 0), 256, "VIA 8233 - AC'97", NULL) < 0)
		goto __nodev;
	return 0;
      __nodev:
	snd_unregister_ioports(card);
	return -ENODEV;
}

static int __init snd_via8233_resources(snd_card_t * card,
					via8233_card_t *scard,
					int dev)
{
	int err;

	if ((err = snd_register_interrupt(card,
			"VIA 8233 Audio", scard->pci->irq,
			SND_IRQ_TYPE_PCI, snd_via8233_card_interrupt,
			scard, NULL, &scard->irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"VIA 8233 - playback", 0,
			SND_DMA_TYPE_PCI, snd_pbk_frame_size[dev],
			NULL, &scard->dma_pbk)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"VIA 8233 - capture", 1,
			SND_DMA_TYPE_PCI, snd_cap_frame_size[dev],
			NULL, &scard->dma_cap)) < 0)
		return err;
	return 0;
}

static int __init snd_via8233_probe(int dev, via8233_card_t *scard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	via8233_t *codec;
	int pcm_dev = 0, mixer_dev = 0;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_via8233_use_inc, snd_via8233_use_dec);
	if (card == NULL)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_VIA8233;
	do {
		if (!snd_via8233_detect(card, scard, PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_8233_5))
			break;
	} while (scard->pci);
	if (scard->pci != NULL)
		goto __found;
	snd_card_free(card);
	return -ENODEV;

      __found:
	if (pci_enable_device(scard->pci)) {
		snd_card_free(card);
		return -ENODEV;
	}

	pci_set_master(scard->pci);

	if (snd_via8233_resources(card, scard, dev) < 0) {
		snd_card_free(card);
		return -ENODEV;
	}

	if (snd_via8233_create(card,
			       scard->pci,
			       scard->dma_pbk,
			       scard->dma_cap,
			       scard->irqptr,
			       &scard->codec) < 0)
		goto __nodev;
	codec = scard->codec;

	if (snd_via8233_mixer(codec, mixer_dev++, 1, &pcm_dev, &mixer) < 0)
		goto __nodev;
	if (snd_via8233_pcm(codec, pcm_dev++, &pcm) < 0)
		goto __nodev;

	strcpy(card->abbreviation, "VIA8233");
	strcpy(card->shortname, "VIA 8233");
	
	sprintf(card->longname, "%s at 0x%lx, irq %li",
		card->shortname, codec->port, scard->irqptr->irq);

	if (!snd_card_register(card)) {
		scard->card = card;
		scard->mixer = mixer;
		scard->pcm = pcm;
		return 0;
	}
	goto __nodev;

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

#ifdef MODULE

static int __exit snd_via8233_card_free(int dev)
{
	via8233_card_t *scard;

	scard = snd_via8233_cards[dev];
	snd_via8233_cards[dev] = NULL;
	if (scard) {
		snd_card_unregister(scard->card);
		snd_kfree(scard);
	}
	return 0;
}

#endif

int __init alsa_card_via8233_init(void)
{
	int dev, cards;
	via8233_card_t *scard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		scard = (via8233_card_t *)
				snd_kcalloc(sizeof(via8233_card_t), GFP_KERNEL);
		if (scard == NULL)
			continue;
		if (snd_via8233_probe(dev, scard) < 0) {
			snd_kfree(scard);
			break;
		}
		snd_via8233_cards[dev] = scard;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		snd_printk("VIA 8233 soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

void __exit alsa_card_via8233_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_via8233_card_free(dev);
}

module_init(alsa_card_via8233_init)
module_exit(alsa_card_via8233_exit)
