/*
 *
 *  (c) COPYRIGHT INRIA, 1996-2004
 *  Please first read the full copyright statement in file COPYRIGHT.
 *
 */

/*
 * This module handles attribute presentation
 *
 * Author: V. Quint (INRIA)
 *
 */
#include "thot_gui.h"
#include "thot_sys.h"
#include "constmedia.h"
#include "typemedia.h"

#include "attributes_f.h"
#include "changeabsbox_f.h"
#include "content_f.h"
#include "memory_f.h"
#include "presrules_f.h"
#include "references_f.h"
#include "schemas_f.h"
#include "tree_f.h"

/*----------------------------------------------------------------------
   CreateInheritedAttrTable

   Allocate and initialize an InheritAttrTable for element pEl in presentation
   schema pSP.
   This table indicates attributes that transmit presentation rules
   to element pEl.
   table[attr] != '\0' if the presentation schema contains in its
   ATTRIBUTES section a rule of the form

      AttrName(ElType): ...
  ----------------------------------------------------------------------*/
void  CreateInheritedAttrTable (PtrElement pEl, PtrPSchema pPS,
				PtrDocument pDoc)
{
  int                 attr;
  int                 rule;
  AttributePres      *pAttrPR;
  InheritAttrTable   *table;

  if (pPS != NULL &&
      pPS->PsInheritedAttr->ElInherit[pEl->ElTypeNumber - 1] == NULL)
    {
      /* table allocation and initialization */
      if ((table = (InheritAttrTable*) TtaGetMemory (pEl->ElStructSchema->SsNAttributes * sizeof (char))) == NULL)
        /* memory exhausted */
        return;
      pPS->PsInheritedAttr->ElInherit[pEl->ElTypeNumber - 1] = table;
      /* for all attributes defined in the structure schema */
      for (attr = 0; attr < pEl->ElStructSchema->SsNAttributes; attr++)
	{
	  (*table)[attr] = '\0';  /* no inheritance by default */
	  pAttrPR = pPS->PsAttrPRule->AttrPres[attr];
	  if (pAttrPR != NULL)
	    /* check all presentation rules associated with that attr */
	    for (rule = 0; rule < pPS->PsNAttrPRule->Num[attr]; rule++)
	      {
		if (pAttrPR->ApElemType == pEl->ElTypeNumber)
		  {
		    if (pAttrPR->ApElemInherits)
		      /* the element inherits some presentation properties
			 from its ancestors or from itself */
		      (*table)[attr] = 'S';
		    else
		      /* the element inherits some presentation properties
			 from its ancestors */
		      (*table)[attr] = 'H';
		  }
		pAttrPR = pAttrPR->ApNextAttrPres;
	      }
	}
    }
}

/*----------------------------------------------------------------------
  CreateComparAttrTable

  Allocate and initialize a ComparAttrTable for attribute pAttr.

  For each attribute, this table indicates the other attributes that compare
  to the attribute for applying some presentation rules.

  table[attr] = TRUE if the presentation schema contains in its
  ATTRIBUTES section a rule of the form

     AttrName > n : ...

  AttrName is the name of attribute attr (rank of attribute definition
  in structure schema).
  ----------------------------------------------------------------------*/
void CreateComparAttrTable (PtrAttribute pAttr, PtrDocument pDoc,
			    PtrElement pEl)
{
  AttributePres      *pAttrPR;
  int                 attNum;
  ComparAttrTable    *table;
  int                 attr;
  int                 rule;
  int                 item;
  PresentSchema      *pPS;
  PtrPSchema          pSchP;
  PtrHandlePSchema    pHd;   

  /* table allocation and initialization */
  pPS = PresentationSchema (pAttr->AeAttrSSchema, pDoc);
  if (!pPS)
    return;
  if ((table = (ComparAttrTable *) TtaGetMemory (pAttr->AeAttrSSchema->SsNAttributes * sizeof (ThotBool))) == NULL)
    /* memory exhausted */
    return;
  attNum = pAttr->AeAttrNum;
  pPS->PsComparAttr->CATable[attNum - 1] = table;
  /* scan all attributes defined in the structure schema */
  for (attr = 0; attr < pAttr->AeAttrSSchema->SsNAttributes; attr++)
    {
    (*table)[attr] = FALSE;
    /* check only integer attributes */
    if (pAttr->AeAttrSSchema->SsAttribute->TtAttr[attr]->AttrType == AtNumAttr)
      {
	/* check the main presentation schema and all its extensions */
	pSchP = pPS;
	pHd = NULL;
	while (pSchP)
	  {
	    if (pSchP->PsNAttrPRule->Num[attr] > 0)
	      {
		pAttrPR = pSchP->PsAttrPRule->AttrPres[attr];
		/* check presentation rules associated with attribute attr */
		for (rule = 0; rule < pSchP->PsNAttrPRule->Num[attr]; rule++)
		  {
		    if (pAttrPR->ApElemType == 0)
		      /* no inheritance */
		      {
			for (item = 0; item < pAttrPR->ApNCases; item++)
			  {
			   if (pAttrPR->ApCase[item].CaComparType == ComparAttr
			       && (pAttrPR->ApCase[item].CaLowerBound == attNum
				   || pAttrPR->ApCase[item].CaUpperBound == attNum))
			     {
			       (*table)[attr] = TRUE;
			       break;
			     }
			  }
			break;
		      }
		    pAttrPR = pAttrPR->ApNextAttrPres;
		  }
	      }
	    /* next P schema */
	    if (pHd == NULL)
	      /* extension schemas have not been checked yet */
	      /* get the first extension schema */
	      pHd = FirstPSchemaExtension (pAttr->AeAttrSSchema, pDoc, pEl);
	    else
	      /* get the next extension schema */
	      pHd = pHd->HdNextPSchema;
	    if (pHd == NULL)
	      /* no more extension schemas. Stop */
	      pSchP = NULL;
	    else
	      pSchP = pHd->HdPSchema;
	  }
      }
    }
}

/*----------------------------------------------------------------------
  TransmitElementContent

  Takes the contents of element pEl and make it the value of the
  attribute named attrName for all documents included in document
  pDoc by the rule inclRule from the pSS structure schema.
  ----------------------------------------------------------------------*/
void TransmitElementContent (PtrElement pEl, PtrDocument pDoc,
			     char *attrName, int inclRule, PtrSSchema pSS)
{
  PtrReferredDescr    pReferredD;
  PtrReference        pRef;
  PtrElement          pIncludedEl;
  PtrDocument         pIncludedDoc;
  PtrAttribute        pAttr;
  DocumentIdentifier  IncludedDocIdent;
  int                 att;
  PtrTtAttribute         AttrDef;
  PtrElement          pChild;
  ThotBool            found;
  int                 len;

  /* Search all documents of the type in question included in document pDoc */
  pReferredD = pDoc->DocReferredEl;
  /* skip the first descriptor (it's a fake descriptor) */
  if (pReferredD != NULL)
    pReferredD = pReferredD->ReNext;
  /* scan all referred element descriptors */
  while (pReferredD != NULL)
    {
      if (pReferredD->ReExternalRef)
	/* this referred element is in another document */
	{
	  pRef = pReferredD->ReFirstReference;
	  /* scan all references to that external element */
	  while (pRef != NULL)
	    {
	      /* consider only inclusion references */
	      if (pRef->RdTypeRef == RefInclusion)
		{
		  /* get the root of the included document */
		  pIncludedEl = ReferredElement (pRef, &IncludedDocIdent,
						 &pIncludedDoc);
		  if (pIncludedEl != NULL)
		    {
		     /* the included document is loaded. Search the */
		     /* attribute in its structure schema */
		     att = 0;
		     found = FALSE;
		     while (att < pIncludedEl->ElStructSchema->SsNAttributes &&
			    !found)
		       {
			 AttrDef = pIncludedEl->ElStructSchema->SsAttribute->TtAttr[att++];
			 if (AttrDef->AttrType == AtTextAttr)
			   /* that's a text attribute */
			   if (AttrDef->AttrOrigName != NULL &&
			       strcmp (AttrDef->AttrOrigName, attrName) == 0)
			     /* that's the rigth attribute */
			     found = TRUE;
		       }
		     if (found)
		       {
			 GetAttribute (&pAttr);
			 pAttr->AeAttrSSchema = pIncludedEl->ElStructSchema;
			 pAttr->AeAttrNum = att;
			 pAttr->AeAttrType = AtTextAttr;
			 GetTextBuffer (&pAttr->AeAttrText);
			 /* copy the contents of element pEl into */
			 /* the attribute */
			 pChild = pEl;
			 /* first, get the first text leaf of pEl */
			 found = FALSE;
			 while (pChild != NULL && !found)
			   if (pChild->ElTerminal)
			     {
			       if (pChild->ElLeafType == LtText)
				 found = TRUE;
			     }
			   else
			     pChild = pChild->ElFirstChild;
			 if (found)
			   {
			     /* copy the contents of that leaf */
			     CopyTextToText ((PtrTextBuffer) pChild->ElText->BuContent,
					     pAttr->AeAttrText, &len);
			     /* associate the attribute with the */
			     /* of the included document */
			     AttachAttrWithValue (pIncludedEl, pIncludedDoc, pAttr, TRUE);
			   }
			 DeleteAttribute (NULL, pAttr);
		       }
		    }
		}
	      pRef = pRef->RdNext;
	    }
	}
      pReferredD = pReferredD->ReNext;
    }
}

/*----------------------------------------------------------------------
  ApplyTransmitRules

  If element pEl (belonging to document pDoc) is a reference that
  includes an external document, apply all Transmit rules that
  assign a value to attributes of the included document.
  ----------------------------------------------------------------------*/
void                ApplyTransmitRules (PtrElement pEl, PtrDocument pDoc)
{
  PtrPSchema          pPSch;
  PtrSSchema          pSS;
  TransmitElem       *pTransR;
  int                 entry;
  int                 rule;
  int                 srcNumType;
  PtrElement          pSrcEl;
  int                 counter;
  Counter            *pCounter;

  if (pEl != NULL)
    if (pEl->ElStructSchema->SsRule->SrElem[pEl->ElTypeNumber - 1]->SrRefImportedDoc)
      {
	/* it's an inclusion of an external document */
	/* search the presentation schema that applies to the element */
	SearchPresSchema (pEl, &pPSch, &entry, &pSS, pDoc);
	/* does this schema contain a Transmit rule that transmits the */
	/* contents of some element to documents of the type of the */
	/* included document ? */
	/* scans all Transmit rule in the schema */
	for (rule = 0; rule < pPSch->PsNTransmElems; rule++)
	  {
	    /* pTransR: pointer to a Transmit rule */
	    pTransR = &(pPSch->PsTransmElem[rule]);
	    if (pTransR->TeTargetDoc == entry)
	      /* this Transmit rule acts on a document of the type of */
	      /* the included document */
	      {
		/* what element type does transmit its contents? */
		for (srcNumType = 0; srcNumType < pSS->SsNRules &&
		       pPSch->PsElemTransmit->Num[srcNumType] != rule + 1; srcNumType++);
		if (pPSch->PsElemTransmit->Num[srcNumType] == rule + 1)
		  {
		    /* elements of type srcNumType+1 transmit their */
		    /* content to the documents of interest */
		    /* Search an element of that type in the document */
		    pSrcEl = FwdSearchTypedElem (pDoc->DocDocElement,
						 srcNumType + 1, pSS, NULL);
		    if (pSrcEl != NULL)
		      /* apply the Transmit rule to the element found */
		      TransmitElementContent (pSrcEl, pDoc,
			    pTransR->TeTargetAttr, pTransR->TeTargetDoc, pSS);
		  }
	      }
	  }
	/* does the presentation schema contain counters that transmit */
	/* their value to some attribute in the included document? */
	/* scan all counters defined in the presentation schema */
	for (counter = 0; counter < pPSch->PsNCounters; counter++)
	  {
	    pCounter = &pPSch->PsCounter[counter];
	    /* scan all Transmit rules for that counter */
	    for (rule = 0; rule < pCounter->CnNTransmAttrs; rule++)
	      if (pCounter->CnTransmSSchemaAttr[rule] == entry)
		{
		  /* this Transmit rule transmit the counter value to */
		  /* a document of the type of the included document */
		  /* apply the Transmit rule */
		  TransmitCounterVal (pEl, pDoc, pCounter->CnTransmAttr[rule], counter + 1, pPSch, pSS);
		}
	  }
      }
}

/*----------------------------------------------------------------------
  RepApplyTransmitRules

  Apply all Transmit rules to all elements of the same type as
  pTransmEl that follow element pEl in document pDoc.
  ----------------------------------------------------------------------*/
void  RepApplyTransmitRules (PtrElement pTransmEl, PtrElement pEl,
			     PtrDocument pDoc)
{
  if (pTransmEl != NULL)
    while (pEl != NULL)
      {
	pEl = FwdSearchTypedElem (pEl, pTransmEl->ElTypeNumber,
				  pTransmEl->ElStructSchema, NULL);
	ApplyTransmitRules (pEl, pDoc);
      }
}
