/*
    Copyright (C) 2006-2009 Fons Adriaensen <fons@kokkinizita.net>
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include "styles.h"
#include "global.h"
#include "confwin.h"



static const char *_r_scale_text [3] = { "N3D", "SN3D", "Furse-Malham" };
static const char *_r_delcm_text [1] = { "Delay comp." };
static const char *_r_levcm_text [1] = { "Gain comp."  };
static const char *_r_nfecm_text [3] = { "None", "On inputs", "On outputs" };

static int dx0 [2] = { 70, 0 };
static int nk0 [2] = { 11, 0 };
static int dx1 [3] = { 40, 50, 0 };
static int nk1 [3] = { 3, 7, 0 };
static int dx2 [2] = { 70, 0 };
static int nk2 [2] = { 11, 0 };
static int dx4 [2] = { 208, 0 };
static int nk4 [2] = { 63, 0 };
static int dy  [2] = { 19, 0 };

static const char *_h_spkr_text  [4] = { "ID", "Dist", "Azim", "Elev" };
static const char *_h_matr_ord30 [7] = { "W", "X", "Y", "U", "V", "P", "Q" };
static const char *_h_matr_ord31 [8] = { "W", "X", "Y", "Z", "U", "V", "P", "Q" };
static const char *_h_matr_ord22 [9] = { "W", "X", "Y", "Z", "R", "S", "T", "U", "V" };



Confwin::Confwin (X_rootwin *parent, X_callback *xcbh, X_resman *xres, int xp, int yp) :
    X_window (parent, xp, yp, XSIZE, YSIZE, XftColors [C_MAIN_BG]->pixel),
    _xcbh (xcbh),
    _conf (0),
    _r_scale (this, 0),
    _r_ipscl (this, this),
    _r_delay (this, this),
    _r_level (this, this),
    _r_nfeff (this, this),
    _focus (0),
    _l_spkr (0),
    _l_conn (0),
    _l_deco_lf (0),
    _l_deco_hf (0),
    _l_gain_lf (0),
    _l_gain_hf (0),
    _h_spkr (0),
    _h_conn (0),
    _h_matr_lf (0),
    _h_matr_hf (0),
    _m_spkr (0),
    _m_conn (0),
    _m_gain_lf (0),
    _m_gain_hf (0),
    _m_matr_lf (0),
    _m_matr_hf (0),
    _m_last (0)
{
    X_hints   H;
    
    _atom = XInternAtom (dpy (), "WM_DELETE_WINDOW", True);
    XSetWMProtocols (dpy (), win (), &_atom, 1);
    _atom = XInternAtom (dpy (), "WM_PROTOCOLS", True);

    H.position (xp, yp);
    x_apply (&H);
    x_set_title ("AmbDec - Configuration");

    _typewin = new Typewin (parent, this, xres, 300, 300);
    _filewin = new Filewin (parent, this, xres);

    Bst0.size.x = 70;
    Bst0.size.y = 18;
    _b_load = new X_tbutton (this, this, &Bst0,   0, 0, "Load",   0, B_LOAD);   
    _b_load->x_map ();
    _b_save = new X_tbutton (this, this, &Bst0,  70, 0, "Save",   0, B_SAVE);   
    _b_save->x_map ();
    _b_new  = new X_tbutton (this, this, &Bst0, 140, 0, "New",    0, B_NEW);   
    _b_new->x_map ();
    _b_canc = new X_tbutton (this, this, &Bst0, 210, 0, "Cancel", 0, B_CANC);   
    _b_canc->x_map ();
    _b_appl = new X_tbutton (this, this, &Bst0, 280, 0, "Apply",  0, B_APPL);   
    _b_appl->x_map ();
    _t_stat = new X_textip (this, 0, &Tst1, 360,  0, 300, 18, 63);
    _t_stat->x_map ();
    (new X_textln (this, &Tst2,  20, 30, 85, 18, "Description", -1))->x_map ();
    _t_comm = new X_textip (this, this, &Tst0, 110, 30, 400, 17, 63);
    _t_comm->x_map ();

    (new X_textln (this, &Tst2,  20, 60, 115, 18, "Matrix scaling", -1))->x_map ();
    (new X_textln (this, &Tst2, 140, 60, 115, 18, "Input scaling",  -1))->x_map ();
    (new X_textln (this, &Tst2, 260, 60, 115, 18, "Speaker distance", -1))->x_map ();
    (new X_textln (this, &Tst2, 380, 60, 115, 18, "Near-field comp.", -1))->x_map ();
    _r_scale.init (&Bst1, &Tst1, _r_scale_text, 3,  20,  82, 115, 18, 0);
    _r_ipscl.init (&Bst1, &Tst1, _r_scale_text, 3, 140,  82, 115, 18, CB_OPT_IPSCL);
    _r_delay.init (&Bst1, &Tst1, _r_delcm_text, 1, 260,  82, 115, 18, CB_OPT_DELCM);
    _r_level.init (&Bst1, &Tst1, _r_levcm_text, 1, 260, 100, 115, 18, CB_OPT_LEVCM);
    _r_nfeff.init (&Bst1, &Tst1, _r_nfecm_text, 3, 380,  82, 115, 18, CB_OPT_NFECM);
    _l_xfreq = new X_textln (this, &Tst2, 500, 60, 120, 18, "Crossover freq.", -1);
    _t_xfreq = new X_textip (this, this, &Tst0, 505, 82, 70, 17, 11);
    _l_ratio = new X_textln (this, &Tst2, 500, 105, 120, 18, "H/L balance (dB)", -1);
    _t_ratio = new X_textip (this, this, &Tst0, 505, 127, 70, 17, 11);

    _b_con = new X_tbutton (this, this, &Bst0,   5, YSIZE - 22, "Speakers", 0, B_CON);   
    _b_con->x_set_win_gravity (SouthWestGravity);
    _b_con->x_map ();
    _b_dec = new X_tbutton (this, this, &Bst0,  75, YSIZE - 22, "Decoder",  0, B_DEC);   
    _b_dec->x_set_win_gravity (SouthWestGravity);
    _b_lfd = new X_tbutton (this, this, &Bst0,  75, YSIZE - 22, "LF Dec",   0, B_LFD);   
    _b_lfd->x_set_win_gravity (SouthWestGravity);
    _b_hfd = new X_tbutton (this, this, &Bst0, 145, YSIZE - 22, "HF Dec",   0, B_HFD);   
    _b_hfd->x_set_win_gravity (SouthWestGravity);
    _b_pol = new X_tbutton (this, this, &Bst0, 245, YSIZE - 22, "Polar",    0, B_POL);   
    _b_pol->x_set_win_gravity (SouthWestGravity);
    _b_vec = new X_tbutton (this, this, &Bst0, 315, YSIZE - 22, "Vector",   0, B_VEC);   
    _b_vec->x_set_win_gravity (SouthWestGravity);
//    _b_pol->x_map ();
//    _b_vec->x_map ();

    _matrwin = new X_window (this, 5, 158, 10, 10, XftColors [C_TEXT_BG]->pixel);
}

 
Confwin::~Confwin (void)
{
}

 
void Confwin::handle_event (XEvent *E)
{
    switch (E->type)
    {
    case ClientMessage: 
	clmesg ((XClientMessageEvent *) E);
	break;
    }
}


void Confwin::clmesg (XClientMessageEvent *E)
{
    if (E->message_type == _atom) close ();
}


void Confwin::open (AD_conf *C)
{
    if (_conf) delete C;
    else
    {
        conf2disp (_conf = C);
	_b_save->set_stat (0);
	_b_appl->set_stat (0);
	_t_stat->set_text (_conf->_fname);
    }
    x_mapraised ();
}


void Confwin::close (void)
{
    _typewin->x_unmap ();
    x_unmap ();
    delete _conf;
    _conf = 0;
}


void Confwin::handle_callb (int type, X_window *W, XEvent *E)
{
    X_button   *B;
    X_textip   *T;
    X_textmat  *M;

    switch (type)
    {
    case X_callback::BUTTON | X_button::PRESS:
    {
        B = (X_button *) W;
        switch (B->cbid ())
	{
	case B_CON:
	    showcon ();
	    break;
	case B_DEC:
	    showdec ();
	    break;
	case B_LFD:
	    showlfd ();
	    break;
	case B_HFD:
	    showhfd ();
	    break;
	case B_NEW:
	    _typewin->x_map ();
	    break;   
	case B_LOAD:
  	    _filewin->load_dialog (".", "ambdec", "");
	    break;
	case B_SAVE:
	    if (disp2conf (_conf)) _t_stat->set_text ("Configuration error");
	    else _filewin->save_dialog (".", "ambdec", _conf->_fname);
	    break;
	case B_CANC:
	    if (_conf)
	    {
		delete _conf;
		_conf = 0;
		_xcbh->handle_callb (CB_CONF_CANC, 0, 0);
		_b_appl->set_stat (0);
	    }
	    break;
        case B_APPL:
	    if (_conf->_state)
	    {
		if (disp2conf (_conf)) _t_stat->set_text ("Configuration error");
		else
	        {
                    _t_stat->set_text (_conf ? _conf->_fname : "");
		    _xcbh->handle_callb (CB_CONF_APPL, 0, 0);
		    conf2disp (_conf);
		    _conf->_state = 0;
		    _b_appl->set_stat (0);
		    if (_focus) _focus->callb_modified ();
		}
	    }
	    break;
	}
	break;
    }
    case CB_OPT_IPSCL:    
    case CB_OPT_NFECM:    
    case CB_OPT_DELCM:    
    case CB_OPT_LEVCM:
    {
	_conf->_state |= AD_conf::ST_OPTS;
	_b_save->set_stat (1);
	_b_appl->set_stat (1);
	break;
    }
    case X_callback::TEXTIP | X_textip::BUT:
    {
        T = (X_textip *) W;
        T->enable ();
	T->callb_modified ();
	_focus = T;
	break;
    }
    case X_callback::TEXTIP | X_textip::MODIF:
    {
	_b_save->set_stat (1);
	_b_appl->set_stat (1);
	_conf->_state |= AD_conf::ST_OPTS | AD_conf::ST_MATR;
	break;
    }
    case X_textmat::CB_BASE | X_textmat::BUT:
    {
        M = (X_textmat *) W;
	if (M == _h_matr_hf)
	{
            if (M->mod () & ControlMask) copytohf ();
	}
	break;
    }
    case X_textmat::CB_BASE | X_textmat::SEL:
    {
        M = (X_textmat *) W;
	if (_m_last && (M != _m_last) && (M->mod () & ShiftMask))
	{
	    M->set_text (M->row (), M->col (), _m_last->text (_m_last->row (), _m_last->col ()));
        }
	_m_last = M;
        break;
    }
    case X_textmat::CB_BASE | X_textmat::MOD:
    {
        M = (X_textmat *) W;
	M->set_modif (M->row (), M->col ()); 
	_b_save->set_stat (1);
	if (M == _m_spkr)
	{
	    if (M->col () == 0)	_conf->_state |= AD_conf::ST_OUTS;
	    if (M->col () == 1) _conf->_state |= AD_conf::ST_OPTS;
	}
        else if (M == _m_conn) _conf->_state |= AD_conf::ST_OUTS;
	else _conf->_state |= AD_conf::ST_MATR;
        if (_conf->_state) _b_appl->set_stat (1);
        break;
    }
    case CB_FILE_LOAD:
    {	
	chdir (_filewin->wdir ());
	if (_conf->load (_filewin->file ()))
	{
	    _b_save->set_stat (0);
            _t_stat->set_text ("LOAD FAILED");
	}
	else
	{
	    conf2disp (_conf);
	    _b_save->set_stat (0);
	    _b_appl->set_stat (1);
	    _t_stat->set_text (_conf->_fname);
	    _conf->_state = AD_conf::ST_ALL;
	}
	break;
    }
    case CB_FILE_SAVE:
    {
	chdir (_filewin->wdir ());
        if (_conf->save (_filewin->file ()))
	{
            _t_stat->set_text ("Save failed");
	}
	else
	{
	    _b_save->set_stat (0);
	    _t_stat->set_text (_conf->_fname);
	}
	break;
    }
    case CB_CONF_NEW:
    {
	newconf (_conf);
	_b_save->set_stat (1);
	_b_appl->set_stat (1);
	break;
    }
    default:
	;
    }
}


void Confwin::newconf (AD_conf *C)
{
    C->_state = AD_conf::ST_ALL;
    C->_fname [0] = 0;
    _t_stat->set_text ("");
    C->reset ();
    C->_dec.h_ord = _typewin->h_ord ();
    C->_dec.v_ord = _typewin->v_ord ();
    C->_dec.nband = _typewin->nband ();
    C->_dec.nspkr = _typewin->nspkr ();
    C->_dec.scale = _typewin->coeff ();
    C->default_label ();
    C->default_jackp ();
    conf2disp (C);
}


int Confwin::conf2disp (AD_conf *C)
{
    int      i;
    char     s [16];
    Speaker  *S;

    initwin (C->_dec.h_ord, C->_dec.v_ord, C->_dec.nband, C->_dec.nspkr);
  
    _r_scale.set_stat (C->_dec.scale);
    _r_ipscl.set_stat (C->_opt.scale);
    _r_delay.set_stat (C->_opt.delay); 
    _r_level.set_stat (C->_opt.level);
    _r_nfeff.set_stat (C->_opt.nfeff);
    sprintf (s, "%4.0lf", C->_opt.xfreq);
    _t_xfreq->set_color (Tst0.color.normal.bgnd, Tst0.color.normal.text);
    _t_xfreq->set_text (s);
    sprintf (s, "%4.1lf", C->_opt.ratio);
    _t_ratio->set_color (Tst0.color.normal.bgnd, Tst0.color.normal.text);
    _t_ratio->set_text (s);
    _t_comm->set_text (C->_descr);

    for (i = 0, S = C->_speakers; i < C->_dec.nspkr; i++, S++)
    {
	_m_spkr->set_text (i, 0, S->_label);
	sprintf (s, "%4.2lf", S->_r);
	_m_spkr->set_text (i, 1, s);
	sprintf (s, "%3.1lf", S->_a);
	_m_spkr->set_text (i, 2, s);
        sprintf (s, "%3.1lf", S->_e);
        _m_spkr->set_text (i, 3, s);
	_m_conn->set_text (i, 0, S->_jackp);
    }	

    if (_m_matr_lf)
    {
	matr2disp (C, C->_lfmatrix, _m_matr_lf);
	gain2disp (C, C->_lfgain, _m_gain_lf);
    }
    if (_m_matr_hf)
    {
	matr2disp (C, C->_hfmatrix, _m_matr_hf);
	gain2disp (C, C->_hfgain, _m_gain_hf);
    }

    return 0;
}


int Confwin::matr2disp (AD_conf *C, Matrow *R, X_textmat *M)
{
    int  i, j;
    char s [16];

    for (i = 0; i < C->_dec.nspkr; i++, R++)
    {
	j = 0;
	sprintf (s, "%6.4lf", R->_w);
	M->set_text (i, j++, s);
	sprintf (s, "%6.4lf", R->_x);
	M->set_text (i, j++, s);
	sprintf (s, "%6.4lf", R->_y);
	M->set_text (i, j++, s);
	if (C->_dec.v_ord >= 1)
	{
	    sprintf (s, "%6.4lf", R->_z);
  	    M->set_text (i, j++, s);
	}
	if (C->_dec.h_ord >= 2)
	{
	    if (C->_dec.v_ord == 2)
	    {
		sprintf (s, "%6.4lf", R->_r);
		M->set_text (i, j++, s);
		sprintf (s, "%6.4lf", R->_s);
		M->set_text (i, j++, s);
		sprintf (s, "%6.4lf", R->_t);
		M->set_text (i, j++, s);
	    }
	    sprintf (s, "%6.4lf", R->_u);
	    M->set_text (i, j++, s);
	    sprintf (s, "%6.4lf", R->_v);
	    M->set_text (i, j++, s);
	}
	if (C->_dec.h_ord >= 3)
	{
	    sprintf (s, "%6.4lf", R->_p);
	    M->set_text (i, j++, s);
	    sprintf (s, "%6.4lf", R->_q);
	    M->set_text (i, j++, s);
	}
    }

    return 0;
}


int Confwin::gain2disp (AD_conf *C, float *G, X_textmat *M)
{
    int  i;
    char s [16];

    for (i = 0; i < C->_dec.h_ord + 1; i++)
    {
	sprintf (s, "%6.4lf", G [i]);
	M->set_text (0, i, s);
    }

    return 0;
}


int Confwin::disp2conf (AD_conf *C)
{
    int      i;
    float    v;
    char     s [64];
    bool     err;
    Speaker  *S;

    C->_opt.scale = _r_ipscl.stat ();
    C->_opt.delay = _r_delay.stat (); 
    C->_opt.level = _r_level.stat ();
    C->_opt.nfeff = _r_nfeff.stat ();
    strcpy (C->_descr, _t_comm->text ());

    err = false;
    if (C->_dec.nband == 2)
    {
	if ((sscanf (_t_xfreq->text (), "%f", &v) != 1) || (v < 50) || (v > 5000))
	{
	    _t_xfreq->set_color (Mst1.color.error.bg, Mst1.color.error.fg);
	    err = true;
	}
	else C->_opt.xfreq = (int) v;
	if ((sscanf (_t_ratio->text (), "%f", &v) != 1) || (v < -30) || (v > 30))
	{
	    _t_ratio->set_color (Mst1.color.error.bg, Mst1.color.error.fg);
	    err = true;
	}
	else C->_opt.ratio = v;
    }

    _m_spkr->deselect ();
    for (i = 0, S = C->_speakers; i < C->_dec.nspkr; i++, S++)
    {
	if (sscanf (_m_spkr->text (i, 0), "%s", s) != 1)
	{
	    _m_spkr->set_error (i, 0);
	    err = true;
	}
	else strcpy (S->_label, s);
	if ((sscanf (_m_spkr->text (i, 1), "%f", &v) != 1) || (v < 0.5f))
	{
	    _m_spkr->set_error (i, 1);
	    err = true;
	}
	else S->_r = v;
	if ((sscanf (_m_spkr->text (i, 2), "%f", &v) != 1) || (v < -360) || (v > 360)) 
	{
	    _m_spkr->set_error (i, 2);
	    err = true;
	}
	else S->_a = v;
        if ((sscanf (_m_spkr->text (i, 3), "%f", &v) != 1) || (v < -90) || (v > 90)) 
	{
	    _m_spkr->set_error (i, 3);
   	    err = true;
	}
	else S->_e = v;
	if (sscanf (_m_conn->text (i, 0), "%s", s) == 1) strcpy (S->_jackp, s);
	else strcpy (S->_jackp, "");
    }		

    if (_m_matr_lf)
    {
	matr2conf (C, C->_lfmatrix, _m_matr_lf);
	gain2conf (C, C->_lfgain, _m_gain_lf);
    }	
    if (_m_matr_hf)
    {
	matr2conf (C, C->_hfmatrix, _m_matr_hf);
	gain2conf (C, C->_hfgain, _m_gain_hf);
    }

    return err ? 1 : 0;
}


int Confwin::matr2conf (AD_conf *C, Matrow *R, X_textmat *M)
{
    int   i, j;
    float v;

    for (i = 0; i < C->_dec.nspkr; i++, R++)
    {
	j = 0;
	R->_w = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
	R->_x = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
	R->_y = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
	R->_z = R->_r = R->_s = R->_t = R->_u = R->_v = 0; 
	if (C->_dec.v_ord >= 1)
	{
	    R->_z = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
	}
        if (C->_dec.h_ord >= 2)
	{
	    if (C->_dec.v_ord == 2)
	    {
		R->_r = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
		R->_s = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
		R->_t = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
	    }
            R->_u = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
            R->_v = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
	}
        if (C->_dec.h_ord >= 3)
	{
            R->_p = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
            R->_q = (sscanf (M->text (i, j++), "%f", &v) == 1) ? v : 0;
	}
    }

    return 0;
}


int Confwin::gain2conf (AD_conf *C, float *G, X_textmat *M)
{
    int   i;
    float v;

    G [0] = G [1] = G [2] = 0;
    for (i = 0; i < C->_dec.h_ord + 1; i++)
    {
	G [i] = (sscanf (M->text (0, i), "%f", &v) == 1) ? v : 0;
    }

    return 0;
}


void Confwin::copytohf (void)
{
    int r, c;

    for (r = 0; r < _m_matr_hf->nrow (); r++)
    {
	for (c = 0; c < _m_matr_hf->ncol (); c++)
        {
	    _m_matr_hf->set_text (r, c, _m_matr_lf->text (r, c));
	    _m_matr_hf->set_modif (r, c);
        }
    }
}


void Confwin::initwin (int h_ord, int v_ord, int nband, int nspkr)
{
    int         i, k, x1, x2, x3, x4, xm, xs, y0, y1, y2, y3, y4, ys;
    bool        twin;
    const char  **p;
    X_hints     H;

    delete _l_spkr;
    delete _l_conn;
    delete _l_deco_lf;
    delete _l_deco_hf;
    delete _l_gain_lf;
    delete _l_gain_hf;
    delete _h_spkr;
    delete _h_conn;
    delete _h_matr_lf;
    delete _h_matr_hf;
    delete _m_spkr;
    delete _m_conn;
    delete _m_matr_lf;
    delete _m_matr_hf;
    delete _m_gain_lf;
    delete _m_gain_hf;
    _m_last = 0;

    switch ((h_ord << 4) | v_ord)
    {
    case 0x10: k = 3; break;
    case 0x11: k = 4; break;
    case 0x20: k = 5; break;
    case 0x21: k = 6; break;
    case 0x22: k = 9; break;
    case 0x30: k = 7; break;
    case 0x31: k = 8; break;
    default: k = 0;	
    }

    switch (v_ord)
    {
    case 0: p = _h_matr_ord30; break;
    case 1: p = _h_matr_ord31; break;
    case 2: p = _h_matr_ord22; break;
    default: p = 0;	
    }

    _h_spkr = new X_textmat (_matrwin, this, &Mst1, 0, 0, 1,     4, dx1, dy, nk1);
    _m_spkr = new X_textmat (_matrwin, this, &Mst1, 0, 0, nspkr, 4, dx1, dy, nk1);
    _h_conn = new X_textmat (_matrwin, this, &Mst1, 0, 0, 1,     1, dx4, dy, nk4);
    _m_conn = new X_textmat (_matrwin, this, &Mst1, 0, 0, nspkr, 1, dx4, dy, nk4);
    _l_spkr = new X_textln (_matrwin, &Tst0, 0, 0, _m_spkr->xsize (), dy [0], "Speakers", 0);
    _l_conn = new X_textln (_matrwin, &Tst0, 0, 0, _m_conn->xsize (), dy [0], "Connections", 0);

    _m_gain_lf = new X_textmat (_matrwin, this, &Mst1, 0, 0, 1, h_ord + 1, dx0, dy, nk0);
    _h_matr_lf = new X_textmat (_matrwin, this, &Mst1, 0, 0, 1,     k, dx2, dy, nk2);
    _m_matr_lf = new X_textmat (_matrwin, this, &Mst1, 0, 0, nspkr, k, dx2, dy, nk2);
    _l_gain_lf = new X_textln (_matrwin, &Tst0, 0, 0, dx0 [0] - 5, dy [0], "G(order)", 1);

    if (nband == 2)
    {
	twin = (k <= 5);
        xm = 630;
        _m_gain_hf = new X_textmat (_matrwin, this, &Mst1, 0, 0, 1, h_ord + 1, dx0, dy, nk0);
        _h_matr_hf = new X_textmat (_matrwin, this, &Mst1, 0, 0, 1,     k, dx2, dy, nk2);
        _m_matr_hf = new X_textmat (_matrwin, this, &Mst1, 0, 0, nspkr, k, dx2, dy, nk2);
        _l_gain_hf = new X_textln (_matrwin, &Tst0, 0, 0, dx0 [0] - 5, dy [0], "G(order)", 1);
        _l_deco_lf = new X_textln (_matrwin, &Tst0, 0, 0, _m_matr_lf->xsize (), dy [0], "LF Decoder", 0);
        _l_deco_hf = new X_textln (_matrwin, &Tst0, 0, 0, _m_matr_hf->xsize (), dy [0], "HF Decoder", 0);
        _l_xfreq->x_map ();
        _t_xfreq->x_map ();
        _l_ratio->x_map ();
        _t_ratio->x_map ();
    }
    else
    {
        twin = false;
        xm = 520;
	_l_gain_hf = 0;
	_m_gain_hf = 0;
	_h_matr_hf = 0;
        _m_matr_hf = 0;
        _l_deco_lf = new X_textln (_matrwin, &Tst0, 0, 0, _m_matr_lf->xsize (), dy [0], "Decoder", 0);
	_l_deco_hf = 0;
        _l_xfreq->x_unmap ();
        _t_xfreq->x_unmap ();
        _l_ratio->x_unmap ();
        _t_ratio->x_unmap ();
    }

    xs = 20 + _h_spkr->xsize () + _h_matr_lf->xsize (); 
    if (twin) xs += _h_matr_hf->xsize ();
    if (xs < xm)
    {
        x1 = (xm - xs) / 2;
	xs = xm;
    }
    else x1 = 5;
    x4 = x3 = x2 = x1 + _h_spkr->xsize ();
    if (twin) x3 += _h_matr_lf->xsize ();
    y0 = 1;
    y1 = y0 + dy [0] + 3;
    y2 = y1 + dy [0] + 3;
    y3 = y2 + dy [0];
    y4 = y3 + nspkr * (dy [0] - 1) + 5;  
    ys = y4 + 190;

    _l_spkr->x_move (x1, y1);
    _h_spkr->x_move (x1, y2);
    _m_spkr->x_move (x1, y3);
    _l_conn->x_move (x4, y1);
    _h_conn->x_move (x4, y2);
    _m_conn->x_move (x4, y3);
    _h_spkr->set_align (0);
    _m_spkr->set_align (1);
    _h_conn->set_align (0);
    _m_conn->set_align (0);
    _l_deco_lf->x_move (x2, y0);
    _l_gain_lf->x_move (x2, y1);
    _m_gain_lf->x_move (x2 + dx0 [0] - 1, y1);
    _h_matr_lf->x_move (x2, y2);
    _m_matr_lf->x_move (x2, y3);
    _h_matr_lf->set_align (0);
    _m_gain_lf->set_align (1);
    _m_matr_lf->set_align (1);
    if (_l_deco_hf)
    {
        _l_deco_hf->x_move (x3, y0);
        _l_gain_hf->x_move (x3, y1);
        _m_gain_hf->x_move (x3 + dx0 [0] - 1, y1);
        _h_matr_hf->x_move (x3, y2);
	_m_matr_hf->x_move (x3, y3);
        _h_matr_hf->set_align (0);
        _m_gain_hf->set_align (1);
        _m_matr_hf->set_align (1);
    }

    for (i = 0; i < 4; i++) _h_spkr->set_text (0, i, _h_spkr_text [i]);
    for (i = 0; i < k; i++) _h_matr_lf->set_text (0, i, p [i]);
    _m_gain_lf->enable ();
    _m_matr_lf->enable ();
    if (_l_deco_hf)
    {
        for (i = 0; i < k; i++) _h_matr_hf->set_text (0, i, p [i]);
        _m_gain_hf->enable ();
        _m_matr_hf->enable ();
    }
    _h_conn->set_text (0, 0, "Destination port");
    _m_conn->enable ();

    _matrwin->x_resize (xs - 10, y4); 
    x_resize (xs, ys);
    H.size (xs, ys);
    H.maxsize (xs, ys);
    H.minsize (xs, ys);
    x_apply (&H); 

    _l_spkr->x_map ();
    _h_spkr->x_map ();
    _m_spkr->x_map ();
    if (_l_deco_hf && !twin)
    {
        _b_dec->x_unmap ();
        _b_lfd->x_map ();
        _b_hfd->x_map ();
        showlfd ();
    }
    else
    {
        _b_dec->x_map ();
        _b_lfd->x_unmap ();
        _b_hfd->x_unmap ();
        showdec ();
    }
    _matrwin->x_map ();
}


void Confwin::showcon ()
{
    _m_spkr->enable ();
    _l_conn->x_map ();
    _h_conn->x_map ();
    _m_conn->x_map ();
    _l_gain_lf->x_unmap ();
    _m_gain_lf->x_unmap ();
    _l_deco_lf->x_unmap ();
    _h_matr_lf->x_unmap ();
    _m_matr_lf->x_unmap ();
    if (_l_deco_hf)
    {
	_l_gain_hf->x_unmap ();
	_m_gain_hf->x_unmap ();
	_l_deco_hf->x_unmap ();
	_h_matr_hf->x_unmap ();
	_m_matr_hf->x_unmap ();
    }
}


void Confwin::showdec ()
{
    _m_spkr->disable ();
    _l_conn->x_unmap ();
    _h_conn->x_unmap ();
    _m_conn->x_unmap ();
    _l_gain_lf->x_map ();
    _m_gain_lf->x_map ();
    _l_deco_lf->x_map ();
    _h_matr_lf->x_map ();
    _m_matr_lf->x_map ();
    if (_l_deco_hf)
    {
	_l_gain_hf->x_map ();
	_m_gain_hf->x_map ();
	_l_deco_hf->x_map ();
	_h_matr_hf->x_map ();
	_m_matr_hf->x_map ();
    }
}


void Confwin::showlfd ()
{
    _m_spkr->disable ();
    _l_conn->x_unmap ();
    _h_conn->x_unmap ();
    _m_conn->x_unmap ();
    _l_gain_hf->x_unmap ();
    _m_gain_hf->x_unmap ();
    _l_deco_hf->x_unmap ();
    _h_matr_hf->x_unmap ();
    _m_matr_hf->x_unmap ();
    _l_gain_lf->x_map ();
    _m_gain_lf->x_map ();
    _l_deco_lf->x_map ();
    _h_matr_lf->x_map ();
    _m_matr_lf->x_map ();
}


void Confwin::showhfd ()
{
    _m_spkr->disable ();
    _l_conn->x_unmap ();
    _h_conn->x_unmap ();
    _m_conn->x_unmap ();
    _l_gain_lf->x_unmap ();
    _m_gain_lf->x_unmap ();
    _l_deco_lf->x_unmap ();
    _h_matr_lf->x_unmap ();
    _m_matr_lf->x_unmap ();
    _l_gain_hf->x_map ();
    _m_gain_hf->x_map ();
    _l_deco_hf->x_map ();
    _h_matr_hf->x_map ();
    _m_matr_hf->x_map ();
}

