//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2011-2012 Mikhail Fedotov <anyremote@mail.ru>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <dirent.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>

#define SERVER     "webserver/1.1"
#define PROTOCOL   "HTTP/1.1"
#define RFC1123FMT "%a, %d %b %Y %H:%M:%S GMT"

#include <glib.h>
#include "utils.h"
#include "conf.h"
#include "pr_web.h"

extern char tmp[MAXMAXLEN];
extern int  gotExitSignal;

GMutex      * mut  = NULL;
GAsyncQueue *q2web = NULL;
GThread     *mainThread = NULL;
int visits  =  0;                  // counts client connections
int clients =  0;                  // counts client connections

int port = 5050;
int sock = -1;
long cookie = 0;
int secure = NO_COOKIE;
int curForm;

gboolean runWeb = TRUE;
gboolean answerReady = FALSE;
gboolean askPassword = FALSE;
gboolean bottomlineSkin = FALSE;

char* confDir = NULL;

int screenSize = 240;  // min(width,heigth)

// Global data

// Predefined templates
#define HTTP_HEAD1 "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html4/strict.dtd\">\n<html>\n<head>\n<title>"
#define HTTP_HEAD2 "</title>\n"
#define HTTP_HEAD3 "<meta "
#define HTTP_HEAD4 " content=\"text/html; charset=utf-8\">\n"
#define HTTP_HEAD5 "</head>\n<body"
#define HTTP_HEAD6 "><center>"
#define HTTP_SCRIPT      "<script>\nfunction clearForms()\n{\n  var i;\n  for (i = 0; (i < document.forms.length); i++) {\n    document.forms[i].reset();\n  }\n}\n</script>\n"
#define HTTP_SCRIPT_CALL " onload=\"clearForms()\""

#define HTTP_HEAD_CSS_0  "<style type=\"text/css\">\n"
#define HTTP_HEAD_CSS_1  "table{\n\ttext-align:left;\n}\n"
#define HTTP_HEAD_CSS_B1 "body {\n\tcolor:#"
#define HTTP_HEAD_CSS_B2 "; \n\tbackground-color:#"
#define HTTP_HEAD_CSS_B3 ";\n}\n"
#define HTTP_HEAD_CSS_L1 "select {\n\tcolor:#"
#define HTTP_HEAD_CSS_L3 "; \n\twidth:"
#define HTTP_HEAD_CSS_BT1 "button {\n\tcolor:#"

#define HTTP_HEAD_CSS_99 "</style>\n"

#define HTTP_TABLE1 "\n<table border=\"1\" cellpadding=\""
#define HTTP_TABLE2 "\" cellspacing=\"2\">\n<tbody>\n<tr>\n"
//#define HTTP_TABLE3 "	<td style=\"vertical-align: top; text-align: center; width=5%;\">"
#define HTTP_TABLE3 "	<td>"

#define HTTP_MENU1 " [<a href=\""
#define HTTP_MENU2 ".menu\">"
#define HTTP_MENU3 "</a>] "

#define HTTP_LIST1 "<select name=\"list\" size=15>\n"

#define HTTP_FORM  "\n<form action=\"/\" method=\"post\">\n"
 
GString* httpEdit1; 
GString* httpEdit2; 
GString* httpEdit2p;
GString* httpEdit3; 

GString* httpTail;
  
// control form
GString* caption;
GString* title;
GString* status;
GString* volume;
gboolean useVolume = FALSE;
GString* font;
GString* fg;
GString* bg;
GString* icons[12];
GString* cover;
GString* window;
int iconPadding   = 0;
int iconSize      = 64;
int iconSizeScale = 64;

// list form
GString* ls_caption;
GString* ls_font;
GString* ls_fg;
GString* ls_bg;
GSList *ls_items = NULL;
int ls_selidx = 1;
	    
// text form
GString* tx_caption;
GString* tx_font;
GString* tx_fg;
GString* tx_bg;
GString* tx_text;

// edit field 
GString* ef_caption;
GString* ef_text;
GString* ef_label;

GSList * user_menu = NULL;

GSList * clientSockets = NULL;

static const char* i2string(int data,char* buf)
{
    sprintf(buf,"%d",data);
    return buf;
}	
static void stringItemFree(gpointer data)
{
    g_string_free(data,(((GString*) data)->str != NULL));
}

static void listItemFree(gpointer data)
{
    listItem* item = (listItem*) data;
    if (item->icon)   g_string_free(item->icon,   (item->icon->str   != NULL));
    if (item->string) g_string_free(item->string, (item->string->str != NULL));
    free(item); 
    item = NULL;
}

void slist_free_full(GSList *list, GDestroyNotify free_func)
{
        GSList *item = list;
	while(item) {
	    if (item->data) {
	        free_func(item->data);
	    }
	    item = g_slist_next(item);
	}
	g_slist_free(list);
}

static void sendToWebServer(wMessage *buf)
{
    if (q2web && buf) {
        g_mutex_lock(mut);
	DEBUG2("send to web server %d %s", buf->button, (buf->string ? buf->string->str : "no text"));
	g_async_queue_push(q2web,buf);
        g_mutex_unlock(mut);
    }
}

static const char *get_mime_type(const char *name)
{
    char *ext = strrchr(name, '.');
    if (!ext) return NULL;
  
    if (strcmp(ext, ".html") == 0 || 
        strcmp(ext, ".htm") == 0)  return "text/html";
    if (strcmp(ext, ".jpg") == 0 || 
        strcmp(ext, ".jpeg") == 0) return "image/jpeg";
    if (strcmp(ext, ".gif") == 0)  return "image/gif";
    if (strcmp(ext, ".png") == 0)  return "image/png";
    if (strcmp(ext, ".css") == 0)  return "text/css";
    if (strcmp(ext, ".au") == 0)   return "audio/basic";
    if (strcmp(ext, ".wav") == 0)  return "audio/wav";
    if (strcmp(ext, ".avi") == 0)  return "video/x-msvideo";
    if (strcmp(ext, ".mpeg") == 0 || 
	strcmp(ext, ".mpg") == 0)  return "video/mpeg";
    if (strcmp(ext, ".mp3") == 0)  return "audio/mpeg";
    return NULL;
}

static int sendData(int fd, const char *s)
{
    int bytes_sent = send(fd,s,strlen(s),0);
    if (bytes_sent != strlen(s)) {
       ERROR2("[WS]: Error on send data: sent %d from %d", bytes_sent, (int) strlen(s));
       return -1;
    }
    return 0;
}

void sendCookie(int fd) 
{
    if (secure == NO_COOKIE) return;
    
    char b[32];
    sendData(fd, "Set-Cookie: anyremote_id=");
    cookie = random();
    sprintf(b,"%ld",cookie);
    sendData(fd, b);
    sendData(fd, "\r\n");
    
    INFO2("[WS]: sendCookie %s",b);
    secure = COOKIE_SENT;
}

static void sendHeaders(int fd, int status, 
                        const char *title, const char *extra, const char *mime, 
                        int length, time_t date, int cookie)
{
    char f[4096];
    //INFO2("[WS]: sendHeaders %s",title);

    sprintf(f, "%s %d %s\r\n", PROTOCOL, status, title);
    sendData(fd, f);
  
    sprintf(f, "Server: %s\r\n", SERVER);
    sendData(fd, f);
  
    time_t now = time(NULL);
    char timebuf[128];
    strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&now));
  
    sprintf(f, "Date: %s\r\n", timebuf);
    sendData(fd, f);
  
    if (extra) {
        sprintf(f, "%s\r\n", extra);
        sendData(fd, f);
    }
    if (mime) {
        sprintf(f, "Content-Type: %s\r\n", mime);
        sendData(fd, f);
    }
    
    sprintf(f, "Cache-Control: public, max-age=36000\r\n");
    sendData(fd, f);

    if (length >= 0) {
        sprintf(f, "Content-Length: %d\r\n", length);
        sendData(fd, f);
    }
    sendCookie(fd);
    
    if (date != -1)
    {
        strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&date));
        sprintf(f, "Last-Modified: %s\r\n", timebuf);
        sendData(fd, f);
    }
  
    sprintf(f, "Connection: close\r\n\r\n");
    sendData(fd, f);
}

static void sendError(int fd, int status, char *title, char *extra, char *text)
{
    char f[4096];
  
    INFO2("[WS]: sendError %d", status);
    sendHeaders(fd, status, title, extra, "text/html", -1, -1, 0);
  
    sprintf(f, "<HTML><HEAD><TITLE>%d %s</TITLE></HEAD>\r\n", status, title);
    sendData(fd, f);

    if (status != 304) { // 304  - Not Modified
    
	sprintf(f, "<BODY><H4>%d %s</H4>\r\n", status, title);
	sendData(fd, f);

	sprintf(f, "%s\r\n", text);
	sendData(fd, f);

	sprintf(f, "</BODY></HTML>");
	sendData(fd, f);
	
    } else {
	time_t now = time(NULL);
	char timebuf[128];
	strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&now));

	sprintf(f, "Date: %s\r\n", timebuf);
        sendData(fd, f);
    }
    
    
    sprintf(f, "\r\n");
    sendData(fd, f);
}

static void sendFile(int fd, char *path, struct stat *statbuf)
{
    char data[4096];
    int n,bytes_sent;

    INFO2("[WS]: sendFile %s",path);

    FILE *file = fopen(path, "r");
    if (!file) {
        ERROR2("[WS]: Access denied: %s", path);
        sendError(fd, 403, "Forbidden", NULL, "Access denied.");
  
    } else {
    
        int length = S_ISREG(statbuf->st_mode) ? statbuf->st_size : -1;
        sendHeaders(fd, 200, "OK", NULL, get_mime_type(path), length, statbuf->st_mtime, 0);

        while ((n = fread(data, 1, sizeof(data), file)) > 0) {
            //INFO2("read %d bytes from file",n);
            
            bytes_sent = send(fd,data,n,0);
            if (n != bytes_sent) {
                ERROR2("[WS]: Error on send file %s", path);
                break;
            }
        }
    
        fclose(file);
    }
    sendData(fd, "\r\n");
}
 
static void sendIcon(int fd, char *path) 
{
    INFO2("sendIcon %s", path);
    
    struct stat statbuf;
    
    GString* icon = g_string_new(confDir ? confDir : ".");
    g_string_append(icon,"/Icons/");
    
    char b[32];
    g_string_append(icon,i2string(iconSize,b));
    if (path[0] != '/') {
         g_string_append(icon,"/");
    }
    g_string_append(icon,path);
    
    INFO2("sendIcon full name %s", icon->str);
    
    if (stat(icon->str, &statbuf) >= 0) {
        sendFile(fd, icon->str, &statbuf);
    }
    g_string_free(icon, TRUE);
}

static void sendFavicon(int fd) 
{
    struct stat statbuf;
    
    GString* icon = g_string_new(confDir ? confDir : ".");
    g_string_append(icon,"/Icons/anyRemote.png");
    
    if (stat(icon->str, &statbuf) >= 0) {
        sendFile(fd, icon->str, &statbuf);
    }
    g_string_free(icon, TRUE);
}

char abuf[8];

static char* getAccessKey(int index)
{           
    if (index == 10) {
        abuf[0] = '*';
        abuf[1] = '\0';
    } else if (index == 11) {
        abuf[0] = '0';
        abuf[1] = '\0';
    } else if (index == 12) {
        abuf[0] = '#';
        abuf[1] = '\0';
    } else {
        sprintf(abuf,"%d",index);
    }
    return abuf;
}

void sendForm(int fd, GString* httpPageH, GString* httpPageT)
{
    sendData(fd, "HTTP/1.1 200 OK\r\nContent-type: text/html\r\nCache-Control: no-cache, must-revalidate\r\n");
    sendCookie(fd);
    sendData(fd, "\r\n");
    
    sendData(fd, httpPageH->str);
    //printf("%s",httpPageH->str);
    
    /*if refresh < 0:
    	rStr = "%s" % (uploadTmout)
    	GString* httpRate = g_string_new('HTTP-EQUIV=REFRESH CONTENT=\"' + rStr + ';URL=/\"');
	sendData(fd, httpRate->str);
	g_string_free(httpRate,  TRUE);
    */
    
    sendData(fd, httpPageT->str);
    //printf("%s",httpPageT->str);
}

static void addDefMenu()
{
    INFO2("[WS]: addDefMenu %d", curForm);
    switch (curForm) {
    
        case TX:
	case LI:
	    user_menu = g_slist_append(user_menu, g_string_new("Back"));
	    break;
	    
	case EF:
            user_menu = g_slist_append(user_menu, g_string_new("Back"));
            user_menu = g_slist_append(user_menu, g_string_new("Ok"));
	    break;
	    
        case CF:
        case WM:
	    break;
    }
}

static void freeList()
{
    // glib 2.28  
    //g_slist_free_full(ls_items, listItemFree);
    slist_free_full(ls_items, listItemFree);
    ls_items = NULL;
    ls_selidx = 1;
}

static void freeMenu()
{
    INFO2("[WS]: freeMenu %d", curForm);
    // glib 2.28  
    //g_slist_free_full(user_menu, stringItemFree);
    slist_free_full(user_menu, stringItemFree);
    user_menu = NULL;
}

static void setupDefMenu()
{
    INFO2("[WS]: setupDefMenu %d", curForm);
    freeMenu();
    addDefMenu();
}

static void addFormMenu(int formId, GString* page)
{
    GSList* list = user_menu;
    
    INFO2("[WS]: addFormMenu %d #%d", curForm, g_slist_length(list));
    switch (formId) {
    
        case CF:
        case TX:
        case WM:
            if (g_slist_length(list) > 0) {
                
                GString* menu = g_string_new("<HR>");
		
		//g_string_append(menu, "<div style=\"width:240px; overflow:auto;\">");
		
		while(list) {
		    
		    //INFO2("[WS]: addFormMenu add item ");
		    if (list->data && ((GString*) list->data)->str) {
			
			g_string_append(menu, HTTP_MENU1);
			g_string_append(menu, ((GString*) list->data)->str);
			g_string_append(menu, HTTP_MENU2);
			g_string_append(menu, ((GString*) list->data)->str);
			g_string_append(menu, HTTP_MENU3);

			//INFO2("[WS]: addFormMenu item %s", ((GString*) list->data)->str);
                    }
		    list = g_slist_next(list);
                }
		//g_string_append(menu, "</div>");
		g_string_append(menu, "<HR>");
 		g_string_append(page, menu->str);
	    }
	    break;
        
	case LI:
	case EF:
	
            if (g_slist_length(list) > 0) {
                
                GString* menu = g_string_new("");
		
		int i = 0;
		while(list) {
		    
		    g_string_append(menu, "<input name=\"action\" type=\"submit\" value=\"");
		    g_string_append(menu, ((GString*) list->data)->str);
		    g_string_append(menu, "\">\n");
		    
		    i++;
		    list = g_slist_next(list);
                }
		g_string_append(menu, "<HR>");
 		g_string_append(page, menu->str);
	    }
	    break;
    }
}

static GString* renderFormHead(int form, GString* caption, GString* fg, GString* bg) 
{
    GString* head = g_string_new(HTTP_HEAD1);
    g_string_append(head, caption->str);
    g_string_append(head, HTTP_HEAD2);
    if (form == LI) {
        g_string_append(head, HTTP_SCRIPT);
    }
    g_string_append(head, HTTP_HEAD3);
    g_string_append(head, HTTP_HEAD4);

    // Do some CSS styling
    g_string_append(head, HTTP_HEAD_CSS_0);
    //g_string_append(head, HTTP_HEAD_CSS_1);
    
    g_string_append(head, HTTP_HEAD_CSS_B1);
    g_string_append(head, fg->str); 
    g_string_append(head, HTTP_HEAD_CSS_B2);
    g_string_append(head, bg->str);
    g_string_append(head, HTTP_HEAD_CSS_B3);
    
    if (form == CF) {
	g_string_append(head, HTTP_HEAD_CSS_BT1);
	g_string_append(head, fg->str); 
	g_string_append(head, HTTP_HEAD_CSS_B2);
	g_string_append(head, bg->str);
        g_string_append(head, ";\n\tborder-style:none;\n}\n"); 
    }
    
    if (form == LI) {
        g_string_append(head, HTTP_HEAD_CSS_L1);
	g_string_append(head, fg->str);
        g_string_append(head, HTTP_HEAD_CSS_B2);
	g_string_append(head, bg->str);
        g_string_append(head, HTTP_HEAD_CSS_L3);
	char b[32];
        g_string_append(head, i2string(screenSize,b));
        g_string_append(head, "px");

        g_string_append(head, HTTP_HEAD_CSS_B3);
    }
    g_string_append(head, HTTP_HEAD_CSS_99);
    
    g_string_append(head, HTTP_HEAD5);
    if (form == LI) {
        g_string_append(head, HTTP_SCRIPT_CALL);
    }
    g_string_append(head, HTTP_HEAD6);
    
    return head;
}

static void renderControlForm(int fd)
{
    INFO2("[WS]: renderControlForm");
    
    GString* httpPageH = renderFormHead(CF,caption, fg, bg);
     
    GString* page = g_string_new("<");
    g_string_append(page, font->str); 
    g_string_append(page, "\n<p style=\"width:");
    char buf[8];
    g_string_append(page, i2string(screenSize,buf));
    g_string_append(page, "px; overflow:auto;\">");
    g_string_append(page, status->str); 
    g_string_append(page, "</p>");
    g_string_append(page, "</");
    g_string_append(page, font->str);
     
        
    if (bottomlineSkin && cover != NULL) {
       g_string_append(page, "\n<img src=\"");
       g_string_append(page,cover->str);
       g_string_append(page, "\" border=\"1\">");
    }

    g_string_append(page, HTTP_FORM);

    g_string_append(page, HTTP_TABLE1);
    g_string_append(page, i2string(iconPadding,buf));
    g_string_append(page, HTTP_TABLE2);

    int i = 0;
    int col = 1;
    while (i < 12) {
    
        if (bottomlineSkin && strcmp(icons[i]->str,"none") == 0) {
            break; // skip if empty icon specified	
	}
        
	g_string_append(page, HTTP_TABLE3);
	
	if (strcmp(icons[i]->str,"none") != 0) {
	
	    char maze[8];
	    
	    /*g_string_append(page, "<a href=\"");
	    g_string_append(page, i2string(i,maze)); 
	    g_string_append(page, ".key\" accesskey=\""); 
	    g_string_append(page, getAccessKey(i+1)); 
	    g_string_append(page, "\"><img src=\""); 
	    g_string_append(page, icons[i]->str);
	    g_string_append(page, ".png");
	    g_string_append(page, "\" border=\"0\"></a>");*/
	    
	    //<button accesskey="XX" type="submit" value="XX"> <img src="XX" alt="" style="vertical-align: middle">label_empty</button>
	    g_string_append(page, "<button name=\"action\" accesskey=\"");
	    g_string_append(page, getAccessKey(i+1));
	    g_string_append(page, "\" type=\"submit\" value=\"");
	    g_string_append(page, i2string(i+1,maze));
	    g_string_append(page, "\"> <img src=\"");
	    g_string_append(page, icons[i]->str);
	    g_string_append(page, ".png\" border=\"0\"");
	    g_string_append(page, " alt=\"\" style=\"vertical-align: middle\">");
	    g_string_append(page, "</button>");
	    
        }
	
        g_string_append(page, "</td>\n");
	if (!bottomlineSkin && col == 3) {
            g_string_append(page, "    \n</tr>\n<tr>\n");
            col = 0;
	}
        i++;
        col++;
    }
    
    g_string_append(page, "    </tr>\n</tbody>\n</table>");
    g_string_append(page, "    </form>\n");
            
    GString* httpPageT = g_string_new(page->str);
    
    g_string_append(httpPageT, "<");
    g_string_append(httpPageT, font->str);
    g_string_append(httpPageT, "\n<p style=\"width:");
    g_string_append(httpPageT, i2string(screenSize,buf));
    g_string_append(httpPageT, "px; overflow:auto;\">\n");
    g_string_append(httpPageT, title->str);
    g_string_append(httpPageT, "</p>");
    
    if (useVolume && strlen(volume->str) > 0) {
        g_string_append(httpPageT, "<p>");
        g_string_append(httpPageT, volume->str);
        g_string_append(httpPageT, "%</p>");
    }
    g_string_append(httpPageT, "</");
    g_string_append(httpPageT, font->str);

    addFormMenu(CF, httpPageT);
    
    g_string_append(httpPageT, httpTail->str);
        
    sendForm(fd, httpPageH, httpPageT);	        

    g_string_free(httpPageT, TRUE);
    g_string_free(page,      TRUE);
    g_string_free(httpPageH, TRUE);
}

static void renderTextForm(int fd)
{
    GString* httpPageH = renderFormHead(TX,tx_caption,tx_fg,tx_bg);
    
    GString* httpPageT = g_string_new("<");
    g_string_append(httpPageT, tx_font->str);
    g_string_append(httpPageT, tx_text->str);
    g_string_append(httpPageT, "</");
    g_string_append(httpPageT, tx_font->str);

    addFormMenu(TX, httpPageT);
                   
    g_string_append(httpPageT, httpTail->str);
    
    sendForm(fd, httpPageH, httpPageT);

    g_string_free(httpPageT, TRUE);
    g_string_free(httpPageH, TRUE);
}

static void renderListForm(int fd)
{
    GString* httpPageH = renderFormHead(LI,ls_caption,ls_fg,ls_bg);
    
    GString* page = g_string_new("<");
    g_string_append(page, ls_font->str);
    g_string_append(page, HTTP_FORM);
    g_string_append(page, HTTP_LIST1);

    int idx = 0;
    GSList* list = ls_items;
    char num[16];
    
    while (list) {
    
        sprintf(num,"%d", idx);
		
	g_string_append(page, "<option value=\"");
	g_string_append(page, num);
	g_string_append(page, "\"");
	if (idx == ls_selidx - 1) {
	    g_string_append(page, " selected ");
	}
	g_string_append(page, ">");
	g_string_append(page, ((listItem*) list->data)->string->str);
	g_string_append(page, "</option>\n");

        list = g_slist_next(list);
	idx++;
    }

    GString* httpPageT = g_string_new(page->str);
    g_string_append(httpPageT, "</select>\n<hr>");
    
    addFormMenu(LI, httpPageT);
                   
    g_string_append(httpPageT, "</form>\n</");
    g_string_append(httpPageT, ls_font->str);
    g_string_append(httpPageT, "\n");
    g_string_append(httpPageT, httpTail->str);
    
    sendForm(fd, httpPageH, httpPageT);

    g_string_free(httpPageT, TRUE);
    g_string_free(page,      TRUE);
    g_string_free(httpPageH, TRUE);
}

static void renderWmanForm(int fd)
{
    INFO2("[WS]: renderWmanForm");
 
    GString* httpPageH = renderFormHead(WM,caption,fg,bg);
    
    GString* httpPageT = g_string_new("<");
    g_string_append(httpPageT, font->str);
    g_string_append(httpPageT, title->str);
    g_string_append(httpPageT, "</");
    g_string_append(httpPageT, font->str);
        
    if (window != NULL) {
       g_string_append(httpPageT, "\n<img src=\"");
       g_string_append(httpPageT,window->str);
       g_string_append(httpPageT, "\" border=\"1\">");
    }

    addFormMenu(WM, httpPageT);

    g_string_append(httpPageT, httpTail->str);

    sendForm(fd, httpPageH, httpPageT);

    g_string_free(httpPageT, TRUE);
    g_string_free(httpPageH, TRUE);
}

static void renderEditForm(int fd)
{
    INFO2("[WS]: renderEditForm askpass=%d", askPassword);
     
    GString* httpPageH = renderFormHead(EF,ef_caption,fg,bg);
    
    GString* httpPageT = g_string_new("<");
    g_string_append(httpPageT, font->str);
    g_string_append(httpPageT, httpEdit1->str);
    g_string_append(httpPageT, ef_label->str);
    
    if (askPassword) {
        g_string_append(httpPageT, httpEdit2->str);
    } else {
        g_string_append(httpPageT, httpEdit2p->str);
    }
    
    g_string_append(httpPageT, ef_text->str);
    g_string_append(httpPageT, httpEdit3->str);
 
    addFormMenu(EF, httpPageT);

    g_string_append(httpPageT, "</form>\n</");
    g_string_append(httpPageT, font->str);
    g_string_append(httpPageT, "\n");
    g_string_append(httpPageT, httpTail->str);

    sendForm(fd, httpPageH, httpPageT);

    g_string_free(httpPageT, TRUE);
    g_string_free(httpPageH, TRUE);
}

static void renderCurForm(int client, int fd) 
{
     int f = curForm;
     INFO2("[WS]: (%d) renderCurForm %d", client, curForm);
     
     switch (f) {

         case TX:
	     renderTextForm(fd);
 	     return;
	 
         case LI:
	     renderListForm(fd);
 	     return;
         
	 case EF:
	     renderEditForm(fd);
 	     return;
         
	 case FM:
 	     return;
         
	 case WM:
	     renderWmanForm(fd);
 	     return;
         
         //case CF: 
	 default:
	     renderControlForm(fd);
 	     return;
    }
}

long parseCookie(char* buffer)
{
    //DEBUG2("[WS]: parseCookie line: %s", buffer);
    if (!buffer) return 0;

    char* start = buffer;
    char* startc = NULL;
    
    while ((start = strstr(start, "anyremote_id="))) {
        startc = start;
        start += 13;  // size("anyremote_id=") == 12
    
    }
    if (!startc) return 0;
    startc += 13;
    
    //DEBUG2("[WS]: parseCookie start: %s", startc);
    
    start = startc;
    while (startc && isdigit(*startc)) {
        startc++;
    }
    *startc = '\0';
    //DEBUG2("[WS]: parseCookie number: %s", start);
    
    return atol(start);
}

gpointer serveRequest(gpointer data)
{
    char buf[4096];
    //char f[4096];
    char *method = NULL;
    char *path   = NULL;
    struct stat statbuf;
    //char pathbuf[4096];
    //int len;
    //int bytes_sent;
    int fd = *((int*) data);
    free(data);
    int tvisits;
    
    INFO2("[WS]: process request handler");
    
    g_mutex_lock(mut);
    tvisits = ++visits;
    clients++;
    INFO2("[WS]: (%d) process request (%d)", tvisits, clients);
    g_mutex_unlock(mut);
    
    /*if (answerReady == FALSE) {
        INFO2("[WS]: (%d) drop request (%d)", tvisits, clients);
        sendError(fd, 304, "Please wait", NULL, "Please wait.");
	return (void*) -1;
    }*/
  
    int wasRead = read(fd, buf, sizeof(buf)-1);
    if (wasRead<0) return (void*) -1;
    buf[wasRead] = '\0';

    INFO2("[WS]: (%d) URL: %s", tvisits, buf);
    
    char * cookieLine = NULL;
    
    char* firstLine = strtok(buf, "\r");
    
    int rq = RQ_UNKNOWN;
    if (!firstLine) {
        INFO2("[WS]: non-valid request");
        return (void*) -1;
    }
    
    if (strstr(firstLine,"POST ")) {
        rq = RQ_POST;
    } else if (strstr(firstLine,"GET ")) {
        rq = RQ_GET;
    }
    
    char * line       = NULL;
    while ((line = strtok(NULL, "\r"))) {
	if (strstr(line,"Cookie:")) {
            cookieLine = line;
	}
	if (rq == RQ_POST) {
	    path = line;  // it will be last one
	}
    }
    
    method = strtok(firstLine, " "); // need to parse path for GET request
    if (rq == RQ_GET) {
        path   = strtok(NULL,      " ");
    }
    
    long c = parseCookie(cookieLine);
    
    INFO2("[WS]: (%d) Path: >%s<", tvisits, path);
    INFO2("[WS]: (%d) Cookie: %ld", tvisits, c);
    
    char* p = NULL;
    
    gboolean wait    = FALSE;
    gboolean allowed = TRUE;
    
    if (secure == COOKIE_SENT) {
    
	INFO2("[WS]: (%d) secure mode, test the cookie", tvisits);
	
	if (c != cookie) {
	    INFO2("[WS]: (%d) wrong cookie (wait for %ld)", tvisits, cookie);
	    
            wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
            wm->button = -1;
            wm->string = g_string_new("_DISCONNECT_");
	    sendToWebServer(wm);
	    
	    wait    = TRUE;
	    secure  = NEED_SEND_COOKIE;
	    allowed = FALSE;
	} else {
	    INFO2("[WS]: (%d) cookie OK (%ld)", tvisits, cookie);
	}
    }
    
    if (allowed) {
    
	if (rq == RQ_UNKNOWN) {

            ERROR2("[WS]: (%d) Method %s is not supported", tvisits, (method?method:"UNKNOWN"));
            sendError(fd, 501, "Not supported", NULL, "Method is not supported.");

	} else if (strcmp(path,"/")  == 0 ||
	           strcmp(path,"\n") == 0) {

            renderCurForm(tvisits,fd);

	} else if (strcmp(path,"/favicon.ico") == 0) {

	    sendFavicon(fd);

	} else if (strstr((path+1),"/")  == NULL && strstr(path,".png")) {	// icon

	    sendIcon(fd, path);

	} else if ((p = strstr(path,".key"))) {  // button pressed

	    *p = '\0';
	    p--;

	    while (isdigit(*p)) {
		p--;
	    }
	    int button = ((++p == '\0') ? -1 : atoi(p)+1);

	    INFO2("[WS]: (%d) Got button %d", tvisits, button);

            wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
            wm->button = button;
            wm->string = (void*) NULL;
	    sendToWebServer(wm);
	    wait = TRUE;

	} else if ((p = strstr(path,".menu"))) {  // menu in text screen

	    *p = '\0';
	    p--;

	    while (*p != '/') {
		p--;
	    }

	    INFO2("[WS]: (%d) Got menu item %s", tvisits, ++p);

            wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
            wm->button = -1;
            wm->string = g_string_new("");
	    char * ptr = p;
	    char * p2 =  NULL;

	    while ((p2 = strstr(ptr,"%20"))) {   // replace %20 to space
		*p2 = '\0';
		g_string_append(wm->string, ptr);
		g_string_append(wm->string, " ");

		ptr = (p2 + 3);
	    }
	    g_string_append(wm->string, ptr);

	    sendToWebServer(wm);
	    wait = TRUE;

	} else if ((p = strstr(path,"list="))) {  // menu in list screen, list item choosed

            char * item = strstr(path,"action=");

	    if (!item) {
		INFO2("[WS]: (%d) No selected item in list, ignore", tvisits);
	    } else {

		if (item) item += 7; // sizeof("action=")

		char* plus = NULL;
		while((plus = strstr(item,"+"))) {
	            *plus = ' ';  // replace "+" back to spaces
		}

		char * amp = strstr(p,"&");
		*amp = '\0';
   		char* index = p + 5;  // size("list=") == 10

		int idx = atoi(index);
		int iLen = g_slist_length(ls_items);

		if (idx >= 0 && idx < iLen) {

        	    wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
        	    wm->button = -1;
        	    wm->string = g_string_new("Msg:");
        	    g_string_append(wm->string, item);
		    g_string_append(wm->string, "(");
		    char b[32];
		    g_string_append(wm->string, i2string(idx+1,b));
		    g_string_append(wm->string, ",");

		    GSList *ptr = g_slist_nth(ls_items, idx);
		    g_string_append(wm->string, ((listItem*) ptr->data)->string->str);
		    INFO2("[WS]: choosed list item (%s)", ((listItem*) ptr->data)->string->str);

		    g_string_append(wm->string, ")");

		    sendToWebServer(wm);
		    wait = TRUE;
		} else {
	            INFO2("[WS]: (%d) Wrong selected index in list, ignore", tvisits);
		}
	    }
	} else if ((p = strstr(path,"editfield="))) {  // edit field

            char * item = strstr(path,"action=");

	    if (!item) {
		INFO2("[WS]: (%d) No data in edit field, ignore", tvisits);
	    } else {

		item += 7; // sizeof("action=")

		char* plus = NULL;
		while((plus = strstr(item,"+"))) {
	            *plus = ' ';  // replace "+" back to spaces
		}

		char * amp = strstr(p,"&");
		*amp = '\0';
   		char* index = p + 10;   // size("editfield=") == 10

        	wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
        	wm->button = -1;
		if (askPassword && strcmp("Ok", item) == 0) {
	            wm->string = g_string_new("Msg:_PASSWORD_");
	            askPassword = FALSE;
		} else {
                    wm->string = g_string_new("Msg:");
                    g_string_append(wm->string, item);
		}
		g_string_append(wm->string, "(");
		g_string_append(wm->string, ",");
		g_string_append(wm->string, index);
		g_string_append(wm->string, ")");

		sendToWebServer(wm);
		wait = TRUE;

	    }    
	} else if ((p = strstr(path,"action="))) {  // menu in list screen, no list item choosed
            					// or key press

	    p += 7; // sizeof("action=")
	   
	    char* btn = p;
	    while (isdigit(*btn)) {
	    	btn++;
	    }
	    
	    int button = -1;
	    if (btn != p) {
	        *btn = '\0';
		button = atoi(p);
	    }
	
	    wMessage* wm = NULL;
	
	    if (button > 0) {

	    	INFO2("[WS]: (%d) Got button %d", tvisits, button);
		
        	wm = (wMessage*) malloc(sizeof(wMessage));
        	wm->button = button;
        	wm->string = (void*) NULL;
	    	
	    } else {
		char* plus = NULL;
		while((plus = strstr(p,"+"))) {
	    	    *plus = ' ';  // replace "+" back to spaces
		}

        	wm = (wMessage*) malloc(sizeof(wMessage));
        	wm->button = -1;
        	wm->string = g_string_new("Msg:");
        	g_string_append(wm->string, p);
		g_string_append(wm->string, "(,)");
            }
	    
            if (wm) {
	    	sendToWebServer(wm);
	    	wait = TRUE;
	    }

	} else if (stat(path, &statbuf) < 0) {

            ERROR2("[WS]: (%d) File %s not found.\n", tvisits, path);
            sendError(fd, 404, "Not Found", NULL, "File not found.");

	/*} else if (S_ISDIR(statbuf.st_mode)) {

            //INFO2("[WS]: (%d)  Directory listing requested", tvisits);

            len = strlen(path);
            if (len == 0 || path[len - 1] != '/') {

        	snprintf(pathbuf, sizeof(pathbuf), "Location: %s/", path);
        	sendError(fd, 302, "Found", pathbuf, "Directories must end with a slash.");

            } else {
        	snprintf(pathbuf, sizeof(pathbuf), "%sindex.html", path);
        	if (stat(pathbuf, &statbuf) >= 0) {

                    sendFile(fd, pathbuf, &statbuf);

        	} else {
                    DIR *dir;
                    struct dirent *de;

                    sendHeaders(fd, 200, "OK", NULL, "text/html", -1, statbuf.st_mtime, 0);
                    sprintf(f, "<HTML><HEAD><TITLE>Index of %s</TITLE></HEAD>\r\n<BODY>", path);
	            bytes_sent = send(fd,f,strlen(f),0); 

                    sprintf(f, "<H4>Index of %s</H4>\r\n<PRE>\n", path);
	            sendData(fd, f);

	            sendData(fd, "Name Last Modified Size\r\n");
	            sendData(fd, "<HR>\r\n");

                    //if (len > 1) {
	            //    sprintf(f, "<A HREF=\"..\">..</A>\r\n");
                    //    sendData(fd, f);
	            //}
		    //INFO2("[Thread] Parse %s\n", path);

                    dir = opendir(path);
                    while ((de = readdir(dir)) != NULL) {
                	char timebuf[32];
                	struct tm *tm;
                	 strcpy(pathbuf, path);
                	strcat(pathbuf, de->d_name);

                	stat(pathbuf, &statbuf);
                	tm = gmtime(&statbuf.st_mtime);
                	strftime(timebuf, sizeof(timebuf), "%d-%b-%Y %H:%M:%S", tm);

                	sprintf(f, "<A HREF=\"%s%s\">", de->d_name, S_ISDIR(statbuf.st_mode) ? "/" : "");
                	sendData(fd, f);

			sprintf(f, "%s%s", de->d_name, S_ISDIR(statbuf.st_mode) ? "/</A>" : "</A> ");
                	sendData(fd, f);

                	if (_D_EXACT_NAMLEN(de) < 32) {
		            sprintf(f, "%*s", (int) (32 - _D_EXACT_NAMLEN(de)), "");
                            sendData(fd, f);
                	}
                	if (S_ISDIR(statbuf.st_mode)) {
                            sprintf(f, "%s\r\n", timebuf);
                            sendData(fd, f);
                	} else {
                            sprintf(f, "%s %10d\r\n", timebuf, (int) statbuf.st_size);
                            sendData(fd, f);
                	}
                    }
                    closedir(dir);

                    sprintf(f, "</PRE>\r\n<HR>\r\n<ADDRESS>%s</ADDRESS>\r\n</BODY></HTML>\r\n", SERVER);
        	}
            }*/
	} else {
	    INFO2("[WS] (%d) Send file", tvisits);

            sendFile(fd, path, &statbuf);
	}
    }
     
    if (wait) {
        g_mutex_lock(mut);
	answerReady = FALSE;
	g_mutex_unlock(mut);
	
	int waitingTime = 0;

	while (!answerReady && waitingTime < 1500) {	// 1/50*1500 = 30 sec
            // Main loop timer (1/50 of second)
            usleep(20000);
	    waitingTime++;
	}
	
	INFO2("[WS]: (%d) Wait answer for %d (1/50 sec)", tvisits, waitingTime);
	renderCurForm(tvisits,fd);
    }
    
    close(fd);
  
    g_mutex_lock(mut);
    //INFO2("[WS]: (%d) clientSockets before remove #%d", tvisits, g_slist_length(clientSockets));
    clientSockets = g_slist_remove(clientSockets, GINT_TO_POINTER(fd));
    //INFO2("[WS]: (%d) clientSockets after remove #%d", tvisits, g_slist_length(clientSockets));
    clients--;
    g_mutex_unlock(mut);
  
    INFO2("[WS]: (%d) Request processed", tvisits);
    g_thread_exit(NULL);
  
    return (void*)0;
}

static gpointer webServerMain(gpointer data)
{
    q2web = g_async_queue_new();
    
    confDir = getCfgDir();    
    
    const char* v = getVarValue("ScreenSize");
    if (v != NULL) {
        screenSize = atoi(v);
        if (screenSize <= 0) screenSize = 240;
	
    }
    INFO2("[WS]: $(ScreenSize) = %d", screenSize);
    
    struct sockaddr_in sin;
    memset((void *) &sin, 0, sizeof(sin));
 
    sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  
    if (sock < 0) {
        logger(L_ERR, "[WS]: webServerMain: Failed to create a socket");

        g_thread_exit(NULL);
        return (void*)0;
    }
    
    int optval = 1;
    setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval));

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = INADDR_ANY;
    sin.sin_port = htons(port);
    int ret = bind(sock, (struct sockaddr *) &sin, sizeof(sin));
    if (ret < 0) {
	logger(L_ERR, "[WS]: webServerMain: Failed to bind a socket");
        printf("ERROR: on binding %d->%s\n", errno, strerror(errno));
        
	gotExitSignal = 1;
	
        g_thread_exit(NULL);
        return (void*)0;
    }

    listen(sock, 100);
    INFO2("[WS]: webServerMain: HTTP server listening on port %d", port);
    
    while (runWeb) {
    
  	logger(L_INF, "[WS]: webServerMain: HTTP server ready to accept connection");
	
     	int s = accept(sock, NULL, NULL);
    	if (s < 0) continue;
    
    	INFO2("[WS]: webServerMain: HTTP server accepts connection %d", s);
    
    	int* ptr = malloc(sizeof(int));
    	*ptr = s;
	
	g_mutex_lock(mut);
	
	clientSockets = g_slist_append(clientSockets, GINT_TO_POINTER(s));
	
        g_mutex_unlock(mut);
	
    	GThread* rq = g_thread_create(serveRequest, (void *) ptr, FALSE, NULL);
    
    	if (!rq) {
    	    logger(L_ERR, "[WS]: webServerMain: Can not run processing thread");
    	}
    }
    
    logger(L_ERR, "[WS]: webServerMain: exit thread");
    if (sock != -1) {
        close(sock);
	sock = -1;
    } 
   
    g_thread_exit(NULL);
    return (void*)0;
}

int openWebPort(int p)
{
    if (mainThread != NULL) {
        INFO2("[WS]: openWebPort, skip initialization");
	return 0;
    }
    
    port = p;
    
    mut = g_mutex_new();
    
    // default control form
    caption = g_string_new("anyRemote");
    title  = g_string_new(" ");
    status = g_string_new(" ");
    volume = g_string_new("");
    font   = g_string_new("h6>");
    fg     = g_string_new("FFFFFF");
    bg     = g_string_new("000000");
    cover  = NULL;
    
 
    // default text form
    tx_caption = g_string_new("");
    tx_font    = g_string_new("h6>");
    tx_fg      = g_string_new("000000");
    tx_bg      = g_string_new("FFFFFF");
    tx_text    = g_string_new("");
    
    // default list form
    ls_caption = g_string_new("");
    ls_font    = g_string_new("h4>");
    ls_fg      = g_string_new("000000");
    ls_bg      = g_string_new("FFFFFF");
    ls_selidx  = 1;

    // default winndow form
    window     = NULL;

    // default edit field form
    ef_caption = g_string_new("");
    ef_text    = g_string_new("");
    ef_label   = g_string_new("");

   
    // Predefined templates for HTML forms
    
    httpEdit1  = g_string_new("<h4>");
    httpEdit2  = g_string_new("</h4></center>\n<form action=\"\" method=\"post\">\n<input type=\"text\" name=\"editfield\" value=\"");
    httpEdit2p = g_string_new("</h4></center>\n<form action=\"\" method=\"post\">\n<input type=\"password\" name=\"editfield\" value=\"");
    httpEdit3  = g_string_new("\" />\n<hr>");

    httpTail   = g_string_new("\n<h6><a href=\"/\">Redraw</a></h6>\n</center></body>\n</html>\n");
   
    int i;
    for (i=0;i<12;i++) {
        icons[i] = g_string_new("default");
    }
        
    mainThread = g_thread_create(webServerMain, NULL, FALSE, NULL);
    
    return 0;
}

int checkWebPort(char* buf, int capacity)
{
    //logger(L_DBG,"[WS]: checkWebPort");
    
    if (!q2web) return 0;
    int l = 0;
    
    // Verify commands from queue (timeout about 1/2 sec)
    wMessage* wm = (wMessage*) g_async_queue_try_pop(q2web);
    if (wm != NULL) {
        logger(L_DBG, "[WS]: Got event");

        if (wm->button > 0) {
            INFO2("[WS]: Button pressed %d", wm->button);
	    char bbuf[16];
	    if (wm->button == 10) {
	        sprintf(bbuf,"Msg:%c",'*');
	    } else if (wm->button == 11) {
	        sprintf(bbuf,"Msg:%c",'0');
	    } else if (wm->button == 12) {
	        sprintf(bbuf,"Msg:%c",'#');
	    } else {
	        sprintf(bbuf,"Msg:%d",wm->button);
	    }
	    strncpy(buf,bbuf,capacity);
	    l = strlen(buf);
        } else if (wm->string) {
            INFO2("[DS]: Data %s", wm->string->str);
	    
	    if (strcmp(wm->string->str,"_DISCONNECT_") == 0) {
	       buf[0] = '\0';
	       l = EOF;
	    } else {
	       strncpy(buf,wm->string->str,capacity);
	       l = strlen(wm->string->str);
	    }
        }
	
	if (wm->string) {
	    g_string_free(wm->string,  TRUE);
	    wm->string = NULL;
	}
        free(wm);
    }
    return l;
}

static void freeCover()
{
    if (cover) {
    	g_string_free(cover, TRUE);
    	cover = NULL;
    }	 
}

static void freeWindow()
{
    if (window) {
    	g_string_free(window, TRUE);
    	window = NULL;
    }	 
}

void closeWebPort(int final)
{
    INFO2("[WS]: closeWebPort %d", final);
    
    // close all sockets
    if (sock != -1) {
        close(sock);
	sock = -1;
    } 
    
    g_mutex_lock(mut);
    
    //INFO2("[WS]: clientSockets before cleanup #%d", g_slist_length(clientSockets));

    GSList* list = clientSockets;
    while (list) {

        int clientFD = GPOINTER_TO_INT(list->data);
	if (clientFD > 0) {
	    close(clientFD);
	}
        list = g_slist_next(list);
    }
    
    g_slist_free(clientSockets);
    clientSockets = NULL;
    
    g_mutex_unlock(mut);

    if (final) {
	runWeb = FALSE;

	g_string_free(caption,TRUE);
	g_string_free(title,  TRUE);
	g_string_free(status, TRUE);
	g_string_free(volume, TRUE);
	g_string_free(font,   TRUE);
	g_string_free(fg,	  TRUE);
	g_string_free(bg,	  TRUE);
	freeCover();

	g_string_free(tx_caption,TRUE);
	g_string_free(tx_font,   TRUE);   
	g_string_free(tx_fg,     TRUE);  
	g_string_free(tx_bg,     TRUE);  
	g_string_free(tx_text,   TRUE); 

	g_string_free(ls_caption,TRUE); 
	g_string_free(ls_font,   TRUE);   
	g_string_free(ls_fg,     TRUE);  
	g_string_free(ls_bg,     TRUE);   
	freeList();

	freeWindow();

	g_string_free(ef_caption,TRUE);
	g_string_free(ef_text,   TRUE);
	g_string_free(ef_label,  TRUE);

	freeMenu();

	g_string_free(httpEdit1,  TRUE); 
	g_string_free(httpEdit2,  TRUE); 
	g_string_free(httpEdit2p, TRUE);
	g_string_free(httpEdit3,  TRUE); 

	g_string_free(httpTail,   TRUE);

	int i;
	for (i=0;i<12;i++) {
            g_string_free(icons[i],TRUE);
	}

	if (confDir) free(confDir);

	g_mutex_free(mut);
	mut = NULL;
    }
    
    return;
}

static void stripCommandEnding(char *s)
{
    if (s && strlen(s) >= 2) {
        s[strlen(s)-2] = '\0';   // strip );
    }
}

static void setCaption(char * s)
{
    g_string_assign(caption,s);
}

static void setTitle(char * s)
{
    gboolean setupMenu = (curForm != CF);
    
    g_string_assign(title,(s?s:" "));
    curForm = CF;

    if (setupMenu) {
        setupDefMenu();
    }
}

static void setStatus(char * s)
{
    gboolean setupMenu = (curForm != CF);
    
    g_string_assign(status,(s?s:" "));
    curForm = CF;
    
    if (setupMenu) {
        setupDefMenu();
    }
    
}

static void setIcons()
{
    gboolean setupMenu = (curForm != CF);
       
    char* token = strtok(NULL,",");
    
    if (token && strcmp(token,"SAME") != 0) {
    	setCaption(token);
    }
    
    token = strtok(NULL,",");
    
    while (token) {

	while (isspace(*token)) {
            ++token;
	}
	
        int ic = -1;
	if (strncmp(token,"*",1) == 0) {
	   ic = 9;
	} else if (strncmp(token,"#",1) == 0) {
	   ic = 11;
	} else if (strncmp(token,"0",1) == 0) {
	   ic = 10;
	} else {
	   ic = atoi(token)-1;
	}
	if (ic >=0 && ic < 12) {
	    token = strtok(NULL,",");
	    while (isspace(*token)) {
                ++token;
	    }

	    g_string_assign(icons[ic],(token?token:"none"));
	    //INFO2("[WS]: setIcons %d %s",ic,icons[ic]->str);
	}
	token = strtok(NULL,",");
    }
    
    curForm = CF;

    if (setupMenu) {
        setupDefMenu();
    }
}

//
// Set(skin,default|bottomline <--- handle only this for now
//    [,keypad_only|joystick_only] 
//    [,ticker|noticker] [,volume] 
//    [,choose,_button_] [,up,_button_] [,down,_button_])
//
static void setSkin()
{
    useVolume      = FALSE;  // reset volume usage flag
    bottomlineSkin = FALSE;
    char* token;
    
    gboolean setupMenu = (curForm != CF);   
    
    while((token = strtok(NULL,","))) {
    
	if (strcmp(token,"bottomline") == 0) {
    	    bottomlineSkin = TRUE;
	} else if (strcmp(token,"volume") == 0) {
    	    useVolume = TRUE;
	} 
    }
    
    if (bottomlineSkin) {
	freeCover();    
    }
     
    curForm = CF;
    
    if (setupMenu) {
        setupDefMenu();
    }
}

static void setCover(const char* file)
{
    freeCover(); 	 
    if (file) {
	cover = g_string_new(file);
    }
}

static void setFont(int form)
{
    char *s = strtok(NULL,",");
    INFO2("[WS]: setFont %d %s",form,(s?s:"NULL"));

    if (!s) return;
    
    while (isspace(*s)) {
    	++s;
    }
    
    if (form == CF) {
	if (strncmp(s,"small",5) == 0) {
    	    g_string_assign(font,"h6>");
	} else if (strncmp(s,"medium",6) == 0) {
    	    g_string_assign(font,"h4>");
	} else if (strncmp(s,"large",5) == 0) {
    	    g_string_assign(font,"h2>");
	}
    } else if (form == TX) {
	if (strncmp(s,"small",5) == 0) {
    	    g_string_assign(tx_font,"h6>");
	} else if (strncmp(s,"medium",6) == 0) {
    	    g_string_assign(tx_font,"h4>");
	} else if (strncmp(s,"large",5) == 0) {
    	    g_string_assign(tx_font,"h2>");
	}
    } else if (form == LI) {
	if (strncmp(s,"small",5) == 0) {
    	    g_string_assign(ls_font,"h6>");
	} else if (strncmp(s,"medium",6) == 0) {
    	    g_string_assign(ls_font,"h4>");
	} else if (strncmp(s,"large",5) == 0) {
    	    g_string_assign(ls_font,"h2>");
	}
    }
}

static void setFgBg(int form, gboolean set_fg)
{
    char buf[16];
    
    char *token = strtok(NULL,",");
    if (!token) return;
    int r = atoi(token);
    
    token = strtok(NULL,",");
    if (!token) return;
    int g = atoi(token);
    
    token = strtok(NULL,",");
    if (!token) return;
    int b = atoi(token);

    sprintf(buf,"%0*x%0*x%0*x",2,r,2,g,2,b);
    
    INFO2("[WS]: setFgBg fg=%d form=%d color=%s",set_fg,form,buf);
    
    if (form == CF) {
	if (set_fg) {
            g_string_assign(fg,buf);
	} else {
            g_string_assign(bg,buf);
	}
    } else if (form == TX) {
	if (set_fg) {
            g_string_assign(tx_fg,buf);
	} else {
            g_string_assign(tx_bg,buf);
	}
    } else if (form == LI) {
	if (set_fg) {
            g_string_assign(ls_fg,buf);
	} else {
            g_string_assign(ls_bg,buf);
	}
    }
}

static void setText()
{
    //logger(L_INF, "[WS]: setText");
    gboolean setupMenu = (curForm != TX);
    
    char *token = strtok(NULL,",");
    if (!token) return;
    while (isspace(*token)) {
    	++token;
    }
      
    if (strncmp(token,"add",3) == 0 || 
        strncmp(token,"replace",7) == 0) {
        
        curForm = TX;
    
        char *token2 = strtok(NULL,","); // can be NULL
	if (!token2) return; 
        
	//logger(L_INF, "[WS]: setText add/replace");
	
	if (strncmp(token,"replace",7) == 0) {
	    g_string_truncate(tx_text,0);
	}
	
	if (strcmp(token2,"SAME") != 0) {
            g_string_assign(tx_caption,token2);
        }
    
	char *token3 = strtok(NULL,"\n");
	while (token3) {
	
	    //INFO2("[WS]: Set(text, ...) %s", token3);
	    
	    g_string_append(tx_text, "<p style=\"width:");
	    char buf[8];
	    g_string_append(tx_text, i2string(screenSize,buf));
	    g_string_append(tx_text, "px; overflow:auto;\">");

	    g_string_append(tx_text, token3);
	    g_string_append(tx_text, "</p>");
	    
	    token3 = strtok(NULL,"\n");
	}
    } else if (strcmp(token,"close") == 0) {
       char *token2 = strtok(NULL,","); // can be NULL
       if (token2 && strcmp(token,"clear") == 0) {
           g_string_truncate(tx_text,0);
       }
       curForm = CF;
       setupDefMenu();
       return;
    } else if (strncmp(token,"clear",5) == 0) {
       g_string_truncate(tx_text,0);
       return;
    } else if (strncmp(token,"fg",2) == 0) {
       setFgBg(TX,TRUE);
       return;
    } else if (strncmp(token,"bg",2) == 0) {
       setFgBg(TX,FALSE);
       return;
    } else if (strncmp(token,"font",4) == 0) {
       setFont(TX);
       return;
    } else if (strncmp(token,"show",4) == 0) {
       curForm = TX;
    } else {
       ERROR2("[WS]: Can to parse Set(text, ...)");
       return;
    }
    if (setupMenu) {
        setupDefMenu();
    }
} 

static void setList(gboolean useIcons)
{
    logger(L_INF, "[WS]: setList");
    gboolean setupMenu = (curForm != LI);
    
    char *token = strtok(NULL,",");
    if (!token) return;
    while (isspace(*token)) {
    	++token;
    }
    
    if (strncmp(token,"add",3) == 0 || 
        strncmp(token,"replace",7) == 0) {
        
        curForm = LI;
	
        char *token2 = strtok(NULL,","); // can be NULL
	if (!token2) return; 
        
	logger(L_INF, "[WS]: setList add/replace");
	
	if (strcmp(token,"replace") == 0) {
	    freeList();
	}
	
	if (strcmp(token2,"SAME") != 0) {
            g_string_assign(ls_caption,token2);
        }
	
	//INFO2("[WS]: setList title %s", token2);
    
	char *token3 = strtok(NULL,",");
	while (token3) {
	
	    if (*token3 == '\n') {        // remove new-line at the start
	        token3++;
	    }
	    
	    listItem * item = malloc(sizeof(listItem));
	    
	    //INFO2("[WS]: Set(list, ...) %s", token3);
	    int l = strlen(token3);
	    if(*(token3 + l) == '\n') {     // remove new-line at the end
	        *(token3 + l) = '\0';
	    }
	    if(*(token3 + l - 1) == '\n') {  // shit happens
	        *(token3 + l - 1) = '\0';
	    }
	    
	    if (strlen(token3) > 0) {	// avoid empty list item
		char * semicolon = index(token3,':');
		if (useIcons && semicolon) {
		    *semicolon = '\0';
		    semicolon++;
	            item->icon   = g_string_new(token3);
		    item->string = g_string_new(semicolon);
		    //INFO2("[WS]: Set(iconlist, ...) %s : %s", token3, semicolon);
		} else {
	            item->icon   = NULL;
		    item->string = g_string_new(token3);
		}

		ls_items = g_slist_append(ls_items, item);

		//INFO2("[WS]: list size #%d", g_slist_length(ls_items));
	    }
	    
	    token3 = strtok(NULL,",");
	}
	
    } else if (strncmp(token,"close",5) == 0) {
       char *token2 = strtok(NULL,","); // can be NULL
       if (token2 && strcmp(token,"clear") == 0) {
            freeList();
       }
       curForm = CF;
       setupDefMenu();
       return;
    } else if (strncmp(token,"clear",5) == 0) {
       freeList();
       return;
    } else if (strncmp(token,"fg",2) == 0) {
       setFgBg(LI,TRUE);
       return;
    } else if (strncmp(token,"bg",2) == 0) {
       setFgBg(LI,FALSE);
       return;
    } else if (strncmp(token,"font",4) == 0) {
       setFont(LI);
       return;
    } else if (strncmp(token,"select",6) == 0) {
       curForm = LI;
       char *token2 = strtok(NULL,",");
       if (token2) {
           ls_selidx = atoi(token2);
       }
    } else if (strncmp(token,"show",4) == 0) {
       curForm = LI;
    } else {
       ERROR2("[WS]: Can to parse Set([icon]list, ...)");
       return;
    }
    if (setupMenu) {
        setupDefMenu();
    }
}

static void setEditfield()
{
    logger(L_INF, "[WS]: setEditfield");
    
    gboolean setupMenu = (curForm != EF);
    
    char *token1 = strtok(NULL,",");
    if (!token1) return;
    
    char *token2 = strtok(NULL,",");
    if (!token2) return;
    
    char *token3 = strtok(NULL,",");  // can be NULL
    
    g_string_assign(ef_caption,token1);
    g_string_assign(ef_label,  token2);
    g_string_assign(ef_text,   token3 ? token3 : "");

    askPassword = FALSE;
    curForm = EF;
    if (setupMenu) {
        setupDefMenu();
    }
    INFO2("[WS]: setEditfield done %d", curForm);
}

//
// Set(image,window,_image_file_name_) 
// Set(image,icon,_icon_name_,_image_file_name_) -- not supported
// Set(image,show|close|cursor|nocursor|remove_all) -- only 1,2 are supported
//
static void setImage(const char* cmd, const char* file)
{
    INFO2("[WS]: setImage %s",cmd);
    gboolean setupMenu = (curForm != WM);
    
    if (!cmd) return;
    while (isspace(*cmd)) {
    	++cmd;
    }
      
    if (strncmp(cmd,"window",6) == 0) {
        
	logger(L_INF, "[WS]: setImage: window");
	
        curForm = WM;
	
	if (!file) return;
	while (isspace(*file)) {
    	    ++file;
	}

        freeWindow(); 	 
        window = g_string_new(file);
 
    } else if (strncmp(cmd,"show",4) == 0) {
       curForm = WM;
    } else if (strncmp(cmd,"close",5) == 0) { 
       curForm = CF;
       setupDefMenu();
       return;
    } else { 
       logger(L_INF, "[WS]: setImage: skip command");
       return;
    }  
           
    if (setupMenu) {
        setupDefMenu();
    }
}

static void setMenu()
{
    char *token = strtok(NULL,",");
    if (!token) return;
    while (isspace(*token)) {
    	++token;
    }
    
    if (strncmp(token,"add",3) == 0 || 
        strncmp(token,"replace",7) == 0) {
        
	logger(L_INF, "[WS]: setMenu add/replace");
	
	if (strcmp(token,"replace") == 0) {
	    setupDefMenu();
	}
    
	char *token3 = strtok(NULL,",");
	while (token3) {  
	 
	    while (isspace(*token3)) {
    	        ++token3;
            }
	
	    INFO2("[WS]: Set(menu, ...) %s", token3);
	    
	    user_menu = g_slist_append(user_menu, g_string_new(token3));
	    token3 = strtok(NULL,",");
	}
	
    } else if (strncmp(token,"clear",5) == 0) {
       freeMenu();
    } else {
        ERROR2("[WS]: Can to parse Set(menu, ...)");
    }

}

static void setParams()
{
    char *token = strtok(NULL,",");
    if (!token) return;
    
    while (token) {

	while (isspace(*token)) {
    	    ++token;
	}
    
        if (strncmp(token,"icon_padding",12) == 0) {
	
	    char *token2 = strtok(NULL,",");
	    if (!token2) return;
	    
	    while (isspace(*token2)) {
    		++token2;
	    }
	    
            iconPadding = atoi(token2);
            if (iconPadding <= 0) iconPadding = 0;
	    
	    INFO2("[WS]: Use icon padding %d", iconPadding);
        
	} else if (strncmp(token,"icon_size",9) == 0) {
	
	    char *token2 = strtok(NULL,",");
	    if (!token2) return;
	    
	    while (isspace(*token2)) {
    		++token2;
	    }
	    
            iconSizeScale = atoi(token2);
            if (iconSizeScale <= 0) iconSizeScale = 64;
	    
 	    if (iconSizeScale <= 32) iconSize = 32;
	    else if (iconSizeScale <= 48) iconSize = 48;
	    else if (iconSizeScale <= 64) iconSize = 64;
	    else iconSize = 128;
	    
	    INFO2("[WS]: Use icon padding %d", iconPadding);
	    
	} else {
            ERROR2("[WS]: Skip Set(parameter, ...) %s", token);
	}
	
	token = strtok(NULL,",");
    }
}

static void getPass()
{
    logger(L_INF, "[WS]: getPass");
    
    gboolean setupMenu = (curForm != EF);
    
    g_string_assign(ef_caption,"Enter password");
    g_string_assign(ef_label,  "Enter password");
    g_string_assign(ef_text,   "");

    askPassword = TRUE;
    secure      = NEED_SEND_COOKIE;
    cookie      = random();

    curForm = EF;
    
    if (setupMenu) {
        setupDefMenu();
    }
}

int writeWebConnStr(char* value)
{
    //INFO2("[WS]: writeWebConnStr %s", value);

    wMessage* wm  = NULL;
    wMessage* wm2 = NULL;
    
    char * cmd = strdup(value);
    
    //gboolean skipSpaces = FALSE;
    //INFO2("[WS]: parse lenght %d", (cmd ? (int) strlen(cmd) : 0));

    stripCommandEnding(cmd);
    
    if (strlen(cmd) < MAXMAXLEN) {
        INFO2("[WS]: parse %d %s", curForm, cmd);
    } else {
        INFO2("[WS]: parse %d ... command too long ...", curForm);
    }

    char* token = strtok(cmd,",");
    while (isspace(*token)) {
    	++token;
    }
    
    g_mutex_lock(mut);
    
    if (strncmp(token,"Set(status",10) == 0) {
       setStatus(cmd+strlen("Set(status,"));
    } else if (strncmp(token,"Set(title",9) == 0) {
       setTitle(cmd+strlen("Set(title,"));
    } else if (strncmp(token,"Set(icons",9) == 0) {
       setIcons();
    } else if (strncmp(token,"Set(font",8) == 0) {
       setFont(CF);
    } else if (strncmp(token,"Set(fg",6) == 0) {
       setFgBg(CF,TRUE);
    } else if (strncmp(token,"Set(bg",6) == 0) {
       setFgBg(CF,FALSE);
    } else if (strncmp(token,"Set(volume",10) == 0) {
       char* sz = strtok(NULL,",");
       if (sz) {
           g_string_assign(volume,sz);
       } else {
           g_string_assign(volume,"");
       }
    } else if (strncmp(token,"Set(skin",8) == 0) {
       setSkin();
    } else if (strncmp(token,"Set(cover",9) == 0) {
       ERROR2("[WS]: Improperly formed command Set(cover,...)");
    } else if (strncmp(token,"Set(list",8) == 0) {
       setList(FALSE);
    } else if (strncmp(token,"Set(iconlist",12) == 0) {
       setList(TRUE);
    } else if (strncmp(token,"Set(text",8) == 0) {
       setText();
    } else if (strncmp(token,"Set(menu",8) == 0) {
       setMenu();
    } else if (strncmp(token,"Set(editfield",13) == 0) {
       setEditfield();
    } else if (strncmp(token,"Set(image",9) == 0) {
       ERROR2("[WS]: Improperly formed command Set(image,...)");
    } else if (strncmp(token,"Set(popup",9) == 0) {
       // ignore
    } else if (strncmp(token,"Set(disconnect",14) == 0) {
       // ignore
    } else if (strncmp(token,"Set(fullscreen",16) == 0) {
       // ignore
    } else if (strncmp(token,"Set(vibrate",11) == 0) {
       // ignore
    } else if (strncmp(token,"Set(parameter",13) == 0) {
       setParams();
    } else if (strncmp(token,"Get(password",12) == 0) {
       getPass();
    } else if (strncmp(token,"Get(screen_size",16) == 0) {
        
	char b[32];
	const char *buf = i2string(screenSize,b);
	
        //SizeX(width,) and SizeY(height,) - use same
	
	wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = g_string_new("SizeX(");
	g_string_append(wm->string,buf);
	g_string_append(wm->string,",)");

	wm2 = (wMessage*) malloc(sizeof(wMessage));
        wm2->button = -1;
        wm2->string = g_string_new("SizeY(");
	g_string_append(wm2->string,buf);
	g_string_append(wm2->string,",)");
   
    } else if (strncmp(token,"Get(icon_padding",16) == 0) {
    
	char b[32];
	const char *buf = i2string(iconPadding,b);
	
	wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = g_string_new("IconPadding(");
	g_string_append(wm->string,buf);
	g_string_append(wm->string,",)");
 	
    } else if (strncmp(token,"Get(model",9) == 0) {
    
 	wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = g_string_new("WebInterface");

    } else if (strncmp(token,"Get(is_exists",13) == 0) {
       
        char* sz = strtok(NULL,",");
	char* icon = strtok(NULL,",");
	if (sz && icon) {
            while (isspace(*sz)) {
    		++sz;
	    }
	    while (isspace(*icon)) {
    		++icon;
	    }

            wm = (wMessage*) malloc(sizeof(wMessage));
            wm->button = -1;
            wm->string = g_string_new("IconExists(");
	    g_string_append(wm->string,sz);
	    g_string_append(wm->string,",");
	    g_string_append(wm->string,icon);
	}
 
    } else if (strncmp(token,"Get(cover_size",14) == 0) {
	
	wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = g_string_new("CoverSize(");
	char b[32];
	g_string_append(wm->string,i2string((screenSize*8)/10,b));
	g_string_append(wm->string,",)");
    
    } else if (strncmp(token,"Get(version",11) == 0) {

        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = g_string_new("Version(,web_interface)");

    } else if (strncmp(token,"Get(cursor",10) == 0) {
        // ignore
    } else if (strncmp(token,"Get(ping",8) == 0) {
    
        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = g_string_new("Ping");

    } else if (strncmp(token,"End(",4) == 0) {
        answerReady = TRUE;
    } else {
    	ERROR2("[WS]: Unknown command %s", token);
    }
    
    g_mutex_unlock(mut);
    
    if (wm) {
       sendToWebServer(wm);
    }
    if (wm2) {
       sendToWebServer(wm2);
    }
    
    free(cmd);
    INFO2("[WS]: parsed %d", curForm);
    return EXIT_OK;
}

int writeWebConnFile(dMessage* dm)
{
    //INFO2("[WS]: writeWebConnFile: %s", (char*) dm->value);
    
    g_mutex_lock(mut);
    
    if (strncmp(dm->value,"Set(cover",9) == 0) {
       setCover(dm->file);
    } else if (strncmp(dm->value,"Set(image",9) == 0) {
       setImage((dm->value + 10), dm->file);  // 10 = 9 + ","
    } else {
    	ERROR2("[WS]: Unknown command file:%s", (char*) dm->value);
    }
    
    g_mutex_unlock(mut);
    
    INFO2("[WS]: parsed %d", curForm);
    return EXIT_OK;
}

int writeWebConn(dMessage* dm)
{
    //INFO2("[WS]: writeWebConn (%d)", dm->size);
    if (dm->type == DM_SET) {
    
       return writeWebConnStr(dm->value);
       
    } else if (dm->type == DM_SETFILE) {

       return writeWebConnFile(dm);
    
    } else {
        ERROR2("[WS]: Not supported");
	return EXIT_OK;
    }
}
