// pkg_tree.cc
//
//  Copyright 1999,2000,2001 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  A package-tree displayer.

#include "aptitude.h"

#include "pkg_tree.h"

#include "download.h"
#include "pkg_node.h"
#include "pkg_item.h"
#include "pkg_columnizer.h"
#include "pkg_subtree.h"
#include "pkg_grouppolicy.h"
#include "pkg_sortpolicy.h"
#include "ui.h"
#include "vs_progress.h"

#include "vscreen/columnify.h"

#include "load_grouppolicy.h"
#include "load_sortpolicy.h"

#include "generic/apt.h"
#include "generic/config_signal.h"
#include "generic/undo.h"

#include <apt-pkg/progress.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/algorithms.h>

#include <sigc++/bind.h>

keybindings *pkg_tree::bindings=NULL;

vs_editline::history_list pkg_tree::limit_history, pkg_tree::grouping_history,
  pkg_tree::sorting_history, pkg_tree::search_history;

class pkg_matcher_search:public vs_tree_search_func
{
  pkg_matcher *matcher;
public:
  pkg_matcher_search(pkg_matcher *_matcher):matcher(_matcher) {}

  bool operator()(const vs_treeitem &item)
  {
    // EWW
    const pkg_item *pitem=dynamic_cast<const pkg_item *>(&item);
    if(pitem)
      return matcher->matches(pitem->get_package(), pitem->visible_version());
    else
      return false;
  }
};

void pkg_tree::init_bindings()
{
  bindings=new keybindings(vs_tree::bindings);
}

void pkg_tree::redirect_cancel_incsearch(string s)
{
  do_cancel_incsearch();
}

pkg_tree::pkg_tree(const char *def_grouping,
		   pkg_grouppolicy_factory *_grouping,
		   const char *def_limit):
  grouping(_grouping), groupingstr(def_grouping),
  sorting(parse_sortpolicy(aptcfg->Find(PACKAGE "::UI::Default-Sorting",
					 "name"))),
  limit(NULL), last_search_matcher(NULL),
  doing_incsearch(false), pre_incsearch_selected(get_end())
{
  aptcfg->connect(PACKAGE "::UI::Incremental-Search",
		  SigC::slot(this, &pkg_tree::redirect_cancel_incsearch));

  if(def_limit)
    limitstr=def_limit;
  else
    limitstr=aptcfg->Find(PACKAGE "::Pkg-Display-Limit", "");

  if(!limitstr.empty())
    limit=parse_pattern(limitstr);

  if(grouping)
    build_tree();

  cache_closed.connect(slot(this, &pkg_tree::handle_cache_close));

  cache_reloaded.connect(slot(this, &pkg_tree::build_tree));
}

void pkg_tree::handle_cache_close()
{
  set_root(NULL);
}

pkg_tree::~pkg_tree()
{
  delete last_search_matcher;
  delete sorting;
}

void pkg_tree::set_grouping(pkg_grouppolicy_factory *_grouping)
{
  if(grouping)
    delete grouping;

  grouping=_grouping;

  if(grouping)
    build_tree();
}

void pkg_tree::set_grouping(string s)
{
  groupingstr=s;

  pkg_grouppolicy_factory *grp=parse_grouppolicy(s);

  if(grp)
    set_grouping(grp);
}

void pkg_tree::set_sorting(pkg_sortpolicy *_sorting)
{
  delete sorting;

  sorting=_sorting;

  // ummmm
  if(grouping)
    build_tree();
}

void pkg_tree::set_sorting(string s)
{
  pkg_sortpolicy *policy=parse_sortpolicy(s);

  if(policy)
    set_sorting(policy);
}

void pkg_tree::build_tree()
{
  doing_incsearch=false;

  set_root(NULL);
  // Clear the current tree.
  doing_incsearch=false;
  pre_incsearch_selected=get_end();

  if(grouping && apt_cache_file)
    {
      pkg_subtree *mytree=new pkg_subtree(_("All Packages"), true);
      pkg_grouppolicy *grouper=grouping->instantiate(&selected_signal,
						     &selected_desc_signal);

      mytree->set_depth(-1);

      for(pkgCache::PkgIterator i=(*apt_cache_file)->PkgBegin(); !i.end(); i++)
	if((!limit) || limit->matches(i, pkg_item::visible_version(i)))
	  grouper->add_package(i, mytree);

      mytree->sort(pkg_sortpolicy_wrapper(sorting));

      set_root(mytree);

      delete grouper;
    }

  vscreen_update();
}

void pkg_tree::set_limit(string _limit)
{
  if(limit)
    delete limit;
  limit=NULL;
  limitstr=_limit;

  pkg_matcher *new_limit=parse_pattern(_limit);
  if(new_limit)
    limit=new_limit;

  build_tree();
}

bool pkg_tree::handle_char(chtype ch)
{
  if(bindings->key_matches(ch, "ChangePkgTreeLimit"))
    prompt_string(_("Enter the new package tree limit: "),
		  limitstr,
		  SigC::slot(this, &pkg_tree::set_limit),
		  NULL,
		  NULL,
		  &limit_history);
  else if(bindings->key_matches(ch, "ChangePkgTreeGrouping"))
    prompt_string(_("Enter the new package grouping mechanism for this display: "),
		  groupingstr,
		  SigC::slot(this,
			     (void (pkg_tree::*) (string)) &pkg_tree::set_grouping),
		  NULL,
		  NULL,
		  &grouping_history);
  else if(bindings->key_matches(ch, "ChangePkgTreeSorting"))
    prompt_string(_("Enter the new package sorting mechanism for this display: "),
		  "",
		  SigC::slot(this,
			     (void (pkg_tree::*) (string)) &pkg_tree::set_sorting),
		  NULL,
		  NULL,
		  &sorting_history);
  else if(bindings->key_matches(ch, "Search"))
    prompt_string(_("Search for: "),
		  "",
		  SigC::slot(this, &pkg_tree::do_search),
		  SigC::slot(this, &pkg_tree::do_cancel_incsearch),
		  SigC::slot(this, &pkg_tree::do_incsearch),
		  &search_history);
  else if(bindings->key_matches(ch, "ReSearch"))
    {
      if(last_search_matcher)
	{
	  pkg_matcher_search searcher(last_search_matcher);
	  search_for(searcher);
	}
      else
	beep();
    }
  else
    return vs_tree::handle_char(ch);

  return true;
}

void pkg_tree::do_search(std::string s)
{
  if(s.size()!=0)
    {
      delete last_search_matcher;
      last_search_term=s;
      last_search_matcher=parse_pattern(s);
    }

  if(doing_incsearch)
    doing_incsearch=false;
  else
    {
      if(last_search_term.size()!=0 && last_search_matcher)
	{
	  pkg_matcher_search searcher(last_search_matcher);
	  search_for(searcher);
	}
      else
	beep();
    }
}

void pkg_tree::do_incsearch(std::string s)
{
  if(!aptcfg->FindB(PACKAGE "::UI::Incremental-Search", true))
    return;

  if(!doing_incsearch)
    {
      doing_incsearch=true;
      pre_incsearch_selected=get_selected();
    }

  pkg_matcher *m=parse_pattern(s, false);

  set_selection(pre_incsearch_selected);

  if(m)
    {
      pkg_matcher_search searcher(m);
      search_for(searcher);
    }

  delete m;
}

void pkg_tree::do_cancel_incsearch()
{
  if(doing_incsearch)
    {
      set_selection(pre_incsearch_selected);
      doing_incsearch=false;
    }
}
