// cmdline_changelog.cc
//
//  Copyright 2004 Daniel Burrows

#include "cmdline_changelog.h"

#include "cmdline_common.h"
#include "cmdline_progress.h"
#include "cmdline_util.h"

#include <aptitude.h>
#include <download_manager.h>

#include <generic/apt.h>
#include <generic/pkg_changelog.h>

#include <apt-pkg/error.h>
#include <apt-pkg/progress.h>

using namespace std;

bool do_cmdline_changelog(const vector<string> &packages)
{
  const char *pager="/usr/bin/sensible-pager";

  if(access("/usr/bin/sensible-pager", X_OK)!=0)
    {
      _error->Warning(_("Can't execute sensible-pager, is this a working Debian system?"));

      pager=getenv("PAGER");

      if(pager==NULL)
	pager="more";
    }

  for(vector<string>::const_iterator i=packages.begin(); i!=packages.end(); ++i)
    {
      string input=*i;

      cmdline_version_source source;
      string package, sourcestr;

      if(!cmdline_parse_source(input, source, package, sourcestr))
	continue;

      pkgCache::PkgIterator pkg=(*apt_cache_file)->FindPkg(package);

      if(pkg.end())
	{
	  _error->Error("Can't find package %s", package.c_str());
	  continue;
	}

      pkgCache::VerIterator ver=cmdline_find_ver(pkg,
						 source, sourcestr);

      if(!ver.end())
	{
	  // Move this to a central location and just display an apt
	  // error?
	  bool in_debian=false;

	  for(pkgCache::VerFileIterator vf=ver.FileList();
	      !vf.end() && !in_debian; ++vf)
	    if(!vf.File().end() && vf.File().Origin()!=NULL &&
	       strcmp(vf.File().Origin(), "Debian")==0)
	      in_debian=true;

	  if(!in_debian)
	    {
	      _error->Error(_("%s is not an official Debian package, cannot display its changelog."), input.c_str());
	      continue;
	    }

	  pkg_changelog *cl=get_changelog(ver,
					  gen_cmdline_download_progress());

	  if(!cl)
	    _error->Error(_("Couldn't find a changelog for %s"), input.c_str());
	  else
	    {
	      // Run the user's pager.
	      system((string(pager)+" "+cl->get_filename()).c_str());

	      delete cl;
	    }
	}
    }
}

// TODO: fetch them all in one go.
int cmdline_changelog(int argc, char *argv[])
{
  _error->DumpErrors();

  OpProgress progress;
  apt_init(&progress, false);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  vector<string> packages;
  for(int i=1; i<argc; ++i)
    packages.push_back(argv[i]);

  do_cmdline_changelog(packages);

  _error->DumpErrors();

  return 0;
}
