# 
# get-patch.sh - retrieve a patch set from an archive
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;



      --help|-h)
                printf "retrieve a patch set from an archive\\n"
                printf "usage: get-patch [options] [archive/]revision [dir]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf "Obtain the patch set for REVISION and store it in the DIR\\n"
                printf "(or a directory named REVISION.patches).  DIR must not already\\n"
		printf "exist.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "get-patch: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "get-patch: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;
			
    -*)                 printf "get-patch: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -lt 1 -o $# -gt 2 ; then
  printf "usage: get-patch [options] [archive/]revision [dir]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

archive_revision="$1"
shift

if test $# -ne 0 ; then
  dir="$1"
  shift
else
  dir=
fi

################################################################
# Sanity Check and Process Defaults
# 

arch=${archive_revision%%/*}
revision=${archive_revision#*/}

if test "$arch" = "$revision" ; then
  arch=`larch my-default-archive -R "$archroot" -A "$archive"`
fi

archive="$arch"

larch valid-archive-name -e get-patch -- "$archive"
larch valid-package-name -e get-patch --patch-level -- "$revision"

category=`larch parse-package-name --basename $revision`
branch=`larch parse-package-name $revision`
vsn=`larch parse-package-name --vsn $revision`
lvl=`larch parse-package-name --lvl $revision`

patch_set=$category/$branch/$branch--$vsn/$lvl/$branch--$vsn--$lvl.patches.tar.gz

if test -z "$dir" ; then
  dir=$branch--$vsn--$lvl.patches
fi

cd "`dirname \"$dir\"`"

parent="`pwd`"
dir="$parent/`basename \"$dir\"`"

if test -e "$dir" ; then
  printf "get-patch: output directory already exists\\n" 1>&2
  printf "  dir: %s\\n" "$dir" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHARCHIVE" != "$archive" ; then

  exec larch with-archive -R "$archroot" -A "$archive" \
      larch get-patch -R "$archroot" -A "$archive" "$archive/$revision" "$dir"
fi

################################################################
# Get a Temporary Directory
# 

tmpdir="$parent/,,get-patch.$$"

bail()
{
  cd "$parent"
  rm -rf "$tmpdir"
  exit 1
}


finish()
{
  cd "$parent"
  rm -rf "$tmpdir"
  exit 0
}

trap "printf \"get-patch: interrupted -- cleaning up\\n\" 1>&2  ; bail" INT

mkdir "$tmpdir"



################################################################
# Get the Patch
# 

cd "$tmpdir"

wftp-home

if ! wftp-get $patch_set > $branch--$vsn--$lvl.patches.tar.gz ; then
  printf "get-patch: unable to retrieve patch set\\n" 1>&2
  printf "\\n"
  bail
fi

tar -m -zxf $branch--$vsn--$lvl.patches.tar.gz 

if test -e "$dir" ; then
  printf "get-patch: output directory already exists\\n" 1>&2
  printf "  dir: %s\\n" "$dir" 1>&2
  printf "\\n" 1>&2
  bail
fi

mv $branch--$vsn--$lvl.patches "$dir"

finish




