# previous-patch-level.sh - compute the name of the preceeding patch level
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "print the name of the preceeding (continuous) revision\\n"
		printf "usage: previous-patch-level [options] revision\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf "Print the name of the revision that precedes REVISION in the\\n"
		printf "history of patches.  Exit with status 1 if no such revision\\n"
		printf "exists (e.g. for a fresh-start revision) or if the archive must be\\n"
		printf "contacted to find the previous revision, but can not be reached.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-category: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-category: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    --)			shift
			break
			;;

    -*)			printf "previous-patch-level: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: previous-patch-level [options] revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

revision="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

larch valid-package-name -l -e previous-patch-level -- "$revision"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$revision\"`"
category=`larch parse-package-name -b $revision`
branch=`larch parse-package-name $revision`
vsn=`larch parse-package-name -v $revision`
lvl=`larch parse-package-name -l $revision`

################################################################
# Compute the Previous Patch Level
# 


# does this revision have a look-aside ancestor?
# 
wftp-home
wftp-cd $category/$branch/$branch--$vsn/$lvl
link_file=`wftp-ls | grep -E "^CONTINUATION" | head -1`

if test "$link_file" != "" ; then
  link="`wftp-get $link_file`"
  # could validate link here...
  # 
  printf "%s\\n" "$link"
  exit 0
fi


# if there is no look-aside, the previous revision of a 
# patch-N revision or versionfix-N revision is easy to 
# compute:
# 
patchtype=${lvl%-*}
patchnum=${lvl##*-}

if test $patchnum -eq 1 -a $patchtype = patch ; then
  printf "%s/%s--%s--base-0\\n" "$archive" "$branch" "$vsn"
  exit 0
fi

if test $patchnum -eq 1 -a $patchtype = versionfix ; then
  printf "%s/%s--%s--version-0\\n" "$archive" "$branch" "$vsn"
  exit 0
fi

if test $patchnum -gt 0 ; then
  printf "%s/%s--%s--%s-%d\\n" "$archive" "$branch" "$vsn" "$patchtype" $(($patchnum - 1))
  exit 0
fi

# patchnum is 0, patchtype is either version or base.
# If it's version, look for the highest numbered patch- or base-0.
# If it's base-0, an internal error has occured.
# 

if test $patchtype = version ; then

  answer_lvl="`larch revisions -r $branch--$vsn | grep -E "patch|base" | head -1`"
  answer="$archive/$branch--$vsn--$answer_lvl"

else

  answer=

fi

if test "$answer" = "" ; then
  printf "previous-patch-level: CORRUPT ARCHIVE\\n" 1>&2
  printf "\\n" 1>&2
  printf "  no previous revision found" 1>&2
  printf "\\n" 1>&2
  printf "  archive: %s\\n" "$archive" 1>&2
  printf "  revision: %s--%s--%s\\n" "$branch" "$vsn" "$lvl" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

printf "%s\\n" "$answer"

