# patch-details.sh: generate a patch details table entry
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

comand_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "generate a patch details table entry\\n"
		printf "usage: patch-details [options] [--] patch-set-dir\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --notes FILE                  patch level annotations\\n"
		printf " --archive-list FILE           list of archives in the library\\n"
		printf "\\n"
		printf "Generate table-format patch report from a patch set.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

notes=
archive_list=

while test $# -ne 0 ; do

  case "$1" in 

    --notes)		shift
    			if test $# -eq 0 ; then
			  printf "patch-details: --notes requires an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			notes="$1"
			shift
			;;

    --archive-list)	shift
    			if test $# -eq 0 ; then
			  printf "patch-details: --archive-list requires an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive_list="$1"
			shift
			;;

    --)			shift
    			break
			;;
			
    -*)			printf "patch-details: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: patch-details [options] [--] dir\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  delta="$1"
  shift
else
  delta=.
fi

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$delta"
delta="`pwd`"


################################################################
# Functions
# 

html_quote()
{
  sed -e '{
	    s/&/\&amp;/g
	    s/</\&lt;/g
	    s/>/\&gt;/g
	    s/"/\&quot;/g
	  }'
}



################################################################
# Do It
# 



cd "$delta"


# BEGIN "the patch set table"
# 
# A patch set is a table with two or more rows.
# 
# The top row is ``the id row'' which names the table and links it to
# a creator, changelog, distributions, etc.
# 
# The second row is the ``the summary row'' containing the "Summary:" 
# section of the log entry.
# 
# The remaining rows, if any, summarize the patch set much as a patch report
# does, adding information about merges and lost log files.
# 
printf "<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" width=\"100%%\">\\n"
printf "\\n"
printf "\\n"


# BEGIN "the id row" of ``the patch set table'' (top row)
# 
# The id row contains the patch level name, creator name, distributions
# whose revision frontier is at this patch level, etc.
# 
# 
printf "<tr> <!--- the id row ---> \\n"
printf "  <td>\\n"

creator="`cat =log.txt | larch log-header-field creator | html_quote`"
date="`cat =log.txt | larch log-header-field standard-date`"
archive="`cat =log.txt | larch log-header-field archive`"
revision="`cat =log.txt | larch log-header-field revision`"
lvl=${revision##*--}

# the id row is a one row table
#
printf "  <table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" width=\"100%%\">\\n"
printf "    <tr>\\n"

# col 1 of the id row is the patch level name and patch level notes
# 
printf "      <td width=\"%%33\">\\n"		# patch level name and revision notes
printf "        <font color=\"#000000\">\\n"
printf "          <code>%s</code>\\n" "$revision"
printf "          <a type=\"text/plain\" target=\"@TARGET@\" href=\"@REPORTS@/%s/=log.txt\">\\n" "$lvl"
printf "            <small>(complete log entry)</small>\\n"
printf "          </a>\\n"
printf "        </font>\\n"
if test ! -z "$notes" ; then
printf "        <font color=\"@NOTESCOLOR@\">\\n"
cat "$notes" \
| awk -v revision="$revision" \
      -v archive="$archive" \
      '{
         if ((archive "/" revision ":") == substr($0, 1, length(archive "/" revision) + 1))
	   {
	     sub("^[^:]*:", "", $0);
	     printf("          <br><small>%s</small>\n", $0);
	   }
       }'
printf "        </font>\\n"
fi
printf "      </td>\\n"

# col 2 of the id row is for post-processors to add extra links
# 
printf "      <td width=\"%%33\" align=\"center\">\\n"
printf "        <font color=\"#000000\">\\n"
printf "           @XTRALINKS@\\n"
printf "        </font>\\n"
printf "      </td>\\n"


# col 3 of the id row is the creator name and date
# 
printf "      <td width=\"%%33\" align=\"right\">\\n"	# author and date
printf "        <font color=\"#000000\">\\n"
printf "          <p align=\"right\">%s<br>\\n" "$creator"
printf "          <code>%s</code>\\n" "$date"
printf "        </font>\\n"
printf "      </td>\\n"

printf "    </tr>\\n"
printf "  </table>\\n"	# end of the id row table

# END "the id row" of ``the patch set table'' (top row)
# 
printf "  </td>\\n"
printf "</tr> <!--- end of the id row --->\\n"
printf "\\n"
printf "\\n"


# BEGIN "the summary row" of ``the patch set table'' (second row)
# 
# The summary row of the patch set table is the summary field
# from the log message.
# 
printf "<tr>\\n"
printf "  <td>\\n"
printf "    <font color=\"#000000\">\\n"
printf "      <b><i><small>summary:</small></i></b><br>\\n"
printf "      <pre>\\n"
cat =log.txt | larch log-header-field --literal summary | html_quote
printf "      </pre>\\n"
printf "    </font>\\n"
# END "the summary row" of ``the patch set table'' (middle row)
# 
printf "  </td>\\n"
printf "</tr>\\n"
printf "\\n"
printf "\\n"


################################################################
# Merge History
# 

cat =log.txt \
| larch log-header-field --list new-patches \
| grep -v -F "$archive/$revision" \
| larch sort-revisions --full --reverse \
> ,,new-patches

cat =log.txt \
| larch log-header-field --list removed-patches \
| larch sort-revisions --full --reverse \
> ,,removed-patches

if test ! -z "`head -n 1 ,,new-patches`" -o ! -z "`head -n 1 ,,removed-patches`" ; then 

  # BEGIN "the merges row" of ``the patch set table''
  # 
  printf "<tr>\\n"
  printf " <td>\\n"
  printf "  <table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" width=\"100%%\">\\n"
  printf "    <tr>\\n"

  printf "      <td width=\"50%%\">\\n"  # new patches
  if test ! -z "`head -n 1 ,,new-patches`" ; then
    printf "        <font color=\"#000000\"><i><b><small>merged in</small></b></i></font>\\n"
    printf "        <table>\\n"
    printf "\\n"
    cat ,,new-patches \
    | awk -v archive_list="$archive_list" \
	   -f "$ARCH_SUBCMD_ROOT/web/patch-list-rows.awk"
    printf "\\n"
    printf "\\n"
    printf "        </table>\\n"
  fi
  printf "      </td>\\n"  # end of new patches
  printf "\\n" 
  printf "\\n" 

  printf "      <td width=\"50%%\">\\n"  # removed patches
  if test ! -z "`head -n 1 ,,removed-patches`" ; then
    printf "        <font color=\"#000000\"><i><b><small>merged in</small></b></i></font>\\n"
    printf "        <table>\\n"
    printf "\\n"
    cat ,,removed-patches \
    | awk -v archive_list="$archive_list" \
	   -f "$ARCH_SUBCMD_ROOT/web/patch-list-rows.awk"
    printf "\\n"
    printf "\\n"
    printf "        </table>\\n"
  fi
  printf "      </td>\\n" # end of removed patches 
  printf "\\n" 
  printf "\\n" 

  # END "the merges row" of ``the patch set table''
  # 
  printf "    </tr>\\n"
  printf "  </table>\\n"
  printf " </td>\\n"
  printf "</tr>\\n"

fi


################################################################
# more details
# 

if test -e patches ; then
  printf "<tr>\\n"
  printf "  <td>\\n"
  printf "    <font color=\"#000000\">\\n"
  printf "      <small>\\n"
  larch patch-report --target new_window --no-intro --link-root "@PATCHSET@"  --omit-empty --compact --html . \
  | sed -e 's/<[hH][123]>/<p><b><i>/g' -e 's,</[Hh][123]>,</i></b><br>,g'
  printf "      </small>\\n"
  printf "    </font>\\n"
  printf "  </td>\\n"
  printf "</tr>\\n"
fi


# END "the patch set table"
# 
printf "</table>\\n"

###############################################################
# Clean Up
# 

rm -f ,,*


# tag: Tom Lord Thu Dec 13 02:30:24 2001 (patch-sets/patch-details.sh)
#
