# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package ArchZoom::TemplateEngine::Native;

use base 'ArchZoom::TemplateEngine';

use ArchZoom::Util;
no strict 'refs';

# accept filters :asis :urlize :htmlize :break
sub escape ($;$) {
	my $str = shift;
	my $filter_str = shift || '';

	$filter_str = ":htmlize$filter_str"
		unless $filter_str =~ /:(asis|htmlize)\b/;
	die "Invalid variable filter string ($filter_str)\n"
		unless $filter_str =~ s/^:(\w+(:\w+)*)$/$1/;
	my @filters = split(':', $filter_str);

	foreach (@filters) {
		if (/^urlize|htmlize|break$/) {
			$str = &$_($str);
		}
		elsif (/^lc|uc|lcfirst|ucfirst$/) {
			$str = eval "$_(\$str)";
		}
	}
	return $str;
}

sub expand_stash ($$) {
	my $stash = shift;
	$_[0] =~ s{(&\w+(?::\w+)*)\(([^\)]*)\)}{
		my ($prefix, $arg) = ($1, $2);
		$arg =~ s/,\s*/\01/g;
		"$prefix\0$arg\0"
	}sge;

	$_[0] =~ s{\$(?:(\w+)\.)?(\w+)((?::\w+)*)}{
		my $value = $1? do {
			my $this = $stash->{$1};
			ref($this) eq 'HASH'? $this->{$2}: $this->$2()
		}: $stash->{$2};
		defined $value? escape($value, $3): $&;
	}sge;

	$_[0] =~ s{\$#(\w+)}{
		my $value = $stash->{$1};
		ref($value) eq 'ARRAY'? scalar(@$value):
			defined $value? length $value: $&;
	}sge;

	$_[0] =~ s{&(\w+)((?::\w+)*)\0([^\0]*)\0}{
		my $match = $&;
		my ($func, $arg) = ("main::$1", $3);
		my $filter_str = !$2 && substr($func, 6, 7) eq 'selfurl' ? ':urlize' : $2;
		# check whether we are ready to expand the function (not perfect)
		if ($arg =~ /(?:^|\01)\$/) {
			$match
		} else {
			my @args = split(/\01/, $arg);
			my $value = &$func(@args);
			escape($value, $filter_str)
		}
	}sge;
	return undef;
}

sub parse ($$;$$) {
	my $self = shift;
	my $file_name = shift || die "No template to parse";

	my $stash = shift || {};
	$stash = { %{$self->{default_stash}}, %$stash };

	return $self->_dump($stash, $self->_parse($file_name));
}

sub _parse ($$;$) {
	my $self = shift;
	my $file_name = shift || die "No template to parse";
	my $options = shift || {};

	my ($content, $error);

	# template from string
	if (ref($file_name) eq 'SCALAR') {
		$content = $$file_name;
		$file_name = "(inline)";

	# via http
 	} elsif ($file_name =~ m!^http://!) {
 		$file_name =~ s!__SERVER__!$ENV{SERVER_NAME} || "localhost"!e;

 		require Arch::LiteWeb;
 		my $web = Arch::LiteWeb->new;
 		$content = $web->get($file_name);

		$error = $web->error . " while processing " . $web->request_url . "\n"
			unless defined $content;

	# from file
	} else {
		my $dir = $self->{set_dir};
		$file_name .= ".html" unless $file_name =~ /\./;
		$file_name = "$dir/$file_name" unless $file_name =~ /^\//;

		open(TMPL, "< $file_name")
			&& ($content = join("", <TMPL>))
			&& close(TMPL);

		$error = $!
			unless defined $content;
	}

	unless (defined $content) {
		my $msg = "Failed to load template '$file_name': $error\n";

		$options->{die_on_error}
			? die $msg
			: warn $msg;

		$content = "<b>$msg</b>"
			if $options->{include_error};

		$options->{dont_parse} = 1;
	}

	# return content verbatim if dont_parse is set
	return [ { type => 'DATA', data => $content } ]
		if $options->{dont_parse};

	# prepare parse context
	my $ctx;
	$ctx = {
		data      => $content,
		file      => $file_name,
		linecount => scalar @{[ $content =~ /\015\012|\015|\012/g ]},
		error     => sub ($$) {
			my ($op, $msg) = @_;
			my $line = &_linecount($ctx);
			my $near = substr($ctx->{data}, 0, 40);
			$near =~ s/\\/\\\\/g;
			$near =~ s/\015/\\r/g;
			$near =~ s/\012/\\n/g;
			$near =~ s/\'/\\\'/g;

			my $error = defined $op 
				? "Syntax error while parsing $op directive: $msg.\n"
				: "Syntax error: $msg.\n";
			$error .= "At $file_name, line $line, near '$near'.\n";
			$error .= "Parse Stack:\n";

			foreach (@{$ctx->{stack}}) {
				$error .= "  $_->[0] at line $_->[1]\n";
			}

			die $error;
		},
		stack => [],
	};

	# parse content
	my @elements = ();
	while (length($ctx->{data})) {
		push @elements, $self->_parse_element($ctx);
	}

	return \@elements;
}

sub _parse_element ($$) {
	my $self = shift;
	my $ctx = shift;

	my $element = {
		file => $ctx->{file},
		line => &_linecount($ctx),
	};
	
	# <[command
	if ($ctx->{data} =~ s/^<\[\s*(\w[\w-]*|\#)\s+//) {
		my $op = $1;

		push @{$ctx->{stack}}, [ $op, $element->{line} ];

		$element->{type} = $op;

		# <[# ... ]>
		if ($op eq '#') {
			while ($ctx->{data} !~ s/^\]>//) {
				$self->_parse_element($ctx); # throw away
			}


		# <[INCLUDE(-MEMOIZE) variables file substitution]>
		} elsif ($op eq 'INCLUDE' || $op eq 'INCLUDE-MEMOIZE') {
			$element->{fields} = {};
			while ($ctx->{data} =~ s/^(\w+)=([\'\"]?)([^\'\"]*)\2\s+//) {
				$element->{fields}->{$1} = $3;
			}

			$ctx->{data} =~ s/^([^\s]+?)(?:\s+(s[^\w].*?))?\s*\]>//
				or $ctx->{error}->($op, "missing filename");

			$element->{file} = $1;
			$element->{filter} = $2;

		# <[FOREACH variables array strictflag ...]>
		} elsif ($op eq 'FOREACH') {
			$element->{varlist} = [];
			while ($ctx->{data} =~ s/^\$(\w+)\s+//) {
				push @{$element->{varlist}}, $1;
			}
			
			$ctx->{data} =~ s/^\@(\w+)(!?)\s//
				or $ctx->{error}->($op, "error in variable list");

			$element->{array} = $1;
			$element->{strict} = !!$2;

			$element->{subcontent} = [];
			while ($ctx->{data} !~ s/^\]>//) {
				push @{$element->{subcontent}}, $self->_parse_element($ctx);
			}

		# <[(ELS)IF condition THEN ...]>
		} elsif ($op eq 'IF' || $op eq 'ELSIF') {
			$ctx->{data} =~ s/^(.*?)\s+THEN\b\s?//
				or $ctx->{error}->($op, "error in condition");

			$element->{condition} = $1;

			$element->{subcontent} = [];
			while ($ctx->{data} !~ s/^\]>//) {
				push @{$element->{subcontent}}, $self->_parse_element($ctx);
			}

		# <[(ELS)IF-DEFINED|ENABLED variables ...]>
		} elsif ($op =~ /^(ELS)?IF-(DEFINED|ENABLED)$/) {
			$ctx->{data} =~ s/^(!?(?:\w+\.)?\w+)\s*//
				or $ctx->{error}->($op, "error in variable list");

			$element->{varlist} = [ $1 ];
			while ($ctx->{data} =~ s/^,\s*(!?(?:\w+\.)?\w+)\s*//) {
				push @{$element->{varlist}}, $1;
			}

			$element->{subcontent} = [];
			while ($ctx->{data} !~ s/^\]>//) {
				push @{$element->{subcontent}}, $self->_parse_element($ctx);
			}

		# <[SET name=value]>
		} elsif ($op eq 'SET') {
			$element->{variables} = [];
			while ($ctx->{data} =~ s/^(\w+)=(?:([\'\"\`])(.*?)\2|(\S+))\s*//) {
				my $val = defined $3 ? $3 : $4;
				push @{$element->{variables}}, [ $1, $self->_parse(\$val) ];
			}

			$ctx->{data} =~ s/\]>//
				or $ctx->{error}->($op, "assignment expected");

		# <[ELSE ...]>
		} elsif ($op eq 'ELSE') {
			$element->{subcontent} = [];
			while ($ctx->{data} !~ s/^\]>//) {
				push @{$element->{subcontent}}, $self->_parse_element($ctx);
			}

		} else {
			$ctx->{error}->(undef, "invalid directive '$op'");
			return undef;
		}

		pop @{$ctx->{stack}};

	# data (anything that does not contain "<[" or "]>")
	} elsif ($ctx->{data} =~ s/^(?:[^\]<]|\](?!>)|<(?!\[))+//) {
		$element->{type} = 'DATA';
		$element->{data} = $&;

	# syntax error
	} else {
		$ctx->{error}->(undef, "directive or data expected");
		return undef;
	}

	return $element;
}

sub _dump ($$$) {
	my $self = shift;
	my $stash = shift;
	my $block = shift;

	my $env = {
		stash       => $stash,
		index_total => 0,
		last_if     => 1,
		error       => sub ($$) {
			my $op = shift;
			my $msg = shift;

			die "Error in $op->{type} directive: $msg.\nIn $op->{file}, line $op->{line}.";
		},
	};
	
	return $self->_dump_block($env, $block);
}

sub _dump_block ($$$) {
	my $self = shift;
	my $env = shift;
	my $block = shift;

	die "come again?" unless ref($env) eq 'HASH';
	local $env->{last_if} = 1;

	my $output = "";
	foreach my $element (@$block) {
		$output .= $self->_dump_element($env, $element);
	}

	return $output;
}

sub _dump_element ($$$) {
	my $self = shift;
	my $env = shift;
	my $element = shift;

	my $op = $element->{type};

	my $output = "";

	# data stuff
	if ($op eq 'DATA') {
		$output = $element->{data};
		expand_stash($env->{stash}, $output);

	# <[#
	} elsif ($op eq '#') {
		# it's a comment buddy

	# <[INCLUDE
	} elsif ($op eq 'INCLUDE') {
		my $file = $element->{file};
		expand_stash($env->{stash}, $file);

		my $substash = { %{$env->{stash}} };
		while (my ($key, $val) = each %{$element->{fields}}) {
			expand_stash($env->{stash}, $val);
			$substash->{$key} = $val;
		}

		my $options = {
			dont_parse => $substash->{dont_parse},
			die_on_error => $substash->{die_on_error},
			include_error => $substash->{include_error},
		};

		$output = $self->parse($file, $substash, $options);
		eval qq(\$output =~ $element->{filter}; 1) || warn $@
			if $element->{filter};

	# <[INCLUDE-MEMOIZE
	} elsif ($op eq 'INCLUDE-MEMOIZE') {
		my $file = $element->{file};
		expand_stash($env->{stash}, $file);

		if (! exists $ArchZoom::_TemplateIncludeCache{$file}) {
			my $substash = { %{$env->{stash}} };
			while (my ($key, $val) = each %{$element->{fields}}) {
				expand_stash($env->{stash}, $val);
				$substash->{$key} = $val;
			}

			my $options = {
				dont_parse => $substash->{dont_parse},
				die_on_error => $substash->{die_on_error},
				include_error => $substash->{include_error},
			};

			$ArchZoom::_TemplateIncludeCache{$file} =
				$self->parse($file, $substash, $options);
		}

		eval qq(\$ArchZoom::_TemplateIncludeCache{\$file} =~ $element->{filter}; 1) || warn $@
			if $element->{filter};

		$output = $ArchZoom::_TemplateIncludeCache{$file};

	# <[FOREACH
	} elsif ($op eq 'FOREACH') {
		my $name = $element->{array};
		my $array = $env->{stash}->{$name};
		my @varlist = @{$element->{varlist}};

		$array ||= []
			unless $element->{strict};
		$env->{error}->($element, "array $name not defined")
			unless ref($array) eq 'ARRAY';

		my $index = 0;
		foreach my $value (@$array) {
			local $env->{stash} = {
				%{$env->{stash}},
				index  => $index,
				count  => $index + 1,
				zebra0 => $index % 2,
				index_total => $env->{index_total},
				count_total => $env->{index_total} + 1,
				zebra       => $env->{index_total} % 2,
			};

			if (@varlist == 1) {
				$env->{stash}->{$varlist[0]} = $value;

			} else {
				if (ref($value) eq 'HASH') {
					$value = [ map { 
						$env->{error}->($element, "\$${name}->[$index] has no key '$_'")
							unless exists $value->{$_};
						$value->{$_}
					} @varlist ];
				}

				$env->{error}->($element, "\$${name}->[$index] is neither a hash nor an array")
					unless (ref($value) eq 'ARRAY');
				$env->{error}->($element, "\$${name}->[$index]: size does not match variable list")
					unless @$value == @varlist;

				foreach (0..$#varlist) {
					$env->{stash}->{$varlist[$_]} = $value->[$_];
				}
			}

			$output .= $self->_dump_block($env, $element->{subcontent});
			++$index; ++$env->{index_total};
		}


	# <[IF/ELSIF
	} elsif ($op eq 'IF' || $op eq 'ELSIF') {
		return "" if $op eq 'ELSIF' && $env->{last_if};

		my $condition = $element->{condition};
		expand_stash($env->{stash}, $condition);

		$env->{last_if} = eval("package main; $condition");

		$output = $self->_dump_block($env, $element->{subcontent})
			if $env->{last_if};

	# <[ELSE
	} elsif ($op eq 'ELSE') {
		$output = $self->_dump_block($env, $element->{subcontent})
			unless $env->{last_if};

	# <[IF/ELSIF-DEFINED/ENABLED
  	} elsif ($op =~ /^(ELS)?IF-(DEFINED|ENABLED)$/) {
  		return "" if defined $1 && $env->{last_if};
  		my $do_defined = $2 eq 'DEFINED';

  		$env->{last_if} = 1;
  		foreach (@{$element->{varlist}}) {
  			my $not = s/^!//;
  			/^(?:(\w+)\.)?(\w+)$/ or die;
  			my $value = $1
				? do {
					my $this = $env->{stash}->{$1};
					ref($this) eq 'HASH'? $this->{$2}: $this->$2()
				}
				: $env->{stash}->{$2};
  			$env->{last_if} &&= ($do_defined? defined $value: $value? 1: 0) ^ $not;
  		}

  		$output = $self->_dump_block($env, $element->{subcontent})
  			if $env->{last_if};

	# <[SET
	} elsif ($op eq 'SET') {
		foreach (@{$element->{variables}}) {
			my ($key, $value) = @$_;

			$env->{stash}->{$key} = $self->_dump_block($env, $value);
		}

	# error
	} else {
		$env->{error}->($element, 'invalid directive; broken template parser');
	}

	return $output;
}

sub _linecount ($) {
	my $ctx = shift;

	return $ctx->{linecount} - @{[ $ctx->{data} =~ /\015\012|\015|\012/g ]} + 1;
}

1;
