
/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (C) 2000-2001 QoSient, LLC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */


/*
 * argus client library
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#ifndef ArgusClient
#define ArgusClient
#endif

#include <argus_client.h>
#include <math.h>


struct ArgusRecord *
RaCopyArgusRecord (struct ArgusRecord *argus)
{
   int length = 0;
   struct ArgusRecord *retn = NULL;

   if (argus && ((length = argus->ahdr.length) > 0))
      if ((retn = (struct ArgusRecord *) ArgusCalloc(1, length)) != NULL)
         bcopy ((char *) argus, (char *) retn, length);

   return (retn);
}


long long
RaGetActiveDuration (struct ArgusRecord *argus)
{
   return (RaGetuSecDuration (argus));
}

long long
RaGetuSecDuration (struct ArgusRecord *argus)
{
   struct ArgusAGRStruct *agr;
   struct ArgusFarHeaderStruct *argusthisfarhdrs[32];
   long long retn = 0;
   int sec, usec;

   ArgusIndexRecord (argus,  argusthisfarhdrs);

   if ((agr = (struct ArgusAGRStruct *) argusthisfarhdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
      int ArgusThisMultiplier = 1000;

      if (agr->status & ARGUS_AGR_USECACTTIME)
         ArgusThisMultiplier = 1;

      if (agr->act.n)
         return (agr->act.mean * ArgusThisMultiplier);
   }

   sec  = argus->argus_far.time.last.tv_sec  - argus->argus_far.time.start.tv_sec;
   usec = argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec;

   if (usec < 0) {
      sec--; usec += 1000000;
   }

   retn = (sec * 1000000) + usec;

   return (retn);
}

int
RaCIDRAddrMatches (u_int addr, struct RaCIDRAddr *cidr)
{
   int retn = 1;

   switch (cidr->type) {

      case RA_CIDR_TYPE:
         if (cidr->addr) {
            if ((addr & cidr->mask) != cidr->addr)
               retn = 0;
         }
         break;

      case RA_NETCLASS_TYPE:
         if (cidr->addr) {
            if ((addr & ipaddrtonetmask(addr)) != cidr->addr)
               retn = 0;
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCIDRAddrMatches(0x%x, 0x%x) returned %d\n", addr, cidr, retn);
#endif

   return (retn);
}

int
RaProtoMatches (u_short p, int proto)
{
   int retn = 1;

   if (proto != 0xFF)
      retn = (p == proto) ? 1 : 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProtoMatches (%d, %d) returned %d\n", p, proto, retn);
#endif

   return (retn);
}


int
RaPortMatches (u_short p1, u_short p2)
{
   int retn = 1;
   
   if (p2 != 0xFFFF)
      retn = (p1 == p2) ? 1 : 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPortMatches (%d, %d) returned %d\n", p1, p2, retn);
#endif

   return (retn);
}

int
RaPolicyMatch (struct ArgusRecord *argus, struct RaPolicyStruct *rap)
{
   int retn = 0;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   u_char proto = flow->ip_flow.ip_p;

   if (RaCIDRAddrMatches (flow->ip_flow.ip_src, &rap->src))

   if (RaCIDRAddrMatches (flow->ip_flow.ip_dst, &rap->dst))

   if (RaProtoMatches (flow->ip_flow.ip_p, rap->proto))

   switch (proto) {
      case IPPROTO_TCP:
      case IPPROTO_UDP:
         if (RaPortMatches (flow->ip_flow.sport, rap->sport))
         if (RaPortMatches (flow->ip_flow.dport, rap->dport))
            retn = 1;
         break;

      case IPPROTO_ICMP:
         if (RaPortMatches (flow->icmp_flow.type, rap->sport))
         if (RaPortMatches (flow->icmp_flow.code, rap->dport))
         retn = 1;
         break;

      default:
         retn = 1;
         break;
   }


#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPolicyMatch(0x%x, 0x%x) returned %d\n", argus, rap, retn);
#endif

   return (retn);
}


struct RaPolicyStruct *
RaFlowModelOverRides (struct ArgusRecord *argus)
{
   struct RaPolicyStruct *retn = NULL;
   int i;

   if (RaThisFlowNum) {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            for (i = 0; i < RaThisFlowNum; i++) {
               if (RaPolicyMatch (argus, RaFlowArray[i])) {
                  retn = RaFlowArray[i];
                  break;
               }
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            break;

         default:
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaFlowModelOverRides(0x%x) returned 0x%x\n", argus, retn);
#endif

   return (retn);
}

#if defined(__OpenBSD__)
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#endif

#include <netinet/ip_icmp.h>

void
RaModifyDefaultFlow (struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;

   switch (argus->ahdr.status & 0xFFFF) {
      case ETHERTYPE_IP:

         if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_FRAG_FLOWTAG) {
            argus->argus_far.flow.ip_flow.tp_p = 0;
            argus->argus_far.flow.ip_flow.ip_id = 0;
         }

         switch (argus->argus_far.flow.ip_flow.ip_p) {
            case IPPROTO_UDP:
            case IPPROTO_TCP:
               flow->icmp_flow.ip_id = 0;
               break;

            case IPPROTO_ICMP:
               if (flow->icmp_flow.type != ICMP_UNREACH)
                  flow->icmp_flow.id = 0xFFFF;

               flow->icmp_flow.ip_id = 0xFFFF;
               break;

            default:
               flow->icmp_flow.ip_id = 0;
               break;
         }
         break;

      case ETHERTYPE_ARP:
      case ETHERTYPE_REVARP:
         break;

      default:
         break;
   }
}

int RaFlowMajorModified = 0;

void
RaModifyFlow (struct RaPolicyStruct *rap, struct ArgusRecord *argus)
{
   struct RaPolicyStruct *model = RaModelArray[rap->RaModelId];
   struct ArgusFlow *flow = &argus->argus_far.flow;
   u_short proto;

   RaFlowMajorModified = 0;

   switch (argus->ahdr.status & 0xFFFF) {
      case ETHERTYPE_IP:
         if (flow->ip_flow.tp_p == ARGUS_FRAG_FLOWTAG) {
            flow->ip_flow.tp_p = 0;
            flow->ip_flow.ip_id = 0;
         }

         if (model) {
            switch (model->src.type) {
               case RA_CIDR_TYPE:
                  flow->ip_flow.ip_src = flow->ip_flow.ip_src & model->src.addr;
                  break;

               case RA_NETCLASS_TYPE:
                  flow->ip_flow.ip_src = flow->ip_flow.ip_src & ipaddrtonetmask(flow->ip_flow.ip_src);
                  break;
            }

            switch (model->dst.type) {
               case RA_CIDR_TYPE:
                  flow->ip_flow.ip_dst = flow->ip_flow.ip_dst & model->dst.addr;
                  break;

               case RA_NETCLASS_TYPE:
                  flow->ip_flow.ip_dst = flow->ip_flow.ip_dst & ipaddrtonetmask(flow->ip_flow.ip_dst);
                  break;
            }

            proto = flow->ip_flow.ip_p;
            flow->ip_flow.ip_p  &= model->proto;

            switch (proto) {
               case IPPROTO_TCP:
               case IPPROTO_UDP:
                  flow->ip_flow.ip_id = 0;

                  switch (flow->ip_flow.ip_p) {
                     case IPPROTO_TCP:
                        if (flow->ip_flow.sport == 20) {
                           if (!(model->sport))
                              flow->ip_flow.dport = 0xFFFF;
                           if (!(model->dport)) {
                              flow->ip_flow.sport = 0xFFFF;
                              RaFlowMajorModified = 1;
                           }
                        } else

                     case IPPROTO_UDP:
                        if (!(model->sport))
                           flow->ip_flow.sport = 0xFFFF;
                        if (!(model->dport)) {
                           flow->ip_flow.dport = 0xFFFF;
                           RaFlowMajorModified = 1;
                        }
                        break;

                     default:
                        flow->ip_flow.sport = 0xFFFF;
                        flow->ip_flow.dport = 0xFFFF;
                        break;
                  }
                  break;
          
               case IPPROTO_ICMP:
                  if (!(model->sport))
                     flow->icmp_flow.type = 0xFF;

                  if (!(model->dport))
                     flow->icmp_flow.code = 0xFF;

                  flow->icmp_flow.id = 0xFFFF;
                  flow->icmp_flow.ip_id = 0xFFFF;
                  flow->icmp_flow.tp_p = 0;
                  break;

               default:
                  flow->ip_flow.ip_id = 0;
                  flow->ip_flow.tp_p = 0;
                  flow->ip_flow.sport = 0xFFFF;
                  flow->ip_flow.dport = 0xFFFF;
                  break;
            }

            if (flow->ip_flow.ip_p == 0) {
               flow->ip_flow.tp_p  = 0;
               flow->ip_flow.ip_id = 0;
            }
         }
         break;

      default:
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaModifyFlow(0x%x, 0x%x) done.\n", rap, argus);
#endif

}


int RaThisHash = 0;
struct ArgusFlow *RaThisFlow = NULL;
int RaThisFlowRev = 0;

struct ArgusRecordStore *
RaFindRevArgusRecord(struct RaHashTableStruct *htable, struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct RaHashTableHeader *hashEntry;
   struct ArgusFlow flow = *RaThisFlow;
   struct ArgusTCPObject *tcp = NULL;

   RaThisFlowRev = 0;

   switch (flow.ip_flow.ip_p) {
      case IPPROTO_UDP:              
         RaThisFlow->ip_flow.ip_src = flow.ip_flow.ip_dst;
         RaThisFlow->ip_flow.ip_dst = flow.ip_flow.ip_src;
         RaThisFlow->ip_flow.sport  = flow.ip_flow.dport;
         RaThisFlow->ip_flow.dport  = flow.ip_flow.sport;
         RaThisFlowRev++;

         if ((hashEntry = RaFindHashObject (htable)) != NULL)
            retn = hashEntry->storeobj;

         *RaThisFlow = flow;
         break;

      case IPPROTO_TCP:              
         if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
            if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL)
               if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT))
                  break;

         RaThisFlow->ip_flow.ip_src = flow.ip_flow.ip_dst;
         RaThisFlow->ip_flow.ip_dst = flow.ip_flow.ip_src;
         RaThisFlow->ip_flow.sport  = flow.ip_flow.dport;
         RaThisFlow->ip_flow.dport  = flow.ip_flow.sport;
         RaThisFlowRev++;

         if ((hashEntry = RaFindHashObject (htable)) != NULL)
            retn = hashEntry->storeobj;

         *RaThisFlow = flow;
         break;

      default:
         break;
   }


   return (retn);
}


struct ArgusRecordStore *
RaFindArgusRecord(struct RaHashTableStruct *htable, struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct RaHashTableHeader *hashEntry;
   unsigned short hash = 0, *ptr;
   int i, len;

   RaThisFlowRev = 0;
   RaThisFlow = &argus->argus_far.flow;
   ptr = (unsigned short *) RaThisFlow;

   for (i = 0, len = (sizeof(*RaThisFlow)) / sizeof(unsigned short); i < len; i++)
      hash += *ptr++;

   RaThisHash = hash;

   if ((hashEntry = RaFindHashObject (htable)) != NULL)
      retn = hashEntry->storeobj;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaFindArgusRecord: returning 0x%x\n", retn);
#endif

   return (retn);
}


struct ArgusRecordStore *
RaNewArgusStore(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct timeval lasttime;

   if ((retn = (struct ArgusRecordStore *) ArgusCalloc (1, sizeof(struct ArgusRecordStore))) != NULL) {
      RaAllocArgusRecordStore++;
      lasttime = argus->argus_far.time.last;

      retn->qhdr.lasttime = lasttime;
      retn->qhdr.logtime  = lasttime;
      if ((retn->data = (struct ArgusRecordData **) ArgusCalloc (RaHistoTimeSeries, sizeof(void *))) == NULL)
         ArgusLog (LOG_ERR, "RaNewArgusStore () ArgusCalloc %s\n", strerror(errno));
   } else
     ArgusLog (LOG_ERR, "RaNewArgusStore () ArgusCalloc %s\n", strerror(errno));

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaNewArgusStore: added 0x%x\n", retn);
#endif

   return (retn);
}

void RaDeleteArgusStore(struct ArgusRecordStore *);
void RaDeleteArgusData(struct ArgusRecordData *);

void
RaDeleteArgusStore(struct ArgusRecordStore *store)
{
   int i = 0;

   if (store->data != NULL) {
      for (i = 0; i < RaHistoTimeSeries; i++) 
         RaDeleteArgusData(store->data[i]);

      ArgusFree(store->data);
      store->data = NULL;
   }

   ArgusFree(store);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaDeleteArgusStore(0x%x): Done\n", store);
#endif
}

int RaAllocArgusRecordData = 0;
int RaTotalDataRecords = 0;

struct ArgusRecordData *
RaNewArgusData(struct ArgusRecord *argus)
{
   struct ArgusRecordData *retn = NULL;
   struct timeval lasttime;

   if ((retn = (struct ArgusRecordData *) ArgusCalloc (1, sizeof (*retn))) != NULL) {
      RaAllocArgusRecordData++;
   
      if ((retn->argus = RaCopyArgusRecord(argus)) != NULL) {
         RaTotalDataRecords++;

         retn->agr.laststartime = argus->argus_far.time.start;
         lasttime = argus->argus_far.time.last;

         retn->agr.type     = ARGUS_AGR_DSR;
         retn->agr.length   = sizeof(retn->agr);
         retn->agr.lasttime = lasttime;
         retn->agr.lasttime = lasttime;
         retn->agr.act.min  = RaThisActiveDuration;
         retn->agr.act.max  = RaThisActiveDuration;
         retn->act.sumtime  = RaThisActiveDuration;
         retn->act.sumsqrd  = pow (RaThisActiveDuration, 2.0);
         retn->act.n        = 1;
         retn->agr.count    = 1;

         retn->agr.idle.min = 0x7FFFFFFF;
         retn->agr.status  |= ARGUS_AGR_USECACTTIME;
         retn->agr.status  |= ARGUS_AGR_USECIDLETIME;

         bcopy (RaArgusFlow, (unsigned char *)&retn->flow, sizeof(retn->flow));

      } else {
         ArgusFree (retn);
         RaAllocArgusRecordData--;
         retn = NULL;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaNewArgusData: added 0x%x\n", retn);
#endif

   return (retn);
}

void
RaDeleteArgusData(struct ArgusRecordData *data)
{
   if (data->argus != NULL)
      ArgusFree(data->argus);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveArgusRecord: done\n");
#endif
}

void
RaRemoveArgusRecord(struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveArgusRecord: done\n");
#endif
}


void
RaMergeArgusRecord(struct ArgusRecord *a1, struct ArgusRecordData *data)
{
   u_char buf[MAXSTRLEN];
   struct ArgusRecord *a2 = NULL, *tmp = (struct ArgusRecord *)buf;
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   struct ArgusFarStruct *far1 = NULL;
   struct ArgusFarStruct *far2 = NULL;
   struct ArgusRecordStore store;
   struct ArgusFarStruct *ArgusThisDataFar;
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0;
   unsigned int i, status = 0, TimeMultiplier;
   long long duration;

   bzero ((char *)&store, sizeof(store));
   if ((store.data = (struct ArgusRecordData **) ArgusCalloc (RaHistoTimeSeries, sizeof(void *))) == NULL)
      ArgusLog (LOG_ERR, "RaMergeArgusRecord(): ArgusCalloc %s\n", strerror(errno));

   store.data[RaThisActiveIndex] = data;

   if ((a2 = data->argus) != NULL) {
      bzero (buf, sizeof(buf));
      bcopy ((char *) a2, buf, a2->ahdr.length);

      ArgusThisDataFar = (struct ArgusFarStruct *) data->farhdrs[ARGUS_FAR_DSR_INDEX];

      a1DSRStatus = ArgusIndexRecord (a1,  a1farhdr);
      a2DSRStatus = ArgusIndexRecord (tmp, a2farhdr);
/*
      if (RaFlowMajorModified) {
         a1DSRStatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
         data->farhdrstatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
      }
*/

      status = a1DSRStatus & a2DSRStatus;

      far1 = (struct ArgusFarStruct *) a1farhdr[ARGUS_FAR_DSR_INDEX];
      far2 = (struct ArgusFarStruct *) a2farhdr[ARGUS_FAR_DSR_INDEX];

      if (!(RaThisFlowRev)) {
         if (((far2->src.count + far1->src.count) < far2->src.count) ||
             ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
 
      } else {
         if (((far2->src.count + far1->dst.count) < far2->src.count) ||
             ((far2->dst.count + far1->src.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
      }

      for (i = 0; i < 32; i++) {
         if (status & 0x01) {
            switch (i) {
               case ARGUS_FAR_DSR_INDEX: {
                  u_int diffstartime = 0;
                  int diffusec = 0;

                  far1 = (struct ArgusFarStruct *) a1farhdr[i];
                  far2 = (struct ArgusFarStruct *) a2farhdr[i];

                  if (!(data->status & ARGUS_FAR_SRCADDR_MODIFIED))
                     if (RaArgusFlow->ip_flow.ip_src != data->flow.ip_flow.ip_src)
                        data->status |= ARGUS_FAR_SRCADDR_MODIFIED;
                  
                  if (!(data->status & ARGUS_FAR_DSTADDR_MODIFIED))
                     if (RaArgusFlow->ip_flow.ip_dst != data->flow.ip_flow.ip_dst)
                        data->status |= ARGUS_FAR_DSTADDR_MODIFIED;

                  if (!(data->status & ARGUS_FAR_PROTO_MODIFIED))
                     if (RaArgusFlow->ip_flow.ip_p != data->flow.ip_flow.ip_p)
                        data->status |= ARGUS_FAR_PROTO_MODIFIED;

                  if (!(data->status & ARGUS_FAR_SRCPORT_MODIFIED))
                     if (RaArgusFlow->ip_flow.sport != data->flow.ip_flow.sport)
                        data->status |= ARGUS_FAR_SRCPORT_MODIFIED;

                  if (!(data->status & ARGUS_FAR_DSTPORT_MODIFIED))
                     if (RaArgusFlow->ip_flow.dport != data->flow.ip_flow.dport)
                        data->status |= ARGUS_FAR_DSTPORT_MODIFIED;

                  if (!(data->status & ARGUS_FAR_TPVAL_MODIFIED))
                     if (RaArgusFlow->ip_flow.tp_p != data->flow.ip_flow.tp_p)
                        data->status |= ARGUS_FAR_TPVAL_MODIFIED;

                  if (!(RaThisFlowRev)) {
                     if (((far2->src.count + far1->src.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(&store);

                     far2->src.count    += far1->src.count;
                     far2->src.bytes    += far1->src.bytes;
                     far2->dst.count    += far1->dst.count;
                     far2->dst.bytes    += far1->dst.bytes;
                     far2->src.appbytes += far1->src.appbytes;
                     far2->dst.appbytes += far1->dst.appbytes;

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                     }

                  } else {
                     if (((far2->src.count + far1->dst.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->src.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(&store);

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                     }

                     far2->src.count    += far1->dst.count;
                     far2->src.bytes    += far1->dst.bytes;
                     far2->dst.count    += far1->src.count;
                     far2->dst.bytes    += far1->src.bytes;
                     far2->src.appbytes += far1->dst.appbytes;
                     far2->dst.appbytes += far1->src.appbytes;
                  }

                  diffstartime  = (far1->time.start.tv_sec  - data->agr.laststartime.tv_sec);
                  if ((diffusec = (far1->time.start.tv_usec - data->agr.laststartime.tv_usec)) < 0) {
                     diffstartime--;
                     diffusec += 1000000;
                  }

                  data->agr.laststartime = far1->time.start;

                  if (a1farhdr[ARGUS_AGR_DSR_INDEX] == NULL) {
                     if (diffstartime > 0) {
                        if (diffstartime > 2147) {
                           if (data->agr.status & ARGUS_AGR_USECIDLETIME) {
                              if (data->agr.idle.min != 0x7FFFFFFF)
                                 data->agr.idle.min /= 1000;
                              data->agr.idle.max /= 1000;
                              data->idle.sumtime /= 1000;
                              data->idle.sumsqrd /= 1000000.0;
                              data->agr.status &= ~ARGUS_AGR_USECIDLETIME;
                              data->agr.status |=  ARGUS_AGR_MSECIDLETIME;
                           }
                        }

                        if (data->agr.status & ARGUS_AGR_USECIDLETIME)
                           TimeMultiplier = 1000000;
                        else
                           TimeMultiplier = 1000;

                        diffstartime *= TimeMultiplier;
                        diffstartime += (diffusec * 1000) / (TimeMultiplier / 1000);

                        if ((data->agr.idle.min > diffstartime) && (diffstartime > 0))
                           data->agr.idle.min = diffstartime;

                        if ((data->agr.idle.max < diffstartime) && (diffstartime > 0))
                           data->agr.idle.max = diffstartime;

                        data->idle.sumtime += diffstartime;
                        data->idle.sumsqrd += pow(diffstartime, 2.0);
                        data->idle.n++;
                     }

                     duration = RaGetuSecDuration(a1);
                     if (duration > 0x6FFFFFFF) {
                        if (data->agr.status & ARGUS_AGR_USECIDLETIME) {
                            if (data->agr.idle.min != 0x7FFFFFFF)
                               data->agr.act.min /= 1000;
                            data->agr.act.max /= 1000;
                            data->act.sumtime /= 1000;
                            data->act.sumsqrd /= 1000000.0;
                            data->agr.status &= ~ARGUS_AGR_USECACTTIME;
                            data->agr.status |=  ARGUS_AGR_MSECACTTIME;
                        }
                     }

                     if (data->agr.status & ARGUS_AGR_USECACTTIME)
                        TimeMultiplier = 1;
                     else
                        TimeMultiplier = 1000;

                     duration /= TimeMultiplier;

                     if ((data->agr.act.min > duration) && (duration > 0))
                        data->agr.act.min = duration;

                     if ((data->agr.act.max < duration) && (duration > 0))
                        data->agr.act.max = duration;

                     data->act.sumtime += duration;
                     data->act.sumsqrd += pow(duration, 2.0);
                     data->act.n++;

                     data->agr.count++;
                  }

                  if (far1->ArgusTransRefNum != far2->ArgusTransRefNum)
                     far2->ArgusTransRefNum = 0;

                  if ((far2->time.start.tv_sec  > far1->time.start.tv_sec) ||
                     ((far2->time.start.tv_sec == far1->time.start.tv_sec) &&
                     (far2->time.start.tv_usec  > far1->time.start.tv_usec)))

                     far2->time.start = far1->time.start;

                  if ((far2->time.last.tv_sec  < far1->time.last.tv_sec) ||
                     ((far2->time.last.tv_sec == far1->time.last.tv_sec) &&
                     (far2->time.last.tv_usec  < far1->time.last.tv_usec)))

                     far2->time.last = far1->time.last;

                  if (RaThisArgusStore->qhdr.lasttime.tv_sec != far2->time.last.tv_sec) {
                     RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     data->agr.lasttime = far2->time.last;
                  } else
                     if (RaThisArgusStore->qhdr.lasttime.tv_usec < far2->time.last.tv_usec) {
                        RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     }

                  break;
               }

#define TCP_FIN_STATE_TIME	360

               case ARGUS_TCP_DSR_INDEX: {
                  struct ArgusTCPObject *tcp1 = (struct ArgusTCPObject *) a1farhdr[i];
                  struct ArgusTCPObject *tcp2 = (struct ArgusTCPObject *) a2farhdr[i];

                  if (ArgusFlowModelFile == NULL) {
                     if ((tcp2->state & (ARGUS_FIN | ARGUS_FIN_ACK | ARGUS_NORMAL_CLOSE | ARGUS_RESET)) > 0) {
                        if ((a2->argus_far.time.last.tv_sec > (a1->argus_far.time.start.tv_sec + TCP_FIN_STATE_TIME)) ||
                            (a1->argus_far.time.last.tv_sec > (a2->argus_far.time.start.tv_sec + TCP_FIN_STATE_TIME))) {

                           goto doneRaMergeArgus;
                        }
                     } else {
/*
                        if ((a2->argus_far.ArgusTransRefNum != a1->argus_far.ArgusTransRefNum)) {
                           goto doneRaMergeArgus;
                        }
*/
                     }
                  }

                  if (RaCumulativeMerge)
                     tcp2->state |= tcp1->state;
                  else
                     tcp2->state = tcp1->state;

                  if (!(tcp2->synAckuSecs))
                     tcp2->synAckuSecs = tcp1->synAckuSecs;
                  else
                     if (tcp2->synAckuSecs > tcp1->synAckuSecs)
                        if (tcp1->synAckuSecs)
                           tcp2->synAckuSecs = tcp1->synAckuSecs;

                  if (!(tcp2->ackDatauSecs))
                     tcp2->ackDatauSecs = tcp1->ackDatauSecs;
                  else
                     if (tcp2->ackDatauSecs > tcp1->ackDatauSecs)
                        if (tcp1->ackDatauSecs)
                           tcp2->ackDatauSecs = tcp1->ackDatauSecs;

                  tcp2->src.seqbase   = 0;
                  tcp2->dst.seqbase   = 0;

                  tcp2->src.ackbytes += tcp1->src.ackbytes;
                  tcp2->dst.ackbytes += tcp1->dst.ackbytes;
                  tcp2->src.bytes    += tcp1->src.bytes;
                  tcp2->dst.bytes    += tcp1->dst.bytes;
                  tcp2->src.rpkts    += tcp1->src.rpkts;
                  tcp2->dst.rpkts    += tcp1->dst.rpkts;        

                  if (tcp2->src.win > tcp1->src.win);
                     tcp2->src.win = tcp1->src.win;
                  if (tcp2->dst.win > tcp1->dst.win);
                     tcp2->dst.win = tcp1->dst.win;        

                  tcp2->src.flags    |= tcp1->src.flags;
                  tcp2->dst.flags    |= tcp1->dst.flags;        

                  break;
               }

               case ARGUS_RTP_DSR_INDEX: {
                  struct ArgusRTPObject *rtp1 = (struct ArgusRTPObject *) a1farhdr[i];
                  struct ArgusRTPObject *rtp2 = (struct ArgusRTPObject *) a2farhdr[i];

                  if ((((rtp2->sdrop + rtp1->sdrop) < rtp2->sdrop) || ((rtp2->ddrop + rtp1->ddrop) < rtp2->ddrop)) ||
                      (((rtp2->ssdev + rtp1->ssdev) < rtp2->ssdev) || ((rtp2->dsdev + rtp1->dsdev) < rtp2->dsdev))) {
                     RaSendArgusRecord(&store);
                     rtp2->sdrop = rtp1->sdrop;
                     rtp2->ddrop = rtp1->ddrop;
                     rtp2->ssdev = rtp1->ssdev;
                     rtp2->dsdev = rtp1->dsdev;

                  } else {
                     rtp2->sdrop += rtp1->sdrop;
                     rtp2->ddrop += rtp1->ddrop;
                     rtp2->ssdev += rtp1->ssdev;
                     rtp2->dsdev += rtp1->dsdev;
                  }
                  break;
               }

               case ARGUS_ICMP_DSR_INDEX:
               case ARGUS_IGMP_DSR_INDEX:
               case ARGUS_ARP_DSR_INDEX:
               case ARGUS_FRG_DSR_INDEX:
                  break;

               case ARGUS_VLAN_DSR_INDEX: {
                  struct ArgusVlanStruct *vlan1 = (struct ArgusVlanStruct *) a1farhdr[i];
                  struct ArgusVlanStruct *vlan2 = (struct ArgusVlanStruct *) a2farhdr[i];

                  if ((vlan2->status & ARGUS_SRC_VLAN) && (vlan1->status & ARGUS_SRC_VLAN))
                     if (vlan2->sid != vlan1->sid)
                        vlan2->status |= ARGUS_SRC_CHANGED;

                  if ((vlan2->status & ARGUS_DST_VLAN) && (vlan1->status & ARGUS_DST_VLAN))
                     if (vlan2->did != vlan1->did)
                        vlan2->status |= ARGUS_DST_CHANGED;
                  
                  vlan2->status |= vlan1->status;
                  vlan2->sid = vlan1->sid;
                  vlan2->did = vlan1->did;
                  break;
               }

               case ARGUS_MPLS_DSR_INDEX: {
                  struct ArgusMplsStruct *mpls1 = (struct ArgusMplsStruct *) a1farhdr[i];
                  struct ArgusMplsStruct *mpls2 = (struct ArgusMplsStruct *) a2farhdr[i];

                  if ((mpls2->status & ARGUS_SRC_MPLS) && (mpls1->status & ARGUS_SRC_MPLS))
                     if (mpls2->slabel != mpls1->slabel)
                        mpls2->status |= ARGUS_SRC_CHANGED;

                  if ((mpls2->status & ARGUS_DST_MPLS) && (mpls1->status & ARGUS_DST_MPLS))
                     if (mpls2->dlabel != mpls1->dlabel)
                        mpls2->status |= ARGUS_DST_CHANGED;
                  
                  mpls2->status |= mpls1->status;
                  mpls2->slabel = mpls1->slabel;
                  mpls2->dlabel = mpls1->dlabel;
                  break;
               }

               case ARGUS_ESP_DSR_INDEX: {
                  struct ArgusESPStruct *esp1 = (struct ArgusESPStruct *) a1farhdr[i];
                  struct ArgusESPStruct *esp2 = (struct ArgusESPStruct *) a2farhdr[i];

                  if (esp1 && esp2) {
                     if (esp1->src.spi != esp2->src.spi)
                        esp2->src.spi = 0;

                     if (esp1->dst.spi != esp2->dst.spi)
                        esp2->dst.spi = 0;

                     esp2->src.lastseq = esp1->src.lastseq;
                     esp2->dst.lastseq = esp1->dst.lastseq;
                     esp2->src.lostseq = esp1->src.lostseq;
                     esp2->dst.lostseq = esp1->dst.lostseq;
                  }
                  break;
               }
    
               case ARGUS_TIME_DSR_INDEX: {
                  struct ArgusTimeStruct *time1 = (struct ArgusTimeStruct *) a1farhdr[i];
                  struct ArgusTimeStruct *time2 = (struct ArgusTimeStruct *) a2farhdr[i];
                  long long sumtime, sumtime1, sumtime2;
                  double sumsqrd, sumsqrd1, sumsqrd2;

                  if (time1 && time2) {
                     if ((time2->src.act.n += time1->src.act.n) > 0) {
                        sumtime1 = (time1->src.act.mean * time1->src.act.n);
                        sumtime2 = (time2->src.act.mean * time2->src.act.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->src.act.mean  = sumtime/time2->src.act.n;
                        if (time1->src.act.n)
                           sumsqrd1 = (time1->src.act.n * pow(time1->src.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.act.n * pow(time2->src.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.act.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->src.act.stdev = sqrt (sumsqrd/time2->src.act.n - pow (sumtime/time2->src.act.n, 2.0));
                        if (time2->src.act.max < time1->src.act.max)
                           time2->src.act.max = time1->src.act.max; 
                        if (time2->src.act.min > time1->src.act.min)
                           time2->src.act.min = time1->src.act.min; 
                     }

                     if ((time2->dst.act.n += time1->dst.act.n) > 0) {
                        sumtime1 = (time1->dst.act.mean * time1->dst.act.n);
                        sumtime2 = (time2->dst.act.mean * time2->dst.act.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->dst.act.mean  = sumtime/time2->dst.act.n;
                        if (time1->dst.act.n)
                           sumsqrd1 = (time1->dst.act.n * pow(time1->dst.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.act.n * pow(time2->dst.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.act.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->dst.act.stdev = sqrt (sumsqrd/time2->dst.act.n - pow (sumtime/time2->dst.act.n, 2.0));
                        if (time2->dst.act.max < time1->dst.act.max)
                           time2->dst.act.max = time1->dst.act.max; 
                        if (time2->dst.act.min > time1->dst.act.min)
                           time2->dst.act.min = time1->dst.act.min; 
                     }

                     if ((time2->src.idle.n += time1->src.idle.n) > 0) {
                        sumtime1 = (time1->src.idle.mean * time1->src.idle.n);
                        sumtime2 = (time2->src.idle.mean * time2->src.idle.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->src.idle.mean  = sumtime/time2->src.idle.n;
                        if (time1->src.idle.n)
                           sumsqrd1 = (time1->src.idle.n * pow(time1->src.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.idle.n * pow(time2->src.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.idle.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->src.idle.stdev = sqrt (sumsqrd/time2->src.idle.n - pow (sumtime/time2->src.idle.n, 2.0));
                        if (time2->src.idle.max < time1->src.idle.max)
                           time2->src.idle.max = time1->src.idle.max; 
                        if (time2->src.idle.min > time1->src.idle.min)
                           time2->src.idle.min = time1->src.idle.min; 
                     }

                     if ((time2->dst.idle.n += time1->dst.idle.n) > 0) {
                        sumtime1 = (time1->dst.idle.mean * time1->dst.idle.n);
                        sumtime2 = (time2->dst.idle.mean * time2->dst.idle.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->dst.idle.mean  = sumtime/time2->dst.idle.n;
                        if (time1->dst.idle.n)
                           sumsqrd1 = (time1->dst.idle.n * pow(time1->dst.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.idle.n * pow(time2->dst.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.idle.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->dst.idle.stdev = sqrt (sumsqrd/time2->dst.idle.n - pow (sumtime/time2->dst.idle.n, 2.0));
                        if (time2->dst.idle.max < time1->dst.idle.max)
                           time2->dst.idle.max = time1->dst.idle.max; 
                        if (time2->dst.idle.min > time1->dst.idle.min)
                           time2->dst.idle.min = time1->dst.idle.min; 
                     }
                  }

                  break;
               }

               case ARGUS_SRCUSRDATA_DSR_INDEX:
               case ARGUS_DSTUSRDATA_DSR_INDEX: {
                  struct ArgusUserStruct *user1 = (struct ArgusUserStruct *) a1farhdr[i];
                  struct ArgusUserStruct *user2 = (struct ArgusUserStruct *) a2farhdr[i];
                  int len, len1 = 0, len2 = 0;

                  if (user1 && user2) {
                     len1 = (user1->length - 1) * 4;
                     len2 = (user2->length - 1) * 4;
                     len = (len1 > len2) ? len2 : len1;
                     if (len2 > len1)
                        bzero (&user2->data, len2);
                     bcopy (&user1->data, &user2->data, len);
                  }
                  
                  break;
               }

               case ARGUS_AGR_DSR_INDEX: {
                  struct ArgusAGRStruct *agr1 = (struct ArgusAGRStruct *) a1farhdr[i];
                  unsigned int agr1sumtime, sumtime;
                  double agr1sumsqrd, sumsqrd;
                  int overlapping = 1;
                  long long newvalue = 0;

                  if (!((data->agr.status & ARGUS_AGR_USECACTTIME) &&
                            (agr1->status & ARGUS_AGR_USECACTTIME))) {

                     if (data->agr.status & ARGUS_AGR_USECACTTIME) {
                         data->agr.act.max   /= 1000;
                         data->agr.act.min   /= 1000;
                         data->agr.act.mean  /= 1000;
                         data->agr.act.stdev /= 1000;
                         data->act.sumtime /= 1000;
                         data->act.sumsqrd /= 1000000.0;
                         data->agr.status &= ~ARGUS_AGR_USECACTTIME;
                         data->agr.status |=  ARGUS_AGR_MSECACTTIME;
                     }

                     if (agr1->status & ARGUS_AGR_USECACTTIME) {
                         agr1->act.max   /= 1000;
                         agr1->act.min   /= 1000;
                         agr1->act.mean  /= 1000;
                         agr1->act.stdev /= 1000;
                         agr1->status &= ~ARGUS_AGR_USECACTTIME;
                         agr1->status |=  ARGUS_AGR_MSECACTTIME;
                     }
                  }

                  if (data->agr.act.min > agr1->act.min)
                     data->agr.act.min = agr1->act.min;
                  if (data->agr.act.max < agr1->act.max)
                     data->agr.act.max = agr1->act.max;

                  if (agr1->act.n) {
                     agr1sumtime = agr1->act.mean * agr1->act.n;
                     agr1sumsqrd = (agr1->act.n * pow(agr1->act.stdev, 2.0)) +
                                    pow(agr1sumtime, 2.0)/agr1->act.n;

                     data->act.sumtime += agr1sumtime;
                     data->act.sumsqrd += agr1sumsqrd;

                     sumtime = data->act.sumtime;
                     sumsqrd = data->act.sumsqrd;

                     data->agr.act.n += agr1->act.n;
                     data->act.n += agr1->act.n;

                     data->agr.act.mean = sumtime/data->act.n;
                     data->agr.act.stdev = sqrt (sumsqrd/data->act.n -
                                           pow (sumtime/data->act.n, 2.0));
                  }

                  if (!((data->agr.status & ARGUS_AGR_USECIDLETIME) &&
                            (agr1->status & ARGUS_AGR_USECIDLETIME))) {

                     if (data->agr.status & ARGUS_AGR_USECIDLETIME) {
                         data->agr.idle.max   /= 1000;
                         data->agr.idle.min   /= 1000;
                         data->agr.idle.mean  /= 1000;
                         data->agr.idle.stdev /= 1000;
                         data->idle.sumtime /= 1000;
                         data->idle.sumsqrd /= 1000000.0;
                         data->agr.status &= ~ARGUS_AGR_USECIDLETIME;
                         data->agr.status |=  ARGUS_AGR_MSECIDLETIME;
                     }

                     if (agr1->status & ARGUS_AGR_USECIDLETIME) {
                         agr1->idle.max   /= 1000;
                         agr1->idle.min   /= 1000;
                         agr1->idle.mean  /= 1000;
                         agr1->idle.stdev /= 1000;
                         agr1->status &= ~ARGUS_AGR_USECIDLETIME;
                         agr1->status |=  ARGUS_AGR_MSECIDLETIME;
                     }
                  }

                  if (data->agr.status & ARGUS_AGR_USECIDLETIME)
                     TimeMultiplier = 1;
                  else
                     TimeMultiplier = 1000;

                  overlapping = 1;

                  if ((a2->argus_far.time.start.tv_sec < a1->argus_far.time.start.tv_sec) ||
                     ((a2->argus_far.time.start.tv_sec == a1->argus_far.time.start.tv_sec) &&
                      (a2->argus_far.time.start.tv_usec < a1->argus_far.time.start.tv_usec))) {

                     if ((a2->argus_far.time.last.tv_sec < a1->argus_far.time.start.tv_sec) ||
                        ((a2->argus_far.time.last.tv_sec == a1->argus_far.time.start.tv_sec) &&
                         (a2->argus_far.time.last.tv_usec < a1->argus_far.time.start.tv_usec))) {

                        newvalue = (((agr1->laststartime.tv_sec  - data->agr.laststartime.tv_sec) * 1000000) +
                                     (agr1->laststartime.tv_usec - data->agr.laststartime.tv_usec))/TimeMultiplier;
                        overlapping = 0;
                     }

                  } else {
                     if ((a1->argus_far.time.last.tv_sec < a2->argus_far.time.start.tv_sec) ||
                        ((a1->argus_far.time.last.tv_sec == a2->argus_far.time.start.tv_sec) &&
                         (a1->argus_far.time.last.tv_usec < a2->argus_far.time.start.tv_usec))) {
                        newvalue = (((data->agr.laststartime.tv_sec  - agr1->laststartime.tv_sec) * 1000000) +
                                     (data->agr.laststartime.tv_usec - agr1->laststartime.tv_usec))/TimeMultiplier;
                        overlapping = 0;
                     }
                  }

                  if (!(overlapping)) {
                     sumtime = (agr1->idle.mean * agr1->idle.n) + newvalue;
                     sumsqrd = ((agr1->idle.n * pow(agr1->idle.stdev, 2.0)) + pow(sumtime, 2.0)/agr1->idle.n) +
                                 pow(newvalue, 2.0);

                     data->idle.sumtime += sumtime;
                     data->idle.sumsqrd += sumsqrd;
                     data->idle.n += agr1->idle.n + 1;

                     if (newvalue) {
                        if (newvalue > data->agr.idle.max) 
                           data->agr.idle.max = newvalue;
                        else
                        if (newvalue < data->agr.idle.min)
                           data->agr.idle.min = newvalue;
                     }

                     data->agr.count += agr1->count;
                  } else {
                     if (data->agr.idle.min >= agr1->idle.min) {
                        data->idle.sumtime += agr1->idle.min;
                        data->idle.sumsqrd += pow(agr1->idle.min, 2.0);
                        data->idle.n++;
                        data->agr.idle.min = agr1->idle.min;
                     }
                  }

                  if ((data->agr.laststartime.tv_sec > agr1->laststartime.tv_sec) ||
                      ((data->agr.laststartime.tv_sec == agr1->laststartime.tv_sec) &&
                       (data->agr.laststartime.tv_usec > agr1->laststartime.tv_usec))) {
                     data->agr.laststartime = agr1->laststartime;
                  }

                  if ((data->agr.lasttime.tv_sec < agr1->lasttime.tv_sec) ||
                      ((data->agr.lasttime.tv_sec == agr1->lasttime.tv_sec) &&
                       (data->agr.lasttime.tv_usec < agr1->lasttime.tv_usec))) {

                     RaThisArgusStore->qhdr.lasttime = agr1->lasttime;
                     data->agr.lasttime = agr1->lasttime;
                  }

                  break;
               }
            }
         } else {
            int datalen = 0;

            switch (i) {
               case ARGUS_SRCUSRDATA_DSR_INDEX:
                  datalen = (ArgusSrcUserDataLen + 3)/ 4;

               case ARGUS_DSTUSRDATA_DSR_INDEX: {
                  struct ArgusUserStruct *user1 = (struct ArgusUserStruct *) a1farhdr[i];
                  struct ArgusUserStruct *user2 = (struct ArgusUserStruct *) a2farhdr[i];

                  datalen = (ArgusDstUserDataLen + 3)/ 4;

                  if (user1 && (user2 == NULL)) {
                     datalen = (datalen > (user1->length)) ? datalen : (user1->length);
                     user1->length = datalen + 1;
                     bcopy ((char *) user1, &buf[tmp->ahdr.length], user1->length*4);
                     tmp->ahdr.length += (user1->length * 4);
                  }

                  break;
               }
            }
         }
         status >>= 1;
      }

      ArgusFree(data->argus);
      data->argus = RaCopyArgusRecord(tmp);

      data->status |= RA_MODIFIED;
      data->argus->ahdr.status |= ARGUS_MERGED;
      data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);

      if (store.data != NULL)
         ArgusFree(store.data);
#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: merged\n");
#endif
      return;

   } else {
      data->argus = RaCopyArgusRecord(a1);
      data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);

#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: swapped");
#endif
      return;
   }

doneRaMergeArgus:
   RaSendArgusRecord(&store);

   if (store.data != NULL)
      ArgusFree(store.data);
   ArgusFree(data->argus);

   data->argus = RaCopyArgusRecord(a1);
   data->status |= RA_MODIFIED;
   data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaMergeArgusRecord: dumped and swapped");
#endif
}

void
RaUpdateArgusStore(struct ArgusRecord *argus, struct ArgusRecordStore *store)
{
   store->status |= RA_MODIFIED;
   store->qhdr.lasttime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaUpdateArgusStore: \n");
#endif
}

void
RaTimeoutArgusStore (struct ArgusRecordStore *store)
{
   struct ArgusRecord *argus;
   int i;

   if (store != NULL) {
      if (store->status & RA_MODIFIED)
         RaSendArgusRecord(store);

      if (store->rahtblhdr)
         RaRemoveHashEntry (store->rahtblhdr->htable, store->rahtblhdr);

      for (i = 0; i < RaHistoTimeSeries; i++) {
         struct ArgusRecordData *data = NULL;
         
         if ((data = store->data[i]) != NULL) {
            if ((argus = data->argus) != NULL) {
               RaRemoveArgusRecord(argus);
               ArgusFree (argus);
               data->argus = NULL;
            }
            ArgusFree(data);
            store->data[i] = NULL;
         }
      }

      ArgusFree (store->data);
      ArgusFree (store);
      RaAllocArgusRecordStore--;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaTimeoutArgusStore: 0x%x\n", store);
#endif
}


int
RaCheckTimeout (struct ArgusRecordStore *store, struct ArgusRecord *argus)
{
   int retn = 0, lapseTime;
   struct ArgusRecordData *data = NULL;
   struct ArgusRecord *sargus = NULL;

   struct timeval *tvp = &ArgusGlobalTime;

   int lapseTimeSecs, lapseTimeuSecs;
   int starTimeSecs, starTimeuSecs;
   int lastTimeSecs, lastTimeuSecs;

   if (store->ArgusTimeout > 0) {
      lapseTime = store->qhdr.logtime.tv_sec + store->ArgusTimeout;

      if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
              (tvp->tv_usec > store->qhdr.logtime.tv_usec))) {
         RaSendArgusRecord(store);

      } else {
         if (argus != NULL) {
            if ((data = store->data[RaThisActiveIndex]) != NULL) {
               sargus = data->argus;

               starTimeSecs  = sargus->argus_far.time.start.tv_sec;
               starTimeuSecs = sargus->argus_far.time.start.tv_usec;
         
               if (argus->argus_far.time.last.tv_sec > sargus->argus_far.time.last.tv_sec) {
                  lastTimeSecs  = argus->argus_far.time.last.tv_sec;
                  lastTimeuSecs = argus->argus_far.time.last.tv_usec;
               } else {
                  lastTimeSecs  = sargus->argus_far.time.last.tv_sec;
                  if (argus->argus_far.time.last.tv_sec == sargus->argus_far.time.last.tv_sec) {
                     if (argus->argus_far.time.last.tv_usec > sargus->argus_far.time.last.tv_usec) {
                        lastTimeuSecs  = argus->argus_far.time.last.tv_usec;
                     } else
                        lastTimeuSecs  = sargus->argus_far.time.last.tv_usec;
                  } else
                     lastTimeuSecs  = sargus->argus_far.time.last.tv_usec;
               }

               lapseTimeSecs  = (lastTimeSecs  - starTimeSecs);
               lapseTimeuSecs = (lastTimeuSecs - starTimeuSecs);

               if (lapseTimeuSecs < 0) {
                  lapseTimeSecs--;
                  lapseTimeuSecs += 1000000;
               }

               if (lapseTimeSecs >= store->ArgusTimeout)
                  RaSendArgusRecord(store);
            }
         }
      }
   }

   if (store->ArgusIdleTimeout > 0) {
      if ((data = store->data[RaThisActiveIndex]) != NULL) {
         sargus = data->argus;
         lastTimeSecs  = sargus->argus_far.time.last.tv_sec;
         lastTimeuSecs = sargus->argus_far.time.last.tv_usec;

         lapseTimeSecs  = (tvp->tv_sec  - lastTimeSecs);
         lapseTimeuSecs = (tvp->tv_usec - lastTimeuSecs);

         if (lapseTimeSecs >= store->ArgusTimeout)
            retn++;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCheckTimeout: returning %d \n", retn);
#endif

   return (retn);
}

/*
int
RaCheckTimeout (struct ArgusRecordStore *store, struct ArgusRecord *argus)
{
   int retn = 0;
   unsigned int lapseTime, lastTimeSecs, lastTimeuSecs;
   struct timeval *tvp = &ArgusGlobalTime;

   if (store->ArgusTimeout > 0) {
      if (argus != NULL) {
         if (argus->argus_far.time.last.tv_sec > store->qhdr.lasttime.tv_sec) {
            lastTimeSecs  = argus->argus_far.time.last.tv_sec;
            lastTimeuSecs = argus->argus_far.time.last.tv_usec;
         } else {
            lastTimeSecs  = store->qhdr.lasttime.tv_sec;
            if (argus->argus_far.time.last.tv_sec == store->qhdr.lasttime.tv_sec) {
               if (argus->argus_far.time.last.tv_usec > store->qhdr.lasttime.tv_usec) {
                  lastTimeuSecs  = argus->argus_far.time.last.tv_usec;
               } else
                  lastTimeuSecs  = store->qhdr.lasttime.tv_usec;
            } else
               lastTimeuSecs  = store->qhdr.lasttime.tv_usec;
         }
      } else {
         lastTimeSecs  = store->qhdr.lasttime.tv_sec;
         lastTimeuSecs = store->qhdr.lasttime.tv_usec;
      }

      lapseTime = lastTimeSecs + store->ArgusTimeout;
      if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
          (tvp->tv_usec > lastTimeuSecs)))
         retn++;
   
      lapseTime = store->qhdr.logtime.tv_sec + store->ArgusTimeout;
      if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
          (tvp->tv_usec > store->qhdr.logtime.tv_usec)))
   
         RaSendArgusRecord(store);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCheckTimeout: returning %d \n", retn);
#endif

   return (retn);
}
*/

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

struct RaCIDRAddr *
RaParseCIDRAddr (char *str)
{
   int error = 0, vlen, notIPAddr = 0, i;
   struct RaCIDRAddr *retn = NULL;
   char *ptr = NULL, chr = '\0';
   unsigned int **alist;
   int maskIndex;

   RaCIDRPtr->type = RA_CIDR_TYPE;
   RaCIDRPtr->addr = 0;
   RaCIDRPtr->mask = 0xFFFFFFFF;

   if ((ptr = strchr(str, '*')) == NULL) {
      if ((strncasecmp (str, "class", 5)) != 0) {
         if ((ptr = strchr (str, ':')) != NULL) {
            chr = *ptr;
            *ptr++ = 0;
         } else {
            if ((ptr = strchr (str, '/')) != NULL) {
               chr = *ptr;
               *ptr++ = 0;
            }
         }
   
         for (i = 0; i < strlen(str); i++)
           if (!(isdigit((int)str[i]) || (str[i] == '.')))
              notIPAddr++;
   
         if (!(notIPAddr)) {
            vlen = __argus_atoin(str, &RaCIDRPtr->addr);
   
            /* Promote short net number */
   
            while (RaCIDRPtr->addr && (RaCIDRPtr->addr & 0xff000000) == 0) {
               RaCIDRPtr->addr <<= 8;
               RaCIDRPtr->mask <<= 8;
            }
   
         } else {
            if ((alist = argus_nametoaddr(str)) != NULL)  {
               RaCIDRPtr->addr = **alist;
            } else
               error++;
         }
      
         if (ptr && (((chr == ':') || (chr == '/')) && isdigit((int) *ptr))) {
            maskIndex = atoi(ptr);
      
            if ((maskIndex) > 0) {
               RaCIDRPtr->mask = 0xFFFFFFFF;
               RaCIDRPtr->mask <<= (32 - maskIndex);
            }
   
            if (RaCIDRPtr->mask)
               RaCIDRPtr->addr &= RaCIDRPtr->mask;
         }
      } else {
         RaCIDRPtr->type = RA_NETCLASS_TYPE;
      }
   }

   if (!error)
      retn = RaCIDRPtr;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParseCIDRAddr: returning 0x%x \n", retn);
#endif
   
   return (retn);
}


struct RaPolicyStruct *
RaParsePolicyEntry (char *str)
{
   struct RaPolicyStruct *retn = NULL;
   struct RaCIDRAddr *addr = NULL;
   int i, type, policyelements, error = 0;
   char buf[MAXSTRLEN], *ptr = NULL, *tmp = buf;

   if (strncmp (str, RA_FLOWTAGSTR, strlen(RA_FLOWTAGSTR)) == 0) {
      type = RA_FLOWLIST;
      policyelements = RA_FLOWPOLICYFIELDNUM;

   } else {
      if ((strncmp (str, RA_MODELTAGSTR, strlen(RA_MODELTAGSTR))) == 0) {
         type = RA_MODELIST;
         policyelements = RA_MODELPOLICYFIELDNUM;

      } else 
         return (NULL);
   }

   if ((retn = (struct RaPolicyStruct *) ArgusCalloc (1, sizeof (*retn))) == NULL)
      return retn;

   bzero (buf, MAXSTRLEN);
   bcopy ((char *) str, buf, strlen(str));

   for (i = 0; ((i < policyelements) && !(error)); i++) {
      if ((ptr = strtok (tmp, " \t")) && (*ptr != '\n')) {
         switch (i) {
            case RA_LABELSTRING: 
               switch (type) {
                  case RA_FLOWLIST:
                     if (strcmp (ptr,  RA_FLOWTAGSTR))
                        error++;
                     else 
                        retn->RaEntryType = RA_FLOWLIST;
                     break;

                  case RA_MODELIST:
                     if (strcmp (ptr, RA_MODELTAGSTR))
                        error++;
                     else 
                        retn->RaEntryType = RA_MODELIST;
                     break;
               }
               break;
   
            case RA_POLICYID: 
               if (!(retn->RaPolicyId = atoi (ptr))) error++;
               break;

            case RA_POLICYSRCADDR:
               if ((addr = RaParseCIDRAddr (ptr)) != NULL)
                  retn->src = *addr;
               else
                  error++;
               break;
   
            case RA_POLICYDSTADDR:
               if ((addr = RaParseCIDRAddr (ptr)) != NULL)
                  retn->dst = *addr;
               else
                  error++;
               break;
   
            case RA_POLICYPROTO: 
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->proto = atoi(ptr);
                     } else {
                        if (*ptr == '*') {
                           retn->proto = 0xFF;
                        } else {
                           struct protoent *proto;
                           if ((proto = getprotobyname(ptr)) != NULL)
                              retn->proto = proto->p_proto;
                           else
                              error++;
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->proto = 0xFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->proto = 0x00;
                        else
                           error++;
                     break;
               }
               break;

            case RA_POLICYSRCPORT:
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->sport = (arg_uint16) atoi (ptr);
                     } else {
                        if (*ptr == '*')
                           retn->sport = 0xFFFF;
                        else {
                           int proto, port;
                           if (retn->proto == IPPROTO_ICMP) {
                              if (!(strcmp (ptr, "echo"))) retn->sport = ICMP_ECHO; else
                              if (!(strcmp (ptr, "unreach"))) retn->sport = ICMP_UNREACH; else
                              if (!(strcmp (ptr, "squench"))) retn->sport = ICMP_SOURCEQUENCH; else
                              if (!(strcmp (ptr, "redirect"))) retn->sport = ICMP_REDIRECT; else
                              if (!(strcmp (ptr, "timexed"))) retn->sport = ICMP_TIMXCEED; else
                              if (!(strcmp (ptr, "param"))) retn->sport = ICMP_PARAMPROB; else
                              if (!(strcmp (ptr, "timestmp"))) retn->sport = ICMP_TSTAMP; else
                              if (!(strcmp (ptr, "info"))) retn->sport = ICMP_IREQ; else
                              if (!(strcmp (ptr, "advert"))) retn->sport = ICMP_ROUTERADVERT; else
                              if (!(strcmp (ptr, "solicit"))) retn->sport = ICMP_ROUTERSOLICIT; else
                                 error++;
                           } else {
                              if (argus_nametoport(ptr, &port, &proto)) {
                                 retn->sport = port;
                              } else
                                 error++;
                           }
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->sport = 0xFFFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->sport = 0x0000;
                        else
                           error++;
                     break;
               }
               break;
   
            case RA_POLICYDSTPORT:
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->dport = (arg_uint16) atoi (ptr);
                     } else {
                        if (*ptr == '*')
                           retn->dport = 0xFFFF;
                        else {
                           if (retn->proto == IPPROTO_ICMP) {
                           } else {
                              int proto, port;
                              if (argus_nametoport(ptr, &port, &proto)) {
                                 retn->dport = port;
                              } else
                                 error++;
                           }
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->dport = 0xFFFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->dport = 0x0000;
                        else
                           error++;
                     break;
               }
               break;
      
            case RA_POLICYMODELST:
               if (type == RA_FLOWLIST) {
                  if (isdigit((int)*ptr))
                     retn->RaModelId = (arg_uint32) atoi (ptr);
                  else
                     error++;
               }
               break;

            case RA_POLICYTIMEOUT:
               if (type == RA_FLOWLIST) {
                  if (isdigit((int)*ptr))
                     retn->ArgusTimeout = (arg_uint32) atoi (ptr);
                  else
                     error++;
               }
               break;
         }
         tmp = NULL;

      } else
         break;
   }

   if (error) {
      ArgusFree(retn);
      retn = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParsePolicyEntry: returning 0x%x \n", retn);
#endif

   return (retn);
}


int
RaCreatePolicyEntry (int index, char *str)
{
   int retn = 0;
   struct RaPolicyStruct *rap;

   if ((rap = RaParsePolicyEntry (str)) != NULL) {
      rap->str = str;

      switch (rap->RaEntryType) {
         case RA_FLOWLIST:
            RaFlowArray[RaThisFlowNum++] = rap;
            break;

         case RA_MODELIST:
            RaModelArray[RaThisModelNum++] = rap;
            break;
      }
   } else {
      fprintf (stderr, "%s: RaCreatePolicyEntry: format error\n", ArgusProgramName);
      fprintf (stderr, "line %d %s", index, str);
      exit (-1);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCreatePolicyEntry: returning %d\n", retn);
#endif

   return (retn);
}


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#define ARGUSMAXMODELFILE	2048
char *modelbuf[ARGUSMAXMODELFILE];

int
RaReadFlowModelFile (char **model)
{
   int RaPolicyLines = 0, RaPolicyLineNumber = 0;
   char strbuf[MAXSTRLEN], *str = strbuf;
   int retn = 0, i = 0, x = 0;
   FILE *fd;

   if (model == NULL) {
      bzero (modelbuf, sizeof(modelbuf));
      if ((fd = fopen (ArgusFlowModelFile, "r")) != NULL) {
         while ((str = fgets (str, MAXSTRLEN, fd)) != NULL)
            modelbuf[i++] = strdup(str);

         model = modelbuf;

      } else
         ArgusLog (LOG_ERR, "RaReadFlowModelFile () ArgusCalloc %s\n", strerror(errno));
   }

   while ((str = *model++) != NULL) {
      RaPolicyLineNumber++;

      while (isspace((int)*str))
         str++;
      if (strlen(str)) {
         switch (*str) {
            case '#':
            case '\n':
            case '!':
               break;

            default:
               if ((str = strdup(str)) != NULL) {
                  if (RaCreatePolicyEntry(RaPolicyLines, str) < 0) {
                     fprintf (stderr, "%s: RaReadFlowModelFile: parse error line %d\n", ArgusProgramName,
                                RaPolicyLineNumber);
                     fprintf (stderr, "%s: RaReadFlowModelFile: %s\n", ArgusProgramName,
                                RaPolicyError[RaParseError]);
                     exit (-1);
                  } else
                     RaPolicyLines++;
                 
               } else {
                  fprintf (stderr, "%s: RaReadFlowModelFile: error %s\n", ArgusProgramName,
                                strerror(errno));
                  exit (-1);
               }
         }
      }
   }

   for (i = 0; i < RaThisFlowNum; i++) {
      if (RaFlowArray[i]) {
         int found = 0;
         for (x = 0; x < RaThisModelNum; x++) {
            if (RaModelArray[x])
               if (RaFlowArray[i]->RaModelId == RaModelArray[x]->RaPolicyId) {
                  RaFlowArray[i]->RaModelId = x;
                  found++;
                  break;
               }
         }
         if (!found) {
            fprintf (stderr, "%s: RaReadFlowModeleFile: Model Index %d not found\n",
                             ArgusProgramName, RaFlowArray[i]->RaModelId);
            exit (-1);
         }
      }
   }

   retn = RaThisFlowNum;

#ifdef ARGUSDEBUG
   ArgusDebug (3, "RaReadFlowModelFile: returning RaFlowArray[%d]  RaModelArray[%d]\n", RaThisFlowNum, RaThisModelNum);
#endif

   return (retn);
}

struct RaHashTableHeader *
RaFindHashObject (struct RaHashTableStruct *thisHashTable)
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *head = NULL, *target;

   if ((target = thisHashTable->array[RaThisHash % thisHashTable->size]) != NULL) {
      head = target;
      do {
         if (!(bcmp ((char *) RaThisFlow, (char *) &target->flow, sizeof(*RaThisFlow)))) {
               retn = target;
               break;
            } else
               target = target->nxt;
      } while (target != head);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaFindHashObject: returning 0x%x\n", retn);
#endif

   return (retn);
}



struct RaHashTableHeader *
RaAddHashEntry (struct RaHashTableStruct *thisHashTable, struct ArgusRecordStore *store)
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *start = NULL;

   if ((retn = (struct RaHashTableHeader *) ArgusCalloc (1, sizeof (struct RaHashTableHeader))) != NULL) {
      RaAllocHashTableHeaders++;
      bcopy((char *) RaThisFlow, (char *)&retn->flow, sizeof (*RaThisFlow));
      retn->storeobj = store;
      retn->hash = RaThisHash;
      
      if ((start = thisHashTable->array[RaThisHash % thisHashTable->size]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      thisHashTable->array[RaThisHash % thisHashTable->size] = retn;
      thisHashTable->count++;
      retn->htable = thisHashTable;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaAddHashEntry (0x%x) returning 0x%x\n", store, retn);
#endif

   return (retn);
}

void
RaRemoveHashEntry (struct RaHashTableStruct *thisHashTable, struct RaHashTableHeader *rahtblhdr)
{
   unsigned short hash = rahtblhdr->hash;

   rahtblhdr->prv->nxt = rahtblhdr->nxt;
   rahtblhdr->nxt->prv = rahtblhdr->prv;
   rahtblhdr->htable = NULL;

   if (rahtblhdr == thisHashTable->array[hash % thisHashTable->size]) {
      if (rahtblhdr == rahtblhdr->nxt)
         thisHashTable->array[hash % thisHashTable->size] = NULL;
      else
         thisHashTable->array[hash % thisHashTable->size] = rahtblhdr->nxt;
   }

   thisHashTable->count--;
   ArgusFree (rahtblhdr);
   RaAllocHashTableHeaders--;
 
#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaRemoveHashObject (0x%x) returning\n", rahtblhdr);
#endif
}


struct RaQueueStruct *
RaNewQueue ()
{
   struct RaQueueStruct *retn =  NULL;

   if ((retn = (struct RaQueueStruct *) ArgusCalloc (1, sizeof (struct RaQueueStruct))) != NULL) {
      retn->count = 0;
      retn->start = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaNewQueue () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
RaDeleteQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *obj = NULL;

   if (queue != NULL) {
      while ((obj = RaPopQueue(queue)))
         ArgusFree(obj);

      ArgusFree(queue);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaDeleteQueue (0x%x) returning\n", queue);
#endif
}


int
RaGetQueueCount(struct RaQueueStruct *queue)
{

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaGetQueueCount (0x%x) returning %d\n", queue, queue->count);
#endif

   return (queue->count);
}

void
RaPushQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   int retn = 0;

   if ((retn = RaAddToQueue (queue, obj)) > 0)
      queue->start = queue->start->prv;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaPushQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}


int
RaAddToQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   int retn = 0;

   if (queue && obj) {
      if (obj->queue == NULL) {
         if (queue->start != NULL) {
            obj->prv = queue->start->prv;
            queue->start->prv = obj;
            obj->nxt = queue->start;
            obj->prv->nxt = obj;
         } else {
            queue->start = obj;
            obj->nxt = obj;
            obj->prv = obj;
         }
         queue->count++;
         obj->queue = queue;
         retn = 1;

      } else
         ArgusLog (LOG_ERR, "RaAddToQueue (0x%x, 0x%x) obj in queue 0x%x\n", queue, obj, obj->queue);
   } else
      ArgusLog (LOG_ERR, "RaAddToQueue (0x%x, 0x%x) parameter error\n", queue, obj);

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaAddToQueue (0x%x, 0x%x) returning %d\n", queue, obj, retn);
#endif

   return (retn);
}


struct ArgusQueueHeader *
RaPopQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *retn = NULL;
   struct ArgusQueueHeader *obj = NULL;

   if (queue && queue->count) {
      if ((obj = (struct ArgusQueueHeader *) queue->start) != NULL)
         retn = RaRemoveFromQueue(queue, obj);
      else
         ArgusLog (LOG_ERR, "RaPopQueue(0x%x) internal queue error count %d start NULL\n", queue, queue->count);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "RaPopQueue (0x%x) returning 0x%x\n", queue, retn);
#endif
   
   return(retn);
}


struct ArgusQueueHeader *
RaRemoveFromQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct ArgusQueueHeader *retn = NULL;

   if ((queue != NULL) && (obj != NULL)) {
      if (queue->count && (obj->queue == queue)) {
         queue->count--;

         if (queue->count) {
            if (queue->start == obj)
               queue->start = obj->nxt;

            obj->prv->nxt = obj->nxt;
            obj->nxt->prv = obj->prv;

         } else
            queue->start = NULL;

         obj->prv = NULL;
         obj->nxt = NULL;
         obj->queue = NULL;
         retn = obj;

      } else
         ArgusLog (LOG_ERR, "RaRemoveFromQueue(0x%x, 0x%x) obj not in queue\n", queue, obj);
   } else
      ArgusLog (LOG_ERR, "RaRemoveFromQueue(0x%x, 0x%x) parameter error\n", queue, obj);

#ifdef ARGUSDEBUG
   ArgusDebug (9, "RaRemoveFromQueue (0x%x, 0x%x) returning 0x%x\n", queue, obj, obj);
#endif

   return (retn);
}


struct ArgusListStruct *
ArgusNewList ()
{
   struct ArgusListStruct *retn = NULL;
 
   if ((retn = (struct ArgusListStruct *) ArgusCalloc (1, sizeof (struct ArgusListStruct))) != NULL) {
      retn->start = NULL;
      retn->count = 0;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusNewList () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
ArgusDeleteList (struct ArgusListStruct *list)
{
   if (list) {
      while (list->start)
         ArgusPopFrontList(list);

      ArgusFree (list);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusDeleteList (0x%x) returning\n", list);
#endif
}

int
ArgusListEmpty (struct ArgusListStruct *list)
{
   return (list->start == NULL);
}

int
ArgusGetListCount(struct ArgusListStruct *list)
{
   return (list->count);
}

void
ArgusPushFrontList(struct ArgusListStruct *list, void *obj)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if (list && obj) {
      if ((lobj = (struct ArgusListObjectStruct *) ArgusCalloc (1, sizeof(*lobj))) != NULL) {
         lobj->obj = obj;
   
         if (list->start) {
            lobj->nxt = list->start;
            lobj->prv = list->start->prv;
            lobj->nxt->prv = lobj;
            lobj->prv->nxt = lobj;
         } else {
            lobj->prv = lobj;
            lobj->nxt = lobj;
         }
   
         list->start = lobj;
         list->count++;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPushFrontList (0x%x, 0x%x) returning 0x%x\n", list, obj);
#endif
}

void
ArgusPushBackList(struct ArgusListStruct *list, void *obj)
{
   ArgusPushFrontList(list, obj);
   list->start = list->start->nxt;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPushBackList (0x%x, 0x%x) returning 0x%x\n", list, obj);
#endif
}

void *
ArgusFrontList(struct ArgusListStruct *list)
{
   void *retn = NULL;

   if (list->start)
      retn = list->start->obj;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusFrontList (0x%x) returning 0x%x\n", list, retn);
#endif

   return (retn);
}


void
ArgusPopBackList(struct ArgusListStruct *list)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if ((lobj = list->start)) {
      list->start = list->start->prv;
      ArgusPopFrontList(list);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPopBackList (0x%x) returning\n", list);
#endif
}

void
ArgusPopFrontList(struct ArgusListStruct *list)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if ((lobj = list->start)) {
      if (--list->count > 0) {
         if (lobj->prv)
            lobj->prv->nxt = lobj->nxt;
 
         if (lobj->nxt)
            lobj->nxt->prv = lobj->prv;
 
        list->start = lobj->nxt;
 
      } else
         list->start = NULL;
 
      ArgusFree(lobj);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPopFrontList (0x%x) returning\n", list);
#endif
}


char *
RaGenerateLabel(struct ArgusRecord *argus) 
{
   int gflagvar = RaPrintDuration;
   int Sflagvar = RaPrintStartTime;
   int Lflagvar = RaPrintLastTime;
   char RaPrecisionPad[32], RaTimePad[32], date[128];
   int x;

   bzero (date, sizeof (date));
   bzero (RaTimePad, sizeof (RaTimePad));
   bzero (RaPrecisionPad, sizeof (RaPrecisionPad));
   bzero (RaLabelStr, sizeof(RaLabelStr));

   RaPrintDuration = 0;
   RaPrintStartTime = 1;
   RaPrintLastTime = 0;
   ArgusPrintDate (date, argus);
   RaPrintDuration = gflagvar;
   RaPrintStartTime = Sflagvar;
   RaPrintLastTime = Lflagvar;

   if (strlen(date) > 10) {
      for (x = 0; x < (strlen(date) - 10)/2; x++)
         strcat (RaTimePad, " ");
   }
    
   if (pflag)
      for (x = 0; x < pflag/2; x++)
         strcat (RaPrecisionPad, " ");

   RaLabel = RaLabelStr;

   if (idflag)
      strcat (RaLabel, "     ProbeId    ");

   if (RaPrintDate) {
      strcat (RaLabel, RaTimePad);

      if (lflag)
         strcat (RaLabel, " LastTime ");
      else
         strcat (RaLabel, "StartTime ");

      if (pflag) {
         if (!uflag) 
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
               sprintf (&RaLabel[strlen(RaLabel)], "Fraction ");
      }

      strcat (RaLabel, RaTimePad);
      if (strlen(date) & 0x01) strcat (RaLabel, " ");

      if (Gflag) {
         strcat (RaLabel, RaTimePad);
         strcat (RaLabel, " Last_Time ");
         strcat (RaLabel, RaTimePad);
         if (strlen(date) & 0x01) strcat (RaLabel, " ");
      }
   }

   if (RaPrintDuration) {
      strcat (RaLabel, RaPrecisionPad);
      strcat (RaLabel, "   Dur   ");

      strcat (RaLabel, RaPrecisionPad);
      if (pflag) {
         strcat (RaLabel, " ");
         if (pflag & 0x01)
            strcat (RaLabel, " ");
      }
   }

   if (RaPrintIndicator)
      strcat (RaLabel, "  Flgs   ");

   if (RaPrintTransactions)
      strcat (RaLabel, "Trans ");

   if (RaPrintProto)
     strcat (RaLabel, "Type ");

   if (mflag) strcat (RaLabel, RamOptionLabel);

   if (ArgusInput->ArgusManStart.ahdr.type & ARGUS_RMON) {
      if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_TOPN)
         strcat (RaLabel, RaTopNFlowLabel);
      else
         strcat (RaLabel, RaMatrixFlowLabel);

   } else {
      if (RaPrintSrcAddr) {
         if (argus->ahdr.type & ARGUS_RMON)
            sprintf (&RaLabel[strlen(RaLabel)], "%*s Addr%*s ", (hfield - 5)/2, " ", (hfield - 5)/2, " ");
         else
            sprintf (&RaLabel[strlen(RaLabel)], "%*sSrcAddr%*s ", (hfield - 7)/2, " ", (hfield - 7)/2, " ");
         if (hfield & 0x00) sprintf (&RaLabel[strlen(RaLabel)], " ");
      }

      if (RaPrintSrcPort) {
         if (pfield > 5) {
            if (!(pfield & 0x01))
               sprintf (&RaLabel[strlen(RaLabel)], " ");
            sprintf (&RaLabel[strlen(RaLabel)], "%*sSport%*s ", (pfield - 5)/2, " ", (pfield - 5)/2, " ");
         } else
            sprintf (&RaLabel[strlen(RaLabel)], "Sport ");
      } else
         sprintf (&RaLabel[strlen(RaLabel)], " ");

      if (RaPrintDir)
         sprintf (&RaLabel[strlen(RaLabel)], "Dir  ");

      if (RaPrintDstAddr) {
         sprintf (&RaLabel[strlen(RaLabel)], "%*sDstAddr%*s ", (hfield - 7)/2, " ", (hfield - 7)/2, " ");
         if (hfield & 0x00) sprintf (&RaLabel[strlen(RaLabel)], " ");
      }

      if (RaPrintDstPort) {
         if (pfield > 5) {
            if (!(pfield & 0x01))
               sprintf (&RaLabel[strlen(RaLabel)], " ");
            sprintf (&RaLabel[strlen(RaLabel)], "%*sDport%*s ", (pfield - 5)/2, " ", (pfield - 5)/2, " ");
         } else
            sprintf (&RaLabel[strlen(RaLabel)], "Dport ");
      } else
         sprintf (&RaLabel[strlen(RaLabel)], " ");
   }

   if (RaPrintCounts)
      strcat (RaLabel, " SrcPkt   Dstpkt    ");

   if (RaPrintBytes) {
      if (Aflag)
         strcat (RaLabel, "SAppBytes    DAppBytes  ");
      else
      if (Rflag)
         strcat (RaLabel, "Response Information    ");
      else
         strcat (RaLabel, "SrcBytes     DstBytes   ");
   }

   if (RaPrintLoad)
      strcat (RaLabel, RaLoadOptionLabel);

   if (RaPrintStatus) {
      if (zflag || Zflag) {
         strcat (RaLabel, " Status");
      } else {
         strcat (RaLabel, "State");
      }
   }

   if (dflag && RaPrintUserData) {
      int slen = ArgusSrcUserDataLen, dlen = ArgusDstUserDataLen;

      if (eflag == ARGUS_ENCODE_64) {
         if (slen > 0)
            slen += slen/2;
         if (dlen > 0)
            dlen += dlen/2;
      }

      if (slen) {
         sprintf (&RaLabel[strlen(RaLabel)], " %*s%s", ((slen + 8) - 7)/2, " ", "SrcData");
      }

      if (dlen) {
         sprintf (&RaLabel[strlen(RaLabel)], "%*s ", ((slen + 8) - 7)/2, " ");
         sprintf (&RaLabel[strlen(RaLabel)], " %*s%s", ((dlen + 8) - 7)/2, " ", "DstData");
      }
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      char tmpbuf[128], *ptr = tmpbuf, *str = RaLabel, lastchr = ' ';

      bzero (tmpbuf, sizeof(tmpbuf));
      lastchr = RaFieldDelimiter;
      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }
      bzero (RaLabel, MAXSTRLEN);
      if (tmpbuf[strlen(tmpbuf) - 1] == RaFieldDelimiter)
         tmpbuf[strlen(tmpbuf) - 1] = '\0';

      bcopy (tmpbuf, RaLabel, strlen(tmpbuf));
   }

   return (RaLabel);
}

void
RaSortQueue (struct RaQueueStruct *queue)
{
   int i = 0, cnt = queue->count;
   struct ArgusQueueHeader *qhdr;

   if (queue->array != NULL) {
      ArgusFree(queue->array);
      queue->array = NULL;
   }

   if (cnt > 0) {
      if ((queue->array = (struct ArgusQueueHeader **) ArgusCalloc(sizeof(struct ArgusQueueHeader *), cnt)) != NULL) {
         qhdr = queue->start;
         do {
            queue->array[i] = qhdr;
            qhdr = qhdr->nxt;
            i++;
         } while (qhdr != queue->start);

         qsort ((char *) queue->array, i, sizeof (struct ArgusQueueHeader *), RaSortRoutine);

      } else
         ArgusLog (LOG_ERR, "RaSortQueue: ArgusCalloc(%d, %d) %s\n", sizeof(struct ArgusRecord *),
                                                                     cnt, strerror(errno));
   }
}

int
RaSortRoutine (const void *void1, const void *void2)
{
   int retn = 0, i = 0;
   struct ArgusRecordStore *store1 = *(struct ArgusRecordStore **)void1;
   struct ArgusRecordStore *store2 = *(struct ArgusRecordStore **)void2;

   if ((store1 && store1->data[RaThisActiveIndex]->argus) && 
      ((store2 && store2->data[RaThisActiveIndex]->argus))) {
      struct ArgusRecord *argus1 = store1->data[RaThisActiveIndex]->argus;
      struct ArgusRecord *argus2 = store2->data[RaThisActiveIndex]->argus;

      for (i = 0; i < ARGUS_MAX_SORT_ALG; i++)
         if (RaSortAlgorithms[i] != NULL)
            if ((retn = RaSortAlgorithms[i](argus2, argus1)) != 0)
               break;
   }

   return (retn);
}




int RaSortStartTime (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   unsigned int retn;

   if ((retn = a2->argus_far.time.start.tv_sec - a1->argus_far.time.start.tv_sec) == 0)
      retn = a2->argus_far.time.start.tv_usec - a1->argus_far.time.start.tv_usec;

   return (retn);
}

int RaSortLastTime (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   unsigned int retn;

   if ((retn = a2->argus_far.time.last.tv_sec - a1->argus_far.time.last.tv_sec) == 0)
      retn = a2->argus_far.time.last.tv_usec - a1->argus_far.time.last.tv_usec;

   return (retn);
}

int RaSortDuration (struct ArgusRecord *a2, struct ArgusRecord *a1)
{
   return ((int)(RaGetuSecDuration(a1) - RaGetuSecDuration(a2)));
}

int RaSortSrcAddr (struct ArgusRecord *a2, struct ArgusRecord *a1)
{
  int retn = 0;

   if ((retn = (a1->ahdr.status & 0xFFFF) - (a2->ahdr.status & 0xFFFF)) == 0) {
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_IP))
         retn = (a1->argus_far.flow.ip_flow.ip_src - a2->argus_far.flow.ip_flow.ip_src);
      else
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_ARP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_ARP))
         retn = (a1->argus_far.flow.arp_flow.arp_spa - a2->argus_far.flow.arp_flow.arp_spa);
      else
         retn = bcmp ((char *)&a2->argus_far.flow.mac_flow.ehdr.ether_shost, (char *)&a1->argus_far.flow.mac_flow.ehdr.ether_shost, 6);
   }

   return (retn);
}

int RaSortDstAddr (struct ArgusRecord *a2, struct ArgusRecord *a1)
{
   int retn = 0;

   if ((retn = (a1->ahdr.status & 0xFFFF) - (a2->ahdr.status & 0xFFFF)) == 0) {
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_IP))
         retn = (a1->argus_far.flow.ip_flow.ip_dst - a2->argus_far.flow.ip_flow.ip_dst);
      else
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_ARP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_ARP))
         retn = (a1->argus_far.flow.arp_flow.arp_tpa - a2->argus_far.flow.arp_flow.arp_tpa);
      else
         retn = bcmp ((char *)&a2->argus_far.flow.mac_flow.ehdr.ether_dhost, (char *)&a1->argus_far.flow.mac_flow.ehdr.ether_dhost, 6);
   }
   return (retn);
}

int RaSortProtocol (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn;

   if ((retn = (a1->ahdr.status & 0xFFFF) - (a2->ahdr.status & 0xFFFF)) == 0)
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_IP))
         if ((retn = a1->argus_far.flow.ip_flow.ip_p - a2->argus_far.flow.ip_flow.ip_p) == 0)
            retn = a1->argus_far.flow.ip_flow.tp_p - a2->argus_far.flow.ip_flow.tp_p;

   return (retn);
}

int RaSortSrcPort (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a2->argus_far.flow.ip_flow.sport - a1->argus_far.flow.ip_flow.sport);
}

int RaSortDstPort (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a2->argus_far.flow.ip_flow.dport - a1->argus_far.flow.ip_flow.dport);
}

int RaSortSrcTos (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn = (a1->argus_far.attr_ip.stos - a2->argus_far.attr_ip.stos);

   return (retn);
}

int RaSortDstTos (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn =  (a1->argus_far.attr_ip.dtos - a2->argus_far.attr_ip.dtos);

   return (retn);
}

int RaSortSrcTtl (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn =  (a1->argus_far.attr_ip.sttl - a2->argus_far.attr_ip.sttl);

   return (retn);
}

int RaSortDstTtl (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn = (a1->argus_far.attr_ip.dttl - a2->argus_far.attr_ip.dttl);

   return (retn);
}

int RaSortByteCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   if (Aflag)
      return ((a1->argus_far.src.appbytes + a1->argus_far.dst.appbytes) -
              (a2->argus_far.src.appbytes + a2->argus_far.dst.appbytes));
   else
      return ((a1->argus_far.src.bytes + a1->argus_far.dst.bytes) -
              (a2->argus_far.src.bytes + a2->argus_far.dst.bytes));
}

int RaSortLoad (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int src1, src2, dst1, dst2, retn = 0;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   double sec1, sec2;

   start = &a1->argus_far.time.start;
   last  = &a1->argus_far.time.last;
 
   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;
 
   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }
 
   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   sec1 = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   start = &a2->argus_far.time.start;
   last  = &a2->argus_far.time.last;
 
   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;
 
   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }
 
   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   sec2 = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   if (Aflag) {
      src1 = a1->argus_far.src.appbytes;
      dst1 = a1->argus_far.dst.appbytes;
      src2 = a2->argus_far.src.appbytes;
      dst2 = a2->argus_far.dst.appbytes;

   } else {
      src1 = a1->argus_far.src.bytes;
      dst1 = a1->argus_far.dst.bytes;
      src2 = a2->argus_far.src.bytes;
      dst2 = a2->argus_far.dst.bytes;
   }

   if (sec1 && sec2) {
      retn = (((double)(((src1 + dst1) * 1.0)/(double)(sec1 * 1.0)) >
               (double)(((src2 + dst2) * 1.0)/(double)(sec2 * 1.0))) ? 1 :
              ((double)(((src1 + dst1) * 1.0)/(double)(sec1 * 1.0)) ==
               (double)(((src2 + dst2) * 1.0)/(double)(sec2 * 1.0))) ? 0 : -1);
   } else {
      if (sec1)
         retn = 1;
      if (sec2)
         retn = -1;
   }

   return (retn);
}

int RaSortSrcByteCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   if (Aflag)
      return (a1->argus_far.src.appbytes - a2->argus_far.src.appbytes);
   else
      return (a1->argus_far.src.bytes - a2->argus_far.src.bytes);
}

int RaSortDstByteCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   if (Aflag)
      return (a1->argus_far.dst.appbytes - a2->argus_far.dst.appbytes);
   else
      return (a1->argus_far.dst.bytes - a2->argus_far.dst.bytes);
}


int RaSortPktsCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return ((a1->argus_far.src.count + a1->argus_far.dst.count) -
           (a2->argus_far.src.count + a2->argus_far.dst.count));
}

int RaSortSrcPktsCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a1->argus_far.src.count - a2->argus_far.src.count);
}

int RaSortDstPktsCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a1->argus_far.dst.count - a2->argus_far.dst.count);
}


char RaUserDataStr[MAXSTRLEN];

char *
RaGetUserDataString (struct ArgusRecord *argus)
{
   char *retn = RaUserDataStr;
   char strbuf[MAXSTRLEN], *str = strbuf;
   char delim = ' ';
   int len = 0;

   bzero (RaUserDataStr, MAXSTRLEN);

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;

   if (ArgusSrcUserDataLen > 0) {
      len = 0;
      if (ArgusThisFarStatus & ARGUS_SRCUSRDATA_DSR_STATUS) {
         struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_SRCUSRDATA_DSR_INDEX];

         len = (user->length - 1) * 4;
         len = (len < argus->argus_far.src.appbytes) ? len : argus->argus_far.src.appbytes;
         len = len > ArgusSrcUserDataLen ? ArgusSrcUserDataLen : len;
   
         if ((len = ArgusEncode (&user->data, len, str, sizeof(strbuf))) != 0)
            sprintf (RaUserDataStr, "%cs[%d]=%s", delim, len, str);
   
      } else
         if (delim != ' ')
            sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%c", delim);

      if (delim == ' ')
         sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%*s", (ArgusSrcUserDataLen - len) + 1, " ");
   }


   if (ArgusDstUserDataLen > 0) {
      len = 0;
      if (ArgusThisFarStatus & ARGUS_DSTUSRDATA_DSR_STATUS) {
         struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_DSTUSRDATA_DSR_INDEX];

         len = (user->length - 1) * 4;
         len = (len < argus->argus_far.dst.appbytes) ? len : argus->argus_far.dst.appbytes;
         len = len > ArgusDstUserDataLen ? ArgusDstUserDataLen : len;
   
         if ((len = ArgusEncode (&user->data, len, str, sizeof(strbuf))) != 0)
            sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%cd[%d]=%s", delim, len, str);
      } else
         if (delim != ' ')
            sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%c", delim);
   }

   return (retn);
}
