
/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (C) 2000-2001 QoSient, LLC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * rahistogram - argus record histogram generator
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */



#include <argus_client.h>
#include <rahistogram.h>

#include <math.h>

void RaProcessRecord (struct ArgusRecord *argus) { }
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
char *RaHistogramFlowModelFile [] = {
"Flow     100       *                  *                 *       *        *         200      3153600",
"Model    200    0.0.0.0            0.0.0.0              no      no       no",
NULL,
};

struct RaQueueStruct *RaArgusRecordQueue;

int RaInitialized = 0;
int RaModeValue = -1;

#define RAHISTOGRAMMODENUM	6
#define RASECONDS		-1
#define RAMINUTE		0
#define RAHOURLY		1
#define RADAILY 		2
#define RAWEEKLY		3
#define RAMONTHLY		4
#define RANNUALY		5

char *RaHistogramModes[RAHISTOGRAMMODENUM] = {
  "minute",
  "hourly",
  "daily",
  "weekly",
  "monthly",
  "annually",
};

extern int ArgusPassNum;
int RaPassNum = 0;

void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode;
   int i, found = 0;

   if (!(RaInitialized)) {
      ArgusPassNum = 2;
      RaPassNum = 1;
      RaWriteOut = 0;

      if (Gflag) {
         RaWriteOut = 0;
         RaPrintDate = 0;
         uflag++;
         RaPrintTime = 0;
         RaPrintLastTime = 0;
         RaPrintIndicator = 0;
         RaPrintTransactions = 0;
         RaPrintDir = 0;
         RaPrintMACAddress = 0;
         RaPrintUserData = 0;
         RaPrintStatus = 0;
         RaPrintDuration = 0;
         RaPrintSrcAddr = 0;
         RaPrintDstAddr = 0;
         RaPrintProto = 0;
         RaPrintSrcPort = 0;
         RaPrintDstPort = 0;
         RaPrintCounts = 0;
         RaPrintBytes = 0;
         RaPrintJitter = 0;
         Iflag = 0;
         gflag = 0;
         pflag = 0;

         if (!((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')))
            RaFieldDelimiter = ':';
      }

      if ((mode = ArgusModeList) != NULL) {
         do {
            for (i = 0, found = 0; i < RAHISTOGRAMMODENUM; i++) {
               if (!(strncasecmp (mode->mode, RaHistogramModes[i], 3))) {
                  RaModeValue = i;
                  found++;
                  if (Gflag) {
                     RaPrintDate = 1;
                     RaPrintCounts = 1;
                     RaPrintBytes = 1;
                  }
               }
            }

            if (Gflag && !found) {
               switch (*mode->mode) {
                  case 'd': {
                     if (!(strncmp (mode->mode, "dst", 3))) {
                        if (!(strncasecmp (mode->mode, "dstpk", 5))) {
                           RaPrintDate = 1;
                           RaPrintCounts = 1;
                           RaPrintDstPackets = 1;
                        } else
                        if (!(strncasecmp (mode->mode, "dstby", 5))) {
                           RaPrintDate = 1;
                           RaPrintBytes = 1;
                           RaPrintDstBytes = 1;
                        } else
                           usage();
                     } else 
                     if (!(strncmp (mode->mode, "dur", 3))) {
                        Hflag++;
                        RaPrintDuration = 1; break;
                     }
                     break;
                  }
                  case 'c':
                  case 'p': {
                     if ((!(strncmp (mode->mode, "co", 2))) ||
                         (!(strncmp (mode->mode, "pa", 2)))) {
                        RaPrintDate = 1;
                        RaPrintCounts = 1;
                        RaPrintSrcPackets = 1;
                        RaPrintDstPackets = 1; break;
                     }
                  }
                  case 'b': {
                     if (!(strncmp (mode->mode, "by", 2))) {
                        RaPrintDate = 1;
                        RaPrintBytes = 1;
                        break;
                     }
                  }
                  case 'j': {
                     RaPrintJitter = 1; break;
                  }
   
                  case 's': {
                     if (!(strncmp (mode->mode, "src", 3))) {
                        if (!(strncasecmp (mode->mode, "srcpk", 5))) {
                           RaPrintDate = 1;
                           RaPrintCounts = 1;
                           RaPrintSrcPackets = 1;
                        } else
                        if (!(strncasecmp (mode->mode, "srcby", 5))) {
                           RaPrintDate = 1;
                           RaPrintBytes = 1;
                           RaPrintSrcBytes = 1;
                        } else
                           usage();
                     } else
                        usage();
   
                     break;
                  }
   
                  case 't': {
                     if (!(strncmp (mode->mode, "trans", 3))) {
                        RaPrintDate = 1;
                        RaPrintTransactions = 1;
                     }
                     break;
                  }
   
                  default:
                     usage();
               }
            }
            mode = mode->nxt;
         } while (mode);

      } else {
         if (Gflag) {
            RaPrintDate = 1;
            RaPrintCounts = 1;
            RaPrintBytes = 1;
         }
      }

      ArgusFlowModelFile = " ";
   
      bzero ((char *) RaFlowArray,  sizeof(RaFlowArray));
      bzero ((char *) RaModelArray, sizeof(RaModelArray));
   
      if ((RaFlowModelFile = RaReadFlowModelFile(RaHistogramFlowModelFile)) < 0)
         exit(0);
   
      if ((RaModelerQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaArgusRecordQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      if (Mflag) {
         for (i = 0, found = 0; i < RAHISTOGRAMMODENUM; i++) {
            if (!(strncasecmp (Mflag, RaHistogramModes[i], 3))) {
               RaModeValue = i;
               found++;
            }
         }
         if (!found)
            usage();
      }

      if (Hflag)
         RaHistoTimeSeries = ARGUS_TIMESERIES;

      if (Bflag) {
         if ((RaHoldingBuffer = (struct ArgusHoldingBuffer *) ArgusCalloc (1, sizeof(*RaHoldingBuffer))) != NULL) {
            if ((RaHoldingBuffer->array = (struct ArgusListStruct **) ArgusCalloc (Bflag, sizeof(void *))) != NULL) {
               RaHoldingBuffer->size = Bflag;
            } else {
               ArgusFree (RaHoldingBuffer);
               RaHoldingBuffer = NULL;
               exit (-1);
            }
         } else
            exit (-1);
      }

      if (nflag) {
         hfield = 15;
         pfield =  5;
      }

      RaInitialized++;
   }
}

int RaParseCompleting = 0;

struct timeval RaStartTime = {0x7FFFFFFF, 0x7FFFFFFF};
struct timeval RaEndTime = {0,0};

int RaSeconds, RaScaleSeconds, RaStartSeconds, RaEndSeconds;
int RaBinCount = 0;
float RaBinSize = 0.0;

#define RA_BASE_GMT_YEAR	70

void RaInitializeMergeArray(void);
void RaMergeQueue (struct ArgusRecordStore *);
void RaPrintMergeArray (void);

struct ArgusRecordStore **RaStoreMergeArray;
struct tm RaThisTmStructBuf, *RaThisTmStruct = &RaThisTmStructBuf;
struct tm RaEndTmStructBuf, *RaEndTmStruct = &RaEndTmStructBuf;
struct tm RaStartTmStructBuf, *RaStartTmStruct = &RaStartTmStructBuf;

double RaBinSeconds;

void
RaInitializeMergeArray()
{
   RaStartTime.tv_usec = 0;
   if (RaEndTime.tv_usec > 0) {
      RaEndTime.tv_sec++;
      RaEndTime.tv_usec = 0;
   }

   if (startime_t > 0) {
      if (RaStartTime.tv_sec > startime_t)
         RaStartTime.tv_sec = startime_t;
      if (RaEndTime.tv_sec < lasttime_t)
         RaEndTime.tv_sec = lasttime_t;
   }

   *RaStartTmStruct = *localtime((time_t *) &RaStartTime.tv_sec);
   *RaEndTmStruct   = *localtime((time_t *) &RaEndTime.tv_sec);

   switch (RaModeValue) {
      case RASECONDS: {
         RaBinSize = 1.0;
         break;
      }

      case RAMINUTE: {
         RaStartTmStruct->tm_sec = 0;

         if (RaEndTmStruct->tm_sec > 0) {
            RaEndTmStruct->tm_sec = 0;
            if (++RaEndTmStruct->tm_min == 0)
               if (++RaEndTmStruct->tm_hour == 0)
                  if (++RaEndTmStruct->tm_mday == 0)
                     if (++RaEndTmStruct->tm_mon == 0)
                        RaEndTmStruct->tm_year++;
         }

         RaBinSize = 60.0;
         break;
      }

      case RAHOURLY: {
         RaStartTmStruct->tm_sec = 0;
         RaStartTmStruct->tm_min = 0;

         if ((RaEndTmStruct->tm_sec > 0) || (RaEndTmStruct->tm_min > 0)) {
            RaEndTmStruct->tm_sec = 0;
            RaEndTmStruct->tm_min = 0;
            if (++RaEndTmStruct->tm_hour == 0)
               if (++RaEndTmStruct->tm_mday == 0)
                  if (++RaEndTmStruct->tm_mon == 0)
                     RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0;
         break;
      }

      case RADAILY: {
         RaStartTmStruct->tm_sec  = 0;
         RaStartTmStruct->tm_min  = 0;
         RaStartTmStruct->tm_hour = 0;

         if ((RaEndTmStruct->tm_sec > 0) || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0)) {
            RaEndTmStruct->tm_sec  = 0;
            RaEndTmStruct->tm_min  = 0;
            RaEndTmStruct->tm_hour = 0;

            if (++RaEndTmStruct->tm_mday == 0)
               if (++RaEndTmStruct->tm_mon == 0)
                  RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0;
         break;
      }

/*
      case RAWEEKLY: {
         if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_{
            if (++RaEndTmStruct->tm_mday == 0)
               if (++RaThisTmStruct->tm_mon == 0)
                  RaThisTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0;
         break;
      }
*/

      case RAMONTHLY: {
         RaStartTmStruct->tm_sec  = 0;
         RaStartTmStruct->tm_min  = 0;
         RaStartTmStruct->tm_hour = 0;
         RaStartTmStruct->tm_mday = 0;

         if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_mday > 0)) {
            RaEndTmStruct->tm_sec  = 0;
            RaEndTmStruct->tm_min  = 0;
            RaEndTmStruct->tm_hour = 0;
            RaEndTmStruct->tm_mday = 0;

            if (++RaEndTmStruct->tm_mon == 0)
               RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0;

         break;
      }

      case RANNUALY: {
         RaStartTmStruct->tm_sec  = 0;
         RaStartTmStruct->tm_min  = 0;
         RaStartTmStruct->tm_hour = 0;
         RaStartTmStruct->tm_mday = 0;
         RaStartTmStruct->tm_mon  = 0;

         if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_mday > 0) ||
             (RaThisTmStruct->tm_mon > 0)) {
            RaEndTmStruct->tm_sec  = 0;
            RaEndTmStruct->tm_min  = 0;
            RaEndTmStruct->tm_hour = 0;
            RaEndTmStruct->tm_mday = 0;
            RaEndTmStruct->tm_mon  = 0;
            RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0;
   
         break;
      }
   }

   RaStartSeconds = mktime(RaStartTmStruct);
   RaEndSeconds   = mktime(RaEndTmStruct);

   RaStartTime.tv_sec = RaStartSeconds;
   RaEndTime.tv_sec = RaEndSeconds;

   RaSeconds = RaEndTime.tv_sec - RaStartTime.tv_sec;
   *RaThisTmStruct  = *localtime((time_t *) &RaSeconds);

   if ((Nflag == 0)) {
      RaScaleSeconds = RaEndSeconds - RaStartSeconds;
      RaBinCount = (RaScaleSeconds + (RaBinSize - 1))/RaBinSize;
   } else {
      RaBinCount = Nflag;
      RaEndSeconds = RaStartSeconds + (Nflag * RaBinSize);
      RaScaleSeconds = RaEndSeconds - RaStartSeconds;
   }

   if ((RaStoreMergeArray = (struct ArgusRecordStore **) ArgusCalloc (sizeof(void *), (RaBinCount + 1))) == NULL)
      ArgusLog (LOG_ERR, "RaInitializeMergeArray() ArgusCalloc %s\n", strerror(errno));

   modf(RaBinSize, &RaBinSeconds);
   if (RaBinSize != RaBinSeconds)
      RaBinSize = RaBinSeconds + 1.0;
}


int RaQueueCount = 0;
int RaAdjustCount = 0;
int RaAdjustBinCount = 0;
int RaAdjustLastBinCount = 0;

float RaSrcPeriod = 0.0;
float RaDstPeriod = 0.0;

struct ArgusRecord * RaAdjustArgusRecord (struct ArgusRecord *);

struct ArgusRecord *
RaAdjustArgusRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *retn = NULL;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   float startsec, duration, thisduration, actduration = 0.0;
   double RaBinSeconds;

   int timeindex, count, bytes, srcABpp, srcBpp, dstABpp, dstBpp, thisCount;

   if ((argus->argus_far.src.count != 0) || (argus->argus_far.dst.count != 0)) {

      startsec = (float)(argus->argus_far.time.start.tv_sec - RaStartSeconds) + 
                        (argus->argus_far.time.start.tv_usec/1000000.0);
       
      timeindex = (int) (startsec/RaBinSize);

      duration = (float)((argus->argus_far.time.last.tv_sec - argus->argus_far.time.start.tv_sec) + 
                         (argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec)/1000000.0);

      retn = RaCopyArgusRecord (argus);

      ArgusThisFarStatus = ArgusIndexRecord(retn, ArgusThisFarHdrs);

      if ((int)((startsec + duration)/RaBinSize) != timeindex) {   /* we span out of this bin */
         struct ArgusAGRStruct *agr;

         if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
            int ArgusThisMultiplier = 1000;

            if (agr->status & ARGUS_AGR_USECACTTIME)
               ArgusThisMultiplier = 1000000;

            if (agr->act.n)
               actduration = agr->act.mean / ArgusThisMultiplier;
         }

         if ((count = (retn->argus_far.src.count + retn->argus_far.dst.count)) == 2) {
            if (actduration == 0.0) {
               startsec = (RaBinSize * (timeindex + 1));               /* where is the start of the next bin */
               modf(startsec, &RaBinSeconds);                          /* where is the start of the next bin */

               tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
               tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
               if (tvp->tv_usec >= 1000000) {
                  tvp->tv_sec++;
                  tvp->tv_usec -= 1000000;
               }
            } else {
               modf(actduration, &RaBinSeconds);                          /* where is the start of the next bin */

               tvp->tv_sec  = RaBinSeconds + retn->argus_far.time.start.tv_sec;
               tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000) + retn->argus_far.time.start.tv_usec;
               if (tvp->tv_usec >= 1000000) {
                  tvp->tv_sec++;
                  tvp->tv_usec -= 1000000;
               }
            }

            if (RaSrcPeriod == RaDstPeriod) {
               retn->argus_far.time.last = *tvp;                       /* this is the new end for this record */
               argus->argus_far.time.start = *tvp;
            } else {
               retn->argus_far.time.last = retn->argus_far.time.start;
               argus->argus_far.time.start = argus->argus_far.time.last;

               if (retn->argus_far.src.count) {
                  if (retn->argus_far.src.count == 1) {
                     retn->argus_far.dst.count = 0;
                     retn->argus_far.dst.bytes = 0;
                  } else {
                     retn->argus_far.src.count = 1;
                     bytes = retn->argus_far.src.bytes;
                     retn->argus_far.src.bytes /= 2;
                     if (bytes & 0x01)
                        retn->argus_far.src.bytes += 1;
                  }
               } else {
                  retn->argus_far.dst.count = 1;
                  bytes = retn->argus_far.dst.bytes;
                  retn->argus_far.dst.bytes /= 2;
                  if (bytes & 0x01)
                     retn->argus_far.dst.bytes += 1;
               }
            }

            argus->argus_far.src.count -= retn->argus_far.src.count;
            argus->argus_far.dst.count -= retn->argus_far.dst.count;
            argus->argus_far.src.bytes -= retn->argus_far.src.bytes;
            argus->argus_far.dst.bytes -= retn->argus_far.dst.bytes;

         } else {
            if (RaAdjustCount == 0) {
               RaAdjustBinCount = 0;
               RaAdjustLastBinCount = 0;
            }

            startsec = (RaBinSize * (timeindex + 1));               /* where is the start of the next bin */
            modf(startsec, &RaBinSeconds);                          /* where is the start of the next bin */
            tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
            tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
            if (tvp->tv_usec >= 1000000) {
               tvp->tv_sec++;
               tvp->tv_usec -= 1000000;
            }

            retn->argus_far.time.last = *tvp;                       /* this is the new end for this record */

            if (((retn->argus_far.time.last.tv_sec - retn->argus_far.time.start.tv_sec) == 1) &&
                ((retn->argus_far.time.last.tv_usec - retn->argus_far.time.start.tv_usec) == 0)) {
               thisduration = 1.0;

            } else {
               thisduration = (float)(((retn->argus_far.time.last.tv_sec  - retn->argus_far.time.start.tv_sec) * 1.0) +
                                       (retn->argus_far.time.last.tv_usec - retn->argus_far.time.start.tv_usec)/1000000.0);
            }

            argus->argus_far.time.start = *tvp;
            
            if (retn->argus_far.src.count) {
               srcABpp = (retn->argus_far.src.appbytes * thisduration)/duration;
               srcBpp  = (retn->argus_far.src.bytes    * thisduration)/duration;
               thisCount = (retn->argus_far.src.count  * thisduration)/duration;

               if (thisCount > retn->argus_far.src.count)
                  thisCount = retn->argus_far.src.count;

               retn->argus_far.src.count    = thisCount;
               retn->argus_far.src.bytes    = srcBpp;
               retn->argus_far.src.appbytes = srcABpp;
            }

            if (retn->argus_far.dst.count) {
               dstABpp = (retn->argus_far.dst.appbytes * thisduration)/duration;
               dstBpp  = (retn->argus_far.dst.bytes    * thisduration)/duration;

               thisCount = (retn->argus_far.dst.count  * thisduration)/duration;

               if (thisCount > retn->argus_far.dst.count)
                  thisCount = retn->argus_far.dst.count;

               retn->argus_far.dst.count    = thisCount;
               retn->argus_far.dst.bytes    = dstBpp;
               retn->argus_far.dst.appbytes = dstABpp;
            }

            argus->argus_far.src.count -= retn->argus_far.src.count;
            argus->argus_far.dst.count -= retn->argus_far.dst.count;
            argus->argus_far.src.bytes -= retn->argus_far.src.bytes;
            argus->argus_far.dst.bytes -= retn->argus_far.dst.bytes;
            argus->argus_far.src.appbytes -= retn->argus_far.src.appbytes;
            argus->argus_far.dst.appbytes -= retn->argus_far.dst.appbytes;
         }                             

      } else {
         argus->argus_far.src.count = 0;
         argus->argus_far.dst.count = 0;
      }
   }

   return(retn);
}


void
RaMergeQueue (struct ArgusRecordStore *store)
{
   struct ArgusRecord *argus = NULL;
   struct ArgusRecord *thisArgus = NULL;

   struct RaPolicyStruct *rap;
   int index, RaTimeout = -1, i;
   float startsec;

   RaQueueCount++;

   if (store) {
      for (i = 0; i < RaHistoTimeSeries; i++) {
         if (store->data[i] && ((argus = store->data[i]->argus) != NULL)) {
            struct ArgusFlow *flow = &argus->argus_far.flow;

            bcopy(flow, RaArgusFlow, sizeof(struct ArgusFlow));

            if ((rap = RaFlowModelOverRides(argus)) != NULL) {
               RaModifyFlow(rap, argus);
               RaTimeout = rap->ArgusTimeout;
            } else {
               RaTimeout = 0x1FFFFFF;
               RaModifyDefaultFlow(argus);
            }

            RaAdjustCount = 0;
            while ((thisArgus = RaAdjustArgusRecord (argus)) != NULL) {
               RaAdjustCount++;
               startsec = (float)(thisArgus->argus_far.time.start.tv_sec - RaStartSeconds) + 
                                 (thisArgus->argus_far.time.start.tv_usec/1000000.0);
          
               if ((index = (int) (startsec/RaBinSize)) < (RaBinCount + 1)) {
                  if ((RaThisArgusStore = RaStoreMergeArray[index]) == NULL) {
                     if ((RaThisArgusStore = RaNewArgusStore(thisArgus)) == NULL) {
                        ArgusLog (LOG_ERR, "RaProcessRecord: RaNewArgusData error %s\n", strerror(errno));
                     } else {
                        RaStoreMergeArray[index] = RaThisArgusStore;
                        flow = &thisArgus->argus_far.flow;
                        bcopy(flow, RaArgusFlow, sizeof(struct ArgusFlow));
                     }
                  }

                  if (RaThisArgusStore->data[i] != NULL)
                     RaMergeArgusRecord(thisArgus, RaThisArgusStore->data[i]);
                  else {
                     if ((RaThisArgusStore->data[i] = RaNewArgusData(thisArgus)) == NULL) {
                        ArgusLog (LOG_ERR, "RaProcessRecord: RaNewArgusData error %s\n", strerror(errno));
                     } else {
                        struct ArgusRecordData *data = RaThisArgusStore->data[i];

                        data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
                        data->status |= RA_MODIFIED;

                        if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                           double sumtime;

                           bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
                           data->act.n        = data->agr.act.n;
                           sumtime            = data->agr.act.mean * data->agr.act.n;
                           data->act.sumtime  = sumtime;
                           data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                           data->idle.n       = data->agr.idle.n;
                           sumtime            = data->agr.idle.mean * data->agr.idle.n;
                           data->idle.sumtime = sumtime;
                           data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
                        }
                     }
                  }

                  RaUpdateArgusStore(thisArgus, RaThisArgusStore);
                  ArgusFree(thisArgus);
               }
            }

            ArgusFree(argus);
            ArgusFree(store->data[i]);
            store->data[i] = NULL;
         }
      }

      ArgusFree(store);
   }
}


void
RaPrintMergeArray ()
{
   struct ArgusRecordStore *store, *blankStore;
   struct ArgusRecord argusbuf, *argus = &argusbuf;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   float startsec = 0.0;
   double RaBinSeconds;
   char stimebuf[128], dtimebuf[128];
   int i, x;

   bzero ((char *)&argusbuf, sizeof(argusbuf));
   argus->ahdr.type = ARGUS_FAR;
   argus->ahdr.status = ARGUS_STATUS;
   argus->ahdr.length = sizeof(struct ArgusRecord);

   RaEndTime.tv_sec -= RaBinSize;

   if (startime_t > 0) {
      if (startime_t > RaStartTime.tv_sec)
         RaStartTime.tv_sec = startime_t;
      if (lasttime_t < RaEndTime.tv_sec)
         RaEndTime.tv_sec = lasttime_t;
   }

   RaEndTime.tv_sec += RaBinSize;

   strcpy (stimebuf, print_time(&RaStartTime));
   strcpy (dtimebuf, print_time(&RaEndTime));

   if (RaLabel == NULL)
      RaLabel = RaGenerateLabel(argus);

   if (!(wfile) && Gflag) {
      fprintf (stdout, "StartTime=%s\n", stimebuf);
      fprintf (stdout, "StopTime=%s\n", dtimebuf);
      fprintf (stdout, "Seconds=%d\n", RaScaleSeconds);
      fprintf (stdout, "BinSize=%f\n", RaBinSize);
      fprintf (stdout, "Bins=%d\n", (int)((RaEndTime.tv_sec - RaStartTime.tv_sec)/RaBinSize));
      fprintf (stdout, "Columns=%s\n", RaLabel);
   }

   if ((blankStore = RaNewArgusStore(argus)) != NULL) {
      bzero(RaArgusFlow, sizeof(struct ArgusFlow));
      if ((blankStore->data[RaThisActiveIndex] = RaNewArgusData(argus)) != NULL) {
         argus = blankStore->data[RaThisActiveIndex]->argus;
         blankStore->data[RaThisActiveIndex]->status |= RA_MODIFIED;
         tvp->tv_sec  = RaStartSeconds;
         tvp->tv_usec = 0;
         argus->argus_far.time.start = *tvp;

         modf(RaBinSize, &RaBinSeconds);            /* where is the start of the first bin */
         tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
         tvp->tv_usec = (int)((RaBinSize - RaBinSeconds) * 1000000);
         if (tvp->tv_usec >= 1000000) {
            tvp->tv_sec++;
            tvp->tv_usec -= 1000000;
         }
         argus->argus_far.time.last = *tvp;
         RaUpdateArgusStore(argus, blankStore);
      }
   }

   for (x = 0; x < RaBinCount + 1; x++) {
      if (RaStoreMergeArray[x] == NULL)
         store = blankStore;
      else
         store = RaStoreMergeArray[x];

      for (i = 0; i < RaHistoTimeSeries; i++) {
         if (store->data[i] && (argus = store->data[i]->argus)) {
            startsec = (RaBinSize * x);               /* where is the start of this bin */
            modf(startsec, &RaBinSeconds);            
            tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
            tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
            if (tvp->tv_usec >= 1000000) {
               tvp->tv_sec++;
               tvp->tv_usec -= 1000000;
            }

            argus->argus_far.time.start = *tvp;

            startsec = (RaBinSize * (i + x));         /* where is the start of the next bin */
            modf(startsec, &RaBinSeconds);            
            tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
            tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
            if (tvp->tv_usec >= 1000000) {
               tvp->tv_sec++;
               tvp->tv_usec -= 1000000;
            }
            argus->argus_far.time.last = *tvp;

            store->data[i]->status |= RA_MODIFIED;
         }
      }

      RaUpdateArgusStore(argus, store);

      if ((argus->argus_far.time.start.tv_sec >= RaStartTime.tv_sec) &&
          (argus->argus_far.time.start.tv_sec <= RaEndTime.tv_sec)) {
         RaSendArgusRecord(store);
      }

      fflush(stdout);
   }
}


void
RaParseComplete (int sig)
{
   if (RaPassNum++ == 1) {
      RaInitializeMergeArray();

   } else {
      if ((sig == 0) && (totalrecords > 0)) {
         if (!RaParseCompleting) {
            RaParseCompleting++;

            RaPrintMergeArray ();
         }
      }
   }
}


int ArgusHourlyUpdate = 1;
int ArgusMinuteUpdate = 1;

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ragator Version %s\n", version);
   fprintf (stderr, "usage:  %s \n", ArgusProgramName);
   fprintf (stderr, "usage:  %s [options] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -M <mode>      [minute | hourly | daily | weekly | monthly | annually] (default seconds)\n");
   fprintf (stderr, "ra-options: -a             print record summaries on termination.\n");
   fprintf (stderr, "            -A             print application bytes.\n");
   fprintf (stderr, "            -b             dump packet-matching code.\n");
   fprintf (stderr, "            -c             print packet and byte counts.\n");
   fprintf (stderr, "            -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>     specify debug level\n");
   fprintf (stderr, "            -f <flowfile>  read flow model from <flowfile>.\n");
   fprintf (stderr, "            -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "            -g             print record time duration.\n");
   fprintf (stderr, "            -G             print both start and last time values.\n");
   fprintf (stderr, "            -h             print help.\n");
   fprintf (stderr, "            -I             print transaction state and option indicators.\n");
   fprintf (stderr, "            -l             print last time values [default is start time].\n");
   fprintf (stderr, "            -m             print MAC addresses.\n");
   fprintf (stderr, "            -n             don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -q             quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R             print out response data when availabile.\n");
   fprintf (stderr, "            -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                            mm/dd[/yy]\n");
   fprintf (stderr, "            -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>      write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z             print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


struct timeval * RaMaxTime  (struct timeval *, struct timeval *);
struct timeval * RaMinTime  (struct timeval *, struct timeval *);
struct timeval * RaDiffTime (struct timeval *, struct timeval *);

struct timeval *
RaMinTime (struct timeval *s1, struct timeval *s2)
{
   struct timeval *retn = s2;

   if ((s1->tv_sec < s2->tv_sec) || ((s1->tv_sec == s2->tv_sec) && (s1->tv_usec < s2->tv_usec)))
      retn = s1;

   return (retn);
}


struct timeval *
RaMaxTime (struct timeval *s1, struct timeval *s2)
{
   struct timeval *retn = s2;

   if ((s1->tv_sec > s2->tv_sec) || ((s1->tv_sec == s2->tv_sec) && (s1->tv_usec > s2->tv_usec)))
      retn = s1;
   
   return (retn);
}

struct timeval RaDiffTimeBuf;

struct timeval *
RaDiffTime (struct timeval *s1, struct timeval *s2)
{
   struct timeval *retn = NULL;

   bzero ((char *)&RaDiffTimeBuf, sizeof(RaDiffTimeBuf));

   if (s1 && s2) {
      RaDiffTimeBuf.tv_sec  = s1->tv_sec - s2->tv_sec;
      RaDiffTimeBuf.tv_usec = s1->tv_usec - s2->tv_usec;

      if (RaDiffTimeBuf.tv_usec < 0) {
         RaDiffTimeBuf.tv_usec += 1000000;
         RaDiffTimeBuf.tv_sec--;
      }
      retn = &RaDiffTimeBuf;
   }

   return (retn);
}

void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecordStore *store = NULL;
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordData *data = NULL;
   struct ArgusFlow *flow = NULL;

   if (argus != NULL) {
      RaThisActiveDuration = RaGetActiveDuration(argus);

      if (Hflag) {
         RaThisActiveIndex = (RaThisActiveDuration < 100)        ? 0 :
                             (RaThisActiveDuration < 500)        ? 1 :
                             (RaThisActiveDuration < 1000)       ? 2 :
                             (RaThisActiveDuration < 5000)       ? 3 :
                             (RaThisActiveDuration < 10000)      ? 4 :
                             (RaThisActiveDuration < 50000)      ? 4 :
                             (RaThisActiveDuration < 100000)     ? 5 :
                             (RaThisActiveDuration < 500000)     ? 6 :
                             (RaThisActiveDuration < 100000)     ? 7 :
                             (RaThisActiveDuration < 250000)     ? 8 :
                             (RaThisActiveDuration < 500000)     ? 9 :
                             (RaThisActiveDuration < 750000)     ? 10:
                             (RaThisActiveDuration < 1000000)    ? 11:
                             (RaThisActiveDuration < 5000000)    ? 12:
                             (RaThisActiveDuration < 10000000)   ? 13:
                             (RaThisActiveDuration < 50000000)   ? 14: 15;
      } else
         RaThisActiveIndex = 0;

      if (RaPassNum == 1) {
         RaStartTime = *RaMinTime (&RaStartTime, &argus->argus_far.time.start);
         RaEndTime =   *RaMaxTime (&RaEndTime, &argus->argus_far.time.last);

      } else {
         if ((store = RaNewArgusStore(argus)) != NULL) {
            flow = &argus->argus_far.flow;

            bcopy(flow, RaArgusFlow, sizeof(struct ArgusFlow));

            if ((data = RaNewArgusData(argus)) == NULL) {
               ArgusLog (LOG_ERR, "RaProcessRecord: RaNewArgusData error %s\n", strerror(errno));

            } else {
               store->data[RaThisActiveIndex] = data;
               data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
               data->status |= RA_MODIFIED;

               if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                  double sumtime;

                  bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
                  data->act.n        = data->agr.act.n;
                  sumtime            = data->agr.act.mean * data->agr.act.n;
                  data->act.sumtime  = sumtime;
                  data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                  data->idle.n       = data->agr.idle.n;
                  sumtime            = data->agr.idle.mean * data->agr.idle.n;
                  data->idle.sumtime = sumtime;
                  data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
               }
            }

            RaMergeQueue (store);

         } else
            ArgusLog (LOG_ERR, "RaNewArgusStore error %s\n", strerror(errno));
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessRecord(0x%x) done.\n", argus);
#endif
}


void RaMergeArray(struct ArgusServiceRecord *);
extern int RaFlowMajorModified;

void
RaMergeArray (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   int RaTimeout = -1;

   bcopy(flow, RaArgusFlow, sizeof(struct ArgusFlow));

   if ((rap = RaFlowModelOverRides(argus)) != NULL) {
      RaModifyFlow(rap, argus);
      RaTimeout = rap->ArgusTimeout;
   } else {
      RaTimeout = 0x1FFFFFF;
      RaModifyDefaultFlow(argus);
   }

   if ((store = RaFindArgusRecord(&RaHashTable, argus)) == NULL)
      store = RaFindRevArgusRecord(&RaHashTable, argus);

   if (store) {
      RaThisArgusStore = store;

/*
      if (RaCheckTimeout (store, argus))
         RaSendArgusRecord(store);

      if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
         RaSendArgusRecord(store);
         store->status &= ~RA_SVCTEST;
         store->status |= (srv->status & RA_SVCTEST);
      }
*/

      if (!(store->data[RaThisActiveIndex])) {
         struct ArgusRecordData *data = NULL;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else {
         RaFlowMajorModified = 1;
         RaMergeArgusRecord(argus, store->data[RaThisActiveIndex]);
      }

      RaUpdateArgusStore(argus, store);

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         struct ArgusRecordData *data = NULL;

         if ((store->rahtblhdr = RaAddHashEntry (&RaHashTable, store)) != NULL)
            RaAddToQueue(RaModelerQueue, &store->qhdr);

         store->ArgusTimeout = RaTimeout;

         RaThisArgusStore = store;

         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;

         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessRecord: done\n");
#endif
}


void
process_man (struct ArgusRecord *argus)
{
}


struct ArgusServiceRecord ArgusThisSrv;

void
process_tcp (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (tcp->state & ARGUS_DST_RESET)
                     if (argus->argus_far.src.count && argus->argus_far.dst.count)
                        if (!(argus->argus_far.src.bytes && argus->argus_far.dst.bytes))
                           ArgusThisSrv.status = RA_SVCFAILED;
       
                  if (tcp->state & ARGUS_SRC_RESET)
                     if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                        ArgusThisSrv.status = RA_SVCFAILED;
               }
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_icmp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_udp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) 
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_arp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_non_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   struct ArgusFlow *flow = NULL;
   struct ArgusAGRStruct *agr = NULL;

   if ((data = store->data[RaThisActiveIndex]) != NULL) {
      argus = data->argus;

      if (argus && (data->status & RA_MODIFIED)) {
         if (data->act.n > 0) {
            data->agr.act.n = data->act.n;
            data->agr.act.mean = data->act.sumtime/data->act.n;
            data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
         }
         if (data->idle.n > 0) {
            data->agr.idle.n = data->idle.n;
            data->agr.idle.mean = data->idle.sumtime/data->idle.n;
            data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
         }

         if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
            bcopy ((char *)&data->agr, agr, data->agr.length);
         }

         bcopy ((char *) argus, buf, argus->ahdr.length);
         argus = (struct ArgusRecord *) buf;

         ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);
   
         flow = &argus->argus_far.flow;

         if (!(data->status & ARGUS_FAR_SRCADDR_MODIFIED))
            flow->ip_flow.ip_src = data->flow.ip_flow.ip_src;

         if (!(data->status & ARGUS_FAR_DSTADDR_MODIFIED))
            flow->ip_flow.ip_dst = data->flow.ip_flow.ip_dst;

         if (!(data->status & ARGUS_FAR_PROTO_MODIFIED))
            flow->ip_flow.ip_p = data->flow.ip_flow.ip_p;

         if (!(data->status & ARGUS_FAR_SRCPORT_MODIFIED))
            flow->ip_flow.sport = data->flow.ip_flow.sport;

         if (!(data->status & ARGUS_FAR_DSTPORT_MODIFIED))
            flow->ip_flow.dport = data->flow.ip_flow.dport;

         if (!(data->status & ARGUS_FAR_TPVAL_MODIFIED))
            flow->ip_flow.tp_p = data->flow.ip_flow.tp_p;

         if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) == NULL) {
            if (ArgusFlowModelFile && (data->agr.count > 1)) {
               bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
               argus->ahdr.length += data->agr.length;
               argus->ahdr.status |= ARGUS_MERGED;
               ArgusFree (data->argus);
               data->argus = RaCopyArgusRecord(argus);
               ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);

            } else {
               argus->ahdr.status &= ~(ARGUS_MERGED);
            }
         }
   
         if (wfile) {

#ifdef _LITTLE_ENDIAN
            ArgusHtoN(argus);
#endif

            ArgusWriteNewLogfile (wfile, argus);
   
#ifdef _LITTLE_ENDIAN
            ArgusNtoH(argus);
#endif

         } else {
            if (argus->ahdr.type & ARGUS_MAR)
               printf ("%s\n", get_man_string (argus));
               
            else {
               printf ("%s\n", get_ip_string(argus));
            }
            fflush (stdout);
         }
   
         argus = data->argus;

         if (argus->ahdr.type & ARGUS_FAR) {
            int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);
            farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
   
            while (length > 0) {
               switch (farhdr->type) {
                  case ARGUS_FAR: {
                     struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                     far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                     far->time.last.tv_sec = 0; far->time.last.tv_usec = 0;
                     far->src.count = 0; far->src.bytes = 0;
                     far->dst.count = 0; far->dst.bytes = 0;
                     break;
                  }
   
                  case ARGUS_TCP_DSR: {
                     struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
   
                     tcp->state = 0;
                     tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                     tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                     tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                     tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
   
                     break;
                  }
   
                  case ARGUS_TIME_DSR: {
                     struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) farhdr;

                     time->src.act.n = 0;
                     time->src.act.mean = 0;
                     time->src.act.stdev = 0;
                     time->src.act.max = 0;
                     time->src.act.min = 0x7FFFFFFF;
                     time->src.idle.n = 0;
                     time->src.idle.mean = 0;
                     time->src.idle.stdev = 0;
                     time->src.idle.max = 0;
                     time->src.idle.min = 0x7FFFFFFF;

                     time->dst.act.n = 0;
                     time->dst.act.mean = 0;
                     time->dst.act.stdev = 0;
                     time->dst.act.max = 0;
                     time->dst.act.min = 0x7FFFFFFF;
                     time->dst.idle.n = 0;
                     time->dst.idle.mean = 0;
                     time->dst.idle.stdev = 0;
                     time->dst.idle.max = 0;
                     time->dst.idle.min = 0x7FFFFFFF;

                     break;
                  }

                  case ARGUS_VLAN_DSR: {
                     struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
                     vlan->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                     break;
                  }

                  case ARGUS_MPLS_DSR: {
                     struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
                     mpls->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                     break;
                  }

                  case ARGUS_AGR_DSR: {
                     struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
   
                     agr->count = 0;
                     agr->act.n = 0;
                     agr->act.min = 0x7FFFFFFF;   agr->act.mean = 0;
                     agr->act.stdev = 0;  agr->act.max = 0;
                     agr->idle.n = 0;
                     agr->idle.min = 0x7FFFFFFF; agr->idle.mean = 0;
                     agr->idle.stdev = 0; agr->idle.max = 0;
                     break;
                  }
               }

               if ((farlen = farhdr->length) == 0)
                  break;
               if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                   (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                  farlen = farlen * 4;
               length -= farlen;
               farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
            }
         }

         data->agr.count = 0;
         data->agr.act.n = 0;
         data->agr.act.mean = 0;
         data->agr.act.stdev = 0;
         data->agr.act.max = 0;
         data->agr.act.min = 0x7FFFFFFF;
         data->agr.idle.n = 0;
         data->agr.idle.mean = 0;
         data->agr.idle.stdev = 0;
         data->agr.idle.max = 0;
         data->agr.idle.min = 0x7FFFFFFF;
   
         data->act.n = 0;
         data->act.sumtime = 0;
         data->act.sumsqrd = 0;
         data->idle.n = 0;
         data->idle.sumtime = 0;
         data->idle.sumtime = 0;
   
         data->argus->ahdr.status &= ~ARGUS_MERGED;
         data->status &= ~RA_MODIFIED;
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}

#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  256
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
            RaTimeoutArgusStore(obj);

         break;

      default:
         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}
