/****************************************************************/
/* file arity.c

ARIBAS interpreter for Arithmetic
Copyright (C) 1996 O.Forster

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Address of the author

    Otto Forster
    Math. Institut der LMU
    Theresienstr. 39
    D-80333 Muenchen, Germany

Email   forster@rz.mathematik.uni-muenchen.de
*/
/****************************************************************/
/*
** arity.c
** factorization algorithms (Rho, continued fractions, quadratic sieve)
** and prime generation (next_prime)
**
** date of last change
** 97-12-09:    created by splitting the file aritx.c
** 97-12-22:    qs_factorize, big prime var
** 98-05-31:    fixed bug in function mkquadpol
** 98-10-11:    function next_prime
** 00-04-26:	fixed bug in ATARI version due to ptr diffs > 32K
*/

#include "common.h"

PUBLIC void iniarity    _((void));

/*--------------------------------------------------------*/
/********* fuer rho-Faktorisierung **********/
typedef struct {
    word2 *x, *y, *Q;
    int xlen, ylen, Qlen;
} T_xyQ;

#define RHO_CYCLEN  256
#define RHO_INC 2

/********* fuer CF- und QS-Faktorisierung ************/

typedef byte sievitem;

typedef struct {
    word2 *bitmat;
    word2 *piv;
    word2 *Aarr;
    word2 *Qarr;
    word2 *fbas;
    int rank;
    int matinc;
    int vlen;
    int ainc;
    int qinc;
    int baslen;
} BMDATA;


#ifdef M_LARGE
#define ALEN        20      /* even! */
#define MAXMROWS    2560
#define MAXSRANGE   128000
#define MEMCHUNK    64000
#define BIGPRIMES 
#else
#define ALEN        12      /* even! */
#define MAXMROWS    512
#define MEMCHUNK    32000       /* < 2**15 */
#define STACKRES    4092
#endif

#define HLEN        (ALEN/2)

typedef struct {
    word2 kNlen;
    word2 kN[ALEN+1];
    word2 m2[HLEN+1];
    word2 R0[HLEN+1];
    word2 Q0[HLEN+1];
    word2 A0[ALEN+1];
    word2 QQ[HLEN+1];
    word2 AA[ALEN];
    int qrsign;
} CFDATA;

typedef struct {
    word2 NNlen;
    word2 NN[ALEN+1];
    word2 qq[HLEN+1];
    word2 qinv[ALEN+1];
    word2 aa[HLEN+1];
    word2 bb[HLEN+1];
    word2 cc[ALEN];
    int xi;
} QPOLY;

typedef struct {
    word2 *fbas;
    word2 *fbroot;
    sievitem *fblog;
    int baslen;
} QSFBAS;

#ifdef BIGPRIMES
/* for big prime variation in quadratic sieve factorization */
#define BPMINLEN    8   /* minimal length for using big prime var */
#define BPMAXIDX    4
PRIVATE unsigned data_anz[BPMAXIDX]  = {16000, 24000, 32000, 64000};
PRIVATE unsigned hashconst[BPMAXIDX] = {31991, 47981, 63997, 127997};
/* primes < 2*data_anz[k] */
#define BIGPRIMEBOUND   1000000

typedef struct {
    word4 bprime;
    word4 qdiff;
    int x;
} QSBP;

typedef struct {
    word2 q0len;
    word2 Q0[HLEN];
    word2 *hashtab;
    unsigned tablen;
    QSBP *QSBPdata;
    unsigned row;
    unsigned maxrows;
    word4 bpbound;
} QSBIGPRIMES;

typedef struct {
    sievitem *Sieve;
    int srange;
    int useBigprim;
    QPOLY *qpol;
    QSFBAS *fbp;
    QSBIGPRIMES *qsbig;
} SIEVEDATA;

#else /* !BIGPRIMES */

typedef struct {
    sievitem *Sieve;
    int srange;
    QPOLY *qpol;
    QSFBAS *fbp;
} SIEVEDATA;
#endif

#ifdef QTEST    /* only for testing */
FILE *qlog;
#endif

/*-------------------------------------------------------------------*/
/* setbit and testbit suppose that vv is an array of word2 */
#define setbit(vv,i)    vv[(i)>>4] |= (1 << ((i)&0xF))
#define testbit(vv,i)   (vv[(i)>>4] & (1 << ((i)&0xF)))
/*-------------------------------------------------------------------*/
PRIVATE truc Frhofact   _((int argn));
PRIVATE int rhocycle    _((int anz, T_xyQ *xyQ, word2 *N, int len,
               word2 *hilf));
PRIVATE void rhomess    _((word4 i));

PRIVATE unsigned banalfact  _((word2 *N, int len));

PRIVATE truc Fcffact    _((int argn));
PRIVATE int brillmorr   _((word2 *N, int len, unsigned v, word2 *fact));
PRIVATE int bm_alloc    _((word2 *buf1, size_t len1, word2 *buf2, size_t len2,
               BMDATA *bmp, int alen, int qlen));
PRIVATE int brill1  _((word2 *N, int len, unsigned u, word2 *fact,
               BMDATA *bmp, CFDATA *cfp, word2 *hilf));
PRIVATE unsigned factorbase  _((word2 *N, int len, word2 *prim, int anz));
PRIVATE int cfracinit   _((word2 *N, int len, unsigned u, CFDATA *cfp,
               word2 *hilf));
PRIVATE int cfracnext   _((word2 *N, int len, CFDATA *cfp, word2 *hilf));
PRIVATE int smoothea    _((word2 *QQ, word2 *fbas, int baslen, int easize));
PRIVATE word4 smooth    _((word2 *QQ, word2 *fbas, int baslen));

PRIVATE int bm_insert   _((BMDATA *bmp, word2 *QQ, int qsign, word2 *AA,
               word2 *hilf));
PRIVATE int gausselim   _((BMDATA *bmp));
PRIVATE int getfactor   _((word2 *N, int len, BMDATA *bmp, word2 *fact,
               word2 *hilf));
PRIVATE truc Fqsfact    _((int argn));
PRIVATE int mpqsfactor  _((word2 *N, int len, word2 *fact));
PRIVATE int qsfact1 _((word2 *N, int len, word2 *fact, BMDATA *bmp,
               SIEVEDATA *qsp, word2 *hilf));

PRIVATE unsigned modpow    _((unsigned x, unsigned n, unsigned mm));
PRIVATE unsigned fpsqrt    _((unsigned p, unsigned a));
PRIVATE unsigned fpsqrt14  _((unsigned p, unsigned a));
PRIVATE void fp2pow _((unsigned p, unsigned D, unsigned *uu, unsigned n));

PRIVATE int startqq _((word2 *N, int len, unsigned srange,
               word2 *qq, word2 *hilf));

PRIVATE int nextqq  _((word2 *N, int len, word2 *q0, int q0len,
                word2 *qq, word2 *hilf));
PRIVATE int dosieve _((SIEVEDATA *qsp));

PRIVATE int p2sqrt  _((word2 *p, int plen, word2 *x, int xlen,
                word2 *z, word2 *hilf));
PRIVATE int mkquadpol   _((word2 *p, int plen, QPOLY *sptr, word2 *hilf));

PRIVATE int quadvalue   _((QPOLY *polp, word2 *QQ, int *signp));
PRIVATE int qresitem    _((QPOLY *polp, word2 *AA));

PRIVATE void workmess   _((void));
PRIVATE void tick   _((int c));
PRIVATE void counttick  _((word4 v, BMDATA *bmp));
PRIVATE void cf0mess    _((int p, int blen));
PRIVATE void cf1mess    _((long n, int nf));
PRIVATE void qs0mess    _((int srange, int p, int blen));
PRIVATE void qs1mess    _((long n, int nf));

PRIVATE int multlarr    _((word2 *x, int n, unsigned a, word2 *y));

#ifdef BIGPRIMES
PRIVATE int hashbigp    _((QSBIGPRIMES *qsbigp, word4 prim, QPOLY *qpolp,
               QPOLY *qpolp2, word2 *hilf));
PRIVATE int combinebp   _((word2 *N, int len, word4 prim, word2 *QQ, word2 *AA,
               word2 *QQ2, word2 *AA2, word2 *hilf));
PRIVATE void qs2mess    _((long n, int nf, int nf2));
#endif

PRIVATE truc Fnextprime _((int argn));
PRIVATE int nextprime32 _((word4 u, word2 *x));

PRIVATE truc nxtprimsym;
PRIVATE truc rhosym, cffactsym, qsfactsym;
PRIVATE int doreport;
/*------------------------------------------------------------------*/
PUBLIC void iniarity()
{
    rhosym = newsymsig("rho_factorize",sFBINARY,(truc)Frhofact, s_13);
    cffactsym = newsymsig("cf_factorize",sFBINARY,(truc)Fcffact, s_13);
    qsfactsym = newsymsig("qs_factorize",sFBINARY,(truc)Fqsfact, s_12);

    nxtprimsym= newsymsig("next_prime",sFBINARY,(truc)Fnextprime, s_12);
}
/*-------------------------------------------------------------------*/
/*
** Messages for factorization algorithms
*/
PRIVATE void workmess()
{
    fnewline(tstdout);
    fprintstr(tstdout,"working ");
}
/*-------------------------------------------------------------------*/
PRIVATE void tick(c)
int c;
{
    char tt[2];

    tt[0] = c;
    tt[1] = 0;
    fprintstr(tstdout,tt);
    fflush(stdout);
}
/*-------------------------------------------------------------------*/
PRIVATE void counttick(v,bmp)
word4 v;
BMDATA *bmp;
{
    char messbuf[80];
    word4 z,w;
    int c;

    if(v&0x7F) {
        tick('_');
    }
    else {
        v >>= 7;
        c = v % 10;
        if(c) {
            tick('0' + c);
        }
        else {
            z = v/10;
            w = bmp->rank;
            w *= 100;
            w /= bmp->baslen;
            s2form(messbuf,"[~D/~D%]",z,w);
            fprintstr(tstdout,messbuf);
            fflush(stdout);
        }
    }
}
/*-------------------------------------------------------------------*/
PRIVATE void rhomess(anz)
word4 anz;
{
    char messbuf[80];

    s1form(messbuf,"~%factor found after ~D iterations",anz);
    fprintline(tstdout,messbuf);
    fnewline(tstdout);
}
/*-------------------------------------------------------------------*/
PRIVATE void cf0mess(p,blen)
int p, blen;
{
    char messbuf[80];

    s2form(messbuf,"~%CF-algorithm: factorbase 2 ... ~D of length ~D",
        (word4)p,(word4)blen);
    fprintline(tstdout,messbuf);
    fprintstr(tstdout,"working ");
}
/*-------------------------------------------------------------------*/
PRIVATE void cf1mess(n,nf)
long n;
int nf;
{
    char messbuf[80];

    s2form(messbuf,
        "~%~D quadratic residues calculated, ~D completely factorized",
        n,(word4)nf);
    fprintline(tstdout,messbuf);
    fnewline(tstdout);
}
/*-------------------------------------------------------------------*/
PRIVATE void qs0mess(srange,p,blen)
int srange, p, blen;
{
    char messbuf[80];

    s1form(messbuf,"~%quadratic sieve, range = ~D, ",(word4)srange);
    fprintstr(tstdout,messbuf);
    s2form(messbuf,"factorbase 2 ... ~D of length ~D",
        (word4)p,(word4)blen);
    fprintline(tstdout,messbuf);
    fprintstr(tstdout,"working ");
}
/*-------------------------------------------------------------------*/
PRIVATE void qs1mess(n,nf)
long n;
int nf;
{
    char messbuf[80];

    s2form(messbuf,
        "~%~D polynomials, ~D completely factorized quadratic residues",
        n,(word4)nf);
    fprintline(tstdout,messbuf);
    fnewline(tstdout);
}
/*------------------------------------------------------------------*/
#ifdef BIGPRIMES
/*------------------------------------------------------------------*/
PRIVATE void qs2mess(n,nf,nf2)
long n;
int nf, nf2;
{
    char messbuf[80];

    s2form(messbuf,"~%~D polynomials, ~D + ",n,(word4)(nf-nf2));
    fprintstr(tstdout,messbuf);

    s2form(messbuf,"~D = ~D factorized quadratic residues",
        (word4)nf2,(word4)nf);
    fprintline(tstdout,messbuf);
    fnewline(tstdout);
}
/*------------------------------------------------------------------*/
#endif
/*------------------------------------------------------------------*/
/*
** Pollardsche rho-Methode zur Faktorisierung;
** Aufruf rho_factorize(N,anz) oder rho_factorize(N); 
** dabei ist anz die maximale Anzahl der Iterationen,
** default anz = 2**16
*/
PRIVATE truc Frhofact(argn)
int argn;
{
    T_xyQ xyQ;

    truc *argptr;
    word2 *N, *z, *d, *Q, *hilf;
    word4 u, i;
    size_t m;
    unsigned rr;
    int k, n, len, sign, ret;

    argptr = argStkPtr-argn+1;
    if(argn >= 2 && *argStkPtr == zero) {
        doreport = 0;
        argn--;
    }
    else {
        doreport = 1;
    }
    if(chkints(rhosym,argptr,argn) == aERROR)
        return(brkerr());

    len = bigref(argptr,&N,&sign);
    m = aribufSize/4;
    if(len >= (m-1)/2) {
        error(rhosym,err_ovfl,*argptr);
        return(brkerr());
    }

    d = AriBuf;
    xyQ.x = d + m;
    xyQ.y = d + 2*m;
    xyQ.Q = d + 3*m;
    Q = AriScratch;
    hilf = AriScratch + aribufSize;

    rr = random2(64000);
    xyQ.x[0] = xyQ.y[0] = rr;
    xyQ.xlen = xyQ.ylen = (rr ? 1 : 0);
    xyQ.Q[0] = 1;
    xyQ.Qlen = 1;

    if(argn == 2) {
        n = bigref(argptr+1,&z,&sign);
        if(n <= 2 && n)
            u = big2long(z,n);
        else
            u = 0x80000000;
    }
    else
        u = 0x10000;
    if(doreport)
        workmess();

    for(i=0; i<u; i+=RHO_CYCLEN) {
        if(doreport)
            tick('.');
        ret = rhocycle(RHO_CYCLEN,&xyQ,N,len,hilf);
        if(ret) {
            k = xyQ.Qlen;
            cpyarr(xyQ.Q,k,Q);
            cpyarr(N,len,d);
            k = biggcd(d,len,Q,k,hilf);
            if(k > 1 || *d > 1) {
                if(doreport)
                    rhomess(i+RHO_CYCLEN);
                return(mkint(0,d,k)); 
            }
        }
        if(INTERRUPT) {
            setinterrupt(0);
            break;
        }
    }
    return(zero);
}
/*-------------------------------------------------------------------*/
/*
** Berechnet anz mal 
** x -> x*x+RHO_INC; x -> x*x+RHO_INC; y -> y*y+RHO_INC mod N
** Q -> Q*(x-y) mod N
** Rueckgabewert Laenge von Q
*/
PRIVATE int rhocycle(anz,xyQ,N,len,hilf)
int anz;
T_xyQ *xyQ;
word2 *N;
int len;
word2 *hilf;
{
    word2 *x1, *y1, *Q1, *z, *z1;
    int n, m, k, cmp;
    int zlen, z1len, nn;

    nn = 2*len + 2;
    z = hilf;
    z1 = hilf + nn;
    hilf = z1 + nn;

    x1 = xyQ->x;
    n = xyQ->xlen;
    y1 = xyQ->y;
    m = xyQ->ylen;
    Q1 = xyQ->Q;
    *Q1 = 1;
    k = 1;

    while(--anz >= 0) {
        zlen = multbig(x1,n,x1,n,z,hilf);
        zlen = incarr(z,zlen,RHO_INC);
        zlen = modbig(z,zlen,N,len,hilf);
        z1len = multbig(z,zlen,z,zlen,z1,hilf);
        z1len = incarr(z1,z1len,RHO_INC);
        n = modbig(z1,z1len,N,len,hilf);
        cpyarr(z1,n,x1);
        zlen = multbig(y1,m,y1,m,z,hilf);
        zlen = incarr(z,zlen,RHO_INC);
        m = modbig(z,zlen,N,len,hilf);
        cpyarr(z,m,y1);
        cmp = cmparr(z,m,z1,n);
        if(cmp > 0)
            zlen = subarr(z,m,z1,n);
        else if(cmp < 0)
            zlen = sub1arr(z,m,z1,n);
        else
            continue;
        z1len = multbig(Q1,k,z,zlen,z1,hilf);
        k = modbig(z1,z1len,N,len,hilf);
        cpyarr(z1,k,Q1);
    }
    xyQ->xlen = n;
    xyQ->ylen = m;
    xyQ->Qlen = k;
    return(k);
}
/*------------------------------------------------------------------*/
/*
** Continued fraction factorization
** cf_factorize(N: integer[; mult: integer]): integer;
*/
PRIVATE truc Fcffact(argn)
int argn;
{
    truc *argptr;
    word2 *N, *x;
    long mm;
    size_t buflen;
    unsigned u;
    int len0, len, n;
    int sign;

    argptr = argStkPtr-argn+1;
    if(argn >= 2 && *argStkPtr == zero) {
        doreport = 0;
        argn--;
    }
    else {
        doreport = 1;
    }
    if(chkints(cffactsym,argptr,argn) == aERROR)
        return(brkerr());

    len = bigref(argptr,&N,&sign);
    u = banalfact(N,len);
    if(u != (unsigned)-1)
        return(mkfixnum(u));

    buflen = auxbufSize * sizeof(word2);
#ifdef M_SMALL
    mm = stkcheck() - STACKRES;
    if(buflen < mm)
        mm = buflen;
#else
    mm = buflen;
#endif
    len0 = ALEN;
    if(mm < MEMCHUNK) {
        error(cffactsym,err_memev,voidsym);
        return(brkerr());
    }

    if(len > len0 || (len == len0 && bitlen(N[len-1]) > 4)) {
        error(cffactsym,err_2big,*argptr);
        return(brkerr());
    }
    if(argn >= 2) {
        n = bigref(argptr+1,&x,&sign);
        u = *x;
        if(n != 1 || u > 1023)
            u = 1;
    }
    else
        u = 1;
    n = brillmorr(N,len,u,AriBuf);

    return(mkint(0,AriBuf,n));
}
/*------------------------------------------------------------------*/
/*
** Falls (N,len) < 2**32 wird der kleinste Primfaktor zurueckgegeben.
** Falls N gerade, wird 2 zurueckgegeben.
** Andernfalls wird (unsigned)-1 zurueckgegeben.
*/
PRIVATE unsigned banalfact(N,len)
word2 *N;
int len;
{
    word4 u;
    unsigned v,d;

    if(len <= 2) {
        u = big2long(N,len);
        v = intsqrt(u);
        d = trialdiv(N,len,2,v);
        return(d);
    }
    else if(!(N[0] & 1)) 
        return(2);
    else
        return((unsigned)-1);
}
/*------------------------------------------------------------------*/
PRIVATE int brillmorr(N,len,v,fact)
word2 *N, *fact;
int len;
unsigned v;
{
#ifdef M_SMALL
    word2 stackpiece[MEMCHUNK/sizeof(word2)];
#endif
    BMDATA bm;
    CFDATA cf;

    word2 *buf1, *buf2, *hilf;
    word4 u;
    size_t b1len, b2len;
    int k, alen, qlen, baslen, maxrows, b0, b1, b, ret;

    b1 = bitlen(N[len-1]);
    alen = (b1 > 4 ? len+1 : len);
    qlen = (alen + 1)/2;
#ifdef M_SMALL
    buf1 = AriScratch;
    b1len = scrbufSize; /* scrbufSize >= auxbufSize in M_SMALL */
    buf2 = stackpiece;
    b2len = MEMCHUNK/sizeof(word2);
    hilf = AuxBuf;
#else
    buf1 = AuxBuf;
    b1len = auxbufSize/2;
    buf2 = AuxBuf + b1len;
    b2len = b1len;
    hilf = AriScratch;
#endif
    maxrows = bm_alloc(buf1,b1len,buf2,b2len,&bm,alen,qlen);
    b0 = maxrows/16;

    u = (len - 1)*16 + b1;     /* bitlength of N */
    b = 1 + (u*u)/384;
    if(b > b0)
        b = b0;
    bm.vlen = b;

    bm.baslen = baslen = b * 16 - 2;
    for(k=0; k<=baslen; k++)
        bm.piv[k] = baslen-k;
        /* bm.piv[k] = k; */

    bm.fbas = hilf;
    hilf += b * 16;

    ret = brill1(N,len,v,fact,&bm,&cf,hilf);

    return(ret);
}
/*-------------------------------------------------------------------*/
/*
** In (buf1,len1) und (buf2,len2) werden zwei Puffer uebergeben
** Aus diesen wird der Struktur *bmp Speicher zugewiesen
**
** Bedarf fuer bmp:
**  bmp->bitmat Platz fuer eine Bitmatrix maxrows*(2*maxrows)
**  bmp->piv    Platz fuer maxrows word2's
**  bmp->fbas   Platz fuer maxrows word2's
**  bmp->Aarr   Platz fuer maxrows bigints der Laenge alen,
**          zuzueglich Laengen-Angabe
**  bmp->Qarr   Platz fuer maxrows bigints der Laenge qlen,
**          zuzueglich Laengen-Angabe
**
** Rueckgabewert ist maxrows; dies ist durch 16 teilbar
*/
PRIVATE int bm_alloc(buf1,len1,buf2,len2,bmp,alen,qlen)
word2 *buf1, *buf2;
size_t len1, len2;
BMDATA *bmp;
int alen, qlen;
{
    word2 *xx, *yy;
    word4 u;
    size_t ll;
    int maxrows;

    /* allocation for bmp->bitmat (from buf1) */
    u = len1;
    u *= 8;
    maxrows = (int)intsqrt(u);
    if(maxrows > MAXMROWS)
        maxrows = MAXMROWS;
    maxrows &= 0x7FF0;   /* make it a multiple of 16 */

    bmp->bitmat = buf1;
    bmp->matinc = maxrows/8;
    bmp->rank = 0;

    /* allocation for Aarr, Qarr, piv (from buf2) */
    alen++; qlen++;     /* one word2 for length specification */
    ll = len2 / (alen + qlen + 1);
    if(ll < maxrows)
        maxrows = (ll & 0x7FF0);

    bmp->Aarr = buf2 + 1;
    bmp->ainc = alen;

    xx = buf2 + alen * maxrows;
    bmp->Qarr = xx + 1;
    bmp->qinc = qlen;

    yy = xx + qlen * maxrows;
    bmp->piv = yy;

    return(maxrows);
}
/*-------------------------------------------------------------------*/
PRIVATE int brill1(N,len,u,fact,bmp,cfp,hilf)
word2 *N, *fact, *hilf;
unsigned u;
int len;
BMDATA *bmp;
CFDATA *cfp;
{
    word2 *zz, *fbase;
    word4 v;
    int k, qrlen, blen, easize, res;
    int count, count1;

    qrlen = cfracinit(N,len,u,cfp,hilf);
    if(qrlen == 0) {
        k = cfp->AA[-1];
        cpyarr(cfp->AA,k,fact);
        zz = cfp->kN;
        cpyarr(N,len,zz);
        return(biggcd(fact,k,zz,len,hilf));
    }
    blen = bmp->baslen;
    fbase = bmp->fbas;
    factorbase(cfp->kN,(int)cfp->kN[-1],fbase,blen);
    
    if(doreport)
        cf0mess(fbase[blen-1],blen);

    easize = intsqrt(6*(word4)blen);    /* ?! */
    for(v=1, count=count1=0; qrlen && count1<blen; v++) {
        if((v & 0xFF) == 0) {
        if(INTERRUPT) {
            setinterrupt(0);
            break;
        }
        if((v & 0x3FF) == 0) {
            if(doreport)
                counttick(v>>10,bmp);
        }
        }
        if(smoothea(cfp->QQ,fbase,blen,easize)) {
        if((++count & 0x3) == 1)
            if(doreport)
                tick('.');
        res = bm_insert(bmp,cfp->QQ,cfp->qrsign,cfp->AA,hilf);
        if(res && gausselim(bmp)) {
            count1++;
            if(doreport)
                tick('!');
            k = getfactor(N,len,bmp,fact,hilf);
            if(k > 0) {
                if(doreport)
                    cf1mess(v,count);
                return(k);
            }
        }
        }
        qrlen = cfracnext(N,len,cfp,hilf);
    }
    return(0);
}
/*------------------------------------------------------------------*/
/*
** Schreibt in das Array fbase die Primzahl 2 und weitere (anz-1)
** ungerade Primzahlen, fuer die jacobi((N,len),p) = 1
** Falls N durch eine Primzahl, die kleiner als das Maximum
** der Faktorbasis ist, teilbar ist, wird diese zurueckgegeben,
** andernfalls 0.
*/
PRIVATE unsigned factorbase(N,len,fbase,anz)
word2 *N, *fbase;
int len, anz;
{
    unsigned m, p;
    unsigned divisor = 0;
    int idx;

    fbase[0] = 2;
    for(idx=1, p=3; idx<anz; p+=2) {
        if(!prime16(p))
            continue;
        m = modarr(N,len,p);
        if(m == 0 && divisor == 0) {
            divisor = p;
        }
        if(jac(m,p) >= 0) {
            fbase[idx] = p;
            idx++;
        }
    }
    return(divisor);
}
/*-------------------------------------------------------------------*/
PRIVATE int cfracinit(N,len,u,cfp,hilf)
word2 *N;
int len;
unsigned u;
CFDATA *cfp;
word2 *hilf;
{
    word2 *temp, *temp1;
    int k, n, rlen;
    int ll = 2*len + 2;

    temp = hilf;
    temp1 = temp + ll;
    hilf = temp1 + ll;

    len = multarr(N,len,u,temp);
    cpyarr(temp,len,cfp->kN);
    cfp->kN[-1] = len;

    k = bigsqrt(temp,len,temp1,&rlen,hilf);
    cpyarr(temp1,k,cfp->AA);
    cfp->AA[-1] = k;

    n = multbig(temp1,k,temp1,k,temp,hilf);
    rlen = sub1arr(temp,n,cfp->kN,len);
    cpyarr(temp,rlen,cfp->QQ);
    cfp->QQ[-1] = rlen;

    k = shlarr(temp1,k,1);
    cpyarr(temp1,k,cfp->m2);
    cfp->m2[-1] = k;

    cfp->R0[-1] = 0;
    cfp->A0[0] = 1;
    cfp->A0[-1] = 1;
    cfp->Q0[0] = 1;
    cfp->Q0[-1] = 1;

    cfp->qrsign = -1;
    return(rlen);
}
/*-------------------------------------------------------------------*/
/*
** m2 - R0 = bb * QQ + rest;
** Qnew = Q0 + (rest - R0) * bb;
** Anew = (AA * bb + A0) mod N
** next R0 = rest
** next Q0 = QQ
** next A0 = AA
** next QQ = Qnew
** next AA = Anew
*/
PRIVATE int cfracnext(N,len,cfp,hilf)
word2 *N;
int len;
CFDATA *cfp;
word2 *hilf;
{
    static word2 rr[HLEN], bb[HLEN], temp1[HLEN];

    word2 *QQ, *Qtemp, *Atemp;
    int m2len, rlen, blen, qtlen, atlen, t1len;
    int cmp;

    Qtemp = Atemp = hilf;
    hilf += 2*ALEN;
    QQ = cfp->QQ;

    m2len = cfp->m2[-1];
    cpyarr(cfp->m2,m2len,rr);
    m2len = subarr(rr,m2len,cfp->R0,(int)cfp->R0[-1]);
    blen = divbig(rr,m2len,QQ,(int)QQ[-1],bb,&rlen,hilf);
    cpyarr(rr,rlen,temp1);
    t1len = rlen;

    cmp = cmparr(temp1,t1len,cfp->R0,(int)cfp->R0[-1]);
    if(cmp >= 0)
        t1len = subarr(temp1,t1len,cfp->R0,(int)cfp->R0[-1]);
    else
        t1len = sub1arr(temp1,t1len,cfp->R0,(int)cfp->R0[-1]);

    cpyarr(rr,rlen,cfp->R0);
    cfp->R0[-1] = rlen;

    qtlen = multbig(temp1,t1len,bb,blen,Qtemp,hilf);
    if(cmp >= 0)
        qtlen = addarr(Qtemp,qtlen,cfp->Q0,(int)cfp->Q0[-1]);
    else
        qtlen = sub1arr(Qtemp,qtlen,cfp->Q0,(int)cfp->Q0[-1]);

    cpyarr(QQ,(int)QQ[-1],cfp->Q0);
    cfp->Q0[-1] = QQ[-1];
    cpyarr(Qtemp,qtlen,QQ);
    QQ[-1] = qtlen;

    atlen = multbig(cfp->AA,(int)cfp->AA[-1],bb,blen,Atemp,hilf);
    atlen = addarr(Atemp,atlen,cfp->A0,(int)cfp->A0[-1]);
    atlen = modbig(Atemp,atlen,N,len,hilf);

    cpyarr(cfp->AA,(int)cfp->AA[-1],cfp->A0);
    cfp->A0[-1] = cfp->AA[-1];
    cpyarr(Atemp,atlen,cfp->AA);
    cfp->AA[-1] = atlen;

    cfp->qrsign = -cfp->qrsign;

    return(qtlen);
}
/*-------------------------------------------------------------------*/
/*
** Rueckgabe = 1, falls quadratischer Rest QQ smooth; sonst = 0
** QQ[-1] enthaelt Laengenangabe
** TODO: big prime variation
*/
PRIVATE int smoothea(QQ,fbas,baslen,easize)
word2 *QQ, *fbas;
int baslen, easize;
{
    word2 Q[ALEN];
    unsigned p;
    int qn, i, bitl, bound;
    word2 r;

    qn = QQ[-1];
    cpyarr(QQ,qn,Q);
    bitl = (qn - 1)*16 + bitlen(Q[qn-1]);
    bound = easize;
    i = 0;
  nochmal:
    while(++i <= bound) {
        p = *fbas++;
        while(modarr(Q,qn,p) == 0) 
            qn = divarr(Q,qn,p,&r);
        if(qn == 1 && Q[0] == 1)
            return(1);
    }
    if(bound < baslen) {
        if(bitl - 16*(qn - 1) - bitlen(Q[qn-1]) >= 10) {
            bound = baslen;
            goto nochmal;
        }
        /* else early abort */
    }
    return(0);
}
/*-------------------------------------------------------------------*/
/*
** QQ[-1] contains length of QQ, which must be <= ALEN
** Extracts from QQ all prime factors in fbas
** returns last cofactor u if u < 2**32;
** else returns 0
*/
PRIVATE word4 smooth(QQ,fbas,baslen)
word2 *QQ, *fbas;
int baslen;
{
    word2 Q[ALEN];
    unsigned p;
    int qn, i;
    word2 r;

    qn = QQ[-1];
    cpyarr(QQ,qn,Q);
    for(i=0; i<baslen; i++) {
        p = *fbas++;
        while(modarr(Q,qn,p) == 0) 
            qn = divarr(Q,qn,p,&r);
        if(qn == 1 && Q[0] == 1) 
            return(1);
    }
    if(qn <= 2)
        return(big2long(Q,qn));
    else
        return(0);
}
/*-------------------------------------------------------------------*/
/*
** qsign*QQ ist quadratischer Rest = AA**2 mod N,
** der sich mit der Faktorbasis faktorisieren laesst.
** QQ, AA und der zugehoerige Bitvektor werden in 
** die Struktur *bmp eingetragen.
** Rueckgabewert: 1, falls QQ vollstaendig faktorisierbar,
**        0  sonst
*/
PRIVATE int bm_insert(bmp,QQ,qsign,AA,aux)
BMDATA *bmp;
word2 *QQ, *AA, *aux;
int qsign;
{
    word2 *Q, *A, *vect, *prime;
    word2 r;
    unsigned p;
    int qn, i, v, alen, baslen;
	size_t bmrk;

	bmrk = bmp->rank;
    Q = bmp->Qarr + bmrk*(bmp->qinc);
    qn = Q[-1] = QQ[-1];
    cpyarr(QQ,qn,Q);
    cpyarr(QQ,qn,aux);
    A = bmp->Aarr + bmrk*(bmp->ainc);
    alen = AA[-1];
    A[-1] = alen;
    cpyarr(AA,(int)A[-1],A);

    vect = bmp->bitmat + bmrk*(bmp->matinc);
    setarr(vect,bmp->vlen,0);
    if(qsign)
        vect[0] = 1;    /* setbit(vect,0); */
    prime = bmp->fbas;
    baslen = bmp->baslen;
    i = 0;
    while(++i<=baslen) {
        p = *prime++;
        v = 0;
        while(modarr(aux,qn,p) == 0) {
            v++;
            qn = divarr(aux,qn,p,&r);
        }
        if(v & 1)
            setbit(vect,i);
        if((qn == 1) && (aux[0] == 1)) {
            return(1);
        }
    }
    return(0);
}
/*-------------------------------------------------------------------*/
#ifdef QTEST
int showvect(vect,len)
word2 *vect;
int len;
{
	int i;
    for(i=0; i<len; i++)
		fprintf(qlog,"%04X",vect[i]);
	fprintf(qlog,"\n");
	return len;
}
#endif
/*-------------------------------------------------------------------*/
/*
** Rueckgabe = 0, falls letzte Zeile unabhaengig; sonst = 1
*/
PRIVATE int gausselim(bmp)
BMDATA *bmp;
{
    word2 *v, *vect, *vectb, *piv;
    unsigned pivot, minc;
    int i;
    int vn, v2n, baslen;
	size_t rk;

    minc = bmp->matinc;
    rk = bmp->rank;
    vn = bmp->vlen;
    v2n = 2*vn;
    baslen = bmp->baslen;   

    vect = bmp->bitmat + rk * minc;
    vectb = vect + vn;
    setarr(vectb,vn,0);
    setbit(vectb,rk);

    v = bmp->bitmat;
    piv = bmp->piv;
    for(i=0; i<rk; i++) {
        pivot = piv[i];
        if(testbit(vect,pivot))
            xorarr(vect,v2n,v);
        v += minc;
    }
    for(i=0; i<vn; i++) {
        if(vect[i])
            	break;
    }
    if(i == vn)
		return(1);	/* then vect is identically zero */
    /* find new pivot */
    for(i=rk; i<=baslen; i++) {
        pivot = piv[i];
        if(testbit(vect,pivot)) {
            piv[i] = piv[rk];
            piv[rk] = pivot;
            break;
        }
    }
    bmp->rank = rk + 1;

    return(0);
}
/*-------------------------------------------------------------------*/
/*
** Tries to find a factor of N
** Hypothesis: The current row of the bit matrix in *bmp
** is linearly dependent from the previous rows.
** fact is a word2 array large enough to hold the factor
** If a factor is found, the number of linear dependent rows 
** which contributed to finding the factor is stored in hilf[0].
*/
PRIVATE int getfactor(N,len,bmp,fact,hilf)
word2 *N, *fact, *hilf;
int len;
BMDATA *bmp;
{
    word2 *relat, *temp, *aux, *A, *AA, *Q, *QQ, *XX, *X1, *X2;
    int cmp, k;
	size_t rk, ainc, qinc;
    int count;
    int alen, qlen, q1len, tlen, xlen, x1len, x2len;
    int ll, lll;

    ll = 2*ALEN + 2;
    lll = 4*ALEN;

    A = hilf;
    XX = A + ll;
    X1 = XX + ll;
    X2 = X1 + ll;
    Q = X2 + ll;
    temp = Q + lll;
    aux = temp + lll;

    rk = bmp->rank;
    relat = bmp->bitmat + rk * (bmp->matinc) + (bmp->vlen);
    ainc = bmp->ainc;
    qinc = bmp->qinc;

    AA = bmp->Aarr + rk*ainc;
    alen = AA[-1];
    cpyarr(AA,alen,A);
    AA = bmp->Aarr;
    XX[0] = 1;
    xlen = 1;
    QQ = bmp->Qarr + rk*qinc;
    qlen = QQ[-1];
    cpyarr(QQ,qlen,Q);
    QQ = bmp->Qarr;

    for(count=1,k=0; k<rk; k++) {
        if(testbit(relat,k)) {
            count++;
            /* build AA */
            alen = multbig(A,alen,AA,(int)AA[-1],temp,aux);
            cpyarr(temp,alen,A);
            alen = modbig(A,alen,N,len,aux);

            /* build QQ */
            q1len = QQ[-1];
            cpyarr(QQ,q1len,X1);
            cpyarr(Q,qlen,temp);
            x1len = biggcd(X1,q1len,temp,qlen,aux);
            cpyarr(XX,xlen,X2);
            xlen = multbig(X2,xlen,X1,x1len,XX,aux);
            xlen = modbig(XX,xlen,N,len,aux);

            cpyarr(QQ,q1len,temp);
            x2len = divbig(temp,q1len,X1,x1len,X2,&tlen,aux);
            tlen = divbig(Q,qlen,X1,x1len,temp,&qlen,aux);
            qlen = multbig(temp,tlen,X2,x2len,Q,aux);
            if(qlen + ALEN >= lll) {
            lll += 2*ALEN;
            temp = Q + lll;
            aux = temp + lll;
            }
        }
        AA += ainc;
        QQ += qinc;
    }
    tlen = bigsqrt(Q,qlen,temp,&qlen,aux);
    qlen = multbig(temp,tlen,XX,xlen,Q,aux);
    xlen = modbig(Q,qlen,N,len,aux);
    cpyarr(Q,xlen,XX);

    cmp = cmparr(A,alen,XX,xlen);
    if(cmp == 0)
        return(0);
    cpyarr(A,alen,fact);
    if(cmp > 0)
        k = subarr(fact,alen,XX,xlen);
    else
        k = sub1arr(fact,alen,XX,xlen);
    cpyarr(N,len,temp);
    k = biggcd(fact,k,temp,len,aux);
    if(k > 1 || (k == 1 && fact[0] != 1)) {
        hilf[0] = count;
        return(k);
    }
    else
        return(0);
}
/*------------------------------------------------------------------*/
/*
** Multi-polynomial quadratic sieve factorization
*/
PRIVATE truc Fqsfact(argn)
int argn;
{
    truc *argptr;
    word2 *N;
    size_t buflen, buf2len;
    unsigned d;
    int len, n, sign, enough;

    argptr = argStkPtr-argn+1;
    if(argn >= 2 && *argStkPtr == zero) {
        doreport = 0;
    }
    else {
        doreport = 1;
    }
    len = bigref(argptr,&N,&sign);
    if(len == aERROR) {
        error(qsfactsym,err_int,*argptr);
        return(brkerr());
    }
    d = banalfact(N,len);
    if(d != (unsigned)-1)
        return(mkfixnum(d));

#ifdef M_SMALL
    buflen = stkcheck() - STACKRES;
#else
    buflen = scrbufSize * sizeof(word2);
#endif
    buf2len = auxbufSize * sizeof(word2);
    enough = (buflen >= MEMCHUNK && buf2len >= MEMCHUNK);
    if(!enough) {
        error(qsfactsym,err_memev,voidsym);
        return(brkerr());
    }
    if(len > ALEN) {
        error(qsfactsym,err_2big,*argptr);
        return(brkerr());
    }

#ifdef QTEST
qlog = fopen("qtest.log","w");
#endif

    n = mpqsfactor(N,len,AriBuf);

#ifdef QTEST
fclose(qlog);
#endif
    return(mkint(0,AriBuf,n));
}
/*-------------------------------------------------------------------*/
PRIVATE int mpqsfactor(N,len,fact)
word2 *N, *fact;
int len;
{
#ifdef M_SMALL
    word2 stackpiece[MEMCHUNK/sizeof(word2)];
#endif
#ifdef BIGPRIMES
    QSBIGPRIMES qsbp;
    int bpv, idx;
#endif
    BMDATA bm;
    SIEVEDATA sv;
    QSFBAS qsfb;
    word2 *buf1, *buf2, *hilf;
    size_t b1len, b2len, slen, restlen;
    word4 bitl;
    int k, alen, qlen, baslen, maxrows, srange, b0, b;

    bitl = (len - 1)*16 + bitlen(N[len-1]);     /* bitlength of N */
    alen = len;
    qlen = (alen + 1)/2 + 2;        /* !! */

#ifdef M_SMALL
    srange = (MEMCHUNK/sizeof(sievitem))/2;
    sv.srange = srange;
    sv.Sieve = (sievitem *)stackpiece;
    buf1 = AuxBuf;
    b1len = auxbufSize;
    buf2 = AriScratch;
    b2len = scrbufSize/2;
    hilf = AriScratch + b2len;
#else /* M_LARGE */
    buf1 = AuxBuf;
    restlen = auxbufSize;
#ifdef BIGPRIMES
    if(len < BPMINLEN)
        bpv = 0;
    else {
        b0 = sizeof(QSBP) + 2*sizeof(word2);
        b = ((auxbufSize/2)*sizeof(word2))/b0;
        idx = BPMAXIDX;
        while(--idx >= 0) {
            if(data_anz[idx] <= b)
                break;
        }
        bpv = (idx >= 0 ? 1 : 0);
    }
    sv.useBigprim = bpv;
    if(bpv) {
        qlen = alen + 2;        /* !! */
        qsbp.maxrows = data_anz[idx];
        qsbp.QSBPdata = (QSBP *)AuxBuf;
        b1len = (qsbp.maxrows*sizeof(QSBP))/sizeof(word2);
        buf1 = AuxBuf + b1len;
        qsbp.tablen = hashconst[idx];
        qsbp.hashtab = buf1;
        b2len = qsbp.tablen + 1;    /* even */
        buf1 += b2len;
        k = (len >= BPMINLEN+4 ? 4 : (len >= BPMINLEN+2 ? 2 : 1));
        qsbp.bpbound = k*BIGPRIMEBOUND;
        sv.qsbig = &qsbp;
        restlen = auxbufSize - (b1len + b2len);
    }
#endif /* BIGPRIMES */
    srange = bitl*100;
    if(bitl > 120)
        srange += (bitl-120)*(25*bitl)/8;
/*
    srange = 1024 + bitl*(64 + bitl);
    srange = bitl*500;
*/

    if(srange > MAXSRANGE)
        srange = MAXSRANGE;
    if(2*srange*sizeof(sievitem) > (restlen/3)*sizeof(word2))
        srange = (restlen/6)*sizeof(word2)/sizeof(sievitem);
    srange &= ~0x3;     /* make it a multiple of 4 */
    sv.srange = srange;
    sv.Sieve = (sievitem *)buf1;
    slen = (2*srange)/sizeof(word2);
    buf1 += slen; 
    b1len = restlen - slen;

    if(scrbufSize*sizeof(word2) > 2*MEMCHUNK)
        b2len = scrbufSize - MEMCHUNK/sizeof(word2);
    else
        b2len = scrbufSize/2;
    buf2 = AriScratch;
    hilf = buf2 + b2len;
#endif /* M_LARGE */

    maxrows = bm_alloc(buf1,b1len,buf2,b2len,&bm,alen,qlen);
    b0 = maxrows/16;
    b = (bitl*bitl)/400;
    if(bitl > 160)
        b += (b*(bitl-160))/80;
    if(b < 1)
        b = 1;
    else if(b > b0)
        b = b0;
    bm.vlen = b;
    baslen = b*16 - 2;
    for(k=0; k<=baslen; k++)
        bm.piv[k] = baslen-k;

    sv.fbp = &qsfb;
    qsfb.fbas = bm.fbas = hilf;
    qsfb.baslen = bm.baslen = baslen;
    qsfb.fbroot = hilf + b*16;
    qsfb.fblog = (sievitem *)(hilf + b*32);
    hilf += b * (32 + sizeof(sievitem)*16/sizeof(word2));

    return qsfact1(N,len,fact,&bm,&sv,hilf);
}
/*---------------------------------------------------------------*/
PRIVATE int qsfact1(N,len,fact,bmp,qsp,hilf)
word2 *N, *fact, *hilf;
int len;
BMDATA *bmp;
SIEVEDATA *qsp;
{
#ifdef BIGPRIMES
    QSBIGPRIMES *qsbigp;
    QPOLY Qpol2;
    word2 Work2[2*ALEN+4];
    word2 *hashtab, *tabptr, *QQ2, *AA2;
    word4 bigpbound, u;
    int sgn2;
    int count2 = 0;
    int useBigprim;
#endif
    QPOLY Qpol;
    QSFBAS *fbp;
    word2 Q0[HLEN], Q1[HLEN], Work[2*ALEN+4];
    word2 *fbase, *fbroot, *QQ, *AA;
    sievitem *fblog, *sieve, *sptr;
    word4 cofac, v;
    int res, haveres, shrieks;
    int k, n, n8, baslen, qlen, srange, count, count1, sgn, xi;
    unsigned p,a;
    sievitem tol,target;

    QQ = Work + 1;
    AA = Work + (ALEN+4);

    fbp = qsp->fbp;
    fbase = fbp->fbas;
    baslen = fbp->baslen;
    p = factorbase(N,len,fbase,baslen);
    if(p > 1) { /* found small prime divisor */
        fact[0] = p;
        return(1);
    }
    fbroot = fbp->fbroot;
    fbroot[0] = 1;
    fblog = fbp->fblog;
    n8 = N[0] & 0x7;
    fblog[0] = (n8 == 1 ? 3 : (n8 == 5 ? 2 : 1));

    for(k=1; k<baslen; k++) {
        p = fbase[k];
        a = modarr(N,len,p);
        fbroot[k] = fpsqrt(p,a);
        fblog[k] = bitlen(p);
    }
    Qpol.NNlen = len;
    cpyarr(N,len,Qpol.NN);
    qsp->qpol = &Qpol;

    sieve = qsp->Sieve;
    srange = qsp->srange;
    qlen = startqq(N,len,srange,Q0,hilf);
    tol = fblog[baslen-1];

#ifdef BIGPRIMES
    useBigprim = qsp->useBigprim;
    if(useBigprim) {
        QQ2 = Work2 + 1;
        AA2 = Work2 + (ALEN+4);
        Qpol2.NNlen = len;
        cpyarr(N,len,Qpol2.NN);
        qsbigp = qsp->qsbig;
        bigpbound = qsbigp->bpbound;
        u = fbase[baslen-1];
        u *= u;
        if(bigpbound > u) {
            bigpbound = u;
        }
        tol = lbitlen(bigpbound) + (len-BPMINLEN);
        /* somewhat arbitrary */

        hashtab = qsbigp->hashtab;
        for(k=qsbigp->tablen, tabptr=hashtab; k>0; k--)
            *tabptr++ = 0xFFFF;
        qsbigp->row = 0;
        cpyarr(Q0,qlen,qsbigp->Q0);
        qsbigp->Q0[-1] = qlen;
    }   
#endif
    target = bit_length(N,len)/2 + lbitlen((word4)srange) - tol;

    if(doreport)
        qs0mess(srange,fbase[baslen-1],baslen);
    for(v=1, count=count1=shrieks=0; shrieks<baslen; v++) {
        if(INTERRUPT) {
        setinterrupt(0);
        break;
        }
        /* calculate polynomial for next sieving */
  nochmal:
        cpyarr(Q0,qlen,Q1);
        qlen = incarr(Q1,qlen,2);
        qlen = nextqq(N,len,Q1,qlen,Q0,hilf);
        res = mkquadpol(Q0,qlen,&Qpol,hilf);
        if(res > 0) {   /* Q0 divides N */
        cpyarr(Q0,qlen,fact);
        return(qlen);
        }
        else if(res < 0) {  /* Q0 not prime */
        goto nochmal;
        }

        dosieve(qsp);
        /* collect sieve results */
        for(sptr=sieve,xi=-srange; xi<srange; xi++,sptr++) {
        if(*sptr >= target) {
            haveres = 0;
            Qpol.xi = xi;
            quadvalue(&Qpol,QQ,&sgn);
            cofac = smooth(QQ,fbase,baslen);
            if(cofac == 1) {
            if((++count1 & 0x3) == 1)
                if(doreport)
                tick('.');  
            qresitem(&Qpol,AA);
            haveres = 1;
            }
#ifdef BIGPRIMES
            else if(useBigprim && (cofac < bigpbound) && (cofac > 1)) {
            res = hashbigp(qsbigp,cofac,&Qpol,&Qpol2,hilf);
            if(res > 0) {
                if((++count2 & 0x3) == 1)
                if(doreport)
                    tick(':');
                qresitem(&Qpol,AA);
                quadvalue(&Qpol2,QQ2,&sgn2);
                qresitem(&Qpol2,AA2);
                sgn = (sgn == sgn2 ? 0 : -1);
                combinebp(N,len,cofac,QQ,AA,QQ2,AA2,hilf);
                haveres = 1;
            }
            }
#endif /* BIGPRIMES */
            if(haveres) {
            count++;
            bm_insert(bmp,QQ,sgn,AA,hilf);
            if(gausselim(bmp)) {
                shrieks++;
                if(doreport)
                tick('!');
                n = getfactor(N,len,bmp,fact,hilf);
                if(n > 0) {
#ifdef BIGPRIMES
                if(useBigprim) {
                    if(doreport)
                        qs2mess(v,count,count2);
                }
                else
#endif
                    if(doreport)
                    qs1mess(v,count);
                return(n);
                }
            }
            }
        }
        }
        if(doreport)
        counttick(v,bmp);
    }
    return(0);
}
/*---------------------------------------------------------------*/
PRIVATE int dosieve(qsp)
SIEVEDATA *qsp;
{
    QPOLY *qpol;
    QSFBAS *fbp;
    sievitem *sieve, *sptr, *fblog;
    word2 *fbas, *fbroot, *aa, *bb, *cc;
    word4 u;
    unsigned a1, ainv, b1, binv, p, r, r1, s, xi, xi0, srange, srange2;
    int k, alen, blen, clen, baslen;
    sievitem z;

    srange = qsp->srange;
    srange2 = 2*srange;
    sieve = qsp->Sieve;
    qpol = qsp->qpol;

    aa = qpol->aa; alen = aa[-1];
    bb = qpol->bb; blen = bb[-1];
    cc = qpol->cc; clen = cc[-1];

    fbp = qsp->fbp;
    fbas = fbp->fbas;
    baslen = fbp->baslen;
    fbroot = fbp->fbroot;
    fblog = fbp->fblog;

    z = fblog[0];
    sieve[0] = sieve[srange2-1] = 0;
    xi0 = ((cc[0]&1) == (srange&1) ? 0 : 1);
    for(sptr=sieve+xi0, xi=xi0+1; xi<srange2; xi+=2) {
        *sptr++ = z;
        *sptr++ = 0;
    }

    for(k=1; k<baslen; k++) {
        p = fbas[k];
        r = fbroot[k];
        z = fblog[k];
        s = srange % p;
        if(s) 
            s = p - s;          
        /* s = (-srange) mod p */
        a1 = modarr(aa,alen,p);
        b1 = modarr(bb,blen,p);
        if(a1) {
            ainv = modpow(a1,p-2,p);
            /* inverse of a1 mod p */
            u = r + (p - b1);
            u *= ainv;
            r1 = u % p;
            /* r1 = (r-b1)*ainv mod p, 
            ** this is one root of qpol mod p
            */
            /* sieving with first root */
            xi0 = (r1 >= s ? r1 - s : r1 + (p-s));
            for(sptr=sieve+xi0,xi=xi0; xi<srange2; sptr+=p,xi+=p)
            *sptr += z;
            /* now calculate the other root of qpol */
            u = (p - r) + (p - b1);
            u *= ainv;
            r1 = u % p;
        }
        else {  
            /* qpol mod p is linear, we calculate
            ** the only root of qpol mod p
            ** r1 = cc/(2*bb) (mod p)
            */
            u = modarr(cc,clen,p);
            binv = modpow(2*b1,p-2,p);
            u *= binv;
            r1 = u % p;
        }
        /* now sieving with second root */
        xi0 = (r1 >= s ? r1 - s : r1 + (p-s));
        for(sptr=sieve+xi0,xi=xi0; xi<srange2; sptr+=p,xi+=p)
            *sptr += z;
    }
    return(0);
}
/*---------------------------------------------------------------*/
#ifdef BIGPRIMES
/*
** inserts bigprime in hash table
** if insertion possible without collision, returns 0
** if matching bigprime is found, returns 1
** if there is collision without match, returns -1
*/
PRIVATE int hashbigp(qsbigp,prim,qpolp,qpolp2,hilf)
QSBIGPRIMES *qsbigp;
word4 prim;
QPOLY *qpolp, *qpolp2;
word2 *hilf;
{
    QSBP *qsdata;
    word2 *hashtab;
    word2 Qtemp[HLEN], pp[2];
    word2 *q0;
    word4 qdiff;
    unsigned idx, row, row0;
    int qlen, q0len, plen;

    qsdata = qsbigp->QSBPdata;
    hashtab = qsbigp->hashtab;
    idx = prim % (qsbigp->tablen);
    row0 = hashtab[idx];
    if(row0 == 0xFFFF) {
        row = qsbigp->row;
        if(row < qsbigp->maxrows)
        hashtab[idx] = row;
        qsdata[row].bprime = prim;
        qsdata[row].x = qpolp->xi;
        qlen = qpolp->qq[-1];
        cpyarr(qpolp->qq,qlen,Qtemp);
        q0 = qsbigp->Q0; q0len = q0[-1];
        qlen = subarr(Qtemp,qlen,q0,q0len);
        qsdata[row].qdiff = big2long(Qtemp,qlen);
        qsbigp->row = ++row;
        return(0);
    }
    else if(prim == qsdata[row0].bprime) {
        qpolp2->xi = qsdata[row0].x;
        qdiff = qsdata[row0].qdiff;
        plen = long2big(qdiff,pp);
        q0 = qsbigp->Q0; q0len = q0[-1];
        cpyarr(q0,q0len,Qtemp);
        qlen = addarr(Qtemp,q0len,pp,plen);
        mkquadpol(Qtemp,qlen,qpolp2,hilf);
        return(1);
    }
    return(-1);
}
/*---------------------------------------------------------------*/
PRIVATE int combinebp(N,len,prim,QQ,AA,QQ2,AA2,hilf)
word2 *N;
int len;
word4 prim;
word2 *QQ, *AA, *QQ2, *AA2, *hilf;
{
    word2 pp[2];
    word2 *xtemp, *ytemp, *ztemp;
    int alen, a2len, plen, pinvlen, qlen, q2len, rlen;

    qlen = QQ[-1];
    q2len = QQ2[-1];
    xtemp = hilf;
    ytemp = xtemp + 2*len;
    ztemp = ytemp + len;
    hilf += 5*len;

    plen = long2big(prim,pp);
    qlen = divbig(QQ,qlen,pp,plen,xtemp,&rlen,hilf);
    cpyarr(xtemp,qlen,QQ);
    q2len = divbig(QQ2,q2len,pp,plen,xtemp,&rlen,hilf);
    cpyarr(xtemp,q2len,QQ2);
    qlen = multbig(QQ,qlen,QQ2,q2len,xtemp,hilf);
    cpyarr(xtemp,qlen,QQ);
    QQ[-1] = qlen;
    alen = AA[-1];
    a2len = AA2[-1];
    alen = multbig(AA,alen,AA2,a2len,xtemp,hilf);
    alen = modbig(xtemp,alen,N,len,hilf);
    pinvlen = modinverse(pp,plen,N,len,ytemp,hilf);
    alen = multbig(xtemp,alen,ytemp,pinvlen,ztemp,hilf);
    alen = modbig(ztemp,alen,N,len,hilf);
    cpyarr(ztemp,alen,AA);
    return(AA[-1] = alen);
}
/*---------------------------------------------------------------*/
#endif  /* BIGPRIMES */
/*---------------------------------------------------------------*/
/*
** Calculates a start q-value, which is
** approx sqrt(sqrt(2*N)/srange) = sqrt(sqrt(N/2)/(srange/2))
*/
PRIVATE int startqq(N,len,srange,qq,hilf)
word2 *N,*qq,*hilf;
int len;
unsigned srange;
{
    word2 NN[ALEN],q0[HLEN];
    word2 rr;
    unsigned sroot;
    int k, len1, dum;

    cpyarr(N,len,NN);
    len1 = shrarr(NN,len,1);
    k = bigsqrt(NN,len1,q0,&dum,hilf);
    k = bigsqrt(q0,k,qq,&dum,hilf);
    sroot = intsqrt((word4)(srange/2))+1;   /* sroot < 2**16 */
    k = divarr(qq,k,sroot,&rr);
    return(k);
}
/*---------------------------------------------------------------*/
/*
** Calculates the smallest prime qq = 3 mod 4, which is >= q0
** and such that jacobi(N,qq) = 1.
** The result is stored in the buffer qq,
** return value is the length of qq.
** hilf is a buffer for auxiliary variables, whose length
** must be >= 11*max(len,q0len)
*/
PRIVATE int nextqq(N,len,q0,q0len,qq,hilf)
word2 *N, *q0, *qq, *hilf;
int len, q0len;
{
    word2 *NN, *QQ, *aux;
    unsigned bound;
    int qqlen, len0;

    qqlen = q0len;
    cpyarr(q0,q0len,qq);
    qq[0] |= 0x3;       /* make it = 3 mod 4 */

    bound = (qqlen > 2 ? 0xFFFF : 0x7FF);
  nochmal:
    while(trialdiv(qq,qqlen,3,bound))
        qqlen = incarr(qq,qqlen,4);

    len0 = (len > qqlen ? len : qqlen+1);
    NN = hilf; QQ = hilf + len0, aux = QQ + len0;
    cpyarr(N,len,NN);            
    cpyarr(qq,qqlen,QQ); 
    if(jacobi(0,NN,len,QQ,qqlen,aux) == 1) {
        if(qqlen <= 2) {
            if(prime32(big2long(qq,qqlen)))
            return(qqlen);
        }
        else if(rabtest(qq,qqlen,aux)) {
            /* not 100% certain that qq is prime */
            return(qqlen);
        }
    }
    qqlen = incarr(qq,qqlen,4);
    goto nochmal;
}
/*---------------------------------------------------------------*/
/*
** (p,plen) must be a prime = 3 mod 4.
** sptr is a pointer to a struct QPOLY which must contain in the
** fields NNlen and NN[] the number N to be factored.
** The function calculates the coefficients of a quadratic polynomial 
**  Q(x) = a*x*x + 2*b*x - c
** such that a = p*p and N = b*b + a*c
** The inverse of (p,len) mod NN is also calculated.
** Return value:
**  0 if OK
**  1 if p and N are not relatively prime
**     -1 in case of error
*/
PRIVATE int mkquadpol(p,plen,sptr,hilf)
word2 *p, *hilf;
int plen;
QPOLY *sptr;
{
    word2 *N, *xx, *zz, *aux;
    int len, qilen, p2len, blen, b2len, clen, rlen, cmp;

    N = sptr->NN;
    len = sptr->NNlen;
    xx = hilf;
    zz = xx + len;
    aux = zz + 4*plen;

    sptr->qq[-1] = plen;
    cpyarr(p,plen,sptr->qq);

    /* now calculate inverse of (p,plen) mod (N,len) */
    qilen = modinverse(p,plen,N,len,xx,aux);
    if(qilen == 0) {
        /* p divides N */
        return(1);
    }
    else {
        cpyarr(xx,qilen,sptr->qinv);
        sptr->qinv[-1] = qilen;
    }

    /* coefficient a is square of p */
    p2len = multbig(p,plen,p,plen,sptr->aa,hilf);
    sptr->aa[-1] = p2len;

    /* coefficient b is square root of N mod p*p */
    blen = p2sqrt(p,plen,N,len,zz,aux);
    sptr->bb[-1] = blen;
    cpyarr(zz,blen,sptr->bb);

    /* calculate c as (n - b*b)/a */
    b2len = multbig(sptr->bb,blen,sptr->bb,blen,zz,aux);
    cpyarr(N,len,xx);
    cmp = cmparr(xx,len,zz,b2len);
    if(cmp < 0) {   /* this case should not happen */
        return(-1);
    }
    len = subarr(xx,len,zz,b2len);
    clen = divbig(xx,len,sptr->aa,p2len,sptr->cc,&rlen,aux);
    sptr->cc[-1] = clen;
    if(rlen != 0) { /* then probably p was not prime */
        return(-1);
    }
    return(0);
}
/*---------------------------------------------------------------*/
/*
** polp describes a polynomial F(X) = a*X*X + 2*b*X - c,
** polp->xi is an argument xi for this function
** quadvalue calculates F(xi).
** The value F(xi) is stored in QQ and *signp;
** the length of QQ is stored in QQ[-1].
*/
PRIVATE int quadvalue(polp,QQ,signp)
QPOLY *polp;
word2 *QQ;
int *signp;
{
    word2 ww1[ALEN], ww2[ALEN];
    word2 *aa, *bb, *cc;
    unsigned u;
    int x, sgn, sgnx, cmp, lenax, len, alen, blen, clen;

    x = polp->xi;
    sgnx = (x < 0 ? -1 : 0);
    u = (sgnx ? -x : x);
    aa = polp->aa; alen = aa[-1];
    bb = polp->bb; blen = bb[-1];
    cc = polp->cc; clen = cc[-1];

    lenax = multlarr(aa,alen,u,ww1);

    cpyarr(bb,blen,ww2);
    blen = shlarr(ww2,blen,1);
    if(!sgnx) {
        len = addarr(ww1,lenax,ww2,blen);
        sgn = 0;
    }
    else if(cmparr(ww1,lenax,ww2,blen) >= 0) {
        len = subarr(ww1,lenax,ww2,blen);
        sgn = -1;
    }
    else {
        len = sub1arr(ww1,lenax,ww2,blen);
        sgn = 0;
    }
    len = multlarr(ww1,len,u,ww1);
    if(len == 0)
        sgn = 0;
    else
        sgn = (sgnx ? -sgn-1 : sgn);

    if(!sgn) {
        cmp = cmparr(ww1,len,cc,clen);
        if(cmp >= 0) {
            len = subarr(ww1,len,cc,clen);
            sgn = 0;
        }
        else {
            len = sub1arr(ww1,len,cc,clen);
            sgn = -1;
        }
    }
    else {
        len = addarr(ww1,len,cc,clen);
        sgn = -1;
    }
    *signp = sgn;
    cpyarr(ww1,len,QQ); 
    return(QQ[-1] = len);
}
/*---------------------------------------------------------------*/
PRIVATE int multlarr(x,n,a,y)
word2 *x, *y;
int n;
unsigned a;
{
#ifdef M_LARGE
    word4 a0, a1, u, v, carry;
    int i;

    if(a <= 0xFFFF)
        return(multarr(x,n,a,y));
#ifdef M32_64
    return(mult4arr(x,n,a,y));
#else /* !M32_64 */
    carry = 0;
    a0 = a & 0xFFFF;
    a1 = a >> 16;
    for(i=0; i<n; i++) {
        u = v = *x++;
        u *= a0;
        v *= a1;
        u += (carry & 0xFFFF);
        *y++ = u & 0xFFFF;
        v += u >> 16;
        carry >>= 16;
        carry += v;
    }
    if(carry) {
        *y++ = carry & 0xFFFF;
        n++;
        if(carry >>= 16) {
            *y = carry;
            n++;
        }
    }
    return(n);
#endif /* ?M32_64 */

#else  /* !M_LARGE; in M_SMALL we always have a < 2**16 */
    return(multarr(x,n,a,y));
#endif /* ?M_LARGE */
}
/*---------------------------------------------------------------*/
/*
** polp describes a polynomial F(X) = a*X*X + 2*b*X - c,
** where a = q*q and b*b + a*c = N. Let qinv := q**-1 mod N.
** The following equation holds: 
**      q*q*F(x) = (a*x + b)**2 mod N
** qresitem calculates AA := (a*x + b)*qinv mod N.
** the length of AA are stored in AA[-1].
*/
PRIVATE int qresitem(polp,AA)
QPOLY *polp;
word2 *AA;
{
    word2 ww0[2*ALEN], ww1[ALEN], ww2[ALEN], hilf[ALEN+1];
    word2 *aa, *bb, *qinv, *N;
    unsigned u;
    int x, sgnx, lenax, len, alen, blen, qlen, nlen;

    x = polp->xi;
    sgnx = (x < 0 ? -1 : 0);
    u = (sgnx ? -x : x);
    aa = polp->aa; alen = aa[-1];
    bb = polp->bb; blen = bb[-1];
    qinv = polp->qinv; qlen = qinv[-1];
    N = polp->NN; nlen = polp->NNlen;

    lenax = multlarr(aa,alen,u,ww1);
    cpyarr(ww1,lenax,ww2);

    if(!sgnx) {
        len = addarr(ww2,lenax,bb,blen);
    }
    else if(cmparr(ww2,lenax,bb,blen) >= 0) {
        len = subarr(ww2,lenax,bb,blen);
    }
    else {
        len = sub1arr(ww2,lenax,bb,blen);
    }
    len = multbig(ww2,len,qinv,qlen,ww0,hilf);
    len = modbig(ww0,len,N,nlen,hilf);
    cpyarr(ww0,len,AA); 
    return(AA[-1] = len);
}
/*---------------------------------------------------------------*/
/*
** Calculates x**n mod mm
** mm must be < 2**16
*/
PRIVATE unsigned modpow(x,n,mm)
unsigned x,n,mm;
{
    word4 u,z;

    if(n == 0)
        return(1);
    z = 1; u = x % mm;
    while(n > 1) {
        if(n & 1)
            z = (z*u) % mm;
        u = (u*u) % mm;
        n >>= 1;
    }
    return((z*u) % mm);
}
/*---------------------------------------------------------------*/
/*
** returns square root of a mod p, where p is a prime
** Hypothesis: jac(a,p) = 1.
** p and a should be 16-bit numbers.
*/
PRIVATE unsigned fpsqrt(p,a)
unsigned p,a;
{
    if((p & 3) == 1)
        return(fpsqrt14(p,a));
    else
        return(modpow(a,(p+1)/4,p));
}
/*---------------------------------------------------------------*/
/*
** returns square root of a mod p, where p is a prime = 1 mod 4.
** Hypothesis: jac(a,p) = 1.
** p and a should be 16-bit numbers.
*/
PRIVATE unsigned fpsqrt14(p,a)
unsigned p,a;
{
    word4 c;
    unsigned D;
    unsigned uu[2];

    a = a % p;
    a = p - a;
    for(c=1; c < p; c++) {
        D = (unsigned)((c*c + a) % p);
        if(jac(D,p) == -1)
            break;
    }
    uu[0] = (unsigned)c;
    uu[1] = 1;
    fp2pow(p,D,uu,(p+1)/2);
    return(uu[0]);
}
/*---------------------------------------------------------------*/
/*
** calculates uu**n in the field Fp(sqrt(D))
** Hypothesis: jac(D,p) = -1,
** p 16-bit prime
** (uu[0],uu[1]) is destructively replaced by result
*/
PRIVATE void fp2pow(p,D,uu,n)
unsigned p,D;
unsigned *uu;
unsigned n;
{
    word4 x,x0,y,y0,X,Y;

    if(n == 0) {
        uu[0] = 1;
        uu[1] = 0;
        return;
    }
    x = uu[0]; y = uu[1];
    X = 1; Y = 0;
    while(n > 1) {
        if(n & 1) {
            x0 = X;
            y0 = (Y*y) % p;
            /*
            ** X = (X*x + D*y0) % p;
            ** Y = (x0*y + Y*x) % p;
            */
            X *= x; X %= p;
            X += D*y0; X %= p;
            Y *= x; X %= p;
            Y += x0*y; Y %= p;
        }
        x0 = x;
        y0 = (y*y) % p;
        /*
        ** x = (x*x + D*y0) % p;
        ** y = (2*x0*y) % p;
        */
        x *= x; x %= p;
        x += D*y0; x %= p;
        y *= x0; y %= p;
        y += y; y %= p;
        n >>= 1;
    }
    x0 = X;
    y0 = (Y*y) % p;
    /*
    ** uu[0] = (X*x + D*y0) % p;
    ** uu[1] = (X*y + Y*x) % p;
    */
    X *= x; X %= p;
    X += D*y0;
    uu[0] = X % p;
    Y *= x;
    Y += x0*y; 
    uu[1] = Y % p;
}
/*---------------------------------------------------------------*/
/*
** Calculates a square root of x mod p**2
** Hypothesis: p prime = 3 mod 4, jacobi(x,p) = 1
** The result is stored in z, its length is returned
** The buffer z must have a length >= 4*plen.
**
** The square root z is calculated using the formula
**  z = x ** ((p*p - p + 2)/4) mod p**2
*/
PRIVATE int p2sqrt(p,plen,x,xlen,z,hilf)
word2 *p, *x, *z, *hilf;
int plen, xlen;
{
    word2 *xx, *ex, *p2, *aux;
    int exlen, p2len, zlen;

    xx = hilf;
    p2len = 2*plen;
    p2 = xx + (xlen > p2len ? xlen : p2len);
    ex = p2 + p2len;
    aux = ex + p2len;

    p2len = multbig(p,plen,p,plen,p2,aux);
    cpyarr(x,xlen,xx);
    xlen = modbig(xx,xlen,p2,p2len,aux);
    cpyarr(p2,p2len,ex);
    exlen = subarr(ex,p2len,p,plen);
    exlen = incarr(ex,exlen,2);
    exlen = shrarr(ex,exlen,2);
    zlen = modpower(xx,xlen,ex,exlen,p2,p2len,z,aux);
    return(zlen);
}
/*------------------------------------------------------------------*/
PRIVATE truc Fnextprime(argn)
int argn;
{
#define ANZRAB  10
    truc *argptr;
    word2 *x;
    word4 u;
    int compos;
    int i, n, sign;

    argptr = argStkPtr-argn+1;
    if(argn >= 2 && *argStkPtr == zero) {
        doreport = 0;
        argn--;
    }
    else {
        doreport = 1;
    }
    n = bigref(argptr,&x,&sign);
    if(n == aERROR) {
        error(nxtprimsym,err_int,*argptr);
        return(brkerr());
    }
    if(n >= aribufSize/9) {
        error(nxtprimsym,err_int,*argStkPtr);
        return(brkerr());
    }
    if(n <= 2) {
        u = big2long(x,n);
        n = nextprime32(u,AriBuf);
    }
    else {
        cpyarr(x,n,AriBuf);
        x = AriBuf;
        if((x[0] & 1) == 0)
            x[0]++;
        compos = 1;
                if(doreport)
                        workmess();
        while(compos) {
            while(trialdiv(x,n,3,0xFFFB))
                n = incarr(x,n,2);
            for(compos=0,i=0; i<ANZRAB; i++) {
                if(rabtest(x,n,AriScratch) == 0) {
                    compos = 1;
                    if(doreport)
                        tick('.');
                    n = incarr(x,n,2);
                    break;
                }
                                if(doreport)
                                        tick(',');
            }
        }
                if(doreport)
                        fprintstr(tstdout," probable prime:");
    }
    return(mkint(0,AriBuf,n));
}
/*------------------------------------------------------------------*/
PRIVATE int nextprime32(u,x)
word4 u;
word2 *x;
{
    int n;  

    if(u <= 2) {
        *x = 2;
        return(1);
    }
    else if(u <= 0xFFFFFFFB) {
            if((u & 1) == 0)
            u++;
        while(prime32(u) == 0)
            u += 2;
        n = long2big(u,x);
        return(n);
    }
    else {  
        x[0] = 0xF;
        x[1] = 0;
        x[2] = 1;
        return(3);
    }
}
/********************************************************************/

