/*
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARKVISUAL_CACHE_H
#define ARKVISUAL_CACHE_H

#include <Ark/ArkObject.h>
#include <Ark/ArkModel.h>
#include <Ark/ArkTexture.h>
#include <Ark/ArkParticle.h>
#include <Ark/ArkCollision.h>

namespace Ark
{

   class CacheP;

   // ========================================================================
   /**
    * A object cache is a structure holding a list of "objects" loaded in
    * memory. A object is a mesh, a model (mesh with skeleton), a texture or
    * a particle template.
    *
    * \par Object loading.
    * When an application needs a object, it calls one of the Get*() method,
    * (* corresponds to the object type), which checks in its list wether the
    * object is in memory or not. If it is, it returns a pointer to the
    * already loaded object and increments its reference counter. If it's not
    * in memory, it creates a new structure and load the object from the
    * specified file, and set the object reference counter to 1.
    *
    * \par Usage.
    * When the object is not needed any more by the application, this one
    * should call the Object::Unref method which deletes the object if the
    * refcount is 0. The object cache is used by most Ark applications,
    * client/server and tools.  All advanced 3D structures need a object
    * cache to be created : World,  HeightField, Model and so on.
    *
    * \par Renderer dependent Cache
    * Most renderers needs additionnal informations to be able to draw
    * models, textures, etc.. So classes derived from Model, Mesh, Texture, 
    * might be  created to hold renderer specific data (ie GLModel, GLMesh
    * and GLTexture, for the OpenGL renderer).
    */
   //  =======================================================================

   class Cache : public DRefCounter
   {
     private:
	 /** Load a generic object. */
	 ARK_DLL_API virtual Object *LoadObject  (ObjectType type, const String &name);
	 
     public:
	 /**
	  * Create an object of the given type. Derived class might overload
	  * this, if they want a specific class (ie GLModel) used instead of
	  * a generic one (ie Model).
	  */
	 ARK_DLL_API virtual Object *CreateObject (ObjectType type, const String &name);
	 	 
      public:
	 /// Create a new object cache.
	 ARK_DLL_API Cache (bool server = false);
	 
	 /**
	  * Destroy the object cache. It also destroys all objects contained
	  * in the cache, even if their reference counter isn't nil, so you
	  * should make sure that all objects won't be used any more before
	  * destroying a cache.
	  */
	 ARK_DLL_API virtual ~Cache ();
	
	 /// Get (load if necessary) an object of the given type.
	 ARK_DLL_API Object *Get (ObjectType type, const String &file);

	 /**
	  * Get and load if this is necessary an object of the given type.
	  * This function returns false and set *object to nil on failure.
	  */
	 template <class T>
	 inline bool Get (ObjectType type, const String &file, 
		   T **object)
	 {
	     assert (object != NULL);
	     *object = static_cast<T*>(Get(type, file));

	     return *object != NULL;
	 }

	 /**
	  * Get and load if this is necessary an object of the given type.
	  * This function returns false and set *object to nil on failure.
	  */
	 template <class T>
	 inline bool Get (ObjectType type, const String &file, Ptr<T>& ptr)
	 {
	     T* object = static_cast<T*>(Get(type, file));

	     // Refed by creation
	     ptr = Ptr<T>(object, 0);

	     return object != NULL;
	 }

	 /**
	  * Destroy the cache data (models, textures). 
	  * \param force :  If set to true, objects are destroyed even if they
          * are still used (reference count is greater than 0).
	  */
	 ARK_DLL_API void Clear (bool force = false);
	 
	 /// Print a list of objects loaded in memory to the standard output.
	 ARK_DLL_API void Dump ();

	 /// Print a report on memory usage to the standard output.
	 ARK_DLL_API void MemReport ();

	 /// Get the collision system associated with this cache...
	 ARK_DLL_API ColSystem *GetColSystem()
	 {
	    return m_ColSystem;
	 }
	 
	 /// Set the collision system associated with this cache...
	 ARK_DLL_API void SetColSystem(ColSystem *system)
	 {
	    if (m_ColSystem) delete m_ColSystem;
	    m_ColSystem = system;
	 }

	 ARK_DLL_API virtual bool IsServer() {return m_Server;}

      private:
	 ColSystem *m_ColSystem;
	 bool m_Server;

	 /// Private implementation of the cache.
	 friend class CacheP;
	 CacheP *m_Objects;
   };

   typedef Ptr<Cache> CachePtr;
   
/* namespace Ark */
}

#endif
