/* $Id: ArkLoaderPNG.cpp,v 1.17 2003/03/15 00:20:35 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Ark/ArkImage.h>
#include <Ark/ArkLoader.h>


#include <stdio.h>

#ifdef HAVE_PNG_H
#include <png.h>
#endif

namespace Ark
{

#ifdef HAVE_PNG_H
   static void
   ImagePngRead (png_structp png, png_bytep data, png_size_t size)
   {
      Stream *file = (Stream *) png->io_ptr;
      
      if (!file->read ((char *)data, size))
	 png_error (png, "Read Error");
   }
   
/* This function is called when there is a warning */
   static void
   ImagePngWarning (png_structp png_ptr, png_const_charp message)
   {
      PNG_CONST char *name = "UNKNOWN (ERROR!)";
      
      if (png_ptr != NULL && png_ptr->error_ptr != NULL)
	 name = (PNG_CONST char *) png_ptr->error_ptr;
      
      Sys()->Warning ("%s: libpng warning: %s\n", name, message);
   }

/* This is the default error handling function. */
   static void
   ImagePngError (png_structp png_ptr, png_const_charp message)
   {
      ImagePngWarning (png_ptr, message);
      /* We can return because png_error calls the default handler, which is
       * actually OK in this case. */
   }

   class LoaderPNG : public Loader
   {
      public:
	 virtual bool CanLoad (ObjectType type, Stream &file,
			       const String &name, const String &args)
	 {
	    if (type == V_IMAGE && args == "")
	    {
	       png_byte sig[3];
	       
	       file.read ((char *)sig, 3);
	       if (png_sig_cmp (sig, 0, 3) == 0)
		  return true;
	    }

	    return false;
	 }
	 
	 /// Load the file pointed to by \c name, and read a object in it.
	 /// It will update the progress every \c granularity percents.
	 virtual bool Load (Object *vis, Stream &file, const String &name,
			    const String &args,
			    Cache *cache,
			    Progress *progress, int granularity)
	 {
	    if (vis == NULL || vis->Type() != V_IMAGE)
	       return false;
	    
	    size_t rowbytes, exp_rowbytes;
	    png_infop info;

	    const char *fname = name.c_str();
	    
	    png_structp png =
	       png_create_read_struct (PNG_LIBPNG_VER_STRING,
				       (png_voidp) fname,
				       &ImagePngError,
				       &ImagePngWarning);
	    
	    if (!png)
	       return false;
	    
	    info = png_create_info_struct (png);
	    if (!info || setjmp (png->jmpbuf))
	    {
	       // If we get here, we had a problem reading the file
	       png_destroy_read_struct (&png,
					(png_infopp) NULL,
					(png_infopp) NULL);
	       return false;
	    }
	    
	    png_set_read_fn (png, &file, ImagePngRead);
	    png_read_info (png, info);
	    
	    // Get picture info
	    png_uint_32 width, height;
	    int bit_depth, color_type;
    
	    png_get_IHDR (png, info, &width, &height, &bit_depth, &color_type,
			  NULL, NULL, NULL);
	    
	    if (bit_depth > 8)
	    {
	       // tell libpng to strip 16 bit/channel files down to
	       // 8 bits/channel
	       png_set_strip_16 (png);
	    }
	    else if (bit_depth < 8)
	    {
	       // Expand pictures with less than 8bpp to 8bpp
	       png_set_packing (png);
	    }
	    
	    Image::Format format = Image::NONE_0;
	    switch (color_type)
		{
			case PNG_COLOR_TYPE_GRAY:
				format = Image::I_8;
				break;
			case PNG_COLOR_TYPE_RGB:
				format = Image::RGB_888;
				break;
			case PNG_COLOR_TYPE_RGB_ALPHA:
				format = Image::RGBA_8888;
				break;
			default: png_error (png, "Unsupported color type");
		}
	    const int bpp = Image::GetBytesPerPixel(format);
	    
	    if (!(color_type & PNG_COLOR_MASK_ALPHA))
	    {
	       // Expand paletted or RGB images with transparency to full alpha
	       // channels so the data will be available as RGBA quartets.
	       if (png_get_valid (png, info, PNG_INFO_tRNS))
		  png_set_expand (png);
	    }
	    
	    // Update structure with the above settings
	    png_read_update_info (png, info);

	    exp_rowbytes = width * bpp;
	    rowbytes = png_get_rowbytes (png, info);
	    
	    if (exp_rowbytes != rowbytes)
	       png_error (png, "Expected bytes/row is not the same as the "
			  "real one (?)");
	    
	    Image *img = (Image*) vis;
	    png_bytep * const row_pointers = new png_bytep[height];
	    
	    if (!img->SetFormat(width, height, format, NULL) || !row_pointers)
	       png_error (png, "No memory to hold image data");
	    
	    if (setjmp (png->jmpbuf)) // Set a new exception handler
	    {
	       delete [] row_pointers;
	       delete img;
	       
	       png_destroy_read_struct (&png,
					(png_infopp) NULL,
					(png_infopp) NULL);
	       return false;
	    }
	    
	    // fill row pointers
	    for (png_uint_32 row = 0; row < height; row++)
	       row_pointers [row] = ((png_bytep) img->m_Data) + row * rowbytes;
	    
	    // Read image data (FIXME: update progress)
	    png_read_image (png, row_pointers);
	    
	    // read rest of file, and get additional chunks in info_ptr
	    png_read_end (png, (png_infop)NULL);
	    
	    // clean up after the read, and free any memory allocated
	    png_destroy_read_struct (&png, &info, (png_infopp) NULL);
	    delete [] row_pointers;
	    
	    return img != 0;
	 }
	 
	 virtual String GetInformations()
	 {
	    return "Portable Network Graphic";
	 }
   };

   void ark_AddPngLoader (Loaders *loaders)
   {
      loaders->Add (new LoaderPNG());
   }
   
#else
   void ark_AddPngLoader (Loaders *loaders)
   {
   }
#endif
   
/* namespace Ark */
}
