/* $Id: ArkObject.h,v 1.10 2003/03/23 20:09:25 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_OBJECT_H
#define ARK_OBJECT_H

#include <Ark/Ark.h>
#include <Ark/ArkString.h>
#include <Ark/ArkRefCounter.h>

namespace Ark
{

   enum ObjectType
   {
      V_UNDEFINED,
      V_MATERIAL,
      V_MODEL,
      V_SEQUENCE,
      V_IMAGE,
      V_TEXTURE,
      V_SKIN,
      V_PARTICLE,
      V_FONT
   };

   //  ======================================================================
   /**
    * A object is an abstract class used which provides refcounting and naming
    * to the object cache. All objects that can be cached (see Cache)  are
    * derived from this class.
    */
   //  ======================================================================
   class ARK_DLL_API Object : public DRefCounter
   {
	 friend class Cache;
	 friend class CacheP;
	 
      protected:
	 /// Reference counter
	 int m_RefCount;

	 /// Name of this object
	 String m_Name;
	 
	 /// Type of the object
	 ObjectType m_Type;
	 
      protected:
	 /**
	  * A object can't be destroyed directly : the Unref() method should
	  * be called and only the cache can decide if it needs to be destroyed
	  * (if the memory is low, or if the object hasn't been accessed for
	  * a long time, by example).
	  */
	 virtual ~Object ();
	 
      public:
	 /**
	  * Create a new object. It simply sets its name.
	  *  \param name is the name of the object
	  */
	 Object (const Ark::String &name,
		 ObjectType type = V_UNDEFINED);
	 
	 /// Return the type of this object.
	 ObjectType Type() const
	 {return m_Type;}
	 
	 /**
	  * Get the name of this object. Most of time, it's the name of the
	  * file this object was read from.
	  */
	 const String &Name() const
	 {return m_Name;}
	 
	 /**
	  * A  object implementation can have a Load method, which tries
	  * to read from \c file, to override the \c Loaders one in Cache
	  */
	 virtual bool Load (Cache* cache, const String &name);
	 
	 /**
	  * Called by the object cache after the object has been loaded. It
	  * can be used to do operations that need data to be loaded 
	  * (ie. create a compiled vertex buffer for a Model, send the texture
	  * to the rendering engine for a Texture, and so on).
	  */
	 virtual bool PostLoad (Cache* cache);

	 /**
	  * Return a description of this object. This is used to dump the state
	  * of the object cache, by example.
	  */
	 virtual String Dump (bool long_version = false);
   };

   typedef Ptr<Object> ObjectPtr;
}

#endif

