/* $Id*
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_PARTICLE_H
#define ARK_PARTICLE_H

#include <Ark/ArkMath.h>
#include <Ark/ArkTexture.h>

#include <vector>

namespace Ark
{
   class Cache;
   class World;
   class Renderer;

   class ParticleSys;
   class ParticleTmpl;
   class Particle;

//  ==========================================================================
/** A particle template keeps informations common to a specific particle type.
 *   It means it has informations on how the particles will interact physically
 *   with the world, if they should die on collision, or only when they reach
 *   a certain age, etc..
 *   It also have informations on how to draw them, which is needed for the
 *   rendering classes.
 */
//  ==========================================================================
   class ARK_DLL_API ParticleTmpl : public Object
   {
      public:
	 enum
	 {
	    DIE_COLLISION = (1<<0),
	    DIE_AGE       = (1<<1),
	    DIE_RESPAWN   = (1<<2),
	    FCT_GRAVITY   = (1<<3),
	    FCT_RANDOM    = (1<<4),
	    FCT_WIND      = (1<<5),
	    FCT_STRENGTH  = (1<<6)
	 };

	 /// Type of the particle. See member m_pType.
	 enum PartType
	 {
	    TYPE_NULL,
	    TYPE_LINE,
	    TYPE_SPRITE
	 };

	 /** Depending on the values of m_pType, the object appearance of the
	  *   particle will change :
	  *     \arg \c TYPE_NULL    means the type hasn't been set yet
	  *     \arg \c TYPE_LINE    means that the particle is a line,
	  *                          m_LineLength should of course be set. The
	  *                          direction of the line will change
	  *                          according to the velocity vector.
	  *      \arg \c TYPE_SPRITE means the particle is a sprite : it's a
	  *                          textured quad which will always face the 
	  *                          viewer.
	  */
	 PartType m_pType;


	 /** Setting the following bits will have different effects :
	  *     \arg \c DIE_COLLISION means the particle will be destroyed if 
	  *                           it hits the world or any entity.
	  *     \arg \c DIE_AGE       it will be destroyed when it has reached
	  *                           a certain age (between m_MinLife and 
	  *                           m_MaxLife)
	  *     \arg \c DIE_RESPAWN   when it dies it is respawned (the
	  *                           particle system will never be empty)
	  *     \arg \c FCT_GRAVITY   gravity applies to this particle
	  *     \arg \c FCT_RANDOM    a random force apply to this particle.
	  *                           m_RandomScale specifies the amount o
	  *                           this strength
	  *     \arg \c FCT_WIND      the wind can move the particle.
	  *     \arg \c FCT_STRENGTH  the  misc. template strength 
	  *                           (m_Strength) has an influence on the
	  *                           particle.
	  */
	 int m_Flags;

	 /// Weight of a particle, in kgs
	 scalar m_Weigth;

	 /// The maximum time (in seconds) a particle of this type can live.
	 scalar m_MaxLife;
	 /// The minimum time (in seconds) a particle of this type can live.
	 scalar m_MinLife;

	 /// Radius of a particle, in meters.
	 scalar m_Radius;

	 /// Scale factor for the random strength.
	 scalar m_RandomScale;
	 /// Template specific strength (such as (0.0,-1.0,0.0) for rain or
	 /// snow)
	 Vector3 m_Strength;

	 /// Color of a particle.
	 Color m_Color;

	 union
	 {
	       /// If this particle is a line, this is the length of a
	       /// particle, in meters
	       scalar m_LineLength;

	       /// Sprite texture used by this particle.
	       Texture *m_Sprite;
	 };

      public:
	 /// Create a new particle template named \c name
	 ParticleTmpl (const String &name);

	 /// Destroy the given template
	 ~ParticleTmpl ();

	 /// Write this particle template definition to \c file
	 void Write (WriteStream &file);

	 /// Read this template from a stream.
	 void Read (const String&, Stream &file, Cache &cache);
   };

//  ==========================================================================
/** A particle is a specific instanciation of a particle template, which
 *   evolves in a system. A template specifies the properties of the particle,
 *   and the system the strength to which it should react. So the particle
 *   itself only contains informations about its current state, its age, its
 *   velocity and so on.
 *
 *   Some data may also be set randomly (like the life time), so they are also
 *   registered for each particle.
 */
//  ==========================================================================
   class ARK_DLL_API Particle
   {
      public:
	 /// Create a new particle.
	 Particle (ParticleSys *system);

	 /// Destroy the particle.
	 ~Particle ();

	 /// Current position.
	 Vector3 m_Pos;

	 /// Current velocity.
	 Vector3 m_Vel;

	 /**
	  * Time elapsed since the creation of the particle. If m_Age is
	  * greater than m_LifeTime, then the particle is destroyed.
	  */
	 scalar m_Age;

	 /**
	  * Maximum life time for this particle. It's set by the Init()
	  * function, and it is between the m_MaxLife and m_MinLife members of
	  * the template
	  */
	 scalar m_LifeTime;

	 /**
	  * True if this particle is dead (it will be respawned by the system
	  * on the next update if the DIE_RESPAWN flag is set in the particle
	  * template)
	  */
	 bool m_Dead;

	 /**
	  * Initialises the particle data according to the particle system
	  * and to the particle template
	  */
	 void Init (ParticleSys *system);

	 /**
	  * Apply the different system stength to this particle, after an
	  * elapsed time of \c delta.
	  */
	 void Update (ParticleSys *system, scalar delta);
   };

//  ==========================================================================
/** A particle system contains a list of particle and also the list of
 *   strength which are applied to those particles. It's needed to create a lot
 *   of special effects, such as smokes, weather (rain, snow), and everything
 *   which can be represented by a lot of moving sprites, lines or points.
 *
 *   It's mostly used on the client side, so only the renderer-specific classes
 *   will have an interest for programmers.
 */
//  ==========================================================================
   class ARK_DLL_API ParticleSys
   {
      public:
	 /**
	  * Used to specify the way each member of the particle will be
	  * initialized:
	  *   \arg \c D_NULL means it will be set to zero
	  *   \arg \c D_GIVEN means it will use the ParticleSys default
	  *   \arg \c D_RANDOM means it will be initialised randomly
	  */
	 enum Default
	 {
	    D_NULL,
	    D_GIVEN,
	    D_RANDOM
	 };

      protected:
	 friend class Particle;
	 friend class ParticleTmpl;

	 /// The world this system belongs to.
	 World *m_World;

	 /// System stopped ? (ie stop spawning particles)
	 bool m_Stopped;

	 /**
	  * Create a new position and velocity, to initialise a particle,
	  * according to the default for each component of each vector (GIVEN,
	  * NULL or RANDOM) When a vector's component default is set to
	  * D_GIVEN the corresponding component of the system default (m_Pos
	  * or m_Velocity) is used.
	  */
	 void GetNew (Vector3 *pos, Vector3 *vel);

      public:
	 /// Particle template.
	 ParticleTmpl *m_Template;

	 /// Default position.
	 Vector3 m_Pos;

	 /// Default velocity.
	 Vector3 m_Velocity;

	 /// Vectors for the gravity applied to each particles (this should be
	 /// (0.0,-9.8,0.0) on earth).
	 Vector3 m_Gravity;

	 /// Vector for another strength applied to the particles
	 Vector3 m_Wind;

	 /// Radius of the particle system
	 scalar m_Radius;

	 /// It specifies which init method should for each component of the
	 /// velocity.
	 Default m_DefVel[3];

	 /// It specifies which init method should for each component of the
	 /// positon.
	 Default m_DefPos[3];

	 /// List of particles type.
	 typedef std::vector< Particle > ParticleList;

	 /// List of particles.
	 ParticleList m_Particles;

      public:
	 /** Create a particle system.
	  *    \param world is the world it will be in.
	  *    \param pos is the position of the particle system (also the
	  *           default position for new particles).
	  *    \param initvel is the initial velocity for new particles.
	  *    \param radius is the radius of the particle system.
	  */
	 ParticleSys (World *world, Vector3 pos,
		      Vector3 initvel, scalar radius = 1.0);

	 /** Create a new particle system, using three definition strings.
	  *    \param world is the world it will be in.
	  *    \param posvel contains the default position p and velocity v and
	  *           the radius R, in the form "(pX, pY, pZ)(vX, vY, vZ)(R)"
	  *    \param defs contains the initialization methods for the three
	  *           components of the velocity and the position respectively 
	  *           "(pX, pY, pZ)(vX, vY, vZ)". X, Y and Z can be R (Random),
	  *           G (Given) or N (Null).
	  *    \param part contains infos about the particles :
	  *           "(n, tmpl_file)", where n is the number particles and
	  *           tmpl_file the file where a description of the template
	  *           can be found.
	  *    \param cache is the object cache, to load particle template from
	  */
	 ParticleSys (World *world,
		      const String &posvel,
		      const String &defs,
		      const String &part,
		      Cache *cache);

	 /// Destroy the particle system (and each of its particle, of course)
	 ~ParticleSys ();

	 /// Insert n particles which will use the given template in the
	 /// particle system
	 void Insert (ParticleTmpl *tpl, int n = 1);

	 /// Remove all particles from the particle system.
	 void Clear ();

	 /// Update the particles after a time delta of dt.
	 void Update (scalar dt);

	 /// Sets the init methods used for each component of the position.
	 void SetDefaultPos (Default X, Default Y, Default Z)
	 {m_DefPos[0] = X; m_DefPos[1] = Y; m_DefPos[2] = Z;}

	 /// Sets the init methods used for each component of the velocity.
	 void SetDefaultVel (Default X, Default Y, Default Z)
	 {m_DefVel[0] = X; m_DefVel[1] = Y; m_DefVel[2] = Z;}

	 /**
	  * Render the particle system. This is done directly by the renderer
	  * (sprites computations can be done faster), so this function is just
	  * some kind of wrapping, to ensure consistency accross the library
	  * (ie you should call stuff.Render(renderer) and no
	  * renderer.RenderStuff (stuff)).
	  */
	 void Render (Renderer &renderer);

	 /** 
	  * Stop generating particles : the Ended() function will return true
	  * when there will be no more particles in the system.
	  */
	 void Stop ()
	 {
	    m_Stopped = true;
	 }

	 /**
	  * Returns true if the system is stopped (ie no more particles are
	  * created).
	  */
	 bool Stopped () const
	 {
	    return m_Stopped;
	 }

	 /**
	  * Returns true if the system is ended (ie it contains no particle).
	  */
	 bool Ended () const
	 {
	    return m_Particles.empty();
	 }
   };

}

#include <Ark/ArkCache.h>

#endif
