/* $Id: ArkPath.cpp,v 1.31 2003/02/14 00:39:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2003 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkPath.h>
#include <Ark/ArkEntity.h>
#include <Ark/ArkWorld.h>

#include <iostream>

namespace Ark
{

#define MAX_FOLLOW_DISTANCE 3.5
#define MIN_FOLLOW_DISTANCE 2.0


   Path::Path (Entity *entity) :
      m_Entity (entity),
      m_GoalReached (false),
      m_EntTarget(NULL),
      m_BrokenPath (false),
      m_NextUpdateTime (1.e10),
      m_hasLastCol(false)
   {
   }

   Path::~Path ()
   {}
   
   void
   Path::AddPoint (const Vector3 &v, scalar speed)
   {
      m_Points.push_back(v);
   }

   void
   Path::Reset ()
   {
      m_Points.resize(0);
      AddPoint (m_Entity->m_MState.m_Position, 0.0);
   }

   bool
   Path::Update (Vector3 *pos, scalar dt)
   {
      assert (pos);

      // Initialize
      if (m_Points.empty())
	 AddPoint (m_Entity->m_MState.m_Position, 0.0);

      Vector3 dir;
      bool found = false;

      if (m_EntTarget == NULL)
      {
	 // Update path if necessary
	 if (m_BrokenPath && Timer::GetTime() > m_NextUpdateTime)
	 {
	    bool res = m_Entity->m_World->FindPath (*this);
	    
	    m_Entity->SetChanged(Entity::PATH);
	    if (res == false)
	    {
	       m_BrokenPath = true;
	       m_NextUpdateTime = Timer::GetTime() + 2.0f;
	    }
	    else m_BrokenPath = false;
	 }
	 
	 while(1)
	 {	 
	    dir = m_Points.front() - m_Entity->m_MState.m_Position;
	    dir.Y = 0.0;
	    
	    if (dir.GetMagnitude() >= 0.5 || m_Points.size() <= 1)
	       break;
	    
	    m_Points.pop_front();
	 }

	 if (m_Points.size() == 1 && dir.GetMagnitude() <= 0.5)
	    found = true;
      }
      else
      {
	 dir = (m_EntTarget->m_MState.m_Position -
		m_Entity->m_MState.m_Position);
	 dir.Y = 0.0;

#define FIXME_FOLLOW_DISTANCE 1.5
	 if (dir.Y <= FIXME_FOLLOW_DISTANCE)
	    found = true;
      }

      if (found)
      {
	 if(m_GoalReachedFlag == false)
	 {
	    m_GoalReached = true;
	    m_GoalReachedFlag = true;
	    Reset();
	    *pos = m_Entity->m_MState.m_Position;
	    return true;
	 }
      }

      dir.Normalize();
      m_GoalDir = dir;

      if (m_hasLastCol)
      {
	 m_hasLastCol = false;

	 Ark::Entity *e = m_Entity;
	 Ark::Entity *c = m_LastCol.m_Collider;

	 //std::cerr << "Lastcol " << e->m_Name << " / " << c->m_Name << std::endl;

	 Ark::Vector3 diff = c->m_MState.m_Position - e->m_MState.m_Position;
	 diff.Y = 0.0;

	 if (c->m_Path.m_EntTarget == this->m_Entity)
	    ;	 
	 else if (c->m_Flags & Entity::STATIC)
	 {
	    Ark::Matrix44 m;	       
	    m.MakeRotationY(90.f);
	    diff.Normalize(); 
	    dir = dir - 1.5 * diff ;

	    //std::cerr << e->m_Name << " " << diff << " " << e->m_Path.m_GoalDir
	    //	 << " " << dir << std::endl;
	 }
	 
	 // c and e are going to collide..
	 else if (c->m_Path.m_GoalDir * e->m_Path.m_GoalDir <= 0)
	 {
	    if (diff * e->m_Path.m_GoalDir >= 0)
	    {
	       Ark::Matrix44 m;
	       m.MakeRotationY(90.f);
	       dir = m.Transform(diff);
	    }
	 }
	 else if(diff * c->m_Path.m_GoalDir > 0)
	    dir = Ark::Vector3();

	 dir.Normalize();
      }
      
      // FIXME: Compute speed. It's a bit complicated so I leave it
      // for later.. :)
      *pos = m_Entity->m_MState.m_Position + (3.0f * dt) * dir;
	 
      return true;
   }

   void
   Path::SetGoal (const Vector3 &goal)
   {
      m_PosTarget = goal;
      m_EntTarget = NULL;
      m_BrokenPath = true;

      m_GoalReached = false;
      m_GoalReachedFlag = false;

      m_NextUpdateTime = Timer::GetTime();
   }

   void
   Path::SetGoal (Entity *goal)
   {
      if (goal == NULL) 
      {
	 Reset();
	 m_GoalReached = true;
	 m_GoalReachedFlag = true;
	 m_BrokenPath = false;
	 m_NextUpdateTime = 0.0;
	 return;
      }

      m_EntTarget = goal;
      m_BrokenPath = true;
      m_GoalReached = false;
      m_GoalReachedFlag = false;

      m_NextUpdateTime = Timer::GetTime();
   }

   void
   Path::PlanifyUpdate (scalar time)
   {
      m_BrokenPath = true;
      m_NextUpdateTime = time;
   }
   
   void
   Path::SetLastCollision(const EntityCollision &et)
   {
      m_hasLastCol = true;
      m_LastCol = et;
   }
}
