/* $Id: HFWorld.cpp,v 1.34 2003/03/20 17:23:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef WIN32
 #include <windows.h>
#endif

#include <GL/gl.h>

#include <sstream>

// For memcpy 
#include <string.h>

#include <Ark/ArkConfig.h>
#include <Ark/ArkFileSys.h>
#include <Ark/ArkLight.h>
#include <Ark/ArkRenderer.h>
#include <Ark/ArkSystem.h>
#include <Modules/HeightField/HFWorld.h>
#include <Modules/HeightField/HFQuadtree.h>
#include <Modules/HeightField/SkyDome.h>

   // =======================================================================
   
   bool g_DrawTerrain = true;
   bool g_DrawEntities = true;
   bool g_DrawSkels = false;

#define DEBUG_FOG 1
   
   // =======================================================================

   
namespace Ark
{

   // Allocate a new "size*size" height field
   HeightField::HeightField (Cache *cache, WorldUpdater *upd) : 
       World (cache,upd),
       m_OffsetX (0.0f), 
       m_OffsetZ (0.0f),
       m_SizeX (0), 
       m_SizeZ (0),
       m_Scale (0.25f),
       m_ScaleY(0.0125f),
       m_Y (0),
       m_G (0),
      // By default there's no pathfinder (it takes a lot of memory)
      // call InitPathfinder() to enable it...
       m_Pathfinder (0),
       m_Quadtree (0),
       m_SkyDome (0)
   {
   }


   // Destroy the given Height-field.
   HeightField::~HeightField ()
   {
      delete[] m_Y;
      delete[] m_G;

      DestroyPathfinder();

      delete m_Quadtree;
      delete m_SkyDome;
   }


   bool HeightField::LoadLight (int aLightNum, Light &light)
   {
      std::string type, prefix;
      std::ostringstream prefixs;
 
      prefixs << "light::Light" << aLightNum << "::";
      prefix = prefixs.str();
 
      type = m_Config.GetStr (prefix + "Type", "");
 
      if (type == std::string("ambient"))
 	 light = Light (LIGHT_AMBIENT);
      else if (type == std::string("directional"))
 	 light = Light (LIGHT_DIRECTIONAL);
      else if (type == std::string("point"))
 	 light = Light (LIGHT_POINT);
      else if (type == std::string("spot"))
 	 light = Light (LIGHT_SPOT);
      else
      {
	 // FIXME: need an explicit message if there is a wrong type...
	 return false;
      }
 
      // Retrieve light color
      light.SetColor
	 (Color (m_Config.GetScalar (prefix+"Color::R", 1.0f),
		 m_Config.GetScalar (prefix+"Color::G", 1.0f),
		 m_Config.GetScalar (prefix+"Color::B", 1.0f)));      
      
      // Retrieve light position
      light.SetPosition
	 (Vector3 (m_Config.GetScalar (prefix + "Position::X", 0.0f),
		   m_Config.GetScalar (prefix + "Position::Y", 0.0f),
		   m_Config.GetScalar (prefix + "Position::Z", 0.0f)));
 
      
      // Retrieve attenuation
      light.SetAttenuation
	  (m_Config.GetScalar (prefix + "Attenuation", 1.0f));
     
      return true;
   }

   /**
    * Some notes on the game data : in the directory pointed to by \c path,
    * there should be a configuration file called world.cfg, which contains
    * general informations about the world, and informations about the
    * heightfield.
    * The variable \c heightfield::HeightData should point to a 8bit gray
    * texture containing elevation data. There should also be a
    * heightfield::GroundData pointing to a 8bit gray image, each level of
    * gray being an index in the groundtype list. The last thing needed is
    * a list of ground types present in the map.
    */
   bool
   HeightField::Load (const String &path)
   {
      if (! m_Config.Load(path + "/world.cfg"))
	 return false;

      if (Sys()->FS()->IsFile(path + "/world-misc.cfg"))
      {
	 Sys()->Log("Loading additional file : %s\n",
		    (path+"/world-misc.cfg").c_str());
	 m_Config.Load(path + "/world-misc.cfg");
      }
       
      m_SkyDome = new SkyDome(m_Cache, m_Config);
      m_SkyDome->Build();
       
      m_OffsetX = m_Config.GetScalar ("heightfield::OffsetX", 0.0f);
      m_OffsetZ = m_Config.GetScalar ("heightfield::OffsetZ", 0.0f);
       
      m_Scale = m_Config.GetScalar ("heightfield::Scale", 1.0f);
      m_ScaleY = m_Config.GetScalar ("heightfield::ScaleY", 1.0f);
       
      /// Load elevation map.
      String hdata = m_Config.GetStr("heightfield::HeightData", String());
      
      const scalar timeOfDay = m_Config.GetScalar ("atmosphere::TimeOfDay",
						   0.11f);
      SetTimeOfDay(timeOfDay);
      
      const scalar fogRed = m_Config.GetScalar ("atmosphere::FogRed", 0.8f);
      const scalar fogGreen = m_Config.GetScalar ("atmosphere::FogGreen", 0.8f);
      const scalar fogBlue = m_Config.GetScalar ("atmosphere::FogBlue", 1.0f);
      SetFogColor(Color(fogRed, fogGreen, fogBlue, 1.0f));
      
      const scalar fogDensity  = m_Config.GetScalar ("atmosphere::FogDensity", 0.0065f);
      SetFogDensity(fogDensity);
      
      // constant size of loaded images
	  int size = -1;

	  // Load elevation map
	  {
      ImagePtr elevation;
      m_Cache->Get (V_IMAGE, hdata, elevation);
      
      if (!elevation || elevation->m_Format != Image::I_8)
	 return false;
      
      m_SizeX = elevation->m_Width;
      m_SizeZ = elevation->m_Height;
      
      // compute image size
      size = m_SizeX * m_SizeZ;
      
      // allocate memory for height field
      m_Y = new Height[size];
      
      for (int elevationIndex=0; elevationIndex<size ; ++elevationIndex)
	 m_Y[elevationIndex] = elevation->m_Data[elevationIndex];
	}
      
      /// Load ground map.
	  {
      ImagePtr groundmap;
      m_Cache->Get (V_IMAGE,
		    m_Config.GetStr("heightfield::GroundData",""),
		    groundmap);

      if (!groundmap || groundmap->m_Format != Image::I_8)
	 return false;
      
      if (groundmap->m_Width != m_SizeX ||
	  groundmap->m_Height != m_SizeZ)
	 return false;


      // Copy grounds
      m_G = new uchar[size];
      memcpy (m_G, groundmap->m_Data, size);
	  }
      
      const size_t ngrounds = m_Config.GetInt ("heightfield::NumGrounds", 0);

      for (int resetGroundIndex=0; resetGroundIndex<size; ++resetGroundIndex)
      {
	 if (ngrounds <= m_G[resetGroundIndex])
	    m_G[resetGroundIndex] = 0;
      }

      /// Load ground list.

      m_Grounds.resize (ngrounds);
      for (size_t j = 0; j < ngrounds; j++)
      {
	 std::ostringstream os;
	 os << "heightfield::Ground" << j;
	 String grd = os.str();
	 String mat = m_Config.GetStr(grd, String());

	 if (!m_Cache->Get(V_MATERIAL, mat, m_Grounds[j]))
	 {
		 Sys()->Warning("Could not load ground material: %s\n", grd.c_str());
	 }
      }


      // Light 0 is always the ambient one
      {
	  m_AmbientColor = Color(0.5f, 0.5f, 0.5f, 1.f);
	  Light light = Light (LIGHT_AMBIENT);

	  // Retrieve light color
	  light.SetColor(m_AmbientColor);

	  // Retrieve light position
	  light.SetPosition(Vector3());
	  ((World*)this)->Add (light);
      }

      /// Load lights
      for (int lightIndex=0 ; lightIndex<7; ++lightIndex)
      {
	 Light light;
	 if (LoadLight(lightIndex, light))
	    ((World*)this)->Add (light);
      }

      return true;
   }

   // ======
   // Write the hf to a file
   // ======
   bool
   HeightField::Write (const String &path, bool mipmaps)
   {
      assert (!"todo Write");
      return false;
   }

   bool HeightField::Init (int init_flags)
   {
     if (((init_flags & WORLD_HAS_RENDERING) ||
           (init_flags & WORLD_HAS_COLLISION)) &&
         m_Quadtree == 0)
     {
       m_Quadtree = new Quadtree(this);
     }


     if ((init_flags & WORLD_HAS_PATHFINDING)
         && m_Pathfinder == 0)
     {
       InitPathfinder ();
     }

     return true;
   }

// ======
// Returns the height (y-value) of a point in this heightmap. The given (x,z)
// are in world coordinates. Heights outside this heightmap are considered to
// be 0. Heights between sample points are bilinearly interpolated from
// surrounding points.
// ======
   scalar
   HeightField::GetHeight (scalar xp_, scalar zp_) const
   {
       // Goes into HF coords
       const scalar xp = xp_ - m_OffsetX;
       const scalar zp = zp_ - m_OffsetZ;

       // Gets rid of negative values (return 0)
       if ( (xp < 0.0f) || (zp < 0.0f) )
	   return 0.0f;

      const scalar ooscl = 1.0f / m_Scale;
      const scalar x = xp * ooscl;
      const scalar z = zp * ooscl;

      const unsigned int fx = static_cast< unsigned int >( x );
      const unsigned int fz = static_cast< unsigned int >( z );

      // Inside the heightfield
      if ( (fx < m_SizeX) && (fz < m_SizeZ) )
      {
	  const scalar lx = x - fx;
	  const scalar lz = z - fz;

	  // Bilinearly interpolate from surrounding points
	  const scalar y0 =  Y(fx,   fz) + ( Y(fx,   fz+1) - Y(fx,   fz) ) * lz;
	  const scalar y1 =  Y(fx+1, fz) + ( Y(fx+1, fz+1) - Y(fx+1, fz) ) * lz;
	  
	  return ( y0 + (y1 - y0)*lx ) * m_ScaleY;
      }
      
      return 0.0f;
   }

   Vector3 
   HeightField::GetRestPosition(const Vector3& position) const
   {
       Vector3 result(position);
       result.Y = GetHeight(position.X, position.Z);
       return result;
   }
   
   void
   HeightField::DeleteGround (unsigned int cur)
   {
     const unsigned int nitems = m_Grounds.size();
     const unsigned int sz = m_SizeX * m_SizeZ;
     uchar *grd = &GetGround(0,0);

     m_Grounds.erase(m_Grounds.begin()+cur);

     if (cur == nitems-1)
     {
       for (unsigned int i = 0; i < sz; i++)
       {
         if (grd[i] == cur)
           grd[i] = nitems-1;
       }
     }
     else if (cur > 0)
     {
       for (unsigned int i = 0; i < sz; i++)
       {
         if (grd[i] > cur)
           grd[i]--;
       }
     }
     else
     {
       for (unsigned int i = 0; i < sz; i++)
       {
         if (grd[i])
           grd[i]--;
       }
     }

     if (m_Quadtree)
       m_Quadtree->SetMaterials (m_Grounds);
   }

   void
   HeightField::AppendGround (const MaterialPtr& mat)
   {
      m_Grounds.push_back(mat);

      if (m_Quadtree)
        m_Quadtree->SetMaterials(m_Grounds);
   }
   
   /**
    * Calls invalidate on the quadtree
    */
   void
   HeightField::Invalidate (scalar minx, scalar minz, scalar maxx, scalar maxz)
   {
       if (m_Quadtree)
	   m_Quadtree->Invalidate(minx, minz, maxx, maxz);
   }
   // =======================================================================


   void HeightField::SetTimeOfDay(scalar f)
   {
	   const scalar t = (f<0.f) ? 0.f : (1.f<f) ? 1.f : f;
	   m_TimeOfDay = t;
   }

   void HeightField::SetFogColor(const Color& color)
   {
	   m_FogColor = color;
   }

   void HeightField::SetFogDensity(scalar density)
   {
	   m_FogDensity = density;
   }
   
   scalar
   HeightField::GetTimeOfDay() const
   {
      return m_TimeOfDay;
   }

   Color
   HeightField::GetFogColor() const
   {
      return m_FogColor;
   }

   scalar
   HeightField::GetFogDensity() const
   {
      return m_FogDensity;
   }

   void HeightField::SetFog()
   {
#if DEBUG_FOG
      glEnable(GL_FOG);

	  const GLenum fogMode = GL_EXP2;
	  const GLfloat fogColor[4] = { m_FogColor.R, m_FogColor.G, m_FogColor.B, m_FogColor.A }; 
      glFogi (GL_FOG_MODE,    fogMode);
      glFogfv(GL_FOG_COLOR,   fogColor);
      glFogf (GL_FOG_DENSITY, m_FogDensity);
#endif
   }

   void HeightField::UnsetFog()
   {
#if DEBUG_FOG
      glDisable(GL_FOG);
#endif
   }

   void
   DrawModelSkel (Skeleton* mdl, Ark::Matrix44 *matrices)
   {
      bool enab = glIsEnabled(GL_TEXTURE_2D) != GL_FALSE;
      bool enabdt = glIsEnabled(GL_DEPTH_TEST) != GL_FALSE;
      if (enab) glDisable (GL_TEXTURE_2D);
      if (enabdt) glDisable (GL_DEPTH_TEST);

      for (size_t i = 0; i < mdl->m_Bones.size(); i++)
      {
         Bone &bone = mdl->m_Bones[i];
	 
         if (bone.m_Parent >= 0)
         {
            glPointSize (3.0f);
            glColor3f (1, 0.7f, 0);

            glBegin (GL_LINES);
            glVertex3f (matrices[bone.m_Parent].M(3,0),
                        matrices[bone.m_Parent].M(3,1),
                        matrices[bone.m_Parent].M(3,2));
            glVertex3f (matrices[i].M(3,0),
			matrices[i].M(3,1),
                        matrices[i].M(3,2));
            glEnd ();

            // Draw points
            glColor3f (0, 0, 0.8f);
            glBegin (GL_POINTS);
            if (mdl->m_Bones[bone.m_Parent].m_Parent != -1)
               glVertex3f (matrices[bone.m_Parent].M(3,0),
                           matrices[bone.m_Parent].M(3,1),
                           matrices[bone.m_Parent].M(3,2));
            glVertex3f (matrices[i].M(3,0),
                        matrices[i].M(3,1),
                        matrices[i].M(3,2));
            glEnd ();
         }
         else
         {
            // Draw parent bone node
            glPointSize (5.0f);
            glColor3f (0.8f, 0, 0);

            glBegin (GL_POINTS);
            glVertex3f (matrices[i].M(3,0),
                        matrices[i].M(3,1),
                        matrices[i].M(3,2));
            glEnd ();
         }
      }

      glPointSize (1.0f);
      if (enab) glEnable(GL_TEXTURE_2D);
      if (enabdt) glEnable (GL_DEPTH_TEST);
   }

   bool
   HeightField::Render (Renderer &renderer, const Camera& camera)
   {
      if (m_Quadtree == NULL)
	 return false;

      scalar height = GetHeight (camera.m_PointOfView.X,
				 camera.m_PointOfView.Z);

      Camera copyCamera = camera;


      copyCamera.m_PointOfView.Y = std::max(camera.m_PointOfView.Y, height + 1.0f);

      renderer.SetCamera( copyCamera );

      if (m_SkyDome)
      {
	 // when fog is enabled, we must disable it for the skydome
	 m_SkyDome->Render(renderer, camera.m_PointOfView, m_TimeOfDay);

	 // gets the horizon color for fog
	 const Color fogColor = m_SkyDome->GetHorizonColor(m_TimeOfDay);
	 SetFogColor(fogColor);

	 // gets the ambient color
	 const Color ambientColor = m_SkyDome->GetAmbientColor(m_TimeOfDay);
	 m_AmbientColor = ambientColor;

	 std::vector<Light>::iterator itLight = m_Lights.begin();
	 itLight->SetColor(ambientColor);
      }

      std::vector<Light>::const_iterator itLight;
      int light = 0;
      for (itLight = m_Lights.begin() ; itLight != m_Lights.end() ; ++itLight)
      {
	 renderer.RenderLight(*itLight, light);
	 light++;
      }

      SetFog();

      // === Render the world
      if (g_DrawTerrain) 
      {
	  m_Quadtree->Render(renderer, camera, m_FogColor);
      }

      std::vector<Entity*>::const_iterator cur;

      if (g_DrawEntities)
      {	 
	  // === Render the entities' models ===
	  const Frustum &frustum = renderer.GetFrustum();

	  for (cur=m_Entities.begin() ; cur != m_Entities.end() ; ++cur)
	  {
	      if ((*cur)->m_MState.GetModel())
	      {
		  // Cull invisible entities using view frustum
		  if (frustum.GetVisibility ((*cur)->m_BBox) != OUTSIDE)
		      (*cur)->m_MState.Render(renderer);

		  if (g_DrawSkels &&(*cur)->m_MState.GetModel()->m_Skeleton)
		     DrawModelSkel((*cur)->m_MState.GetModel()->m_Skeleton,
				   (*cur)->m_MState.m_BoneMatrices);
	      }
	  }
      }

      // === Draw the particle systems  ===
      // I do this without z buffer writing so a particle won't occlude another
      // at a point where it is transparent. It also avoid having to sort
      // the particle in Z order, which can be quite time consuming.
      // Finally, it doesn't matter visually...
      
      // === Draw the "athmospheric" particle system (snow, rain, etc) ===
      // if (m_System != NULL)
      // {
      //  m_System->Update (dt);
      //  m_System->Render ();
      // }
      
      // === Draw the particle systems bound to the entity ===
      for (cur=m_Entities.begin() ; cur != m_Entities.end() ; ++cur)
      {
	  std::vector<EntityPSys*> list = (*cur)->m_Particles;
	  std::vector<EntityPSys*>::iterator i;

	  for (i=list.begin() ; i != list.end(); ++i)
	  {
	      if (*i == NULL)
		  continue;

	      (*i)->Render (renderer);
	  }

      }
   
      // === Draw the entity path (debugging) ===
      for (cur=m_Entities.begin(); cur != m_Entities.end(); ++cur)
      {
	 const std::list<Vector3> path = (*cur)->m_Path.GetPoints();

	 if (path.empty())
	    continue;

	 glColor3f (1.0, 0.0, 0.0);
	 glLineWidth (3.0);
	 glBegin (GL_LINES);
	 
	 glVertex3fv (&path.front().X);
	    
	 std::list<Vector3>::const_iterator i;
	 for (i = path.begin(); i != path.end(); ++i)
	 {
	    glVertex3fv (&i->X);
	    glVertex3fv (&i->X);
	 }
	 
	 glVertex3fv (&path.back().X);
	 
	 glEnd ();
	 glLineWidth (1.0);
      }
      glColor4f (1.0, 1.0, 1.0, 1.0);

      UnsetFog();
      return true;
   }

/* namespace Ark */
}

#define ARK_MODULE
#include <Ark/ArkFactoryReg.h>
class HFWorldFactory : public Ark::WorldFactory
{
   public:
      virtual ~HFWorldFactory(){}
      virtual Ark::World *NewWorld(Ark::Cache *cache,
				   Ark::WorldUpdater *updater)
      { return new Ark::HeightField(cache, updater); }
};

ARK_REGISTER("ark::World::HeightField", HFWorldFactory);
