/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	encoder options class implementation
 *
 *	by Tony Sideris	(01:16PM May 20, 2002)
 *================================================*/
#include "arson.h"

#include <qtextstream.h>
#include <qfile.h>
#include <qxml.h>

#include <klocale.h>

#include "encoderopts.h"
#include "process.h"
#include "xmlwriter.h"
#include "utils.h"

#define ENCODERRC	"arsonencoder.xml"

/*========================================================*/

QStringList ArsonEncoderOpts::m_warnings;

const QString ArsonEncoderOpts::encoder_names[_encoder_max] = {
	I18N_NOOP("Any"),
	I18N_NOOP("LAME"),
	I18N_NOOP("BladeEnc"),
	I18N_NOOP("oggenc"),
#ifdef FLAC
	I18N_NOOP("FLAC"),
#endif
};

/*========================================================*/

ArsonEncoderOpts::ArsonEncoderOpts (void)
	: m_bitrate(128),
	m_channels(chanStereo),
	m_vbr(-1),
	m_flags(0),
	m_manual(encoderNone)
{
	//	Nothing...
}

/*========================================================*/

void ArsonEncoderOpts::applyTo (ArsonLameProcess &proc)
{
	if (useManual(encoderLAME, proc))
		return;

	if (m_vbr >= 0)
		proc << "-V" << QString::number(vbr());
	else
		proc << "-b" << QString::number(bitrate());

	proc << "-m";

	switch (channels())
	{
	case chanStereo: proc << "s"; break;
	case chanJoint: proc << "j"; break;
	default: proc << "m" << "-a";	//	Mono, downmix
	}

	if (is(flagCRC)) proc << "-p";
	if (is(flagCopyright)) proc << "-c";
	if (is(flagCopy)) proc << "-o";
}

/*========================================================*/

void ArsonEncoderOpts::applyTo (ArsonBladeProcess &proc)
{
	if (useManual(encoderBlade, proc))
		return;
	
	proc << "-br" << QString::number(bitrate());

	switch (channels())
	{
	case chanMono: proc << "-mono"; break;
	case chanLeftMono: proc << "-leftmono"; break;
	case chanRghtMono: proc << "-rightmono"; break;
	}

	if (is(flagCRC)) proc << "-crc";
	if (is(flagCopyright)) proc << "-copyright";
	if (is(flagCopy)) proc << "-copy";
}

/*========================================================*/

void ArsonEncoderOpts::applyTo (ArsonOggencProcess &proc)
{
	if (useManual(encoderOgg, proc))
		return;
	
	proc << "-b" << QString::number(bitrate());
}

/*========================================================*/
#ifdef FLAC
void ArsonEncoderOpts::applyTo (ArsonFlacEncoderProcess &proc)
{
	useManual(encoderFLAC, proc);
}
#endif
/*========================================================*/

void ArsonEncoderOpts::applyTo (ArsonEncoderProcess &proc)
{
	Trace("WARNING: base encoder opts called\n");
}

/*========================================================*/

bool ArsonEncoderOpts::useManual (int enc, ArsonEncoderProcess &proc) const
{
	if (m_manual == encoderNone)
		return false;

	if (m_manual == enc || m_manual == encoderAny)
	{
		const QStringList sl = QStringList::split(QString(" "), m_switches);

		for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
			 it != end; ++it)
			proc << (*it);

		return true;
	}

	if (!arsonWarning(
			i18n("The manual encoder options selected are not for the current encoder, continue with the default quality setting for THIS encoder?"),
			"manual-enc"))
		throw ArsonError(QString::null);

	return true;	//	Use default, no switches
}

/*========================================================*/

void ArsonEncoderOpts::setManual (int enc, const QString &switches)
{
	m_switches = (enc == encoderNone)
		? QString::null : switches;

	m_manual = enc;
}

/*========================================================*/

void ArsonEncoderOpts::load (const QXmlAttributes &attr)
{
	bool ok = false;
	const int man = attr.value("manual").toInt(&ok);

	if (ok && man != encoderNone)
		setManual(man, attr.value("switches"));
	else
	{
		setBitrate(attr.value("bitrate").toInt());
		setChannels(attr.value("channels").toInt());
		setVbr(attr.value("vbr").toInt());
		setFlags(attr.value("flags").toInt());
	}
}

void ArsonEncoderOpts::save (const QString &name, ArsonXmlWriter &writer)
{
	ArsonXmlTag tag (writer, "preset");
	const int man = manualEncoder();

	tag.addAttribute("name", name);
	
	if (man != encoderNone)
	{
		tag.addAttribute("manual", man);
		tag.addAttribute("switches", manualSwitches());
	}
	else
	{
		tag.addAttribute("bitrate", bitrate());
		tag.addAttribute("channels", channels());
		tag.addAttribute("vbr", vbr());
		tag.addAttribute("flags", flags());
	}

	tag.doit();
}

/*========================================================*/
/*	Preset collection class impl
 *========================================================*/

class arsonEncoderParser : public ArsonUserCfg
{
public:
	arsonEncoderParser (ArsonEncoderPresets *ptr)
		: ArsonUserCfg(),
		m_ptr(ptr) { }

	virtual bool startElement (const QString &ns, const QString &local,
		const QString &name, const QXmlAttributes &attr)
	{
		if (name == "encoderopts")
			return true;

		else if (name == "preset")
		{
			ArsonEncoderOpts opt;
			const QString name = attr.value("name");

			if (name.isEmpty())
				return false;

			opt.load(attr);
			m_ptr->addPreset(name, opt);
		}

		return true;
	}

private:
	ArsonEncoderPresets *m_ptr;
};

ArsonEncoderPresets::ArsonEncoderPresets (void)
{
	arsonEncoderParser parser (this);
	parser.readConfig(ENCODERRC);
}

/*========================================================*/

void ArsonEncoderPresets::save (void)
{
	const QString path = arsonDataFile(ENCODERRC, "config", true);

	if (path != QString::null)
	{
		QFile file (QFile::encodeName(path));

		if (file.open(IO_WriteOnly | IO_Truncate))
		{
			QTextStream ts (&file);
			ENCODERMAP::Iterator it, end;
			ArsonXmlWriter writer (ts, "encoderopts");

			writer.begin();
			
			/*	Have each preset create a preset node,
			 *	and save itself in that new node.
			 */
			for (it = m_presets.begin(), end = m_presets.end(); it != end; ++it)
				it.data().save(it.key(), writer);

			writer.end();
		}
	}
}

/*========================================================*/

void ArsonEncoderPresets::addPreset (const QString &name, const ArsonEncoderOpts &opts)
{
	m_presets[name] = opts;
	Trace("Encoder count: %d\n", count());
}

/*========================================================*/

void ArsonEncoderPresets::delPreset (const QString &name)
{
	ENCODERMAP::Iterator it = m_presets.find(name);

	if (it != m_presets.end())
		m_presets.remove(it);
}

/*========================================================*/

QStringList ArsonEncoderPresets::names (void) const
{
	QStringList sl;
	ENCODERMAP::ConstIterator it, end;

	for (it = m_presets.begin(), end = m_presets.end(); it != end; ++it)
		sl.append(it.key());


	return sl;
}

/*========================================================*/

const ArsonEncoderOpts *ArsonEncoderPresets::opts (const QString &name) const
{
	ENCODERMAP::ConstIterator it = m_presets.find(name);
	return (it == m_presets.end()) ? NULL : &(it.data());
}

ArsonEncoderOpts *ArsonEncoderPresets::opts (const QString &name)
{
	ENCODERMAP::Iterator it = m_presets.find(name);
	return (it == m_presets.end()) ? NULL : &(it.data());
}

/*========================================================*/
