#include <errno.h>
#include <glib.h>
#include <stdio.h>
#include <string.h>

#include "asdlib.h"

GStaticPrivate _asd_errno_private = G_STATIC_PRIVATE_INIT;

static gchar* _text(guint16 e)
{
  switch (e)
    {
    case PROTOCOL_ASD_ERROR_SUCCESS : return "Success";
    case PROTOCOL_ASD_ERROR_VERSION_NOT_SUPPORTED : return "Version not supported";
    case PROTOCOL_ASD_ERROR_UNKNOWN_COMMAND : return "Unknown command";
    case PROTOCOL_ASD_ERROR_SIZE_MISMATCH : return "Size mismatch";
    case PROTOCOL_ASD_ERROR_ACCESS_DENIED : return "Access denied";
    case PROTOCOL_ASD_ERROR_AUTH_FAILURE : return "Authorization failure";
    case PROTOCOL_ASD_ERROR_DEVICE_NOT_FOUND : return "Device not found";
    case PROTOCOL_ASD_ERROR_SAMPLE_TYPE_INVALID : return "Invalid sample type";
    case PROTOCOL_ASD_ERROR_SAMPLE_TYPE_NOT_SUPPORTED : return "Sample type not supported";

    case PROTOCOL_ASD_LOCAL_ERROR_SYSTEM : return "(LOCAL) System error";
    case PROTOCOL_ASD_LOCAL_ERROR_BROKEN_SERVER_SPEC : return "(LOCAL) Broken server specification";
    case PROTOCOL_ASD_LOCAL_ERROR_PROTOCOL_CORRUPT : return "(LOCAL) Protocol corrupt";
    case PROTOCOL_ASD_LOCAL_ERROR_NOT_IN_CONTROL_STATE : return "(LOCAL) Connection not in control state";
    case PROTOCOL_ASD_LOCAL_ERROR_SAMPLE_TYPE_INVALID : return "(LOCAL) Invalid sample type";
    default:
      if (e >= 100)
	return "Unknown local error";
      else
	return "Unknown error";
    }
}

guint16 asd_get_errno()
{
  guint16 *e;
  e = g_static_private_get(&_asd_errno_private);
  
  return e ? *e : PROTOCOL_ASD_ERROR_SUCCESS;
}

void asd_set_errno(guint16 e)
{
  guint16 *p;
  
  if (!(p = g_static_private_get(&_asd_errno_private)))
    {
      p = g_new(guint16, 1);
      g_static_private_set(&_asd_errno_private, p, g_free);
    }

  *p = e;   
}

gchar* asd_strerror(guint16 e)
{
  if (e == PROTOCOL_ASD_LOCAL_ERROR_SYSTEM)
    return strerror(e);
  
  return _text(e);
}

gchar* asd_strerror_r(guint16 e, gchar *s, gulong l)
{
  if (e == PROTOCOL_ASD_LOCAL_ERROR_SYSTEM)
    return strerror_r(e, s, l);

  return _text(e);
}

void asd_perror(gchar* msg)
{
  gchar buf[256];
  gchar *s;

  s = asd_strerror_r(asd_get_errno(), buf, sizeof(buf));

  if (msg)
    if (*msg != 0)
      {
	g_printerr("%s: %s\n", msg, s);
	return;
      }

  g_printerr("%s\n", s);
}
