#include <glib.h>

#include <recursive-lock.h>

#include <conjunction.h>

static GPtrArray *_conjunctions = NULL;

static GStaticMutex _conjunction_mutex = G_STATIC_MUTEX_INIT;
static GStaticPrivate _conjunction_private = G_STATIC_PRIVATE_INIT;

typedef struct 
{
  gpointer from;
  gpointer to;

  ConjunctionDeathProc from_death;
  ConjunctionDeathProc to_death;

  BlockQueue *block_queue;
} Conjunction;

#define _conjunction_lock() recursive_lock(&_conjunction_mutex, &_conjunction_private)
#define _conjunction_unlock() recursive_unlock(&_conjunction_mutex, &_conjunction_private)

static void _conjunction_free(gpointer p);

static Conjunction* _conjunction_new(gpointer from, ConjunctionDeathProc from_death, gpointer to, ConjunctionDeathProc to_death, guint len, guint hold)
{
  Conjunction *c;

  g_assert(from && to);
  g_assert(c = g_new0(Conjunction, 1));

  c->from = gc_ref_inc(from);
  c->from_death = from_death;
  c->to = gc_ref_inc(to);
  c->to_death = to_death;

  c->block_queue = block_queue_new(len, hold);
  
  return gc_add(c, _conjunction_free, "conjunction");
}

static void _conjunction_free(gpointer p)
{
  Conjunction *c;
  g_assert(c = (Conjunction*) p);
  
  if (c->from_death)
    c->from_death(c->from, c->block_queue);

  if (c->to_death)
    c->to_death(c->to, c->block_queue);

  gc_ref_dec(c->from);
  gc_ref_dec(c->to);
  gc_ref_dec(c->block_queue);
}

static Conjunction* conjunction_find(gpointer from, gpointer to)
{
  Conjunction *c = NULL;
  _conjunction_lock();
  
  if (_conjunctions)
    {
      guint i;
      
      for (i = 0; i < _conjunctions->len; i++)
	{
	  Conjunction *d;

	  g_assert(d = (Conjunction*) g_ptr_array_index(_conjunctions, i));
	  if ((d->from == from) && (d->to == to))
	    {
	      c = d;
	      break;
	    }
	}
    }
  
  _conjunction_unlock();
  return c;
}

gboolean conjunction_exists(gpointer from, gpointer to)
{
  gboolean b;

  _conjunction_lock();
  b = conjunction_find(from, to) ? TRUE : FALSE;
  _conjunction_unlock();

  return b;
}

BlockQueue* conjunction_add(gpointer from, ConjunctionDeathProc from_death, gpointer to, ConjunctionDeathProc to_death, guint len, guint hold)
{
  BlockQueue* bq = NULL;

  _conjunction_lock();
  if (!conjunction_exists(from, to))
    {
      Conjunction *c;
      g_assert(c = _conjunction_new(from, from_death, to, to_death, len, hold));
      if (!_conjunctions)
	_conjunctions = g_ptr_array_new();
      g_ptr_array_add(_conjunctions, c);
      bq = gc_ref_inc(c->block_queue);
    }
  _conjunction_unlock();

  return bq;
}

static void _check_empty()
{
  if (_conjunctions)
    if (_conjunctions->len == 0)
      {
	g_ptr_array_free(_conjunctions, FALSE);
	_conjunctions = NULL;      
      }
}

gboolean conjunction_remove(gpointer from, gpointer to)
{
  gboolean r;
  Conjunction *c;

  _conjunction_lock();
  if ((r = (c = conjunction_find(from, to)) ? TRUE : FALSE))
    {
      g_ptr_array_remove_fast(_conjunctions, c);
      gc_ref_dec(c);
    }

  _check_empty();
  
  _conjunction_unlock();

  return r;
}

void conjunction_remove_item(gpointer item)
{
  _conjunction_lock();

  if (_conjunctions)
    {
      guint i;
      
      for (i = 0; i < _conjunctions->len; i++)
	{
	  Conjunction *c;
	  g_assert(c = (Conjunction*) g_ptr_array_index(_conjunctions, i));

	  if ((c->from == item) || (c->to == item))
	    {
	      g_ptr_array_remove_fast(_conjunctions, c);
	      gc_ref_dec(c);
	    }
	}
    }

  _check_empty();

  _conjunction_unlock();
}

void conjunction_reset()
{
  _conjunction_lock();

  if (_conjunctions)
    {
      guint i;

      for (i = 0; i < _conjunctions->len; i++)
	gc_ref_dec(g_ptr_array_index(_conjunctions, i));

      g_ptr_array_set_size(_conjunctions, 0);
    }

  _check_empty();
  
  _conjunction_unlock();
}

void link_source_sink(Source *source, Sink *sink, guint len, guint hold)
{
  BlockQueue *bq;

  g_assert(source && sink);

  _conjunction_lock();
  g_assert(bq = conjunction_add(source, source_output_block_queue_death_proc, 
				sink, sink_input_block_queue_death_proc, 
                                len, hold));
  source_add_output_block_queue(source, bq);
  sink_add_input_block_queue(sink, bq);
  _conjunction_unlock();

  gc_ref_dec(bq);
}

void link_sink_sink(Sink *sink1, Sink* sink2, guint len, guint hold)
{
  BlockQueue *bq;

  g_assert(sink1 && sink2);

  _conjunction_lock();
  g_assert(bq = conjunction_add(sink1, sink_monitor_block_queue_death_proc, 
				sink2, sink_input_block_queue_death_proc, 
                                len, hold));
  sink_add_monitor_block_queue(sink1, bq);
  sink_add_input_block_queue(sink2, bq);
  _conjunction_unlock();

  gc_ref_dec(bq);

}
