#include <signal.h>

#include <volume.h>
#include <asdutil.h>

#include <gc.h>
#include <namespace.h>
#include <sink.h>
#include <asd.h>
#include <sink-default.h>
#include <conjunction.h>
#include <thread-manager.h>


static void _sink_free(gpointer p);

Sink* sink_new(gchar *shortname, gchar *name, gboolean ns_reg)
{
  Sink *s;

  if (ns_reg)
    if (!namespace_register(shortname))
      return NULL;

  g_assert(s = g_new(Sink, 1));

  s->shortname = g_strndup(shortname, ASD_SHORTNAME_LENGTH);
  s->name = g_strndup(name, ASD_NAME_LENGTH);
  s->type = "ABSTRACT";

  s->sample_type = default_sample_type;
  volume_max(&s->volume);

  s->mode = SINK_DISABLED;
  s->flags = 0;

  s->thread_running = FALSE;

  s->throughput = throughput_new();

  s->input_block_queues = NULL;
  s->input_block_queues_mutex = recursive_mutex_new();

  s->monitor_block_queues = NULL;
  s->monitor_block_queues_mutex = recursive_mutex_new();

  s->direct_source = NULL;

  notify_init(&s->notify);

  latency_bank_reset(&s->latency_bank);

  s->byte_counter_mutex = recursive_mutex_new();
  byte_counter_reset(&s->byte_counter);
  origin_reset(&s->current_block_origin);
  s->byte_counter_since_open = 0;
  
  s->private_data = NULL;
  s->open         = sink_default_open_impl;
  s->close        = sink_default_close_impl;
  s->reopen       = sink_default_reopen_impl;
  s->write        = sink_default_write_impl;
  s->direct_write = NULL;
  s->write_block  = sink_default_write_block_impl;
  s->get_current_byte = NULL;
  s->free_private = sink_default_free_private_impl;
  s->save_private = sink_default_save_private_impl;

  namespace_set(s->shortname, s, NAMESPACE_SINK);

  return gc_add(s, _sink_free, shortname);
}

static void _sink_free(gpointer p)
{
  Sink *s;
  g_assert(s = (Sink*) p);

  namespace_unregister(s->shortname);

  if (s->free_private)
    s->free_private(s);

  notify_done(&s->notify);

  g_assert(s->input_block_queues == NULL);
  recursive_mutex_free(s->input_block_queues_mutex);

  g_assert(s->monitor_block_queues == NULL);
  recursive_mutex_free(s->monitor_block_queues_mutex);

  g_free(s->shortname);
  g_free(s->name);
  g_free(s->private_data);

  recursive_mutex_free(s->byte_counter_mutex);

  gc_ref_dec(s->throughput);
}

void sink_add_input_block_queue(Sink *s, BlockQueue *b)
{
  g_assert(s && b);

  recursive_mutex_lock(s->input_block_queues_mutex);
  g_assert(!g_slist_find(s->input_block_queues, b));
  block_queue_set_push_notify(b, &s->notify);
  block_queue_set_pop_wait(b, FALSE);
  block_queue_set_pop_enabled(b, s->mode == SINK_RUNNING);
  s->input_block_queues = g_slist_prepend(s->input_block_queues, gc_ref_inc(b));
  recursive_mutex_unlock(s->input_block_queues_mutex);
}

void sink_remove_input_block_queue(Sink *s, BlockQueue *b)
{
  g_assert(s && b);

  recursive_mutex_lock(s->input_block_queues_mutex);
  g_assert(g_slist_find(s->input_block_queues, b));
  s->input_block_queues = g_slist_remove(s->input_block_queues, b);
  block_queue_set_push_notify(b, NULL);
  block_queue_set_pop_enabled(b, FALSE);
  gc_ref_dec(b);
  recursive_mutex_unlock(s->input_block_queues_mutex);
}

void sink_add_monitor_block_queue(Sink *s, BlockQueue *b)
{
  g_assert(s && b);

  recursive_mutex_lock(s->monitor_block_queues_mutex);
  g_assert(!g_slist_find(s->monitor_block_queues, b));
  block_queue_set_pop_notify(b, NULL);
  block_queue_set_push_wait(b, FALSE);
  block_queue_set_push_enabled(b, s->mode == SINK_RUNNING);
  s->monitor_block_queues = g_slist_prepend(s->monitor_block_queues, gc_ref_inc(b));
  recursive_mutex_unlock(s->monitor_block_queues_mutex);
}

void sink_remove_monitor_block_queue(Sink *s, BlockQueue *b)
{
  g_assert(s && b);

  recursive_mutex_lock(s->monitor_block_queues_mutex);
  g_assert(g_slist_find(s->monitor_block_queues, b));
  s->monitor_block_queues = g_slist_remove(s->monitor_block_queues, b);
  block_queue_set_pop_notify(b, NULL);
  block_queue_set_push_enabled(b, FALSE);
  gc_ref_dec(b);
  recursive_mutex_unlock(s->monitor_block_queues_mutex);
}

void sink_input_block_queue_death_proc(gpointer p, BlockQueue*b)
{
  sink_remove_input_block_queue((Sink*) p, b);
}

void sink_monitor_block_queue_death_proc(gpointer p, BlockQueue*b)
{
  sink_remove_monitor_block_queue((Sink*) p, b);
}

Block* sink_input_read_block(Sink *s)
{
  GSList *l;
  Block *blocks[ASD_MIX_BLOCKS_MAX];
  guint c = 0, silence = FALSE;

  g_assert(s);
  
  pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->input_block_queues_mutex);
  recursive_mutex_lock(s->input_block_queues_mutex);  

  l = s->input_block_queues;
  while (l)
    {
      Block *b;
      if ((b = block_queue_pop((BlockQueue*) l->data)))
	{
	  if (b == silence_block)
	    {
	      silence = TRUE;
	      gc_ref_dec(b);
	    }
	  else
	    blocks[c++] = b;
	}
	  
      l = l->next;
    }

  pthread_cleanup_pop(1);

  if (c == 0)
    {
      if (silence)
	return gc_ref_inc(silence_block);
      else
	return NULL;
    }
  else
    return block_mix_array(blocks, c);
}

gboolean sink_monitor_write_block(Sink *s, Block *b)
{
  GSList *l;
  gboolean ok = FALSE;
  
  g_assert(s && b);

  pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->monitor_block_queues_mutex);
  recursive_mutex_lock(s->monitor_block_queues_mutex);  

  l = s->monitor_block_queues;
  while (l)
    {
      if (block_queue_push((BlockQueue*) l->data, b))
	ok = TRUE;
      l = l->next;
    }

  pthread_cleanup_pop(1);
  return ok;
}


/* void sink_empty_block_queues(Sink *s) */
/* { */
/*   GSList *l; */
/*   g_assert(s); */

/*   pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->block_queues_mutex); */
/*   recursive_mutex_lock(s->block_queues_mutex);   */

/*   l = s->block_queues; */
/*   while (l) */
/*     { */
/*       block_queue_empty((BlockQueue*) l->data); */
/*       l = l->next; */
/*     } */
  
/*   pthread_cleanup_pop(1); */
/* } */

void sink_input_block_queues_enable(Sink *s, gboolean b)
{
 GSList *l;
  g_assert(s);

  pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->input_block_queues_mutex);
  recursive_mutex_lock(s->input_block_queues_mutex);  

  l = s->input_block_queues;
  while (l)
    {
      block_queue_set_pop_enabled((BlockQueue*) l->data, b);
      l = l->next;
    }
  
  pthread_cleanup_pop(1);
}

void sink_monitor_block_queues_enable(Sink *s, gboolean b)
{
 GSList *l;
  g_assert(s);

  pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->monitor_block_queues_mutex);
  recursive_mutex_lock(s->monitor_block_queues_mutex);  

  l = s->monitor_block_queues;
  while (l)
    {
      block_queue_set_push_enabled((BlockQueue*) l->data, b);
      l = l->next;
    }
  
  pthread_cleanup_pop(1);
}


static void _sink_process_cleanup(gpointer p)
{
  Sink *s;
  g_assert(s = (Sink*) p);

  if (s->close)
    s->close(s);
}

static void _sink_process(Sink *s)
{
  gboolean quit = FALSE;
  g_assert(s);

  if (s->open)
    if (!s->open(s))
      return;

  g_assert(s->write_block);

  pthread_cleanup_push(_sink_process_cleanup, s);

  for (;;)
    {
      for (;;)
	{
	  Block *b;

	  if (!(b = sink_input_read_block(s))) break;

          b = block_volume(b, &s->volume);

	  pthread_cleanup_push(gc_ref_dec, b);

          recursive_mutex_lock(s->byte_counter_mutex);
          s->current_block_origin = b->origin;
          recursive_mutex_unlock(s->byte_counter_mutex);

	  quit = !s->write_block(s, b);

	  sink_monitor_write_block(s, b);

          if (b->dynamic)  // Only include in latency calculation if dynamic
            {
              latency_global_process(b);
              latency_bank_process(&s->latency_bank, b);
            }

          recursive_mutex_lock(s->byte_counter_mutex);
          byte_counter_process(&s->byte_counter, b);
          s->byte_counter_since_open += b->size;
          recursive_mutex_unlock(s->byte_counter_mutex);

	  pthread_cleanup_pop(1);
	  if (quit) break;
	}

      if (quit) break;

      notify_wait(&s->notify);
    }

  pthread_cleanup_pop(1);
}


static void _sink_thread_cleanup(gpointer p)
{
  Sink *s;
  g_assert(s = (Sink*) p);

  if (s->flags & SINK_AUTOCLEAN)
    conjunction_remove_item(s);

  thread_unregister(pthread_self());
  s->thread_running = FALSE;
  gc_ref_dec(s);
}

static void* _sink_thread(void *arg)
{
  sigset_t ss;
  Sink *s;

  g_assert(s = (Sink*) arg);

  s->thread_running = TRUE;
  thread_register(pthread_self());
  pthread_cleanup_push(_sink_thread_cleanup, s);

  pthread_sigmask(SIG_BLOCK, NULL, &ss);
  sigaddset(&ss, SIGINT);
  sigaddset(&ss, SIGQUIT);
  //  sigaddset(&ss, SIGHUP);
  pthread_sigmask(SIG_BLOCK, &ss, NULL);

  pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

  _sink_process(s);

  pthread_cleanup_pop(1);

  pthread_detach(pthread_self());

  return NULL;
}

void sink_start(Sink *s)
{
  g_assert(s);
  g_assert(!s->thread_running);
  gc_ref_inc(s);
  s->thread_running = TRUE;
  g_assert(!pthread_create(&s->thread, NULL, _sink_thread, (void*) s));
}

void sink_stop(Sink *s)
{  
  g_assert(s);
  if (s->thread_running)
    {
      pthread_cancel(s->thread);
      pthread_join(s->thread, NULL);
      s->thread_running = FALSE;
    }
}

void sink_kill(Sink *s)
{
  g_assert(s);
  sink_stop(s);
  conjunction_remove_item(s);
}

void sink_set_mode(Sink *s, SinkMode m)
{
  g_assert(s);
  
  s->mode = m;

  sink_input_block_queues_enable(s, s->mode == SINK_RUNNING);
  sink_monitor_block_queues_enable(s, s->mode == SINK_RUNNING);
}


gulong sink_get_current_byte(Sink*s, OriginId id)
{
  gulong p;
  g_assert(s);

  recursive_mutex_lock(s->byte_counter_mutex);

  p = byte_counter_get(&s->byte_counter, id);

  if (s->mode == SOURCE_RUNNING)
    if (origin_is_set(&s->current_block_origin, id))
      if (s->get_current_byte)
        p+= s->get_current_byte(s);
  
  recursive_mutex_unlock(s->byte_counter_mutex);

  return p;
}

gboolean sink_direct(Sink *s, gboolean enable, Source* source)
{
  g_assert(s);

  if (enable)
    g_assert(source);

  if (!(s->flags & SOURCE_DIRECT_SUPPORTED))
    return FALSE;

  if ((enable && s->mode != SINK_RUNNING) || (!enable && s->mode != SINK_DIRECT))
    return FALSE;

  s->direct_source = source;

  sink_set_mode(s, enable ? SINK_DIRECT : SINK_RUNNING);
  
  return TRUE;
}

void sink_set_volume(Sink *s, Volume *v)
{
  g_assert(s && v);
  s->volume = *v;
}
