#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>

#include <sink-default.h>
#include <asd.h>
#include <dsp.h>
#include <sample-convert.h>
#include <sample-type.h>

#include "sink-dsp.h"

static gboolean _sink_dsp_open(Sink *s)
{
  SinkDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SinkDefaultPrivate *) s->private_data);
  g_assert(p->filename && !p->is_open);
  g_assert(s->mode == SINK_DISABLED);

  if ((p->is_open = (p->fd = dsp_open(p->filename, FALSE, &s->sample_type)) >= 0))
    {
      g_message("Sink '%s' successfully opened.", s->shortname);

      if (!sample_type_equal(&s->sample_type, &default_sample_type))
	{
	  p->buffer_size = sample_convert_length(&default_sample_type, &s->sample_type, default_block_size, FALSE);
	  p->buffer = g_new(guint8, p->buffer_size);
	}
    }
  else
    g_message("Sink '%s' open failed.", s->shortname);

  sink_set_mode(s, p->is_open ? SINK_RUNNING : SINK_DISABLED);
  s->byte_counter_since_open = 0;

  return p->is_open;
}

static void _sink_dsp_close(Sink *s)
{
  SinkDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SinkDefaultPrivate*) s->private_data);
  
  if (p->is_open)
    {
      dsp_close(p->filename, FALSE);
      p->is_open = FALSE;  
      g_message("Sink '%s' closed.", s->shortname);
    }

  sink_set_mode(s, SINK_DISABLED);
  s->byte_counter_since_open = 0;
}

static gulong _sink_dsp_get_current_byte(Sink *s)
{
  count_info ci;
  SinkDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SinkDefaultPrivate*) s->private_data);
  
  if (!p->is_open)
    return 0;

  if (ioctl(p->fd, SNDCTL_DSP_GETOPTR, &ci) == -1)
    return 0;

  return sample_convert_length(&s->sample_type, &default_sample_type, ci.bytes, FALSE)-s->byte_counter_since_open;
}

Sink* sink_dsp_new(gchar *shortname, gchar *fname)
{
  Sink *s;
  char name[ASD_NAME_LENGTH];

  g_snprintf(name, sizeof(name), "DSP %s", fname);
  g_assert(s = sink_new(shortname, name, TRUE));

  g_assert(s->private_data = sink_default_alloc_private(fname));
  
  s->type = "DSP";
  s->open = _sink_dsp_open;
  s->close = _sink_dsp_close;
  s->get_current_byte = _sink_dsp_get_current_byte;
  
  return s;
}
