// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/ACIntroAnalyzer.h"
#include "Puma/CSemDatabase.h"
#include "Puma/CCNameLookup.h"
#include "Puma/CScopeInfo.h"
#include "Puma/CTokens.h"
#include <string.h>

namespace Puma {

int ACIntroAnalyzer::ttype(int pos) {
  return _intro->Entry (pos)->token ()->type ();
}

bool ACIntroAnalyzer::is_in (int tok, int *set) {
  while (*set) {
    if (*set == tok)
      return true;
    set++;
  }
  return false;
}

bool ACIntroAnalyzer::block (int &outer_pos, int open, int close) {
  if (ttype (outer_pos) != open)
    return false;
    
  int pos = outer_pos;
  pos++;
  int level = 1;
  while (level > 0 && pos < _intro->Entries ()) {
    int tok = ttype (pos);
    if (tok == open)
      level++;
    else if (tok == close)
      level--;
    pos++;
  }

  if (level != 0)
    return false;
    
  outer_pos = pos;
  return true;
}

bool ACIntroAnalyzer::skip (int &outer_pos, int *term) {
  // save the current position
  int pos = outer_pos;
  int entries = _intro->Entries ();
  int tok = ttype (pos);
  while (pos < entries && !is_in (tok, term)) {    
    if (tok == TOK_OPEN_ROUND) // skip "(...)"
      block (pos, TOK_OPEN_ROUND, TOK_CLOSE_ROUND);
    else if (tok == TOK_OPEN_SQUARE) // skip "[...]"
      block (pos, TOK_OPEN_SQUARE, TOK_CLOSE_SQUARE);
    else if (tok == TOK_OPEN_CURLY) // skip "{...}"
      block (pos, TOK_OPEN_CURLY, TOK_CLOSE_CURLY);
    else // skip one token
      pos++;
    tok = ttype (pos);
  }
  
  if (pos == entries)
    return false;
  outer_pos = pos;
  return true;
}


bool ACIntroAnalyzer::constructor_args (int &outer_pos) {
  int pos = outer_pos;
  bool result = false;
  if (block (pos, TOK_OPEN_ROUND, TOK_CLOSE_ROUND) &&
      pos < _intro->Entries ()) {
    int tok = ttype (pos);
    if (tok == TOK_SEMI_COLON || tok == TOK_OPEN_CURLY)
      result = true;
    else if (tok == TOK_COLON) {
      int term[] = { TOK_OPEN_CURLY, 0 };
      if (skip (pos, term))
        result = true;
    }
  }
  if (result)
    outer_pos = pos;
  return result;
}


bool ACIntroAnalyzer::nested_name (int &outer_pos, bool &is_declarator,
  bool is_decl_name) {

  // save the current position
  int pos = outer_pos;

  // by default this name is not necessarily part of the declarator
  is_declarator = false;
  
  // scope is 0 at the beginning
  CScopeInfo *last_scope = 0;
  
  // root qualified name ("::" is optional)?
  bool root = false;
  CScopeInfo *lookup_scope = _current_scope;
  CScopeInfo *relevant_scope = 0;
  if (ttype (pos) == TOK_COLON_COLON) {
    pos++;
    root = true;
    while (!lookup_scope->GlobalScope ())
      lookup_scope = lookup_scope->Parent ();
    last_scope = lookup_scope;
  }

  int names = 0;
  int relevant_name_index = -1;
  bool is_destructor = false;
  bool done;
  do {
    done = true;
    int tok = ttype (pos);
    if (tok == TOK_TILDE) {
      pos++;
      tok = ttype (pos);
      if (tok != TOK_ID)
        return false;
      names++;
      pos++;
      tok = ttype (pos);
      if (tok == TOK_COLON_COLON)
        return false;
      is_destructor = true;
    }
    else if (tok == TOK_ID) {
      ErrorStream err;
      CCNameLookup nl (err);
      nl.lookup (_intro->Entry (pos)->token ()->dtext (),
        lookup_scope->Structure (), !last_scope);
      CObjectInfo *obj = nl.Objects () ? nl.Object (0) : (CObjectInfo*)0;
      if (obj && obj->ScopeInfo ()) {
        if (!relevant_scope && ((_is_slice && _db->SliceInfo (obj)) ||
            (!_is_slice && is_decl_name && obj->ClassInfo () &&
            _db->AspectInfo (obj->ClassInfo ())))) {
          relevant_scope = obj->ScopeInfo ();
          relevant_name_index = pos;
        }
      }
      names++;
      pos++;
      tok = ttype (pos);
      if (tok == TOK_COLON_COLON) {
        if (obj && obj->ScopeInfo ()) {
          lookup_scope = obj->ScopeInfo ();
          last_scope = lookup_scope;
        }
        done = false;
        pos++;
      }
    }
    else if (tok == TOK_OPERATOR) {
      int term[] = { TOK_COMMA, TOK_ASSIGN, TOK_OPEN_CURLY, TOK_SEMI_COLON,
                     TOK_CLOSE_ROUND, 0 };
      if (!skip (pos, term))
        return false;
      is_declarator = true;
      outer_pos = pos;
      return true;
    }
  } while (!done && pos < _intro->Entries ());

  if (relevant_name_index >= 0) {  
    _intro->AddNameIndex (outer_pos, relevant_name_index);
  }
  
  // a special name?
  if (names >= 2) {
    // qualified constructor name?
    const DString &last_name = _intro->Entry (pos - 1)->token ()->dtext ();
    if (last_name == _intro->Entry (pos - 3)->token ()->dtext ()) {
      is_declarator = true;
    }
    // qualified destructor name?
    else if (is_destructor &&
             last_name == _intro->Entry (pos - 4)->token ()->dtext ()) {
      is_declarator = true;
    }
    if (is_declarator && is_decl_name &&
      relevant_scope && last_name == relevant_scope->Name ()) {
        // store the detected name in the CT_Intro object
        _intro->AddNameIndex (pos - 1);
    }
  }
  
  if (is_destructor)
    is_declarator = true;
    
  if (names && !(!is_decl_name && is_declarator)) {
    if (relevant_scope) {
      if (_intro->Scope () && relevant_scope != _intro->Scope ())
        return false;
      _intro->Scope (relevant_scope->Structure ());
    }
    
    outer_pos = pos;
  }
  return (names > 0);
}


bool ACIntroAnalyzer::ptr_operator (int &outer_pos) {
  // save the current position
  int pos = outer_pos;

  int tok = ttype (pos);
  if (tok == TOK_AND) {
    pos++;
    outer_pos = pos;
    return true;
  }
  
  if (tok == TOK_COLON_COLON) {
    pos++;
    tok = ttype (pos);
    if (tok != TOK_ID)
      return false;
  }
    
  while (tok == TOK_ID) {
    pos++;
    tok = ttype (pos);
    if (tok != TOK_COLON_COLON)
      return false;
    pos++;
    tok = ttype (pos);
  }
  
  if (tok != TOK_MUL)
    return false;
    
  pos++;
  tok = ttype (pos);  
  
  while (tok == TOK_CONST || tok == TOK_VOLATILE) {
    pos++;
    tok = ttype (pos);
  }

  outer_pos = pos;
  return true;
}

bool ACIntroAnalyzer::declarator (int &outer_pos) {
  int term[] = { TOK_COMMA, TOK_ASSIGN, TOK_OPEN_CURLY, TOK_SEMI_COLON,
                 TOK_CLOSE_ROUND, 0 };
  // save the current position
  int pos = outer_pos;

  // first skip (optional) pointer operators
  while (ptr_operator (pos));
    
  // check and parse a nested declarator
  if (ttype (pos) == TOK_OPEN_ROUND) {
    pos++;
    if (!declarator (pos))
      return false;
    if (ttype (pos) != TOK_CLOSE_ROUND)
      return false;
    pos++;
  }
  else {
    // now parse the name
    bool special;
    if (!nested_name (pos, special, true))
      return false;
  }
  if (!skip (pos, term))
    return false;

  // move the current position forward
  outer_pos = pos;
  return true;
}

bool ACIntroAnalyzer::init_declarator (int &outer_pos) {
  // save the current position
  int pos = outer_pos;

  if (!declarator (pos))
    return false;

  if (ttype (pos) == TOK_ASSIGN) {
    int term[] = { TOK_COMMA, TOK_SEMI_COLON, 0 };
    pos++;
    if (!skip (pos, term))
      return false;
  }
  outer_pos = pos;
  return true;
}

const char *ACIntroAnalyzer::error_msg () const {
  switch (_state) {
    case IA_OK:              return "no error in introduction";
    case IA_UNDEFINED:       return "introduction not analyzed";
    case IA_INVALID:         return "invalid introduction";
    case IA_INV_SCOPE:       return "introduction definition not in aspect scope";
    case IA_INV_SLICE:       return "invalid slice declaration";
    case IA_INV_DESTR:       return "invalid destructor name";
    case IA_SLICE_IN_SLICE:  return "slice in slice definition";
    case IA_ASPECT_IN_SLICE: return "aspect in slice definition";
    case IA_ADVICE_IN_SLICE: return "advice in slice definition";
    case IA_PCT_IN_SLICE:    return "pointcut in slice definition";
  }
  return 0;
}


void ACIntroAnalyzer::analyze_intro_member (CT_Intro *intro) {
  _intro = intro;
  
  int pos;
  _state = IA_OK;
  
  // check for constructor
  pos = 0;
  if (ttype (pos) == TOK_EXPLICIT)
    pos++;
  if (ttype (pos) == TOK_ID &&
    _intro->Entry (pos)->token ()->dtext () == _current_scope->Name ()) {
     int name_pos = pos;
     pos++;
     if (constructor_args (pos)) {
       _intro->AddNameIndex (name_pos);
     }
  }

  // check for destructor
  pos = 0;
  if (ttype (pos) == TOK_VIRTUAL)
    pos++;
  if (ttype (pos) == TOK_TILDE) {
    pos++;
    if (ttype (pos) == TOK_ID &&
      _intro->Entry (pos)->token ()->dtext () == _current_scope->Name ()) {
      int name_pos = pos;
      pos++;
      if (ttype (pos) == TOK_OPEN_ROUND && ttype (pos + 1) == TOK_CLOSE_ROUND) {
        _intro->AddNameIndex (name_pos);
      }
    }
    else {
      _state = IA_INV_DESTR;
    }
  }
  
  // check for errors
  for (pos = 0; pos < _intro->Entries () && _state == IA_OK; pos++) {
    int tok = ttype (pos);
    if (tok == TOK_SLICE)
      _state = IA_SLICE_IN_SLICE;
    else if (tok == TOK_ASPECT)
      _state = IA_ASPECT_IN_SLICE;
    else if (tok == TOK_ADVICE)
      _state = IA_ADVICE_IN_SLICE;
    else if (tok == TOK_POINTCUT)
      _state = IA_PCT_IN_SLICE;
  }
}

void ACIntroAnalyzer::analyze_intro (CT_Intro *intro) {

  // should become attributes
  _intro = intro;
  _is_slice = (intro->token ()->type () == TOK_SLICE);

  _state = IA_INVALID;
  _intro->Scope (0);

  // skip the declaration specifier sequence
  int pos = 0;
  bool have_type = false;
  while (pos < intro->Entries ()) {

    int saved_pos = pos;
    bool special;
    if (nested_name (pos, special)) {
      // if we already have a type, this name belongs to the declarator
      if (have_type || special) {
        pos = saved_pos;
        intro->RollbackNameIndex (pos);
        break;
      }
      have_type = true;
      continue;
    }
    
    CTree *curr = intro->Entry (pos);
    assert (curr);
    Token *token = curr->token ();
    int tok = token->type ();
    
    if (tok == TOK_OPEN_CURLY) { // a struct
      if (!block (pos, TOK_OPEN_CURLY, TOK_CLOSE_CURLY))
        return;
      have_type = true;
      continue;
    }
    
    if (tok == TOK_BOOL ||
        tok == TOK_CHAR ||
        tok == TOK_INT ||
        tok == TOK_VOID ||
        tok == TOK_FLOAT ||
        tok == TOK_DOUBLE ||
        tok == TOK_WCHAR_T ||
        tok == TOK_SIGNED ||
        tok == TOK_UNSIGNED ||
        tok == TOK_LONG ||
        tok == TOK_SHORT) {
      have_type = true;
    }
    else if (tok == TOK_TYPEOF) {
      pos++;
      if (ttype (pos) != TOK_OPEN_ROUND)
        return;
      if (!block (pos, TOK_OPEN_ROUND, TOK_CLOSE_ROUND))
        return;
      have_type = true;
      continue;
    }
    else if (tok == TOK_OPEN_ROUND ||
             tok == TOK_SEMI_COLON ||
             tok == TOK_OPERATOR ||
             tok == TOK_TILDE ||
             tok == TOK_AND ||
             tok == TOK_MUL) {
      break; // begin of the declarator found
    }
    pos++; // default: just skip the token
  }
  
  if (init_declarator (pos)) {
    int tok = ttype (pos);
    while (tok == TOK_COMMA) {
      pos++;
      if (!init_declarator (pos))
        return;
      tok = ttype (pos);
    }     
  }
  
  // check if the end of the declaration is reached
  int tok = ttype (pos);
  if (tok != TOK_SEMI_COLON && tok != TOK_OPEN_CURLY) {
    return;
  }

  //if we were able to determine the scope, the result is valid
//  cout << "slice: " << _intro->token ()->location () << endl;
  if (_intro->Scope ()) {
//    cout << "  ";
//    for (int e = 0; e < _intro->Entries (); e++)
//      cout << _intro->Entry (e)->token ()->text () << " ";
//    cout << endl;
//    cout << "  => indices: ";
//    for (int i = 0; i < _intro->NameIndices (); i++) {
//      cout << "(" << _intro->NameIndex (i) << "," << _intro->NameToIndex (i);
//      if (_intro->NameQual (i))
//        cout << " qual";
//      cout << ") ";
//    }
//    cout << endl;
//    cout << "  => scope: " << _intro->Scope ()->QualName () << endl;
    _state = IA_OK;
  }
  else
    _state = IA_INVALID;
}


} // namespace Puma
