// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//
// This program is free software;  you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
// MA  02111-1307  USA

#include "CodeWeaver.h"
#include "AspectInfo.h"
#include "AdviceInfo.h"
#include "ACModel/Utils.h"
#include "ACUnit.h"
#include "Naming.h"
#include "IntroductionUnit.h"
#include "NamespaceAC.h"
#include "ResultBuffer.h"
#include "ModelBuilder.h"

#include "Puma/ErrorSink.h"
#include "Puma/CProtection.h"
#include "Puma/CTemplateInfo.h"
#include "Puma/CClassInfo.h"
#include "Puma/CUnionInfo.h"
#include "Puma/CAttributeInfo.h"
#include "Puma/CFileInfo.h"
#include "Puma/CArgumentInfo.h"
#include "Puma/CFunctionInfo.h"
#include "Puma/CProject.h"
#include "Puma/FileUnit.h"
#include "Puma/CSemDatabase.h"

void CodeWeaver::setup_tjp (ThisJoinPoint &tjp, CFunctionInfo* func) {
  assert (func->Tree ()->NodeName () == CT_FctDef::NodeId ());

  CT_CmpdStmt *body = ((CT_FctDef*)func->Tree ())->Body ();
  for (int i = 0; i < body->Entries (); i++)
    setup_tjp (tjp, body->Entry (i));
}

void CodeWeaver::setup_tjp (ThisJoinPoint &tjp, CTree *node) {

  const char *nodename = node->NodeName ();

  if (nodename == CT_QualName::NodeId () ||
      nodename == CT_RootQualName::NodeId()) {
    CT_QualName *qual_name = (CT_QualName*)node;
    CT_SimpleName *first_name = (CT_SimpleName*)qual_name->Entry (0);
    if (first_name->NodeName () == CT_SimpleName::NodeId () &&
        tjp.check_type (first_name->Text ())) {
      if (qual_name->Entries () == 2) {
        const char *text = qual_name->Text ();
        tjp.check_field (text);
      }
    }
  }
  else if (nodename == CT_SimpleName::NodeId ()) {
    CT_SimpleName *simple_name = (CT_SimpleName*)node;
    if (!tjp.check_obj (simple_name->Text ()))
      tjp.check_type (simple_name->Text ());
  }
  else if (nodename == CT_MembPtrExpr::NodeId () &&
      node->Son (0)->NodeName() == CT_SimpleName::NodeId () &&
      tjp.check_obj (node->Son (0)->token ()->text ())) {
    if (node->Son (2)->IsSimpleName ())
      tjp.check_field (node->Son (2)->IsSimpleName ()->Text (), true);
  }

  for (int s = 0; s < node->Sons (); s++)
    setup_tjp (tjp, node->Son (s));
}


void CodeWeaver::type_check (const ACM_Class *cls, const string &name, bool result) {

  // TODO: type checks in unions might be necessary for call advice - not handled
  // TODO: aggregate initialization might be broken by adding virtual functions
  const TI_Class *ti = TI_Class::of(*cls);

  // create the code
  ostringstream code;
  code << endl;
  if (ti->is_class ())
    code << "public:" << endl;
  code << "  virtual bool " << name << " () const { return "
       << (result ? "true" : "false") << "; }"	<< endl;
  if (ti->is_class ())
    code << "private:" << endl;

  // paste the function after "{" of the class/struct definition
  const WeavePos &pos = weave_pos (ti->body_start_pos(), WeavePos::WP_AFTER);
  insert (pos, code.str (), true);
}

void CodeWeaver::add_aspect_include (ACM_Any *jpl, AspectInfo &aspect_info,
  AspectRef::Kind kind) {

  if (jpl && is_pseudo(*jpl))
    return;

  // find the unit (file, macro, or intro) in which the joinpoint is located
  Unit *iu = TransformInfo::unit (*jpl);

  // if the 'insert unit' is an 'introduction unit', find the intro target unit
  IntroductionUnit *intro_unit = 0;
  while ((intro_unit = IntroductionUnit::cast (iu)) != 0)
    iu = intro_unit->target_unit ();

  _aspect_includes.insert (iu, &aspect_info, kind);
}

void CodeWeaver::aspect_includes (CProject &project) {

  // weave the aspect includes in all registered unit
  for (AspectIncludes::const_iterator iter = _aspect_includes.begin ();
    iter != _aspect_includes.end (); ++iter) {

    // determine the insertion position for the current unit
    Unit *unit = _aspect_includes.unit (iter);

    // generate the includes for the current unit
    string includes = _aspect_includes.generate (iter, problems ());

    // paste the includes (and optional guard head) at the beginning of the unit
    insert (header_pos (unit), includes);
  }
}


void CodeWeaver::insert_namespace_ac () {
  insert (header_pos (), NamespaceAC::def (_problems._size_type));
}

void CodeWeaver::insert_invocation_functions (ACM_Aspect *jpl_aspect,
    const string &defs) {

  const TI_Aspect *ti = TI_Aspect::of(*jpl_aspect);
  insert (weave_pos (ti->objdecl_end_pos(), WeavePos::WP_AFTER), defs);
}


void CodeWeaver::open_namespace (ostream &out, CObjectInfo *obj) {
  CClassInfo *cls = obj->ClassInfo();
  if (cls && cls->TemplateInfo ())
    obj = (CObjectInfo*)cls->TemplateInfo ();

  Array<CObjectInfo*> namespaces;
  while (!obj->Scope ()->GlobalScope () && obj->Scope ()->NamespaceInfo ()) {
    obj = obj->Scope ();
    namespaces.append (obj);
  }

  for (int i = namespaces.length () - 1; i >= 0; i--) {
    if (namespaces[i]->isAnonymous ())
      out << "namespace {" << endl;
    else
      out << "namespace " << namespaces[i]->Name () << " {" << endl;
  }
}


void CodeWeaver::close_namespace (ostream &out, CObjectInfo *obj) {
  CClassInfo *cls = obj->ClassInfo();
  if (cls && cls->TemplateInfo ())
    obj = (CObjectInfo*)cls->TemplateInfo ();

  ostringstream ns_close;
  while (!obj->Scope ()->GlobalScope () && obj->Scope ()->NamespaceInfo ()) {
    obj = obj->Scope ();
    out << "} // closed " << (obj->isAnonymous () ? "" : obj->QualName ()) << endl;
  }
}


// generate the code, which proceeds an intercepted flow of control, e.g.
// continues a call or execution
void CodeWeaver::make_proceed_code (ostream &out, ACM_Any *badly_typed_loc,
                                    bool action, vector<string> *arg_names) {
  ACM_Code *loc = (ACM_Code*)badly_typed_loc;
  TI_Code *ti = (TI_Code*)TransformInfo::of (*loc);

  // set flags that describe the kind of join point
  bool is_exec = false, is_call = false, is_cons = false, is_dest = false;
  if (loc->type_val () == JPT_Execution)
    is_exec = true;
  else if (loc->type_val () == JPT_Call)
    is_call = true;
  else if (loc->type_val () == JPT_Construction)
    is_cons = true;
  else if (loc->type_val () == JPT_Destruction)
    is_dest = true;
  else
    return; // should not happen!

  // find out which function has to be called by proceed
  ACM_Function *func = 0;
  if (is_call)
    func = ((ACM_Call*)loc)->get_target();
  else if (is_exec || is_cons || is_dest)
    func = (ACM_Function*)loc->get_parent ();

  // find out if the function is a member function of class
  ACM_Class *dst_cls = 0;
  JoinPointType parent_type = ((ACM_Name*)func->get_parent())->type_val();
  if (parent_type & (JPT_Class|JPT_Aspect))
    dst_cls = (ACM_Class*)func->get_parent();

  // analyze the target object of this call
  bool target_is_ptr = false; // initialization to avoid warning
  CT_Expression *target_expr = 0;
  if (is_call)
    target_expr = ((TI_MethodCall*)ti)->target_expr (target_is_ptr);

  CFunctionInfo *dstfunc = ((TI_Function*)func->transform_info())->func_info ();
  bool obj_needed = TransformInfo::needs_this (dstfunc);
  bool obj_used = obj_needed || (target_expr && !action);

  // nothing is done to proceed implicitly defined special member functions
  if ((is_cons || is_dest) && func->get_builtin())
    return;

  // create a local class that enters/leaves the cflow triggers
  const CFlowList &cflows = TI_Code::of(*loc)->cflows();
  if (cflows.size () > 0) {
    out << "{" << endl;
    cflows.gen_trigger_obj (out);
  }

  // generate the expression that calculates the result
  stringstream res;
  int args = 0;

  bool bypass = (is_call && ((TI_MethodCall*)ti)->needs_rights());
  CClassInfo *dstcls = cscope (dstfunc);
  if (bypass) {
    // check privileged calls to external functions
    if (!in_project(*func)) {
      // calls to "private" externals are not supported, thus the
      // target function is "protected"
      // => we have to use the target object's class bypass
      if (((ACM_Call*)loc)->has_target_class())
        dst_cls = ((ACM_Call*)loc)->get_target_class();
      else
        dst_cls = (ACM_Class*)lexical_scope(*loc);
    }

    res << "::" << signature(*dst_cls) << "::"
        << Naming::bypass_caller_class(dst_cls) << "<";
    Naming::bypass_id_class (res, (ACM_Call*)loc);
    res << ">::call(";
    if (obj_needed) {
      res << "(::" << signature(*dst_cls) << "*)";
      args++;
      if (action)
        res << "__TJP::target ()";
      else {
        res << "((";
        if (dstfunc->TypeInfo()->isConst())
          res << "const ";
        stringstream scope;
        dstcls->TypeInfo()->TypeText(scope, "", true, true);
        res << cleanup_name (scope.str ()) << "*)&dst)";
      }
    }
  }
  // if an object is needed for the call, issue 'obj->'
  else if (obj_used) {
    if (is_call) {
      if (action)
        res << "__TJP::target ()";
      else {
        res << "((";
        if (dstfunc->TypeInfo()->isConst())
          res << "const ";
        stringstream scope;
        dstcls->TypeInfo()->TypeText(scope, "", true, true);
        res << cleanup_name (scope.str ()) << "*)&dst)";
      }
      res << "->";
    }
    else if (is_exec || is_cons || is_dest) {
      if (action)
        res << "__TJP::that ()->";
      else
        res << "this->";
    }
  }

  // generate the function name
  if (is_exec || (!dstfunc->isBuiltin () && (is_cons || is_dest))) {
    if (!obj_used && nscope (func)) {
      string scope_name = string ("::") + signature (*nscope (func));
      res << cleanup_name(scope_name) << "::";
    }
    // use __old_<name> for execution join points
    Naming::exec_inner (res, loc);
  } else if (is_call && !bypass) {
    // destination function name
    if (((TI_MethodCall*)TransformInfo::of (*loc))->is_qualified () ||
        dstfunc->isStaticMethod () || !dstfunc->isMethod ()) {
      string name = cleanup_name (dstfunc->QualName (true, true));
      res << name;
    }
    else
      res << dstfunc->Name ();

    // add the template arguments if it is a function template instance
    bool call_uses_template_params =
        ((TI_MethodCall*)TransformInfo::of (*loc))->has_explicit_template_params();
    CTemplateInstance *instance = dstfunc->TemplateInstance ();
    if (instance && call_uses_template_params) {
      res << "< ";
      for (unsigned a = 0; a < instance->DeducedArgs (); a++) {
        if (a > 0) res << ",";
        DeducedArgument *arg = instance->DeducedArg (a);
        if (arg->Type ())
          arg->Type ()->TypeText (res, "", true, true);
        else if (arg->Value ()) {
          if (arg->Value ()->isSigned ())
            res << arg->Value ()->convert_to_int ();
          else if (arg->Value ()->isUnsigned ())
            res << arg->Value ()->convert_to_uint ();
          else if (arg->Value ()->isFloat ())
            res << arg->Value ()->convert_to_float ();
        }
        else
          res << "*invalid template arg*";
      }
      res << " >";
    }
  }

  // destination function arguments
  if (!bypass)
    res << "(";
  for (int a = 0 ; a < get_arg_count (*loc); a++, args++) {
    if (args > 0) res << ", ";
    if (action) {
      // make sure that if the argument type was a reference we use the
      // referred type here
      res << "*(typename __TJP::template Arg<" << a
          << ">::ReferredType*)__TJP::arg(" << a << ")";
    }
    else {
      if (arg_names)
        res << (*arg_names)[a];
      else
        res << "arg" << a;
    }
  }
  res << ")";

  // generate the 'result =' code
  // TODO: better pass result_buffer as parameter
  ResultBuffer result_buffer (func, is_call);
  out << "    ";
  if (action)
    out << result_buffer.action_result_assignment(res.str ());
  else
    out << result_buffer.result_assignment(res.str ());

  out << ";" << endl;

  if (cflows.size () > 0)
    out << "}" << endl;
}

// create an action wrapper function, which may be invoked by 'proceed()'
void CodeWeaver::make_action_wrapper(ostream &impl, ACM_Code *loc, int depth) {
  // generate the action function, which is used to obtain an action object
  impl << "  AC::Action &action() {" << endl;
  impl << "    this->__TJP::_wrapper = &";
  Naming::action_wrapper (impl, loc, depth);
  impl << ";" << endl;
  impl << "    return *this;" << endl;
  impl << "  }" << endl;

  // make the function static if it is defined in class scope
  impl << "  static void ";
  Naming::action_wrapper (impl, loc, depth);
  impl << " (AC::Action &action) {" << endl;
  // convert the action object ref into a TJP_... object ref and call proceed
  impl << "    ((__TJP&)action).proceed ();" << endl;

  impl << "  }" << endl;
}

// create the 'proceed()' function for the current level
void CodeWeaver::make_proceed_func(ostream &impl, ACM_Any *loc, ACM_CodePlan *plan) {
  // the signature
  impl << "  void proceed () {" << endl;

  if (depth(*plan) == 0) {
    // this is a proceed function for the last around advice
    make_proceed_code (impl, loc, true);
  }
  else {
    // generate __TJP type (for the next level)
    impl << "    typedef ";
    Naming::tjp_struct (impl, (ACM_Code*)loc, depth (*plan) - 1);
    impl << "<TResult, TThat, TTarget, TArgs> __TJP;" << endl;
    // generate advice calls
    make_advice_calls (impl, plan->get_next_level(), loc, true);
  }
  impl << "  }" << endl;
}

// generate the name of the joinpoint-specific TJP type
void CodeWeaver::make_tjp_typename (ostream &out, ACM_Code *loc, int depth, ResultBuffer &rb) {
  TI_Code *ti = (TI_Code*)TransformInfo::of (*loc);
  Naming::tjp_struct (out, loc, depth);
  out << "< " << rb.tjp_result_type() << ", ";
  ti->that_type ()->TypeText (out, "", true, true);
  out << ", ";
  bool is_ptr;
  if (loc->type_val () == JPT_Call && ((TI_MethodCall*)ti)->target_expr(is_ptr)) {
    if (((TI_MethodCall*)ti)->target_type()->isConst())
      out << "const ";
    out << "TTarget";
  }
  else
    ti->target_type ()->TypeText (out, "", true, true);
  // argument types
  unsigned arg_count = get_arg_count (*loc);
  out << ", ";
  list<ACM_Type*> arg_types;
  get_arg_types (*loc, arg_types);
  for (list<ACM_Type*>::iterator i = arg_types.begin(); i != arg_types.end(); ++i) {
    CTypeInfo *argtype = TI_Type::of (**i)->type_info ();
    out << " AC::TL< ";
    argtype->TypeText (out, "", true, true);
    out << ",";
  }
  out << " AC::TLE";
  for (unsigned a = 0; a < arg_count; a++)
    out << " >";
  out << " >";
 }

// insert TJP class definition before join point
void CodeWeaver::make_tjp_struct(ostream &out, ACM_Code *loc,
    ACM_CodePlan *plan, const ThisJoinPoint &tjp) {

  int depth = ::depth (*plan);
  // recursively call the function for the previous TJP classes
  if (depth > 0) {
    make_tjp_struct (out, loc, plan->get_next_level(), tjp);
  }

  // generate the definition
  out << endl;
  tjp.gen_tjp_struct (out, loc, _problems, depth);

  // generate the proceed function for the current structure
  if (plan->has_around()) {
    const ThisJoinPoint &curr_tjp =
      TI_AdviceCode::of (*plan->get_around()->get_advice())->this_join_point ();
    if (curr_tjp.proceed () || curr_tjp.action() ||
        !TI_CodeAdvice::of(*plan->get_around())->get_advice_info()->pointcut().cflow_triggers ().empty ())
      make_proceed_func (out, loc, plan);

    // generate action wrapper if needed
    if (curr_tjp.action())
      make_action_wrapper (out, loc, depth);
  }

  if (depth == 0) {
    stringstream jpname;
    Naming::tjp_struct(jpname, loc, 0);
    gen_binding_templates (out, loc->get_plan(), jpname.str ().c_str ());
  }

  // closing bracket of the TJP class! Opened in gen_tjp_struct.
  out << "};" << endl;

  out << endl;
}


// generates the signature of a wrapper function for exec/cons/dest join pts
string CodeWeaver::wrapper_function_signature (ACM_Code *loc,
    const SyntacticContext &sctxt, bool def) {

  ACM_Function *function = (ACM_Function*)loc->get_parent ();

  set<string> used_args;
  bool search_args = (def && function->get_kind() == FT_CONSTRUCTOR);
  if (search_args)
    sctxt.search_used_args (used_args);

  stringstream name_args;
  if (sctxt.qualified_scope() != "" &&
      sctxt.syntactical_scope() != sctxt.qualified_scope())
    name_args << sctxt.qualified_scope() << "::";
  Naming::exec_inner (name_args, loc);
  name_args << "(";
  for (unsigned a = 0; a < sctxt.args(); a++) {
    if (sctxt.arg_type(a) == "void")
      break;
    if (a > 0) name_args << ",";
    string name = sctxt.arg_name(a);
    if (name != "" && search_args && used_args.find (name) == used_args.end ())
      name = "";
    name_args << sctxt.arg_type(a, name);
  }
  name_args << ")";

  ostringstream wrapped;

  if (_problems._use_always_inline) // GNU extension
    wrapped << "__attribute__((always_inline)) ";
  wrapped << "inline "; // the wrapped function should always be inlined
  if (function->get_kind() == FT_STATIC_MEMBER && sctxt.is_in_class_scope())
    wrapped << "static ";

  if (function->get_kind() == FT_CONSTRUCTOR ||
      function->get_kind() == FT_DESTRUCTOR ||
      function->get_kind() == FT_VIRTUAL_DESTRUCTOR ||
      function->get_kind() == FT_PURE_VIRTUAL_DESTRUCTOR)
    wrapped << "void " << name_args.str ().c_str ();
  else {
    wrapped << sctxt.result_type (name_args.str ());
    switch (function->get_cv_qualifiers()) {
    case CVQ_CONST:          wrapped << " const";          break;
    case CVQ_VOLATILE:       wrapped << " volatile";       break;
    case CVQ_CONST_VOLATILE: wrapped << " const volatile"; break;
    default: break;
    }
  }

  return wrapped.str ();
}

void CodeWeaver::cons_join_point (ACM_Construction *loc) {
  if (((ACM_Name*)loc->get_parent ())->get_builtin())
    gen_special_member_function (loc);
  else
    wrap_function (loc);
}

void CodeWeaver::dest_join_point (ACM_Destruction *loc) {
  if (((ACM_Name*)loc->get_parent ())->get_builtin())
    gen_special_member_function (loc);
  else
    wrap_function (loc);
}


// checks whether an implicitely defined constructor or destructor may be
// generated
// TODO: this implementation is not very efficient. Results should be cached.
bool CodeWeaver::check_special_member_function (CFunctionInfo *func) {
  // the argument should be a valid class member
  assert (func && func->ClassScope ());
  CClassInfo *cls = func->ClassScope ()->ClassInfo ();

  // check if any base class has a private constructor/copy constructor/destr.
  for (unsigned b = 0; b < cls->BaseClasses (); b++) {
    CClassInfo *base = cls->BaseClass (b)->Class ();
    for (unsigned f = 0; f < base->Functions (); f++) {
      CFunctionInfo *curr = base->Function (f);
      if ((curr->isConstructor () && func->isConstructor () &&
           curr->Arguments () == func->Arguments ()) ||
          (curr->isDestructor () && func->isDestructor ())) {
        if (curr->Protection () == CProtection::PROT_PRIVATE ||
            !check_special_member_function (curr))
          return false; // function or any base is private -> problem!
      }
    }
  }

  // check the types of non-static attributes (or array thereof) for private constructor/copy constructor/destr.
  for (unsigned a = 0; a < cls->Attributes (); a++) {
    CObjectInfo *attr = cls->Attribute(a);
    if (attr->isStatic() || attr->isAnonymous())
      continue;
    CTypeInfo *type = attr->TypeInfo();
    while (type->isArray())
      type = type->BaseType();
    CRecord *base = type->Record();
    if (!base ||             // no problem if the attribute is no class/struct/union instance
        base->isAnonymous()) // anonymous structs/classes/unions cannot have a private constrcutor/destructor
      continue;
    for (unsigned f = 0; f < base->Functions (); f++) {
      CFunctionInfo *curr = base->Function (f);
      if ((curr->isConstructor () && func->isConstructor () &&
           curr->Arguments () == func->Arguments ()) ||
          (curr->isDestructor () && func->isDestructor ())) {
        if (curr->Protection () == CProtection::PROT_PRIVATE ||
            !check_special_member_function (curr))
          return false; // function or any base is private -> problem!
      }
    }
  }

  return true; // no problem
}


void CodeWeaver::gen_special_member_function (ACM_Code *loc) {

  ostringstream code;
  CFunctionInfo *func = TransformInfo::of (*loc)->assoc_obj ()->FunctionInfo ();
  assert (func && func->ClassScope ());
  CClassInfo *cls = func->ClassScope ()->ClassInfo ();

  // check if the special member would have to call a private member
  if (!check_special_member_function (func))
    return; // can be ignored, because the implicit special member can't be
            // called anyway

  Token *inspos = ((CT_ClassDef*)cls->Tree ())->Members ()->end_token ();

  // create thisJoinPoint class
  ThisJoinPoint tjp;
  tjp.merge_flags(*loc->get_plan());
  if (tjp.type_needed ()) {
    ostringstream wrappers;
    make_tjp_struct(wrappers, loc, loc->get_plan(), tjp);
    insert (weave_pos (inspos, WeavePos::WP_BEFORE), wrappers.str ());
  }

  // generate the function itself
  code << endl << "public:" << endl << "inline " << func->Name ();

  // constructors or destructors don't have more than one argument
  assert (func->Arguments () <= 1);

  if (func->Arguments () == 0) {
    // default constructor or destructor
    code << " () {" << endl;
  }
  else {
    CT_ClassDef *cd = (CT_ClassDef*)cls->Tree ();
    // wrap attributes that are non-static arrays in a class.
    wrap_attribute_arrays (cd->Members ());
    // the same for introduced members
    if (cd->IntroMembers ()) {
      wrap_attribute_arrays ((CT_MembList*)cd->IntroMembers ());
    }

    // generate the copy constructor
    ostringstream copy_cons;
    bool arg_needed = tjp.arg_needed ((ACM_Code*)loc);
    // copy the baseclass members
    bool first_initializer = true;
    for (unsigned b = 0; b < cls->BaseClasses (); b++) {
      if (first_initializer) {
        first_initializer = false;
        copy_cons << ": ";
      }
      else
        copy_cons << ", ";
      CClassInfo *base = cls->BaseClass (b)->ClassPseudoInstance () ?
        cls->BaseClass (b)->ClassPseudoInstance () :
        cls->BaseClass (b)->Class ();
      copy_cons << *base->TypeInfo () << " (arg0)";
      arg_needed = true;
    }

    set<CUnionInfo*> anon_unions;
    for (unsigned a = 0; a < cls->Attributes (); a++) {
      CAttributeInfo *attr = cls->Attribute (a);

      // only certain members are initialized
      if (attr->isStatic () || attr->isAnonymous () || attr->EnumeratorInfo ())
        continue;

      // attributes that are member of anonymous unions have to be copied
      // differently
      if (attr->Scope ()->isUnion ()) {
        anon_unions.insert (attr->Scope ()->UnionInfo ());
        continue;
      }

      // write ':' or ','
      if (first_initializer) {
        first_initializer = false;
        copy_cons << ": ";
      }
      else
        copy_cons << ", ";
      // initialize the member by copy-constructor
      copy_cons << attr->Name () << " (arg0." << attr->Name () << ")";
      arg_needed = true;
    }
    copy_cons << " {" << endl;

    // copy anonymous unions
    int count = 0;
    for (set<CUnionInfo*>::const_iterator iter = anon_unions.begin ();
      iter != anon_unions.end (); ++iter, ++count) {
      CUnionInfo *curr_union = *iter;
      CAttributeInfo *attr = curr_union->Attribute (0);
      // anonymous unions are very simple to copy. They have no base classes,
      // no function members, no static members, ...

      // create a named local copy of the union definition
      copy_cons << "  union __ac_union_" << count << " ";
      print_tree (copy_cons, ((CT_UnionDef*)curr_union->Tree ())->Members ());
      copy_cons << ";" << endl;
      copy_cons << "  *(union __ac_union_" << count << "*)&" << attr->Name ()
           << " = *(union __ac_union_" << count << "*)&arg0."
           << attr->Name () << ";" << endl;
      arg_needed = true;
    }

    code << " (";
    func->Argument (0u)->TypeInfo ()->TypeText (code, arg_needed ? "arg0" : "");
    code << ") " << copy_cons.str ();
  }

  ResultBuffer result_buffer (&CTYPE_VOID);
  // generate __TJP type (for this level)
  if (tjp.type_needed ()) {
    code << "  typedef ";
    make_tjp_typename (code, loc, depth (*loc->get_plan()), result_buffer);
    code << " __TJP;" << endl;
  }

  // generate common JoinPoint initialization
  tjp.gen_tjp_init(code, loc, _problems, depth (*loc->get_plan()), false, 0);

  // generate calls to advice code or original function
  make_advice_calls(code, loc->get_plan(), loc);

  code << endl << "}" << endl;

  // insert the definition
  insert (weave_pos (inspos, WeavePos::WP_BEFORE), code.str ());
}

// wrap attributes that are non-static arrays in a class.
void CodeWeaver::wrap_attribute_arrays (CT_MembList *members) {
  for (int m = 0; m < members->Entries (); m++) {
    if (members->Entry (m)->NodeName () != CT_ObjDecl::NodeId ())
      continue;
    CT_ObjDecl *decl = (CT_ObjDecl*)members->Entry (m);
    CT_DeclSpecSeq *decl_specs = decl->DeclSpecs ();
    CT_DeclaratorList *dlist = decl->Declarators ();
    bool del_decl_specs = false;
    for (int init_decls = 0, s = 0; s < dlist->Sons (); s++) {
      if (dlist->Son (s)->NodeName () != CT_InitDeclarator::NodeId ())
        continue;
      CT_InitDeclarator *init_decl = (CT_InitDeclarator*)dlist->Son (s);
      CObjectInfo *obj = init_decl->SemObject ()->Object ();

      // if this object has no type, continue. might be an alias
      if (!obj->TypeInfo ())
        continue;

      // if this object is a typedef, nothing is of interest here
      if (obj->TypedefInfo ())
        break;

      // replace comma with semi colon
      if (init_decls > 0) {
        // replace the "," of the init declarator list with ";" if needed
        CTree *comma = dlist->Son (s - 1);
        const WeavePos &from = weave_pos (comma->token (), WeavePos::WP_BEFORE);
        const WeavePos &to   = weave_pos (comma->token (), WeavePos::WP_AFTER);
        replace (from, to, ";");
      }

      // only non-static arrays are to be wrapped
      if (obj->TypeInfo ()->TypeArray () && !obj->isStatic ()) {
        // it is a non-static array

        // generate the code of the wrapper class
        ostringstream wrapper_class_code;
        gen_wrapped_array (wrapper_class_code, obj);

        // replace the original init declarator with the wrapper object decl
        const WeavePos &from =
          weave_pos (init_decl->token (), WeavePos::WP_BEFORE);
        const WeavePos &to =
          weave_pos (init_decl->end_token (), WeavePos::WP_AFTER);
        replace (from, to, wrapper_class_code.str ());

        // if this is the first init declarator, remove the decl spec seq
        if (init_decls == 0)
          del_decl_specs = true;
      }
      else {
        // it is no array; copy the declaration specifier list
        if (decl_specs && decl_specs->token () && init_decls > 0) {
          ostringstream ds;
          print_tree(ds, decl_specs);
          insert (weave_pos (init_decl->token (), WeavePos::WP_BEFORE),
              ds.str() + " ");
        }
      }

      // this was an init declarator and not a comma
      init_decls++;
    }

    // delete the decl specs
    if (del_decl_specs && decl_specs && decl_specs->token ())
      kill (decl_specs);
  }
}

void CodeWeaver::gen_wrapped_array (ostream &out, CObjectInfo *obj) {
  assert (obj->TypeInfo ()->TypeArray ());
  CTypeArray *t_array = obj->TypeInfo ()->TypeArray ();
  out << endl;
  out << "  struct ";

  // generate name to satisfy VC++ 2002
  out << "__ac_wrapper_" << obj->Name ();

  out << " {" << endl;
  out << "    typedef ";
  t_array->BaseType ()->TypeText (out, "E", true, true);
  out << "; typedef E A[" << t_array->Dimension () << "]; A _data;" << endl;
  out << "    operator A& () { return _data; }" << endl;
  out << "    operator A& () const { return (A&)*(";
  CTypeArray *t_first_array = t_array;
  while (t_first_array->BaseType ()->TypeArray ())
    t_first_array = t_first_array->BaseType ()->TypeArray ();
  t_first_array->BaseType ()->TypeText (out, "*", true, true);
  out << ")_data; }" << endl;
  out << "    operator const A& () { return _data; }" << endl;
  out << "    operator const A& () const { return _data; }" << endl;
  out << "    operator void* () { return _data; }" << endl;
  out << "    operator void* () const { return (void*)_data; }" << endl;
  out << "    operator const void* () { return _data; }" << endl;
  out << "    operator const void* () const { return _data; }" << endl;
  out << "    template <typename I> E& operator [] (I i) "
      << "{ return _data[i]; } // for VC++ 2003" << endl;
  out << "    template <typename I> const E& operator [] (I i) const "
      << "{ return _data[i]; } // for VC++ 2003" << endl;
  out << "  } " << obj->Name ();
}

void CodeWeaver::exec_join_point (ACM_Execution *loc) {
  wrap_function (loc);
}

void CodeWeaver::wrap_function (ACM_Code *loc) {

  ACM_Function *function = (ACM_Function*)loc->get_parent ();
  
  const vector<SyntacticContext> &decl_contexts =
      TI_Function::of(*function)->syntactic_contexts();
  if (function->get_variadic_args ()) {
    _err << sev_warning << decl_contexts.begin()->error_pos()->location ()
         << "can't weave execution advice for function '"
         << signature(*function).c_str() << "' with variable arguments"
         << endMessage;
         return;
    /* 
     * Functions with variable arguments require a dedicated code
     * transformation. This could be implemented via function-template
     * wrappers.
     * 
     * Example:
     * 
     * void printf(char*, ...);
     * 
     *        ||
     *        \/
     * 
     * template<typename T1>
     * void printf(char* s, T1 t1) {
     *   [...]; __exec_old_printf(s, t1); }
     * 
     * template<typename T1, typename T2>
     * void printf(char* s, T1 t1, T2 t2) {
     *   [...]; __exec_old_printf(s, t1, t2); }
     * 
     * 
     * These template wrappers need to be generated for a fixed amount
     * of parameters, which could be made configurable via a command
     * line option. For C++11, a single variadic template would be
     * sufficient.
     * 
     * However, this code transformation does not fit so well into the
     * current transformation scheme. As opposed to it, these template
     * wrapper have to be generated at each function declaration (a
     * preprocessor guard could be used to avoid multiple definitions).
     * Additionally, separate JoinPoint structures have to be generated
     * for each number of template parameters, since the proceed()
     * function needs to know the amount of arguments to pass to
     * __exec_old_printf.
     * 
     * At the functions definition, these wrappers have to be generated
     * as well, for instance if no declaration was found before. The
     * body of these template wrappers needs to be generated in both
     * cases (at each function declaration and its definition). The
     * original function just has to be renamed (__exec_old_printf).
     * 
     * Finally, the generation of the separate JoinPoint structures,
     * for each number of template parameters, would require to extend
     * the current (already complicatad) JoinPoint-structure generation.
     * 
     * Caveats:
     *  - C linkage of the the function is lost (template wrappers)
     *  - it has be ensured that no similar function template exists
     *    in the same scope (as the wrapper), which would otherwise lead
     *    to multiple definitions
     *  - gcc won't inline the __exec_old vararg function,
     *    see: http://gcc.gnu.org/bugzilla/show_bug.cgi?id=10980
     *    -> __attribute__((always_inline)) must not be used here
     * 
     */
  }

  // handle each declaration and the optional definition separately
  // the transform info object yield the necessary syntactic context
  bool wrapped_decl = false;
  const SyntacticContext *def_context = 0;
  for (vector<SyntacticContext>::const_iterator i = decl_contexts.begin ();
      i != decl_contexts.end (); ++i) {
    if ((*i).is_definition ())
      def_context = &(*i);
    else {
      if (wrap_function_decl (loc, *i))
        // friend declarations have an assigned scope. They are no real
        // declarations. Another declaration has to be generated.
        if (!(*i).has_assigned_scope ())
          wrapped_decl = true;
    }
  }
  // if a definition has been found, wrap it
  if (def_context)
    wrap_function_def (loc, *def_context, wrapped_decl);

}

// create a new function implementation and paste it behind the original
void CodeWeaver::wrap_function_def (ACM_Code *loc, const SyntacticContext &sctxt,
  bool wrapped_decl) {

  ACM_Function *function = (ACM_Function*)loc->get_parent ();
  ostringstream pre_sig, pre_body;

  bool have_pre_sig = false;
  pre_sig << endl;

  // the first is the declaration of the inner function (the original)
  // not necessary for methods and if there is a declaration anyway
  if (!wrapped_decl &&
      (function->get_kind() == FT_NON_MEMBER ||
      function->get_kind() == FT_STATIC_NON_MEMBER)) {
    have_pre_sig = true;
    pre_sig << wrapper_function_signature (loc, sctxt, true) << ";" << endl;
  }

  // generate the JoinPoint class
  ThisJoinPoint tjp;
  tjp.merge_flags(*loc->get_plan());
  if (tjp.type_needed ()) {
    have_pre_sig = true;
    if (sctxt.is_in_extern_c_block())
      pre_sig << "}" << endl << endl;
    make_tjp_struct(pre_sig, loc, loc->get_plan(), tjp);
    if (sctxt.is_in_extern_c_block())
      pre_sig << endl << "extern \"C\" {" << endl;
  }

  // paste pre-signature code in front of the old function signature
  if (have_pre_sig) // check only for better readability
    insert (weave_pos (sctxt.wrapper_pos(), WeavePos::WP_BEFORE), pre_sig.str ());

  // rename the arguments of the original declaration
  vector<string> arg_names;
  rename_args (sctxt, "arg", arg_names);

  // if the function is the "main()" function and if it does not contain a
  // return statement at the end, generate one ("return 0;")
  if (function->get_name() == "main" &&
      signature(*(ACM_Name*)function->get_parent()) == "::" &&
      !sctxt.ends_with_return())
    insert (weave_pos (sctxt.fct_body_end(), WeavePos::WP_BEFORE),
           "return 0;\n");

  // generate the body of the wrapper function
  ResultBuffer result_buffer (function);

  pre_body << "{" << endl;

  // generate __TJP type (for this level)
  if (tjp.type_needed ()) {
    pre_body << "  typedef ";
    make_tjp_typename (pre_body, loc, depth (*loc->get_plan()), result_buffer);
    pre_body << " __TJP;" << endl;
  }

  // declare the result object
  pre_body << "  " << result_buffer.result_declaration ();

  // generate common JoinPoint initialization
  tjp.gen_tjp_init(pre_body, loc, _problems, depth (*loc->get_plan()),
      false, &arg_names);

  // generate calls to advice code or original function
  make_advice_calls (pre_body, loc->get_plan(), loc, false, &arg_names);

  pre_body << "  " << result_buffer.result_return ();
  pre_body << endl << "}" << endl;

  // generate the signature of the inner function (the original)
  // special treatment for operator new/delete/new[]/delete[]
  if (function->get_kind () == FT_MEMBER && !needs_this(*function) && !wrapped_decl)
    pre_body << "static ";
  // add a wrapper function declaration
  pre_body << wrapper_function_signature (loc, sctxt, true);

  // insert the new body and new signature in front of the old body
  insert(weave_pos (sctxt.fct_body_start(), WeavePos::WP_BEFORE), pre_body.str ());
}


// create a new declaration for a wrapped function
bool CodeWeaver::wrap_function_decl (ACM_Code *loc, const SyntacticContext &sctxt) {

  ACM_Function *function = (ACM_Function*)loc->get_parent ();

  // check if the function declaration belongs to the project and is not built-in
  // if it does not, we cannot insert the wrapper declaration
  if (!sctxt.is_in_project())
    return false;

  string inner_decl;
  if (sctxt.is_in_class_scope ())
    inner_decl += "public: ";
  // special treatment for friend declarations
  if (sctxt.has_assigned_scope ())
    inner_decl += "friend ";
  // special treatment for operator new/delete/new[]/delete[] => implicitly static
  if (function->get_kind () == FT_MEMBER && !needs_this(*function))
    inner_decl += "static ";
  // add the wrapper function declaration
  inner_decl += wrapper_function_signature (loc, sctxt, false);
  inner_decl += ";";
  // switch to the right protection if needed
  inner_decl += "\n";
  if (sctxt.protection() != "" && sctxt.protection() != "public")
    inner_decl += sctxt.protection() + ":\n";

  // move the generated declaration in front of the original declaration
  insert (weave_pos (sctxt.wrapper_pos(), WeavePos::WP_BEFORE), inner_decl);

  return true;
}


void CodeWeaver::make_advice_call(ostream &out, ACM_Any *loc,
				                          AdviceInfo *ai, bool inter, int depth) {

//  CFunctionInfo *srcfunc;
  const char *srcthis_name = "srcthis";

  if (loc->type_val () == JPT_Execution ||
      loc->type_val () == JPT_Construction ||
      loc->type_val () == JPT_Destruction) {
//    srcfunc = TransformInfo::of (*loc)->assoc_obj ()->DefObject ()->FunctionInfo ();
    if (!inter)
      srcthis_name = "this";
  }
  else if (loc->type_val () == JPT_Call) {
//    srcfunc = ((TI_MethodCall*)TransformInfo::of (*loc))->caller ();
  }
  else {
    out << "// invalid join point type in make_advice_call" << endl;
    return; // should never happen
  }

  PointCut &pc = ai->pointcut ();
  PointCut::iterator jp_iter = pc.find (loc);
  assert (jp_iter != pc.end ());
  const JoinPoint &jp = *jp_iter;

  if (jp.condition ()) {
    out << "  if(";
    jp.condition ().gen_check (out, ai, srcthis_name);
    out << ") {" << endl;
  }

  // make the advice call itself
  stringstream tjp_tp;
//  make_tjp_typename (tjp_tp, (JPL_Code*)jp.location (), depth);
//  tjp_tp << " ";
  tjp_tp << "__TJP";

  stringstream tjp_obj;
  if (inter)
    tjp_obj << "this";
  else {
    tjp_obj << "&";
    Naming::tjp_instance (tjp_obj, jp.location ());
  }

  out << "  ";
  ai->gen_invocation_func_call (out, tjp_tp.str ().data (),
                                tjp_obj.str ().data ());
  out << endl;

  if (jp.condition ()) {
    out << "  }" << endl;
    if (ai->type () == ACT_AROUND) {
      out << "  else {" << endl << "    ";
      if (inter)
        out << tjp_tp.str ().data () << "::";
      else {
        Naming::tjp_instance(out, loc);
        out << ".";
      }
      out << "proceed ();" << endl << "  }" << endl;
    }
  }
}


void CodeWeaver::make_advice_calls (ostream &out, ACM_CodePlan *plan,
    ACM_Any *loc, bool inter, vector<string> *arg_names) {

  typedef ACM_Container<ACM_CodeAdvice, true> Container;
  Container &before = plan->get_before();
  for (Container::iterator i = before.begin (); i != before.end (); ++i)
    make_advice_call (out, loc, TI_CodeAdvice::of(**i)->get_advice_info(), inter, depth (*plan));

  if (plan->has_around())
    make_advice_call (out, loc, TI_CodeAdvice::of(*plan->get_around())->get_advice_info(), inter, depth (*plan));
  else
    make_proceed_code (out, loc, inter, arg_names);

  Container &after = plan->get_after();
  for (Container::iterator i = after.begin (); i != after.end (); ++i)
    make_advice_call (out, loc, TI_CodeAdvice::of(**i)->get_advice_info(), inter, depth (*plan));
}


// this function weaves the invocation code for a tree of advice functions
// at a call join point
void CodeWeaver::call_join_point (ACM_Call *loc) {

  // determine the weaving parameters
  TI_MethodCall &ti = *(TI_MethodCall*)TI_MethodCall::of(*loc);
  const SyntacticContext &sctxt = ti.call_context();
  ACM_Name *src_obj = (ACM_Name*)loc->get_parent();
  ACM_Function *src_func = ((src_obj->type_val() == JPT_Function) ?
      (ACM_Function*)src_obj : (ACM_Function*)0);
  ACM_Function *dst_func = loc->get_target ();
  CTypeInfo *result_type = ti.result_type_info ();
  bool deduce_result_type = ti.has_result (); // TODO: don't do it always

  // analyze the target object of this call
  bool target_is_ptr = ti.target_is_ptr ();
  bool target_expr   = ti.has_target_expr ();

  // determine the weaving position for wrapper code
  bool extern_c_block = sctxt.is_in_extern_c_block();
  const WeavePos &pos = weave_pos (sctxt.wrapper_pos(), WeavePos::WP_BEFORE);

  // generate and insert a forward declaration for the target function if needed:
  // direct recursive call of a non-member function that has no forward declaration
  if (src_func && src_func == dst_func &&
      (dst_func->get_kind() == FT_NON_MEMBER ||
      dst_func->get_kind() == FT_STATIC_NON_MEMBER) &&
      dst_func->get_source().size() == 1) {
    // TODO: a forward declaration has to be generated and default argument
    //       initializers have to be moved into the forward declaration
//    cout << "FORWARD" << endl;
  }

  // generate JoinPoint-ID-Class and caller function of necessary
  if (ti.needs_rights()) {
    insert_id_class (loc, src_obj, pos, *dst_func);
  }

  // generate and insert the JoinPoint class
  ThisJoinPoint tjp;
  tjp.merge_flags(*loc->get_plan());
  if (tjp.type_needed ()) {
    ostringstream tjp_code;
    if (extern_c_block)
      tjp_code << "}" << endl;
    make_tjp_struct (tjp_code, loc, loc->get_plan(), tjp);
    // insert the definition
    insert (pos, tjp_code.str ());
  }

  // generate replacement function for dstfunc
  ostringstream code;
  code << endl;

  bool havesrcthis = src_func && needs_this(*src_func) && (tjp.that() || tjp.useAction());
  bool havedstthis = needs_this(*dst_func) || target_expr;

  // argument list
  int arg_count = get_arg_count (*loc);

  bool wrapper_is_template =
    (ti.has_result () || havesrcthis || havedstthis || arg_count > 0);
  if (wrapper_is_template) {
    code << "template <";
    int targs = 0;
    if (havesrcthis) {
      if (targs > 0)
        code << ", ";
      code << "typename TSrc";
      targs++;
    }
    if (arg_count > 0) {
      for (int a = 0; a < arg_count; a++, targs++) {
        if (targs > 0)
          code << ", ";
        code << "typename TArg" << a;
      }
    }
    if (ti.has_result ()) {
      if (targs > 0)
        code << ", ";
      code << "typename TResult";
      targs++;
    }
    if (havedstthis) {
      if (targs > 0)
        code << ", ";
      code << "typename TTarget";
      targs++;
    }
    code << ">" << endl;
  }

  // if the lexical scope in which the associated object of the call was
  // defined is a class scope (not global or namespace) the function definition
  // has to be preceded by a 'static' keyword to avoid an implicit 'this'.
  if (sctxt.is_in_class_scope())
      code << "static ";
  if (_problems._use_always_inline) // GNU extension
    code << "__attribute__((always_inline)) ";
  code << "inline ";

  stringstream new_call;
  stringstream asig;
  stringstream impl;

  // add result type
  code << (ti.has_result() ? "TResult " : "void ");
  if (result_type->isAddress()) {
    if (result_type->BaseType ()->isConst())
      code << "const ";
    code << "&";
  }

  // add function name
  Naming::call_wrapper (code, loc, depth (*loc->get_plan ()));
  code << " (";
  Naming::call_wrapper (new_call, loc, depth (*loc->get_plan()));
  if (wrapper_is_template) {
    new_call << "< ";
    int targs = 0;
    if (havesrcthis) {
      if (targs++ > 0)
        new_call << ", ";
      string scope = string("::") + signature(*cscope (src_func));
      new_call << cleanup_name (scope) << "*";
    }
    if (arg_count > 0) {
      // argument types
      list<ACM_Type*> arg_types;
      get_arg_types (*loc, arg_types);
      int a = 0;
      for (list<ACM_Type*>::iterator i = arg_types.begin(); i != arg_types.end();
           ++i, ++a) {
        CTypeInfo *type = TI_Type::of (**i)->type_info ();
        if (targs++ > 0)
          new_call << ", ";
        type->TypeText (new_call, "", true, true);
      }
    }
    if (ti.has_result () && !deduce_result_type) {
      if (targs++ > 0)
        new_call << ", ";
      CTypeInfo *t = result_type;
      if (t->isAddress()) t = t->BaseType ();
      bool is_const = false;
      if (t->TypeQualified() && t->TypeQualified()->isConst()) {
        is_const = true;
        t->TypeQualified()->isConst(false);
      }
      t->TypeText(new_call, "", true, true);
      if (is_const)
        t->TypeQualified()->isConst(true);
    }
    new_call << " >";
  }
  new_call << " (";

  unsigned argnum = 0;
  const char *calltype = ti.CallNode ()->NodeName ();

  // source and destination this pointer arguments (if needed)
  if (havesrcthis) {
    string scope = string("::") + signature(*cscope (src_func));
//    string scope = string("::") + cscope (srcfunc)->QualName();
    code << "TSrc srcthis";
    new_call << "(" << cleanup_name (scope) << "*)this";
    argnum++;
  }

  if (havedstthis) {
    if (argnum > 0)
      code << ", ";
    code << "const TTarget &dst";

    // move/paste the target object pointer
    if (argnum > 0)
      new_call << ",";
    if (target_expr) {
      if (target_is_ptr)
        new_call << "*(";
    }
    else {
      new_call << "*this";
    }

    argnum++;
  }

  // argument list
  for (int a = 0 ; a < arg_count; a++) {
    if (argnum > 0)
      code << ", ";
    code << "TArg" << a << " arg" << a;
    argnum++;
  }

  if (deduce_result_type) {
    if (argnum > 0)
      code << ", ";
    code << "AC::RT<TResult>";
    argnum++;
  }
  code << ")";

  ResultBuffer result_buffer (result_type, wrapper_is_template);

  // replacement function body
  code << "{" << endl;

  // generate __TJP type (for the next level)
  if (tjp.type_needed ()) {
    code << "  typedef ";
    make_tjp_typename (code, loc, depth (*loc->get_plan ()), result_buffer);
    code << " __TJP;" << endl;
  }

  // declare the result store
  code << "  " << result_buffer.result_declaration();

  // generate common JoinPoint initialization
  tjp.gen_tjp_init(code, loc, _problems, depth (*loc->get_plan ()),
      target_expr || deduce_result_type, 0);

  // add calls to advice code
  make_advice_calls(code, loc->get_plan(), loc);

  code << "  " << result_buffer.result_return();

  code << " }" << endl;

  if (extern_c_block)
    code << endl << "extern \"C\" {" << endl;

  // now we perform the code manipulation
  // insert the call wrapper body
  insert (pos, code.str ());

  // TODO: make Puma check for macro problems and warn if requested

  // replace call with call to replacement function
  const WeavePos &before_pos = weave_pos (ti.before_pos(), WeavePos::WP_BEFORE);
  const WeavePos &after_pos  = weave_pos (ti.after_pos(), WeavePos::WP_AFTER);

  // provide wrapper call arguments by transforming the call expression
  CTree *node;
  // call was a binary operator call
  if (calltype == CT_BinaryExpr::NodeId ()) {
    if (target_expr) {
      if (target_is_ptr)
        insert (after_pos, ")");
    }
    else if (havesrcthis && !havedstthis)
      insert (before_pos, ",");
    node = ((CT_BinaryExpr*)ti.CallNode ())->Son (1);
    insert (weave_pos (node->end_token (), WeavePos::WP_AFTER), ",");
    kill (node);
  }
  // call was a unary operator call
  else if (calltype == CT_UnaryExpr::NodeId () ||
      calltype == CT_DerefExpr::NodeId ()) {
    if (target_expr) {
      if (target_is_ptr)
        insert (after_pos, ")");
    }
    else if (havesrcthis && !havedstthis)
      insert (before_pos, ",", true);
    node = ((CT_UnaryExpr*)ti.CallNode ())->Son (0);
    kill (node);
  }
  // call was a postfix operator call
  else if (calltype == CT_PostfixExpr::NodeId ()) {
    if (target_expr && target_is_ptr)
      insert (after_pos, ")");
    if (dst_func->get_name() == "operator ++" ||
        dst_func->get_name() == "operator --")
      insert (after_pos, ", 0");
    node = ((CT_BinaryExpr*)ti.CallNode ())->Son (1);
    kill (node);
  }
  // call was an index operator call
  else if (calltype == CT_IndexExpr::NodeId ()) {
    if (target_expr && target_is_ptr)
      insert (after_pos, ")");
    // replace the opening '[' with ','
    node = ((CT_BinaryExpr*)ti.CallNode ())->Son (1);
    insert (weave_pos (node->end_token (), WeavePos::WP_AFTER), ",");
    kill (node);
    // delete the closing ']'
    node = ((CT_BinaryExpr*)ti.CallNode ())->Son (3);
    kill (node);
  }
  // the destination function is a conversion function
  else if (calltype == CT_ImplicitCall::NodeId ()) {
  }
  // call was an ordinary function call
  else {
    CT_ExprList *args = ((CT_CallExpr*)ti.CallNode ())->Arguments ();
    CTree *fctnode = ti.CallNode()->Son (0);
    // if the function is given in parenthesis, remove it
    while (fctnode->NodeName () == CT_BracedExpr::NodeId ()) {
      kill (fctnode->Son (0));
      kill (fctnode->Son (2));
      fctnode = fctnode->Son (1);
    }
    const char *fctnodetype = fctnode->NodeName ();
    // now perform the transformation
    if (args->Entries () > 0 && (havedstthis || havesrcthis))
      insert (weave_pos (args->token (), WeavePos::WP_BEFORE), ",", true);
    if ((target_expr && target_is_ptr) &&
         (fctnodetype == CT_MembRefExpr::NodeId () ||
          fctnodetype == CT_MembPtrExpr::NodeId ())) {
      insert (weave_pos (args->token (), WeavePos::WP_BEFORE), ")", true);
    }
    if (fctnodetype == CT_MembRefExpr::NodeId () && havedstthis) {
      kill (((CT_MembRefExpr*)fctnode)->Son (1));
      kill (((CT_MembRefExpr*)fctnode)->Son (2));
    }
    else if (fctnodetype == CT_MembPtrExpr::NodeId () && havedstthis) {
      kill (((CT_MembPtrExpr*)fctnode)->Son (1));
      kill (((CT_MembPtrExpr*)fctnode)->Son (2));
    }
    else {
      if (!ti.is_call_op())
        kill (fctnode);
    }
    kill (args->Son (0));
    kill (args->Son (args->Sons () - 1));
  }

  // if the result type needs to be deduced, add expression copy here
  if (deduce_result_type) {
    ostringstream typeof_call;
    if (argnum > 1)
      typeof_call << ", ";
    typeof_call << "__AC_TYPEOF(("; // the two brackets are needed, because the call might contain a ','
                                    // e.g. func<a,b>() => ACTYPEOF(func<a,b>()) would be regarded a
                                    // macro call with *two* arguments => use ACTYPEOF((...))!
    print_tree (typeof_call, ti.CallNode(), true);
    typeof_call << "))";
    insert (after_pos, typeof_call.str ());
  }

  // paste a closing bracket behind the call
  insert (after_pos, ")");

  // paste the first (generated) part of the call (in front)
  insert (before_pos, new_call.str (), true);
}

void CodeWeaver::insert_id_class (ACM_Call *loc, ACM_Name *src_obj,
    const WeavePos &pos, ACM_Function &dst_func) {
  TI_MethodCall &ti = *(TI_MethodCall*)TransformInfo::of (*loc);
  const SyntacticContext &sctxt = ti.call_context();
  const SyntacticContext &dfctxt = ti.dest_func_context();
  CObjectInfo *srcobj = TransformInfo::of (*src_obj)->obj_info();
  ostringstream bypass;
  const WeavePos *bypass_pos = &pos;
  CClassInfo *srccls = srcobj->Scope ()->ClassInfo();
  CFunctionInfo *dstfunc = TI_Function::of(dst_func)->func_info();
  CClassInfo *dstcls = cscope (dstfunc);
  CTypeInfo *result_type = ti.result_type_info ();

  // check privileged calls to external functions
  Unit *unit = dstcls->SourceInfo ()->SrcUnit ();
  bool dst_external = false;
  if (!dstcls->SemDB ()->Project ()->isBelow (unit) &&
      !IntroductionUnit::cast (unit)) {
    dst_external = true;
  }

  if (dst_external) {
    // calls to private externals are not supported
    if (dfctxt.protection() == "private") {
      _err << sev_error << ti.before_pos()->location ()
           << "can't weave call advice for private external function '"
           << signature(dst_func).c_str () << "'" << endMessage;
      return;
    }
    // target function is protected
    if (ti.has_target_expr ()) {
      ACM_Class *target_class = loc->get_target_class ();
      dstcls = TI_Class::of(*target_class)->class_info();
    }
    else {
      dstcls = cscope (srcobj);
    }
  }

  bool before_class = (srccls && srcobj->Scope () != dstcls);
  if (before_class) {
    bypass_pos = &weave_pos (srccls->Tree ()->token(), WeavePos::WP_BEFORE);
    close_namespace (bypass, srccls);
  }
  else if (srcobj->Scope()->isNamespace())
    close_namespace (bypass, srcobj);

  map<CClassInfo*, ACM_Class *>::iterator i = _bypass_map.find(dstcls);
  assert (i != _bypass_map.end ());
  string ns_open, ns_close, classname = Naming::bypass_caller_class(i->second);
  if (srcobj->Scope() != dstcls) {
    CObjectInfo *curr = dstcls;
    while (curr->ClassInfo()) {
      classname = string (curr->Name().c_str()) + "::" + classname;
      curr = curr->Scope();
    }
    while (curr && curr->NamespaceInfo() && !curr->FileInfo()) {
      ns_open = string ("namespace ") + curr->Name ().c_str() + " { " + ns_open;
      ns_close += "} ";
      curr = curr->Scope();
    }
  }
  if (ns_open != "")
    bypass << ns_open << endl;
  if (srccls && !before_class)
    bypass << "public:" << endl;
  bypass << "// unique ID-class, just a type" << endl;
  bypass << "struct ";
  Naming::bypass_id_class (bypass, loc);
  bypass << ";" << endl;
  bypass << "template<int X> struct " << classname << "<";
  Naming::bypass_id_class (bypass, loc);
  bypass << ", X> {" << endl;
  bypass << "  static ";
  if (_problems._use_always_inline) // GNU extension
    bypass << "__attribute__((always_inline)) ";
  bypass << "inline ";
  stringstream sig, impl;
  sig << "call (";
  int argnum = 0;
  bool dstthis_needed = needs_this (dst_func);
  // pass object pointer if needed
  if (dstthis_needed) {
    if (dstfunc->TypeInfo()->isConst())
      sig << "const ";
    stringstream scope;
    dstcls->TypeInfo()->TypeText(scope, "", true, true);
    sig << cleanup_name (scope.str ()) << " *dstthis";
    argnum++;
  }
  // argument list
  list<ACM_Type*> arg_types;
  get_arg_types (*loc, arg_types);
  int a = 0;
  for (list<ACM_Type*>::iterator i = arg_types.begin(); i != arg_types.end();
       ++i, ++a) {
    CTypeInfo *type = TI_Type::of (**i)->type_info ();
    if (argnum > 0)
      sig << ", ";
    stringstream arg_name;
    arg_name << "arg" << a;
    type->TypeText (sig, arg_name.str ().data (), true, true);
    argnum++;
  }
  sig << ")";
  // add result type
  result_type->TypeText(bypass, sig.str ().data (), true, true);
  bypass << "{" << endl;
  bypass << "    ";
  if (!result_type->isVoid())
    bypass << "return ";
  if (dstthis_needed)
    bypass << "dstthis->";
  // TODO: how to check if a fully qualified name must be used here?
  if (ti.is_qualified())
    bypass << dstfunc->QualName () << " (";
  else
    bypass << dstfunc->Name () << " (";
  a = 0;
  for (list<ACM_Type*>::iterator i = arg_types.begin(); i != arg_types.end();
     ++i, ++a) {
    if (a > 0)
      bypass << ", ";
    bypass << "arg" << a;
  }
  bypass << ");" << endl;
  bypass << "  }" << endl;
  bypass << "};";
  if (ns_close != "")
    bypass << ns_close << endl;
  if (before_class) {
    open_namespace (bypass, srccls);
  } else if (sctxt.is_in_namespace())
    open_namespace (bypass, srcobj);
  // switch to the right protection if needed
  if (srccls && !before_class && !srccls->isStruct() && !srccls->isUnion() &&
      sctxt.protection() != "" && sctxt.protection() != "public")
    bypass << "\n" << sctxt.protection() << ":\n";

  // finally insert the generated code at the appropriate position
  insert (*bypass_pos, bypass.str ());
}

void CodeWeaver::bypass_info_clear () {
  _bypass_blacklist.clear ();
  _bypass_map.clear ();
}

bool CodeWeaver::bypass_in_blacklist (ACM_Class *cls) {
  const TI_Class *ti = TI_Class::of (*cls);
  Token *token = ti->objdecl_start_pos();
  return !token || (_bypass_blacklist.find (token) != _bypass_blacklist.end ());
}

void CodeWeaver::bypass_insert (ACM_Class *acm_cls) {
  const TI_Class *ti = TI_Class::of (*acm_cls);

  if (!ti->is_defined() || ti->is_template_instance() || ti->is_extern_c())
    return;

  string bypass =
      string("\n  template <typename, int = 0> struct ") +
      Naming::bypass_caller_class (acm_cls) +
      " {};\n  template <typename, int> friend struct " +
      Naming::bypass_caller_class (acm_cls) + ";";
  if (ti->is_class() && !ti->is_struct()) // don't weave this into structs
    bypass = string ("public:") + bypass + "\nprivate:";

  insert (weave_pos (ti->body_start_pos(), WeavePos::WP_AFTER), bypass, true);

  CClassInfo *cls = ti->class_info();
  _bypass_map.insert(map<CClassInfo*, ACM_Class*>::value_type(cls, acm_cls));

  // put a class in the backlist if is defined inside a template instance
  // -> it can appear more than once in the model!
  if (ti->is_in_template_instance())
    _bypass_blacklist.insert (ti->objdecl_start_pos());
}

// Transform aspect declarations: Simply replace the keyword 'aspect' by 'class'
void CodeWeaver::transform_aspect(Token *start) {
  const WeavePos &from = weave_pos (start, WeavePos::WP_BEFORE);
  const WeavePos &to   = weave_pos (start, WeavePos::WP_AFTER);
  replace(from, to, "class");
}

// Transform the definition of a named pointcut:
//
// pointcut [virtual] foo(int i) = call("C") && args(i);
// ==>
// void foo(int i);
void CodeWeaver::transform_pointcut_def(Token *start, Token *virtual_token,
    Token *assignment_token, Token *pct_end_token) {
  // TODO: at the moment the Puma parser checks whether base classes of aspects
  // are abstract. Therefore, a pure virtual pointcut has to become a pure virtual
  // function. Later this will not be necessary.
  insert (weave_pos(start, WeavePos::WP_BEFORE), "void");
//  transform_delete(start, virtual_token ? virtual_token : start);
  transform_delete(start, start);
  if (string(pct_end_token->text()) != "0")
    transform_delete(assignment_token, pct_end_token);
}

// Transform the advice code into an ordinary C++ template function:
//
// advice call("C") && args(i) : before(int i) { ... }
// ==>
// template <typename __Dummy> void __a42_before (int i) { ... }
// template <typename JoinPoint> void __a42_before (JoinPoint *tjp, int i) { ... }
void CodeWeaver::transform_advice_code (int no, string kind, AdviceCodeContext context,
    bool has_args, Token *arg_begin, Token *body_begin, Token *body_end,
    const string &prot) {

  // create and insert the function name
  ostringstream func_decl;
  func_decl << endl << "public: ";
  if (context >= ACC_TYPE)
    func_decl << "template <typename JoinPoint> ";
  func_decl << "void __a" << no << "_" << kind << " ";
  insert (weave_pos(arg_begin, WeavePos::WP_BEFORE), func_decl.str ());
  if (context >= ACC_OBJ) {
    string tjp_param = "JoinPoint *tjp";
    if (has_args) tjp_param += ", ";
    insert (weave_pos(arg_begin, WeavePos::WP_AFTER), tjp_param);
  }
  if (context >= ACC_TYPE) {
    // also generate some typedefs for That, Target, and Result
    // the corresponding typedef must replace each JoinPoint::(That|Target|Result)
    ostringstream typedefs;
    typedefs << endl;
    typedefs << "  typedef typename JoinPoint::That ";
    Naming::tjp_typedef (typedefs, "That");
    typedefs << ";" << endl;
    typedefs << "  typedef typename JoinPoint::Target ";
    Naming::tjp_typedef (typedefs, "Target");
    typedefs << ";" << endl;
    typedefs << "  typedef typename JoinPoint::Result ";
    Naming::tjp_typedef (typedefs, "Result");
    typedefs << ";" << endl;
    insert (weave_pos (body_begin, WeavePos::WP_AFTER), typedefs.str ());
  }

  if (prot != "public" && prot != "")
    insert (weave_pos (body_end, WeavePos::WP_AFTER), string ("\n") + prot + ":\n");
}

// Transform the aspect function into an ordinary C++ template function:
//
// static Foo *aspectof () { ... }
// ==>
// [template <typename JoinPoint>] static Foo *aspectof ([JoinPoint *tjp]) { ... }
void CodeWeaver::transform_aspectof (AdviceCodeContext context,
    Token *start, Token *name, Token *arg_begin, Token *body_begin, Token *body_end,
    const string &prot) {

  // create and insert the function name
  ostringstream func_decl;
  func_decl << endl << "public: ";
  if (context >= ACC_TYPE)
    func_decl << "template <typename JoinPoint> ";
  insert (weave_pos(start, WeavePos::WP_BEFORE), func_decl.str ());
  // TODO: issue warning saying that deprecated name "aspectOf" is used
  if (string(name->text ()) == "aspectOf")
    replace (weave_pos(name, WeavePos::WP_BEFORE),
        weave_pos (name, WeavePos::WP_AFTER), "aspectof");
  if (context >= ACC_OBJ) {
    string tjp_param = "JoinPoint *tjp";
    insert (weave_pos(arg_begin, WeavePos::WP_AFTER), tjp_param);
  }
  if (context >= ACC_TYPE) {
    // also generate some typedefs for That, Target, and Result
    // the corresponding typedef must replace each JoinPoint::(That|Target|Result)
    ostringstream typedefs;
    typedefs << endl;
    typedefs << "  typedef typename JoinPoint::That ";
    Naming::tjp_typedef (typedefs, "That");
    typedefs << ";" << endl;
    typedefs << "  typedef typename JoinPoint::Target ";
    Naming::tjp_typedef (typedefs, "Target");
    typedefs << ";" << endl;
    typedefs << "  typedef typename JoinPoint::Result ";
    Naming::tjp_typedef (typedefs, "Result");
    typedefs << ";" << endl;
    insert (weave_pos (body_begin, WeavePos::WP_AFTER), typedefs.str ());
  }

  if (prot != "public" && prot != "")
    insert (weave_pos (body_end, WeavePos::WP_AFTER), string ("\n") + prot + ":\n");
}

void CodeWeaver::transform_delete (Token *from, Token *to) {
  const WeavePos &from_pos = weave_pos (from, WeavePos::WP_BEFORE);
  const WeavePos &to_pos   = weave_pos (to, WeavePos::WP_AFTER);
  kill(from_pos, to_pos);
}

ACM_Class *CodeWeaver::cscope (ACM_Name *obj) {
  ACM_Name *parent = (ACM_Name*)obj->get_parent();
  return (parent->type_val() == JPT_Class ? (ACM_Class*)parent : (ACM_Class*)0);
}

CClassInfo *CodeWeaver::cscope(CObjectInfo *obj) {
  CRecord *crec = obj->ClassScope ();
  return crec ? crec->ClassInfo () : (CClassInfo*)0;
}

string CodeWeaver::cleanup_name(const string& in) {
  string result = in;
  string::size_type pos = result.length ();
  while ((pos = result.rfind ("::<unnamed>", pos)) != string::npos)
    result = result.erase (pos, 11);
  if (result.find ("<unnamed>") == 0)
    result = result.erase (0, 9);
  return result;
}

ACM_Name *CodeWeaver::nscope (ACM_Function *func) {
  ACM_Name *result = (ACM_Name*)func->get_parent ();
  if (result->get_name() == "::" || result->get_name () == "<unnamed>")
    return 0;
  return result;
}

void CodeWeaver::print_tree (ostream &out, CTree *node, bool expand_implicit_calls) {

  if (expand_implicit_calls && node->NodeName () == CT_ImplicitCall::NodeId() &&
      !((CT_ImplicitCall*)node)->Object()->FunctionInfo()->isConstructor()) {
    out << "(";
    print_tree (out, node->Son (0), expand_implicit_calls);
    out << ")." << ((CT_ImplicitCall*)node)->Object()->Name().c_str () << "()";
  }
  else if (node->NodeName () == CT_Token::NodeId ())
    out << node->token ()->text () << " ";
  else
    for (int s = 0; s < node->Sons (); s++)
      print_tree (out, node->Son (s), expand_implicit_calls);
}

void CodeWeaver::rename_args (const SyntacticContext &sctxt, const char * new_name,
    vector<string> &arg_names) {
  CFunctionInfo *func = (CFunctionInfo*)sctxt.object();

  // first rename the arguments in the argument list
  for (unsigned arg = 0; arg < func->Arguments (); arg++) {
    if (func->Argument (arg)->isAnonymous()) {
      // create a new name
      ostringstream name;
      name << new_name << arg;
      // insert the generated name
      rename (func->Argument (arg), name.str ());
      arg_names.push_back (name.str ());
    }
    else {
      arg_names.push_back (func->Argument (arg)->Name ().c_str());
    }
  }
}

void CodeWeaver::rename (CArgumentInfo *arg, const string &new_name) {
  // don't change anything with void arguments
  if (arg->TypeInfo ()->is_void ())
    return;

  // find the syntax tree node of the argument name
  CT_ArgDecl *arg_decl = (CT_ArgDecl*)arg->Tree ();
  CT_Declarator *declarator = (CT_Declarator*)arg_decl->Declarator ();
  CT_SimpleName *arg_name = name (declarator);

  if (arg_name->NodeName () != CT_PrivateName::NodeId ()) {
    const WeavePos &lpos = weave_pos (arg_name->token (), WeavePos::WP_BEFORE);
    const WeavePos &rpos = weave_pos (arg_name->end_token (), WeavePos::WP_AFTER);
    replace (lpos, rpos, new_name);
  }
  else {
    // in the case of an anonymous argument I have to find an insert pos
    CTree *left = 0, *right = 0;
    if (declarator) {
      if (declarator->NodeName () == CT_InitDeclarator::NodeId ())
        right = ((CT_InitDeclarator*)declarator)->Initializer ();
      else if (declarator->NodeName () == CT_BracedDeclarator::NodeId ())
        right = declarator->Son (declarator->Sons () - 1);
      else if (declarator->NodeName () == CT_ArrayDeclarator::NodeId ())
        right = declarator->Son (1);
      else if (declarator->NodeName () == CT_FctDeclarator::NodeId ())
        right = declarator->Son (1);
      else if (declarator->NodeName () == CT_RefDeclarator::NodeId ())
        left = declarator->Son (0);
      else if (declarator->NodeName () == CT_PtrDeclarator::NodeId ())
        left = declarator->Son (declarator->Sons () - 2);
      else if (declarator->NodeName () == CT_MembPtrDeclarator::NodeId ())
        left = declarator->Son (declarator->Sons () - 2);
      else if (declarator->NodeName () == CT_BitFieldDeclarator::NodeId ())
        right = declarator->Son (1);
      else {
        _err << sev_fatal << declarator->token ()->location ()
             << "unknown declarator type" << endMessage;
        return;
      }
    }
    if (!left && !right) {
      // no declarator + anonymous => there must be at least a decl spec seq!
      left = arg_decl->DeclSpecs ();
    }
    if (left) {
      const WeavePos &lp = weave_pos (left->end_token (), WeavePos::WP_AFTER);
      insert (lp, " ");
      insert (lp, new_name);
    }
    else if (right) {
      const WeavePos &rp = weave_pos (right->token (), WeavePos::WP_BEFORE);
      insert (rp, " ");
      insert (rp, new_name);
    }
    else
      _err << sev_fatal << declarator->token ()->location ()
           << "found no hook to replace ananymous arg name" << endMessage;
  }
}

CT_SimpleName *CodeWeaver::is_name (CTree *node) {
  if (node->NodeName () == CT_QualName::NodeId () ||
      node->NodeName () == CT_SimpleName::NodeId () ||
      node->NodeName () == CT_OperatorName::NodeId () ||
      node->NodeName () == CT_ConversionName::NodeId () ||
      node->NodeName () == CT_RootQualName::NodeId () ||
      node->NodeName () == CT_PrivateName::NodeId ())
    return (CT_SimpleName*)node;
  else
    return (CT_SimpleName*)0;
}

CT_FctDeclarator *CodeWeaver::fct_declarator (CT_Declarator *declarator) {
  do {
    CT_Declarator *next = (CT_Declarator*)declarator->Declarator ();
    if (declarator->NodeName () == CT_FctDeclarator::NodeId () &&
  is_name (next))
      return (CT_FctDeclarator*)declarator;
    declarator = next;
  } while (!is_name (declarator));
  return (CT_FctDeclarator*)0;
}

CT_SimpleName *CodeWeaver::name (CT_Declarator *&declarator) {
  CTree *node = declarator;
  declarator  = (CT_Declarator*)0;
  while (!is_name (node)) {
    declarator = (CT_Declarator*)node;
    node = declarator->Declarator ();
  }

  return (CT_SimpleName*)node;
}

// return LinkageSpec node for extern "C" <decl> like declarations
CTree *CodeWeaver::linkage_adjust (CT_Decl *decl) {
  CT_LinkageSpec *linkage = decl->Linkage ();
  return (!linkage || linkage->isList ()) ? decl : linkage;
}

void CodeWeaver::gen_binding_templates (ostream &out, ACM_CodePlan *plan,
    const char *jpname) {
  // later the individual bindings have to be passed from here, too.
  typedef ACM_Container<ACM_CodeAdvice, true> Container;
  Container &before = plan->get_before();
  for (Container::iterator i = before.begin (); i != before.end (); ++i)
    TI_CodeAdvice::of(*(*i))->get_advice_info()->gen_binding_template (out, jpname, _problems);
  if (plan->has_around())
    TI_CodeAdvice::of(*plan->get_around())->get_advice_info()->gen_binding_template (out, jpname, _problems);
  if (plan->has_next_level())
    gen_binding_templates (out, plan->get_next_level(), jpname);
  Container &after = plan->get_after();
  for (Container::iterator i = after.begin (); i != after.end (); ++i)
    TI_CodeAdvice::of(*(*i))->get_advice_info()->gen_binding_template (out, jpname, _problems);
}
