// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __SyntacticContext_h__
#define __SyntacticContext_h__

#include "IntroductionUnit.h"

// Puma includes
#include "Puma/CFunctionInfo.h"
#include "Puma/CTree.h"
#include "Puma/CProject.h"
#include "Puma/CSemDatabase.h"
using namespace Puma;

class SyntacticContext {
  CObjectInfo *_obj;

  // return LinkageSpec node for extern "C" <decl> like declarations
  static CTree *linkage_adjust (CT_Decl *decl) {
    CT_LinkageSpec *linkage = decl->Linkage ();
    return (!linkage || linkage->isList ()) ? decl : linkage;
  }

public:
  SyntacticContext (CObjectInfo *o = 0) : _obj (o) {}

  void set_object (CObjectInfo *o) { _obj = o; }

  bool is_definition () const {
    if (_obj->FunctionInfo ())
      return _obj->FunctionInfo ()->isFctDef ();
    return false;
  }

  bool has_assigned_scope () const {
    return _obj->AssignedScope ();
  }

  bool is_in_extern_c_block () const {
    CTree *tree = _obj->Tree ();
    assert (tree);
    if (tree->NodeName() == CT_FctDef::NodeId()) {
      CT_FctDef *fctdef = (CT_FctDef*)tree;
      return fctdef->Linkage () && fctdef->Linkage ()->isList ();
    }
    else if (tree->NodeName() == CT_InitDeclarator::NodeId()) {
      CT_ObjDecl *obj_decl = ((CT_InitDeclarator*)tree)->ObjDecl ();
      return obj_decl->Linkage () && obj_decl->Linkage ()->isList ();
    }
    // TODO: more cases?
    return false;
  }

  bool is_in_class_scope () const {
    return _obj->ClassScope () && (_obj->Scope() == _obj->ClassScope());
  }

  bool is_in_namespace () const {
    return _obj->Scope ()->isNamespace();
  }

  bool is_in_project () const {
    if (_obj->isBuiltin())
      return false;
    Unit *unit = _obj->SourceInfo ()->SrcUnit ();
    if (!_obj->SemDB ()->Project ()->isBelow (unit) &&
        !IntroductionUnit::cast (unit))
      return false;
    return true;
  }

  string protection () const {
    string result;
    switch (_obj->Protection()) {
    case CProtection::PROT_PUBLIC:    result = "public";    break;
    case CProtection::PROT_PRIVATE:   result = "private";   break;
    case CProtection::PROT_PROTECTED: result = "protected"; break;
    case CProtection::PROT_NONE:      result = "";
    }
    return result;
  }

  Token *wrapper_pos () const {
    CTree *tree = _obj->Tree ();
    assert (tree);
    CTree *pos = 0;
    if (tree->NodeName() == CT_InitDeclarator::NodeId()) {
      // move the generated declaration in front of the original declaration
      CT_ObjDecl *objdecl = ((CT_InitDeclarator*)tree)->ObjDecl ();
      // handle extern "C" declarations
      pos = linkage_adjust (objdecl);
    }
    if (tree->NodeName() == CT_FctDef::NodeId()) {
      pos = linkage_adjust((CT_FctDef*)tree);
    }
    assert (pos);
    return pos->token ();
  }

  Token *error_pos () const {
    return _obj->Tree()->token();
  }

  Token *fct_body_start () const {
    CTree *tree = _obj->Tree ();
    assert (tree && tree->NodeName() == CT_FctDef::NodeId());
    return ((CT_FctDef*)tree)->Body()->token ();
  }

  Token *fct_body_end () const {
    CTree *tree = _obj->Tree ();
    assert (tree && tree->NodeName() == CT_FctDef::NodeId());
    return ((CT_FctDef*)tree)->Body()->end_token ();
  }

  bool ends_with_return () const {
    assert (_obj->FunctionInfo() && _obj->Tree ()->NodeName () == CT_FctDef::NodeId());
    CT_FctDef *fctdef = (CT_FctDef*)_obj->Tree ();
    CTree *last_stmt = 0;
    int entries = fctdef->Body()->Entries();
    if (entries > 0)
      last_stmt = fctdef->Body()->Entry (entries - 1);
    return (last_stmt && last_stmt->NodeName() == CT_ReturnStmt::NodeId());
  }

  // search the set of actually *used* arguments (needed only for constructor
  // wrapper functions).
  void search_used_args (set<string> &args, CTree *tree = 0) const;

  string result_type (const string &name = "") const {
    CFunctionInfo *func = _obj->FunctionInfo();
    assert(func);
    CTypeInfo *type = func->isConversion () ? func->ConversionType () :
                                              func->TypeInfo ()->BaseType ();
    ostringstream type_str;
    type->TypeText (type_str, name.c_str (), true, true);
    return type_str.str ();
  }

  unsigned int args () const {
    CFunctionInfo *func = _obj->FunctionInfo();
    assert(func);
    return func->Arguments();
  }

  string arg_name (unsigned int no) const {
    CFunctionInfo *func = _obj->FunctionInfo();
    assert(func);
    CArgumentInfo *arg = func->Argument(no);
    return arg->isAnonymous() ? "" : arg->Name ().c_str();
  }

  string arg_type (unsigned int no, const string &name = "") const {
    CFunctionInfo *func = _obj->FunctionInfo();
      assert(func);
    ostringstream type_str;
    CTypeInfo *type = func->Argument(no)->TypeInfo();
    type->TypeText(type_str, name.c_str (), true, true);
    return type_str.str ();
  }

  string qualified_scope () const {
    if (_obj->QualifiedScope())
      return _obj->QualifiedScope()->QualName();
    return "";
  }

  string syntactical_scope () const {
    return _obj->Scope()->QualName();
  }

  // TODO: still needed?
  CObjectInfo *object () const { return _obj; }
};

#endif // __SyntacticContext_h__


