/*
 * wrapconnection.cxx
 *
 * OpenH323 Wrapper Library
 *
 * Copyright (c) 2002-2005 InAccess Networks
 * Michalis Manousos <manousos@inaccessnetworks.com>
 * Dimitris Economou <decon@inaccessnetworks.com>
 *
 * This file is part of "H.323 support for ASTERISK"
 *
 * "H.323 support for ASTERISK" is free software;
 * you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version. 
 *
 * "H.323 support for ASTERISK" is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. 
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * $Id: wrapconnection.cxx,v 1.38.2.4 2005/01/18 16:45:49 manousos Exp $
 *
 */

/************************************************************************/

#include <ptlib.h>
#include <h323.h>
#include <h323pdu.h>
#include <mediafmt.h>
#include <lid.h>

#define cplusplus
#include "version.h"
#include "wrapper_misc.hxx"
#include "wrapper.hxx"
#include "wrapendpoint.hxx"
#include "wrapconnection.hxx"
#include "asteriskaudio.hxx"

using namespace std;

// External stuff used inside the wrapper library
extern start_logchan_cb		on_start_logical_channel; 
extern clear_con_cb			on_connection_cleared;
extern alert_con_cb			on_connection_alert;
extern h323_exception_cb	on_h323_exception;
extern init_con_cb			on_connection_init;

/********************************************
 * Class WrapH323Connection
 ********************************************/

/*****************************************************************************
 *
 */
WrapH323Connection::WrapH323Connection(WrapH323EndPoint & ep,
				   unsigned callReference)
: H323Connection(ep, callReference)
{
	ConnectionFastStart = ep.IsFastStartDisabled();
	ConnectionCleared = 0;

	WRAPTRACE(4, "WrapH323Connection created.");
	return;
}

/*****************************************************************************
 *
 */
WrapH323Connection::~WrapH323Connection()
{
	WRAPTRACE(4, "WrapH323Connection deleted.");
	return;
}

/*****************************************************************************
 * Set the capabilities of the connection.
 */
void
WrapH323Connection::SetLocalCapabilities(H323Capability *caps[])
{
	int i;

	if (caps != NULL) {
		localCapabilities.RemoveAll();
		i = 0;
		while ((H323Capability *)caps[i]) {
			localCapabilities.SetCapability(0, 0, caps[i]);
			WRAPTRACE(3, "Outgoing capability " << (caps[i])->GetFormatName());
			i++;
		}
	}
}

/*****************************************************************************
 * Set the callerID number/name of the connection (valid only for 
 * outgoing calls).
 */
void
WrapH323Connection::SetCallerID(char *callerID, char *callerIDname)
{
	if (callerIDname) {
		if (strlen(callerIDname) != 0) {
			PString  cidname = PString(callerIDname);
			localAliasNames.RemoveAll();
			SetLocalPartyName(callerIDname);
			WRAPTRACE(3, "Caller ID name on outgoing call " << cidname);
			WRAPTRACE(3, "LocalPartyName " << localPartyName);
			WRAPTRACE(3, "DestExtraCallInfo " << destExtraCallInfo);

			if (callerID) {
				if (strlen(callerID) != 0) {
					PString cid = PString(callerID);
					localAliasNames.AppendString(cid);
					WRAPTRACE(3, "Caller ID on outgoing call " << cid);
				}
			}
		} else {
			if (callerID) {
				if (strlen(callerID) != 0) {
					PString cid = PString(callerID);
					localAliasNames.RemoveAll();
					SetLocalPartyName(cid);
					WRAPTRACE(3, "Caller ID on outgoing call " << cid);
				}
			}
		}
	}
}

/*****************************************************************************
 *
 */
H323Connection::AnswerCallResponse
WrapH323Connection::OnAnswerCall(const PString & caller,
					const H323SignalPDU & setupPDU,
					H323SignalPDU & /*connectPDU*/)
{
	call_details_t cdet;
	int res = -1;
	PString tmp_str;
	PIPSocket::Address plocal, premote;
	PString sourceE164, destE164, rdnis;

	WRAPTRACE(2, "User " << caller << " is calling us...");

	if (Lock()) {
		// Fill the structure
		memset((char*)&cdet, 0, sizeof(cdet));
		cdet.call_reference = GetCallReference();
		strncpy(cdet.call_token, (const char*)GetCallToken(), 
						sizeof(cdet.call_token) - 1);
		strncpy(cdet.call_source_alias, (const char *)setupPDU.GetSourceAliases(), 
						sizeof(cdet.call_source_alias) - 1);
		strncpy(cdet.call_dest_alias, (const char *)setupPDU.GetDestinationAlias(),
						sizeof(cdet.call_dest_alias) - 1);
		if (setupPDU.GetSourceE164(sourceE164)) {
			strncpy(cdet.call_source_e164, (const char *)sourceE164,
						sizeof(cdet.call_source_e164) - 1);
		}
		if (setupPDU.GetDestinationE164(destE164)) {
			strncpy(cdet.call_dest_e164, (const char *)destE164,
						sizeof(cdet.call_dest_e164) - 1);
		}
		if (setupPDU.GetQ931().GetRedirectingNumber(rdnis)) {
			strncpy(cdet.call_rdnis, (const char *)rdnis,
						sizeof(cdet.call_rdnis) - 1);
		}
		strncpy(cdet.remote_app, (const char *)GetRemoteApplication(),
						sizeof(cdet.remote_app) - 1);
		if (GetSignallingChannel() != NULL) {
			(GetSignallingChannel()->GetRemoteAddress()).GetIpAddress(premote);
			(GetSignallingChannel()->GetLocalAddress()).GetIpAddress(plocal);
			snprintf(cdet.local_addr, sizeof(cdet.local_addr)-1, 
						"%hhu.%hhu.%hhu.%hhu", 
						plocal.Byte1(), plocal.Byte2(), plocal.Byte3(), plocal.Byte4());
			snprintf(cdet.remote_addr, sizeof(cdet.remote_addr)-1, 
						"%hhu.%hhu.%hhu.%hhu", 
						premote.Byte1(), premote.Byte2(), premote.Byte3(), premote.Byte4());
		}

		// DEBUG START
		WRAPTRACE(3, "Call reference: " << cdet.call_reference);
		WRAPTRACE(3, "Call token: " << cdet.call_token);
		WRAPTRACE(3, "Call source alias: "<< cdet.call_source_alias<<"("<<strlen(cdet.call_source_alias)<<")");
		WRAPTRACE(3, "Call dest alias: "<< cdet.call_dest_alias<<"("<<strlen(cdet.call_dest_alias)<<")");
		WRAPTRACE(3, "Call source e164: "<< cdet.call_source_e164<<"("<<strlen(cdet.call_source_e164)<<")");
		WRAPTRACE(3, "Call dest e164: "<< cdet.call_dest_e164<<"("<<strlen(cdet.call_dest_e164)<<")");
		WRAPTRACE(3, "Call RDNIS: "<< cdet.call_rdnis<<"("<<strlen(cdet.call_rdnis)<<")");
		WRAPTRACE(3, "Remote Party number: "<<GetRemotePartyNumber());
		WRAPTRACE(3, "Remote Party name: "<<GetRemotePartyName());
		WRAPTRACE(3, "Remote Party address: "<<GetRemotePartyAddress());
		WRAPTRACE(3, "Remote Application: "<< cdet.remote_app<<"("<<strlen(cdet.remote_app)<<")");
		// DEBUG END

		// Invoke the application callback
		if (on_connection_init != NULL)
			res = on_connection_init(cdet);
		else {
			cout << "H.323 WARNING: No call initiation handling!" << endl;
			res = -1;
		}
		Unlock();
	} else {
		WRAPTRACE(1, "Failed to lock connection.");
		return H323Connection::AnswerCallDenied;
	}

	if (res < 0) {
		WRAPTRACE(2, "Failed to initialize incoming H.323 call. Dropping it.");
		return H323Connection::AnswerCallDenied;
	}

#if 0
	if (ConnectionFastStart) {
		// Send CALL_PROCEEDING with media open.
		return H323Connection::AnswerCallDeferredWithMedia;
	} else {
		return H323Connection::AnswerCallPending;
	}
	//return H323Connection::AnswerCallDeferred;
#endif
	return H323Connection::AnswerCallDeferredWithMedia;
}

/*****************************************************************************
 *
 */
BOOL 
WrapH323Connection::OnAlerting(const H323SignalPDU & /*alertingPDU*/,
		const PString & username)
{
	call_details_t cdet;

	WRAPTRACE(2, "Ringing phone for \"" << username << "\" ...");

	if (Lock()) {
		// Invoke the application callback
		cdet.call_reference = GetCallReference();
		strncpy(cdet.call_token, (const char*)GetCallToken(), 
						sizeof(cdet.call_token) - 1);
		if (on_h323_exception) {
			on_h323_exception(cdet, OH323EXC_CALL_ALERTED, NULL);
		} else {
			cout << "H.323 WARNING: No alert handling!" << endl;
			Unlock();
			return FALSE;
		}
		Unlock();
	} else {
		WRAPTRACE(1, "Failed to lock connection.");
		return FALSE;
	}

	return TRUE;
}

/*****************************************************************************
 *
 */
BOOL 
WrapH323Connection::OnReceivedSignalSetup(const H323SignalPDU & setupPDU)
{
	// Upon receiption of a SETUP message, we must respond soon
	// enough, because the remote endpoint has fired a 4-second (or greater)
	// timer waiting for an initial reply (not necessarily a CONNECT).
	// After the receiption of this initial reply, the remote endpoint
	// will wait additionally another 180 seconds (at least) for a
	// final reply (like CONNECT or RELEASE COMPLETE).
	WRAPTRACE(2, "Received SETUP message...");

	sourceAliases = setupPDU.GetSourceAliases();
	destAliases = setupPDU.GetDestinationAlias();
	sourceE164 = "";
	setupPDU.GetSourceE164(sourceE164);
	destE164 = "";
	setupPDU.GetDestinationE164(destE164);
	rdnis = "";
	setupPDU.GetQ931().GetRedirectingNumber(rdnis);

	return H323Connection::OnReceivedSignalSetup(setupPDU);
}

/*****************************************************************************
 *
 */
BOOL 
WrapH323Connection::OnSendSignalSetup(H323SignalPDU & setupPDU)
{
	WRAPTRACE(2, "Sending SETUP message...");

	if (localAliasNames.GetSize() > 0) {
		WRAPTRACE(3, "Setting display name " << localAliasNames[0]);
		setupPDU.GetQ931().SetDisplayName(localAliasNames[0]);
		if (localAliasNames.GetSize() > 1) {
			WRAPTRACE(3, "Setting calling party number " << localAliasNames[1]);
			setupPDU.GetQ931().SetCallingPartyNumber(localAliasNames[1]);
		}
	}

	/* XXX Set 'Called Party Number' from DNID field */

	sourceAliases = setupPDU.GetSourceAliases();
	destAliases = setupPDU.GetDestinationAlias();
	sourceE164 = "";
	setupPDU.GetSourceE164(sourceE164);
	destE164 = "";
	setupPDU.GetDestinationE164(destE164);
	rdnis = "";

	return H323Connection::OnSendSignalSetup(setupPDU);
}

/*****************************************************************************
 *
 */
void
WrapH323Connection::OnReceivedReleaseComplete(const H323SignalPDU & pdu)
{
	WRAPTRACE(2, "Received RELEASE COMPLETE message [" << GetCallToken() << "]");
	return H323Connection::OnReceivedReleaseComplete(pdu);
}

/*****************************************************************************
 *
 */
void
WrapH323Connection::OnEstablished()
{
	WRAPTRACE(3, "WrapH323Connection [" << GetCallToken() << "] established ("
		<< fastStartState << "/" 
		<< (h245Tunneling ? "H245Tunneling":"noH245Tunneling") << ")");

	H323Connection::OnEstablished();
}

/*****************************************************************************
 *
 */
BOOL
WrapH323Connection::OnClosingLogicalChannel(H323Channel & channel)
{
	WRAPTRACE(2, "Closing logical channel [" << GetCallToken() << "]");
//	if (channel.GetCodec() != NULL) {
//		(channel.GetCodec())->CloseRawDataChannel();
//	}    

	return H323Connection::OnClosingLogicalChannel(channel);
}

/*****************************************************************************
 *
 */
BOOL
WrapH323Connection::OnReceivedCapabilitySet(
		const H323Capabilities & remoteCaps,
		const H245_MultiplexCapability * muxCap,
		H245_TerminalCapabilitySetReject & reject)
{
	BOOL res;
	H323Capability *commonCapability = NULL;
	H323Capability *capability;
	H323Capabilities caps;

	if (!Lock()) {
		WRAPTRACE(1, "Failed to lock connection.");
		return FALSE;
	}
	WRAPTRACE(3, "Connection [" << GetCallToken() << "]"
			" received remote capabilities:\n" << remoteCaps <<
			"local capabilities are:\n" << localCapabilities);
	// We select the codec to use according to the remote's 
	// capabilities set (for now).
	for (PINDEX i = 0; i < remoteCaps.GetSize(); i++) {
		H323Capability & remoteCapability = remoteCaps[i];
		commonCapability = localCapabilities.FindCapability(remoteCapability);
		if (commonCapability != NULL) {
			WRAPTRACE(2, "Selecting " << *commonCapability);
			break;
		}
	}
	if (commonCapability == NULL) {
		WRAPTRACE(1, "Connection [" << GetCallToken() << "]"
				", no common codecs found.");
	} else {
		caps.RemoveAll();
		capability = caps.Copy(*commonCapability);
		caps.SetCapability(0, 0, capability);
		localCapabilities.RemoveAll();
		capability = localCapabilities.Copy(*capability);
		localCapabilities.SetCapability(0, 0, capability);
		remoteCapabilities.RemoveAll();
		capability = remoteCapabilities.Copy(*capability);
		remoteCapabilities.SetCapability(0, 0, capability);
	}
	//res = H323Connection::OnReceivedCapabilitySet(remoteCaps, muxCap, reject);
	res = H323Connection::OnReceivedCapabilitySet(caps, muxCap, reject);
	WRAPTRACE(3, "Connection [" << GetCallToken() << "]"
			" forced remote capabilities:\n" << remoteCapabilities <<
			"forced local capabilities are:\n" << localCapabilities);
	Unlock();
	return(res);
}

/*****************************************************************************
 *
 */
BOOL
WrapH323Connection::OnReceivedFacility(const H323SignalPDU & pdu)
{
	WRAPTRACE(2, "Received FACILITY message [" << GetCallToken() << "]");
	return H323Connection::OnReceivedFacility(pdu);
}

/*****************************************************************************
 *
 */
BOOL
WrapH323Connection::OnControlProtocolError(ControlProtocolErrors errorSource,
				const void *errorData)
{
	int res;
	call_details_t cdet;

	cout << "*** [" << GetCallToken() << "] H.323 CONTROL PROTOCOL ERROR ";
	switch (errorSource) {
		case H323Connection::e_MasterSlaveDetermination:
			cout << "(Master-Slave Determination";
			break;
		case H323Connection::e_CapabilityExchange:
			cout << "(Capability Exchange";
			break;
		case H323Connection::e_LogicalChannel:
			cout << "(Logical Channel";
			break;
		case H323Connection::e_ModeRequest:
			cout << "(Mode Request";
			break;
		case H323Connection::e_RoundTripDelay:
			cout << "(Roundtrip Delay";
			break;
		default:
			cout << "(Unknown";
			break;
	}
	if (errorData)
		cout << " : " << (char *)errorData << ")" << endl;
	else
		cout << ")" << endl;

	// Special case for round trip delay errors
	if ((errorSource == H323Connection::e_RoundTripDelay) &&
		(endpoint.ShouldClearCallOnRoundTripFail() == FALSE)) 
		return TRUE;

	// Invoke the application callback
	if (on_h323_exception) {
		// Fill the structure
		cdet.call_reference = GetCallReference();
		strncpy(cdet.call_token, (const char*)GetCallToken(), 
					sizeof(cdet.call_token) - 1);
		res = on_h323_exception(cdet, OH323EXC_CTRL_ERROR, NULL);
	} else {
		cout << "H.323 WARNING: No exception handling!" << endl;
		res = -1;
	}

	if (res < 0)
		return FALSE;
	return TRUE;
}

// End of file //////////////////////////////////////////////////////////////
