/* menufunc.c: menu functions
 *
 * Copyright (C) 1995-97 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *               1996-97 Michael Schlueter <schlue00@marvin.informatik.uni-dortmund.de>
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */

/* $Id: menufuncs.c,v 1.29 1998/09/19 01:29:00 rnhodek Exp $
 *
 * $Log: menufuncs.c,v $
 * Revision 1.29  1998/09/19 01:29:00  rnhodek
 * New clear_partitions() function.
 *
 * Revision 1.28  1998/02/07 21:22:59  rnhodek
 * Switch from xfmt_unknown to xfmt_AHDI if an extended partition has
 * been created (various places).
 * Handle case xfmt_unknown in select_xfmt() (no internal error...)
 * Tell the user which format is current after action in select_xfmt().
 * new_partition() (non-expert) made seriously wrong decisions if
 * xpart_fmt was xfmt_unknown... fixed and rewritten.
 *
 * Revision 1.27  1997/08/22 15:36:15  rnhodek
 * Implemented basic support for ICD format. Should work, but conversion
 * AHDI<->ICD isn't very clever yet.
 *
 * Revision 1.26  1997/08/22 13:11:13  rnhodek
 * Forgot some tests for IS_EXTENDED in change_partition_size()
 *
 * Revision 1.25  1997/07/10 23:02:27  rnhodek
 * Additional newline after progress report in move_part() necessary.
 *
 * Revision 1.24  1997/07/10 22:56:11  rnhodek
 * Grbml... fix typos.
 *
 * Revision 1.23  1997/07/10 22:55:18  rnhodek
 * In write_table(), add check for correct ars locations.
 *
 * Revision 1.22  1997/07/10 22:31:36  rnhodek
 * Add progress report while moving partition.
 * Fix calculation of max. sector for moving to.
 *
 * Revision 1.21  1997/07/10 21:50:06  rnhodek
 * Change base value of new start sector in change_part_size() so that a
 * new size can be entered easily.
 *
 * Revision 1.20  1997/07/10 20:48:02  rnhodek
 * Terminate new partition ID string in change_partid() so that strcmp
 * works.
 *
 * Revision 1.19  1997/07/10 20:36:36  rnhodek
 * New menu function move_partition_ars().
 * Fixed some minor bugs.
 *
 * Revision 1.18  1997/07/10 19:54:27  rnhodek
 * Changed sector limits on 'GEM' partitions.
 *
 * Revision 1.17  1997/07/10 15:10:34  rnhodek
 * Improved aux. rootsector handling of change_partition_size()
 *
 * Revision 1.16  1997/07/10 14:46:49  rnhodek
 * Fixed bug if no free sectors before possible aux. rs
 *
 * Revision 1.15  1997/07/08 19:00:44  rnhodek
 * can quit from bad_sector_list() now
 *
 * Revision 1.14  1997/07/08 18:59:03  rnhodek
 * Ouch, too many changes without commits in between...
 * Implemented moving of partitions
 * Implemented checking of "extended start condition" (great changes in
 *   new_partition())
 * Some more tests in verify()
 *
 * Revision 1.13  1997/06/22 10:30:55  rnhodek
 * Add __attribute__((unused)) to cvid
 *
 * Revision 1.12  1997/06/21 20:51:30  rnhodek
 * Fixed verify stuff
 *
 * Revision 1.11  1997/06/21 20:47:45  rnhodek
 * Added RCS keywords
 *
 * Revision 1.10  1997/06/21 20:25:53  rnhodek
 * Add RCS keyword headers
 * 
 * Revision 1.9  1997/06/21 20:21:41  rnhodek
 * Extend verify: Warn if empty bad sector list (HDX), show unallocated
 * sectors
 * 
 * Revision 1.8  1997/06/21 20:13:57  rnhodek
 * Show if bsl is HDX compat. if editing
 * 
 * Revision 1.7  1997/06/13 12:51:22  rnhodek
 * Bug fixes and improvements to new_partition; little fixes in verify
 * 
 * Revision 1.6  1997/06/13 10:18:45  rnhodek
 * Implemented settings bootflags on XGM part.
 * allow write command if opened read-only and ALPHA_VERSION; actual writing
 *   skipped in swrite
 * nicer output of list_table
 * 
 * Revision 1.5  1997/06/12 14:11:20  rnhodek
 * Fix minor things again; correct wrong calculations in new_partition
 * 
 * Revision 1.4  1997/06/12 13:43:38  rnhodek
 * Fix many small bugs here and there. The ones you find when first running a
 * program...
 * 
 * Revision 1.3  1997/06/11 19:49:15  rnhodek
 * Implemented bad sector list handling
 * 
 * Revision 1.2  1997/06/11 14:41:58  rnhodek
 * Added 'b' and 'f' menu items
 * 
 * Revision 1.1  1997/06/11 14:36:36  rnhodek
 * Initial revision
 * 
 * Revision 1.1.1.1  1997/06/11 14:36:36  rnhodek
 * Started using CVS for atafdisk
 *
 */

#ifndef lint
static char vcid[] __attribute__ ((unused)) =
"$Id: menufuncs.c,v 1.29 1998/09/19 01:29:00 rnhodek Exp $";
#endif /* lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <limits.h>

#include "fdisk.h"
#include "partable.h"
#include "writepart.h"
#include "input.h"
#include "disk.h"
#include "util.h"


void set_bootable( void )
{
    int i, j, n_ext, dflt, max = n_bootflags;
    int *pflag;

    if (!partitions) {
	printf( "No partitions!\n" );
	return;
    }
    table_infos( NULL, &n_ext, NULL, NULL );
    if (n_ext > 0) {
	i = read_int( 0, partitions, partitions, ignore,
		      "Partition number (0 = XGM)" );
    }
    else {
	i = (partitions == 1) ? 1 : read_int( 1, partitions, partitions,
					      ignore, "Partition number" );
    }
    --i;
    pflag = (i < 0) ? &XGM_flag : &(part_table[i].flag);
    
    printf( "\nCurrent boot flag is: " );
    if (*pflag) {
	printf( "0x%02x %s\n", *pflag, atari_boot_type( *pflag) );
    }
    else {
	printf( "not bootable\n" );
    }
    
    if (i >= 0 && IS_EXTENDED(part_table[i]))
	printf( "\nWarning: Bootflags usually not supported on extended "
		"partitions!\n" );
    
    dflt = 0;
    printf( "\nPossible boot flags are:\n" );
    printf( "  0: not bootable\n" );
    for( j = 0; j < max; ++j ) {
	printf( "  %d: 0x%02x %s\n",
		j+1, bootflags[j].flag, bootflags[j].name );
	if (bootflags[j].flag == *pflag)
	    dflt = j+1;
    }
    j = read_int( 0, dflt, max, deflt, "New flag");
    *pflag = j > 0 ? bootflags[j-1].flag : 0;
}

void bad_sector_list( void )
{
    
    if (bsl_size == 0) {
	printf( "There is no bad sector list currently (start=%lu size=0)\n",
		bsl_start );
    }
    else {
	printf( "The bad sector list resides from %lu to %lu (size %lu)%s\n",
		bsl_start, bsl_start+bsl_size-1, bsl_size,
		bsl_HDX_compat ? " (HDX compatible)" : "" );
    }

    printf( "  0: quit\n" );
    printf( "  1: remove bad sector list\n" );
    printf( "  2: define bad sector list\n" );
    printf( "  3: create HDX compatible empty bad sector list\n" );
    printf( "     (Note: Only necessary if you want to use also HDX on this disk)\n" );
    
    switch( read_int( 0, 0, 3, ignore, "Action") ) {

      case 0:
	break;
	
      case 1:
	bsl_start = bsl_size = 0;
	bsl_HDX_compat = 0;
	break;
	
      case 2:
      { unsigned long old_start, old_size;
	unsigned long start, def_start, end, max_size;

	/* free before new allocation */
	old_start = bsl_start;
	old_size  = bsl_size;
	bsl_start = bsl_size = 0;

	def_start = 1 + allocated_sectors_at( 1 );
	if (def_start >= hd_size) {
	    printf( "There are no unallocated sectors!\n" );
	    bsl_start = old_start;
	    bsl_size  = old_size;
	    return;
	}
	for(;;) {
	    start = read_int( 1, def_start, hd_size-1, deflt,
			      "First sector" );
	    if ((max_size = free_sectors_at( start )) > 0)
		break;
	    printf( "Sector %lu is not free!\n", start );
	}
	end = read_int( start, start+1, start+max_size-1, lower,
			"Last sector or +size[MK]" );
	bsl_size = end+1 - start;
	bsl_start = start;
	bsl_HDX_compat = 0;
	break;
      }
	
      case 3:
      { unsigned long old_start, old_size;

        old_start = bsl_start;
	old_size  = bsl_size;
	bsl_start = bsl_size = 0;

	/* use first unallocated sector */
	bsl_start = 1 + allocated_sectors_at( 1 );
	if (bsl_start >= hd_size) {
	    printf( "There are no unallocated sectors!\n" );
	    bsl_start = old_start;
	    bsl_size  = old_size;
	    return;
	}
	bsl_size = 1;
	bsl_HDX_compat = 1;
	break;
      }
    }
}

void change_partition_size( int i )
{
    long start, end, roots, free_before, free_after, new_size;
    int ars_included = 0, ars_immed_before;
    PARTITION *p = &part_table[i];

    if (!expert_mode && p->contents_valid) {
	if (!read_yesno( "This will make the partition contents invalid. "
			 "Continue", 0 ))
	    return;
    }
    
    start = p->start;
    end   = start + p->size;
    roots = p->rootsec;
    ars_immed_before = IS_EXTENDED(*p) && roots == start-1;
    
    free_before = free_sectors_before( start );
    /* if free area is limited by this partition's ars, search again */
    if (IS_EXTENDED(*p) && start - free_before - 1 == roots) {
	free_before += 1 + free_sectors_before( roots );
	ars_included = 1;
    }

    free_after = free_sectors_at( end );

    start = read_int( start-(free_before-ars_included),	/* lower   */
		      start,				/* default */
		      end-1,				/* max     */
		      deflt,				/* relto   */
		      "New first sector or [+-]size[MK]" );

    end = read_int( start,				/* lower   */
		    end - 1,				/* default */
		    end + free_after - 1,		/* max     */
		    lower,				/* relto   */
		    "New last sector or [+-]size[MK]" ) + 1;

    new_size = end - start;
    p->start = start;
    p->size  = new_size;

    if (ars_immed_before)
	/* there is always 1 sector free before start, because if
	 * ars_immed_before is set, ars_included is true also, and we allowed
	 * only one sector less than free_before! */
	p->rootsec = p->start-1;
    else if (ars_included && roots >= p->start) {
	/* there is a sector free for the ars, see above;
	 * but if the ars wasn't immediately before the data, we move it to
	 * there only if necessary, i.e. if it would be inside the data now */
	p->rootsec = p->start-1;
	if (expert_mode)
	    printf( "Auxiliary root sector now at %lu\n", p->rootsec );
    }
    
    printf( "New size: %lu sectors = %s MB\n",
	    new_size, round_mb( new_size ) );
    p->contents_valid = 0;
}

void clear_partitions( void ) {
    partitions = 0;
}

void delete_partition( int i )
{
    if (!check_ext_start_condition( DEL, i, 0 )) {
	printf( "Deleting this partition would cause the aux. rootsector "
		"of the\n"
		"then-first extended partition isn't at the start "
		"of the XGM area.\n" );
	return;
    }
    
    memmove( &part_table[i], &part_table[i+1],
	     (partitions-1-i)*sizeof(PARTITION) );
    partitions--;
}

void convert_partition( int i )
{
    int n_prim, n_ext, first_ext, last_ext;
    PARTITION *p = &part_table[i];
    int is_ext = IS_EXTENDED(*p);

    if (xpart_fmt == xfmt_ICD) {
	printf( "No extended partitions with ICD format -- cannot convert\n" );
	return;
    }
    
    /* can convert only partitions that are at the "edge" of the XGM area;
     * furthermore, the very first partition must be primary, as the XGM may
     * not reside in the first slot */

    table_infos( &n_prim, &n_ext, &first_ext, &last_ext );

    if (is_ext) {
	/* convert extended -> primary */
	if (n_prim + (n_ext > 1) >= 4) {
	    printf( "No free slots for more primary partitions\n" );
	    return;
	}
	if (!(i == first_ext || i == last_ext)) {
	    printf( "Can only convert partitions at the edge of the "
		    "extended area\n" );
	    return;
	}
	if (!check_ext_start_condition( DEL, i, 0 )) {
	    printf( "This conversion would cause that the aux. rootsector "
		    "of the\n"
		    "then-first extended partition isn't at the start "
		    "of the XGM area.\n" );
	    return;
	}

	if (p->rootsec == p->start - 1 && !p->contents_valid) {
	    p->start--;
	    p->size++;
	}
	else
	    printf( "Cannot integrate aux. root sector into partition. "
		    "It's unassigned now.\n" );
	p->rootsec = 0;
    }
    else {
	/* convert primary -> extended */
	if (i == 0) {
	    printf( "First partition must always be primary, "
		    "cannot convert\n" );
	    return;
	}
	if (n_ext > 0 && !(i == first_ext-1 || i == last_ext+1)) {
	    printf( "Can only convert partitions next to the existing "
		    "extended area\n" );
	    return;
	}
	if (!check_ext_start_condition( ADD, i, p->start )) {
	    printf( "This conversion would cause that the aux. rootsector "
		    "of the\n"
		    "first extended partition isn't at the start "
		    "of the XGM area.\n" );
	    return;
	}

	if (!expert_mode && p->contents_valid) {
	    if (!read_yesno( "The conversion will make the partition "
			     "contents invalid!\nContinue", 0 ))
		return;
	}

	p->rootsec = p->start;
	p->start++;
	p->size--;
	/* now have a extended part, this must be xfmt_AHDI */
	if (xpart_fmt == xfmt_unknown)
	    xpart_fmt = xfmt_AHDI;
    }
}

void select_xfmt( void )
{
    int n_prim, n_ext;
    
    table_infos( &n_prim, &n_ext, NULL, NULL );

    /* XXX: later I can also implement automatic ext<->prim conversions to
     * remove the restrictions below */
    
    switch( xpart_fmt ) {

      case xfmt_unknown:
	printf( "Extended partition format to use wasn't known yet (both "
		"equal until now)\n"
		"Selecting AHDI format.\n" );
	xpart_fmt = xfmt_AHDI;
	break;
	
      case xfmt_AHDI:
	if (n_ext > 0) {
	    printf( "There are extended partitions -- "
		    "can't convert to ICD format\n" );
	    return;
	}
	printf( "Extended partition format set to ICD\n" );
	xpart_fmt = xfmt_ICD;
	break;

      case xfmt_ICD:
	if (n_prim > MAX_PRIMARY_AHDI) {
	    printf( "Too many primary partitions (%d) for AHDI format --\n"
		    "cannot convert to AHDI format\n", n_prim );
	    return;
	}
	printf( "Extended partition format set to AHDI\n" );
	xpart_fmt = xfmt_AHDI;
	break;
	
      default:
	printf( "Internal error: Bad extended partition format.\n" );
	return;
    }
}

void change_contents_flag( int i )
{
    PARTITION *p = &part_table[i];

    if (!p->contents_valid)
	p->contents_valid = 1;
    else {
	printf( "Partition %d's contents are currently marked valid\n", i+1 );
	if (!read_yesno( "Do you really want to mark them invalid", -1 ))
	    return;
	p->contents_valid = 0;
    }
    
    printf( "Partition %d's contents are now marked %svalid\n",
	    i+1, p->contents_valid ? "" : "in" );
}

void new_partition( void )
{
    PARTITION *p;
    unsigned long sec, n, start, def_start, min_start, max_start;
    unsigned long end, size, max_size;
    int header_printed, i, new_num;
    int n_prim, n_ext, first_ext, last_ext;
    int make_ext, prim_possible, ext_possible;
    char *why;
    
    if (partitions >= MAXIMUM_PARTS) {
	printf( "The maximum number of partitions has been created\n" );
	return;
    }
    if (xpart_fmt == xfmt_ICD && partitions >= MAX_PRIMARY_ICD) {
	printf( "The maximum number of partitions for ICD format "
		"has been created\n" );
	return;
    }

    /* Scan partition table for some data */
    table_infos( &n_prim, &n_ext, &first_ext, &last_ext );

    /* print list of free areas */
    header_printed = 0;
    def_start = min_start = max_start = 0;
    n = 0;
    for( sec = 1; sec < hd_size; sec += n ) {
	sec += allocated_sectors_at( sec );
	if (sec >= hd_size || !(n = free_sectors_at( sec )))
	    break;
	if (!header_printed) {
	    printf( "Free areas on disk are:\n" );
	    printf( "    Begin       End    Blocks     MB\n" );
	    header_printed = 1;
	}
	printf( "%9lu %9lu %9lu%c  %s\n",
		sec, sec+n-1, n/2, (n&1) ? '+' : ' ', round_mb(n) );
	if (!def_start) {
	    new_num = canonical_number( sec );
	    prim_possible = primary_possible( new_num, n_prim, n_ext,
					      first_ext, last_ext, NULL );
	    ext_possible = extended_possible( new_num, n_prim, n_ext,
					      first_ext, last_ext, sec, NULL );
	    if (prim_possible || ext_possible)
		def_start = sec;
	}
	if (!min_start)
	    min_start = sec;
	max_start = sec+n-1;
    }
    if (!min_start) {
	printf( "There is no unallocated space on the disk!\n" );
	return;
    }
    if (!def_start) {
	if (expert_mode)
	    def_start = min_start;
	else {
	    printf( "With the current layout, can't create any partitions "
		    "in non-expert mode!\n" );
	    return;
	}
    }

    /* Ask user for start of new partition */
    for(;;) {
	start = read_int( min_start, def_start, max_start, deflt,
			  "First sector" );
	if ((max_size = free_sectors_at( start )) > 0)
	    break;
	printf( "Sector %lu is not free!\n", start );
    }
    /* ...and for the end */
    end = read_int( start, start+max_size-1, start+max_size-1, lower,
		    "Last sector or +size[MK]" );
    size = end+1 - start;

    if (!expert_mode) {
	/* In non-expert mode, new partitions always have canonical numbering
	 * (same order as data on disk). If it's not possible to create the
	 * partition there (no primary slot free and not in or next to
	 * XGM area), the create simply fails. */
	new_num = canonical_number( start );
	printf( "New partition will be number %d\n", new_num+1 );

	prim_possible = primary_possible( new_num, n_prim, n_ext,
					  first_ext, last_ext, NULL );
	ext_possible = extended_possible( new_num, n_prim, n_ext,
					  first_ext, last_ext, start, NULL );

	if (!prim_possible && !ext_possible) {
	    printf( "\nNo more slots available in primary partition table, "
		    "or this partition \n"
		    "can't be an extended one, would break the XGM chain, "
		    "or cannot place the\n"
		    "auxiliary root sector.\n"
		    "Try converting some primary partition to extended, "
		    "deleting some other\n"
		    "partitions or use expert mode to solve this.\n" ); 
	    /* TODO: Maybe automatically convert prim->ext for a partition
	     * where the "contents valid" flag is cleared. */
	    return;
	}
	else if (!ext_possible)
	    make_ext = 0;
	else if (!primary_possible)
	    make_ext = 1;
	else
	    /* Create an extended partition if this is the 4th part. (to avoid
	     * prim->ext conversion later), or if already adjacent to XGM
	     * area. */
	    make_ext = ((n_prim == 3 && n_ext == 0) ||
			(n_ext > 0 &&
			 first_ext-1 <= new_num && new_num <= last_ext+1));
	if (make_ext &&
	    !check_ext_start_condition( ADD, new_num, start )) { 
	    /* shouldn't happen if partitions are ordered by disk location */
	    printf( "This partition will make the aux. root sector of "
		    "the first extended\n"
		    "partition not be the same as the start of the XGM "
		    "area, which\n"
		    "violates the partition format.\n" );
	    return;
	}
	if (partitions > 0)
	    memmove( &part_table[new_num+1], &part_table[new_num],
		     (partitions-new_num)*sizeof(PARTITION) );
	++partitions;
	p = &part_table[new_num];

	p->start = start;
	p->size  = size;
	p->rootsec = 0;

	/* Create an extended partition if this is the 4th part. (to avoid
	 * prim->ext conversion later), or if already adjacent to XGM area. */
	if (make_ext) {
	    p->rootsec = start;
	    p->start++;
	    p->size--;
	    if (xpart_fmt == xfmt_unknown)
		/* after created an extended partition, format is AHDI */
		xpart_fmt = xfmt_AHDI;
	}
    }
    else {
	/* In expert mode, we allow the user to give the partition an
	 * arbitrary number, choose its type (prim/ext), and to locate the aux
	 * root sector freely if extended. */
	char possible[MAXIMUM_PARTS];
	unsigned n_possible = 0, num_possible = 0;
	
	for( i = 0; i <= partitions; ++i ) {
	    possible[i] = 0;
	    if (primary_possible( i, n_prim, n_ext, first_ext,
				  last_ext, NULL ))
		possible[i] |= 1;
	    if (extended_possible( i, n_prim, n_ext, first_ext,
				   last_ext, start, NULL ))
		possible[i] |= 2;

	    if (possible[i]) {
		++n_possible;
		num_possible = i;
	    }
	}

	if (n_possible == 0) {
	    printf( "Sorry, this partition isn't possible (can't be "
		    "either primary or extended).\n" );
	    return;
	}
	else if (n_possible == 1) {
	    new_num = num_possible;
	    printf( "Can be only partition number %d\n",
		    new_num+1 );
	}
	else {
	    for(;;) {
		printf( "The following numbers are possible for this "
			"partition:\n" );
		for( i = 0; i <= partitions; ++i ) {
		    if (possible[i])
			printf( "  %d(%s%s%s)\n", i+1,
				possible[i] & 1 ? "p" : "",
				(possible[i] & 3) == 3 ? "," : "",
				possible[i] & 2 ? "e" : "" );
		}
		new_num = read_int( 1, partitions+1, partitions+1, ignore,
				    "Partition number" ) - 1;
		if (possible[new_num])
		    break;
		printf( "This number is not possible for this partition\n" );
	    }
	}

	if (!(possible[new_num] & 1)) {
	    primary_possible( new_num, n_prim, n_ext, first_ext,
			      last_ext, &why );
	    printf( "Cannot be primary (%s)\n", why );
	}
	
	if (!(possible[new_num] & 2)) {
	    extended_possible( new_num, n_prim, n_ext, first_ext,
			       last_ext, start, &why );
	    printf( "Cannot be extended (%s)\n", why );
	}

	if ((possible[new_num] & 3) == 3) {
	    make_ext = read_yesno( "Make an extended partition?", 0 );
	}
	else if (possible[new_num] & 1) {
	    printf( "This will be a primary partition\n" );
	    make_ext = 0;
	}
	else if (possible[new_num] & 2) {
	    printf( "This will be an extended partition\n" );
	    make_ext = 1;
	}
	else {
	    /* cannot happen, but initialize make_ext to avoid gcc warning */
	    make_ext = 0;
	}

	if (make_ext) {
	    /* aux. root sector must be between (inclusive) the start of the
	     * first extended partition and the start of data of this
	     * partition. If this will be the new first ext. part., the lower
	     * bound is 1. */
	    unsigned long lower_bound, upper_bound;

	    if (n_ext == 0 || new_num <= first_ext) {
		lower_bound = 1;
		upper_bound = min( start, lowest_ars()-1 );
	    }
	    else {
		lower_bound = lowest_ars()+1;
		upper_bound = start;
	    }
	    lower_bound += allocated_sectors_at( lower_bound );
	    upper_bound -= allocated_sectors_before( upper_bound+1 );
	    /* results in -1 if all sectors allocated before upper_bound */

	    if (lower_bound > upper_bound || upper_bound == UINT_MAX) {
		printf( "Oops, no space for the aux. root sector. This "
			"shouldn't happen.\n" );
		return;
	    }
	    if (lower_bound == upper_bound) {
		n = lower_bound;
		printf( "Auxiliary root sector must be at %lu\n", n );
	    }
	    else {
		for(;;) {
		    n = read_int( lower_bound, upper_bound, upper_bound, upper,
				  "Location for auxiliary root sector" );
		    if (n == 0)
			return;
		    if (!(n == start || free_sectors_at( n ) > 0)) {
			printf( "Sector %ld already allocated (and not "
				"beginning of new partition)\n", n );
			continue;
		    }
		    if (!check_ext_start_condition( ADD, new_num, n )) {
			/* shouldn't happen... */
			printf( "Then the aux. root sector of the first "
				"extended partition would be\n"
				"the first sector of the XGM area.\n" );
			continue;
		    }
		    break;
		}
	    }
	}

	if (partitions > 0)
	    memmove( &part_table[new_num+1], &part_table[new_num],
		     (partitions-new_num)*sizeof(PARTITION) );
	++partitions;
	p = &part_table[new_num];

	p->start = start;
	p->size  = size;
	p->rootsec = 0;

	if (make_ext) {
	    if (n == start) {
		p->start++;
		p->size--;
	    }
	    p->rootsec = n;
	    if (xpart_fmt == xfmt_unknown)
		/* after created an extended partition, format is AHDI */
		xpart_fmt = xfmt_AHDI;
	}
    }

    /* common settings for new partition */
    strcpy( p->id, "LNX" );
    p->flag = 0;
    p->contents_valid = 0;
}


#define MOVEBUF_SIZE	(128*SECTOR_SIZE)

void move_partition( int i )
{
    PARTITION *p = &part_table[i];
    unsigned long start, size, min_start, max_start, newstart;
    unsigned long free_secs, n, m, cnt;
    int ars_included = 0;
    char *buf;
    
#ifndef ALPHA_VERSION
    /* for ALPHA_VERSION, read-only devices are trapped later */
    if (type_open == O_RDONLY) {
	printf( "%s is opened read-only; can't move data\n", disk_device );
	return;
    }
#endif

    start = p->start;
    size  = p->size;
    if (IS_EXTENDED(*p) && p->rootsec == start - 1) {
	/* If is extended and aux. root sector is immediately before data
	 * sectors, move it with the part. */
	ars_included = 1;
	printf( "Will keep aux. root sector before partition data\n" );
    }

    min_start = 1;
    if (IS_EXTENDED(*p)) {
	if (!ars_included)
	    /* data must be after aux. rootsec */
	    min_start = p->rootsec+1;
	else {
	    int first_ext;
	    table_infos( NULL, NULL, &first_ext, NULL );
	    if (i != first_ext)
		/* cannot move non-first ext. part. before first ars */
		min_start = lowest_ars()+1;
	}
    }
    min_start += allocated_sectors_at( min_start ) + ars_included;
    max_start = hd_size - allocated_sectors_before( hd_size ) - size;
    
    newstart = read_int( min_start,
			 min_start <= start && start <= max_start ?
			     start : min_start,
			 max_start, deflt,
			 "New start sector or [+-]move[MK]" );
    if (start == newstart) {
	printf( "No move\n" );
	return;
    }

    /* temporarily set the size to 0, so the partition's sectors look free to
     * free_sectors_at() */
    p->size = 0;
    if (ars_included)
	p->rootsec = 0;
    free_secs = free_sectors_at( newstart-ars_included );
    p->size = size;
    if (ars_included)
	p->rootsec = start-1;
    
    if (free_secs < size+ars_included) {
	printf( "Not enough free space at %lu (%lu sectors, need %lu)\n",
		newstart-ars_included, free_secs, size+ars_included );
	return;
    }

    if (!check_ext_start_condition( CHANGE, i,
				    ars_included ? newstart-1 : p->rootsec )) {
	/* shouldn't happen */
	printf( "This would cause the aux. rootsector of the first "
		"extended partition\n"
		"not to be at the start of the XGM area anymore\n" );
	return;
    }

    printf( "\nTo move a partition, direct modifications on disk are "
	    "necessary.\n"
	    "If the moving process is somehow interrupted, probably both, "
	    "the source and\n"
	    "destination areas on disk will be unusable!\n" );
    if (!read_yesno( "Continue", -1 ))
	return;
#ifdef ALPHA_VERSION
    if (type_open != O_RDONLY &&
	!read_yesno( "This is an ALPHA version!! Really continue??", -1 ))
	return;
#endif
    
    if (!(buf = malloc( MOVEBUF_SIZE ))) {
	printf( "Out of memory for buffer!\n" );
	return;
    }

    printf( "Now moving sectors...\n" );
    if (newstart < start) {
	/* copy upwards */
	for( n = 0; n < size; n += cnt ) {
	    cnt = MOVEBUF_SIZE / SECTOR_SIZE;
	    if (cnt > size-n)
		cnt = size-n;
	    if (msread( buf, start+n, cnt ) < 0)
		goto fail;
	    if (mswrite( buf, newstart+n, cnt ) < 0)
		goto fail;
	    printf( "  %9lu/%9lu sectors moved\r", n+cnt, size );
	    fflush( stdout );
	}
    }
    else {
	/* copy downwards */
	for( n = 0; n < size; n += cnt ) {
	    cnt = MOVEBUF_SIZE / SECTOR_SIZE;
	    if (cnt > size-n)
		cnt = size-n;
	    if (msread( buf, start+size-n-cnt, cnt ) < 0)
		goto fail;
	    if (mswrite( buf, newstart+size-n-cnt, cnt ) < 0)
		goto fail;
	    printf( "  %9lu/%9lu sectors moved\r", n+cnt, size );
	    fflush( stdout );
	}
    }
    printf( "\nOK, done.\n" );
    free( buf );

    p->start = newstart;
    if (ars_included)
	p->rootsec = newstart - 1;
    /* partition contents (hopefully :-) still valid, don't change
     * p->contents_valid */

    return;

  fail:
    /* some error occured, try to restore overwritten contents of current
     * partition (n is the number of already copied sectors) */
    printf( "Sh*! Some error occured, trying to revert the changes "
	    "already done\n" );

    printf( "Moving sectors back...\n" );
    if (newstart < start) {
	/* restore downwards */
	for( m = 0; m < n; m += cnt ) {
	    cnt = MOVEBUF_SIZE / SECTOR_SIZE;
	    if (cnt > n-m)
		cnt = n-m;
	    if (msread( buf, start+n-m-cnt, cnt ) < 0)
		goto fail2;
	    if (mswrite( buf, newstart+n-m-cnt, cnt ) < 0)
		goto fail2;
	}
    }
    else {
	/* restore upwards */
	for( m = 0; m < n; m += cnt ) {
	    cnt = MOVEBUF_SIZE / SECTOR_SIZE;
	    if (cnt > m-n)
		cnt = m-n;
	    if (msread( buf, newstart+m, cnt ) < 0)
		goto fail2;
	    if (mswrite( buf, start+m, cnt ) < 0)
		goto fail2;
	}
    }
    printf( "OK, done. Old state should be restored\n" );
    free( buf );
    return;

  fail2:
    free( buf );
    printf( "Real sh* now :-( Another error while moving the sectors back.\n"
	    "The partition and the destination area are now "
	    "probably corrupted.\n" );
}

void move_partition_ars( int i )
{
    PARTITION *p = &part_table[i];
    unsigned long n, lower_bound, upper_bound;
    int j, n_ext, first_ext;
    
    if (!IS_EXTENDED(*p)) {
	printf( "Partition %d is not extended and thus has no aux. "
		"root sector\n", i+1 );
	return;
    }

    printf( "Aux. root sector currently at %lu\n", p->rootsec );
    table_infos( NULL, &n_ext, &first_ext, NULL );
    if (n_ext == 0 || i == first_ext) {
	lower_bound = 1;
	/* upper bound is minimum of all other aux. root sectors and the
	 * partition data start */
	upper_bound = p->start-1;
	for( j = 0; j < partitions; ++j ) {
	    if (j != i && IS_EXTENDED(part_table[j]))
		upper_bound = min( upper_bound, part_table[j].rootsec );
	}
    }
    else {
	lower_bound = lowest_ars()+1;
	upper_bound = p->start-1;
    }
    lower_bound += allocated_sectors_at( lower_bound );
    lower_bound = min( lower_bound, p->rootsec );
    upper_bound -= allocated_sectors_before( upper_bound+1 );
    /* results in -1 if all sectors allocated before upper_bound */
    upper_bound = max( upper_bound, p->rootsec );
    
    if (lower_bound > upper_bound || upper_bound == UINT_MAX) {
	printf( "No space for the aux. root sector.\n" );
	return;
    }
    if (lower_bound == upper_bound) {
	printf( "Auxiliary root sector must be at %lu\n", lower_bound );
	return;
    }

    for(;;) {
	n = read_int( lower_bound, p->rootsec, upper_bound, deflt,
		      "Location for auxiliary root sector" );
	if (n == 0)
	    return;
	if (n != p->rootsec && free_sectors_at( n ) == 0) {
	    printf( "Sector %lu not free\n", n );
	    continue;
	}
	if (!check_ext_start_condition( CHANGE, i, n )) {
	    /* shouldn't happen... */
	    printf( "Then the aux. root sector of the first "
		    "extended partition would be\n"
		    "the first sector of the XGM area.\n" );
	    continue;
	}
	break;
    }

    p->rootsec = n;
}

void list_table( void )
{
    PARTITION *p;
    PARTITION XGM_part;
    char *type, numbuf[4];
    int i, slotno = 0, xgm_printed = -1;
    int w = strlen(disk_device);

    if (expert_mode || XGM_flag) {
	XGM_boundary( &XGM_part.start, &XGM_part.size );
	XGM_part.flag = XGM_flag;
	XGM_part.rootsec = 0;
	XGM_part.contents_valid = 0;
	strcpy( XGM_part.id, "XGM" );
	xgm_printed = 0;
    }
    
    printf( "\nDisk %s: %ld * %d bytes, %ld%s kBytes, %s MBytes\n",
	    disk_device, hd_size, SECTOR_SIZE,
	    hd_size/2, (hd_size&1) ? "+" : "",
	    round_mb( hd_size ) );
    if (w < 5)
	w = 5;
    printf("%*s  Boot Val Rootsec.    Begin      End   Blocks  Id   System\n",
	   w + 1, "Device");
    for( i = 0 ; i < partitions; i++ ) {
	p = &part_table[i];
	if (IS_EXTENDED(*p) && xgm_printed == 0) {
	    p = &XGM_part;
	    xgm_printed = 1;
	    --i;
	}

	if (p == &XGM_part)
	    strcpy( numbuf, "x " );
	else
	    sprintf( numbuf, "%-2d", i+1 );
	
	printf( expert_mode && p->rootsec == 0 ?
	    "%*s%s %3s   %c (slot %d)0%9ld%9ld%9ld%c %3s  %s\n" :
	    "%*s%s %3s   %c %9ld%9ld%9ld%9ld%c %3s  %s\n",
/* device          */	w, disk_device, numbuf,
/* boot flag       */	p->flag ? atari_boot_type_short(p->flag) : "",
/* contents valid  */	p == &XGM_part ? '-' : p->contents_valid ? '*' : '-',
/* rootsec         */	!expert_mode || p->rootsec ? p->rootsec : slotno++,
/* begin           */	p->start,
/* end             */	p->start + p->size - 1,
/* blocks          */	p->size / 2,
/* odd flag on end */	(p->size & 1) ? '+' : ' ',
/* type id         */	p->id,
/* type name       */	(type = atari_partition_type(p->id)) ? type :
			    p == &XGM_part ? "" : "Unknown"
	    ); 
    }
}

void change_partid( int i )
{
    int cnr;
    PARTITION *p = &part_table[i];

    for(;;) {
	list_atari_types();
	printf( "ID is %s. Enter new ID (3 letters): ", p->id );
	read_line();
	cnr = strlen( line_ptr );
	if (1 <= cnr && cnr <= 4 && isalnum(line_ptr[0]) &&
	    isalnum(line_ptr[1]) && isalnum(line_ptr[2]) )
	    break;
	printf( "ID must be three letters or digits\n" );
    }
    
    memcpy( &p->id, line_ptr, 3 );
    p->id[3] = 0;
    printf( "ID changed to %s\n", p->id );

    if (!(isupper(line_ptr[0]) &&
	  isupper(line_ptr[1]) &&
	  isupper(line_ptr[2])))
	printf( "(Don't you want to use three uppercase letters, "
		"which is the standard?)\n" );
}

#if 0
void select_options( void )
{
    int i;
    int n = n_global_options;
    
    printf( "Current option settings:\n" );
    for( i = 0; i < n; ++i ) {
	printf( "  %d) %s: %s\n", i+1,
		global_options[i].name,
		*global_options[i].flag ? "on" : "off" );
	printf( "    %s\n", global_options[i].description );
    }

    while( (i = read_int( 0, 0, n, ignore, "Change option (0 = quit)" )) ) {
	*global_options[i-1].flag ^= 1;
	printf( "%s is now %s\n",
		global_options[i].name,
		*global_options[i].flag ? "on" : "off" );
    }
}

void parse_options( char *options )
{
    char *p;
    int i, value;
    
    for( p = strtok( options, "," ); p; p = strtok( NULL, "," ) ) {
	if (strncmp( p, "no-", 3 ) == 0) {
	    value = 0;
	    p += 3;
	}
	else
	    value = 1;

	for( i = 0; i < n_global_options; ++i ) {
	    if (strcmp( global_options[i].name, p ) == 0)
		break;
	}
	if (i == n_global_options) {
	    fprintf( stderr, "Unrecognized option %s\n", p );
	    continue;
	}

	*global_options[i].flag = value;
    }
}
#endif

/* Do some consistency checks */
void verify(void)
{
    int i, j, k = 0, header_printed, first_ext;
    unsigned errors = 0, warnings = 0, notes = 0;
    unsigned long start1, start2, end1, end2, rs, sec, n, free;
    unsigned long xgm_start, xgm_size, xgm_end;
    
    /* are all partitions mutually disjoint? (and disjoint with the bsl?) */
    /* do they start past zero and before end-of-disk? */
    for( i = 0; i < partitions; ++i ) {
	start1 = part_table[i].start;
	end1   = start1 + part_table[i].size;

	if (start1 < 1) {
	    printf( "Error: Partition %d starts before sector 1 (at %lu)\n",
		    i+1, start1 );
	    ++errors;
	}
	if (end1 > hd_size) {
	    printf( "Error: Partition %d ends after end of disk "
		    "(at %lu > %lu)\n", i+1, end1-1, hd_size-1 );
	    ++errors;
	}

	for( j = i+1; j < partitions; ++j ) {
	    start2 = part_table[j].start;
	    end2   = start2 + part_table[j].size;
	    if (end1 > start2 && end2 > start1) {
		printf( "Error: Partitions %d and %d overlap!\n", i+1, j+1 );
		++errors;
	    }
	}

	if (end1 > bsl_start && bsl_start+bsl_size > start1) {
	    printf( "Error: Partition %d overlaps with "
		    "the bad sector list!\n", i+1 );
	    ++errors;
	}
    }

    /* are all aux. rootsectors not contained in another partition or in the
     * bsl? And auxiliary root sectors must be before their data partitions */
    for( i = 0; i < partitions; ++i ) {
	rs = part_table[i].rootsec;
	if (!rs) continue;
	
	if (rs >= hd_size) {
	    printf( "Error: Partition %d has aux. root sector after end of "
		    "disk (at %lu > %lu)\n", i+1, rs, hd_size-1 );
	    ++errors;
	}

	if (rs >= part_table[i].start) {
	    printf( "Error: Partition %d's aux. root sector (%lu) is after "
		    "the partition start (%lu)\n",
		    i+1, rs, part_table[i].start );
	    ++errors;
	}
	if (rs != part_table[i].start-1) {
	    printf( "Note: Partition %d's aux. root sector (%lu) isn't "
		    "immediately before\n"
		    "  the partition data (%lu)\n",
		    i+1, rs, part_table[i].start );
	    ++notes;
	}
	
	for( j = 0; j < partitions; ++j ) {
	    if (i == j) continue;
	    if (inside_part( &part_table[j], rs )) {
		printf( "Error: Partition %d's aux. root sector is inside "
			"partition %d!\n", i+1, j+1 );
		++errors;
	    }
	}

	if (bsl_start <= rs && rs < bsl_start+bsl_size) {
	    printf( "Error: Partition %d's aux. root sector is inside "
		    "the bad sector list\n", i+1 );
	    ++errors;
	}
    }
    
    /* first primary slot may not be XGM */
    if (partitions > 0 && IS_EXTENDED(part_table[0])) {
	printf( "Error: First partition may not be extended "
		"(first primary slot must not be XGM)\n" );
	++errors;
    }
    
    /* only one XGM chain */
    first_ext = -1;
    for( i = j = 0; i < partitions; ++i ) {
	if (IS_EXTENDED(part_table[i])) {
	    if (i == 0 || !IS_EXTENDED(part_table[i-1]))
		++j;
	    k = i;
	    if (first_ext < 0)
		first_ext = i;
	}
    }
    if (xpart_fmt == xfmt_ICD && j > 0) {
	printf( "Error: ICD format and extended partitions found (??)\n" );
	++errors;
    }
    else if (j > 1) {
	printf( "Error: More than 1 (%d) XGM chains\n", j );
	++errors;
    }

    /* better no primary partitions after XGM chain */
    if (j > 0 && k != partitions-1) {
	printf( "Note: There are primary partitions after the XGM chain "
		"(unusual but ok).\n"
		"  But maybe some weird TOS hd drivers can have problems "
		"with this.\n" );
	++notes;
    }

    if (j > 0) {
	/* does the XGM start past zero and before end-of-disk? */
	XGM_boundary( &xgm_start, &xgm_size );
	xgm_end = xgm_start + xgm_size;
	
	if (xgm_start < 1) {
	    printf( "Error: XGM partition starts before sector 1 (at %lu)\n",
		    xgm_start );
	    ++errors;
	}	
	if (xgm_end > hd_size) {
	    printf( "Error: XGM partition ends after end of disk "
		    "(at %lu > %lu)\n", xgm_end-1, hd_size-1 );
	    ++errors;
	}

	/* is the aux. rs of the first ext. part. also the start of the XGM
         * area? */
	if (part_table[first_ext].rootsec != xgm_start) {
	    printf( "Error: Aux. rootsector of first extended partition "
		    "(%lu) must be\n"
		    "first sector of XGM area (%lu).\n",
		    part_table[first_ext].rootsec, xgm_start );
	    ++errors;
	}
	
	/* are the extended partitions ordered, and are primary partitions
	 * contained inside the XGM area? */
	/* are all extended data area after their aux. rs? */
	end2 = 0; /* last end of extd */
	j = 1; /* are ordered */
	for( i = 0; i < partitions; ++i ) {
	    start1 = part_table[i].start;
	    end1   = start1 + part_table[i].size;
	    if (IS_EXTENDED(part_table[i])) {
		if (start1 < end2)
		    j = 0;
		else
		    end2 = end1;
	    }
	    else {
		if (xgm_end > start1 && end1 > xgm_start) {
		    printf( "Note: Primary partition %d inside XGM area!\n",
			    i+1 );
		    ++notes;
		}
	    }
	}
	if (!j) {
	    printf( "Note: Extended partitions are not ordered by disk "
		    "location\n" );
	    ++notes;
	}
    }

    /* check for invalid boot flags (more than one bit, flags set on extended
     * part.s */
    for( i = 0; i < partitions; ++i ) {
	j = part_table[i].flag;
	if (j == 0)
	    continue;
	if (!strcmp( atari_boot_type( j ), "???" )) {
	    printf( "Warning: Partition %d has invalid boot type (%02x)\n",
		    i+1, j );
	    ++warnings;
	}
	if (IS_EXTENDED(part_table[i])) {
	    printf( "Warning: Partition %d has boot type set but is extended\n"
		    "  (booting from extended partitions is usually "
		    "not supported)\n", i+1 );
	    ++warnings;
	}
    }

    /* check if partition IDs look ok */
    for( i = j = 0; i < partitions; ++i ) {
	if (!is_valid_PID( part_table[i].id )) {
	    printf( "Warning: Partition %d has invalid ID (%s) "
		    "(not alphanumeric)\n", i+1, part_table[i].id );
	    ++warnings;
	}
	if (!(isupper(part_table[i].id[0]) &&
	      isupper(part_table[i].id[1]) &&
	      isupper(part_table[i].id[2]))) {
	    printf( "Note: Partition %d has unusual ID (%s) "
		    "(not three uppercase letters)\n", i+1, part_table[i].id );
	    ++notes;
	}
    }
    
    /* 'GEM' type partitions cannot be bigger than 65532 (TOS 1.04+) or 32766
     * (up to TOS 1.02) sectors */
    for( i = 0; i < partitions; ++i ) {
	if (strcmp( part_table[i].id, "GEM" ) == 0) {
	    if (part_table[i].size > 65532) {
		printf( "Warning: Partition %d has ID \"GEM\", but has more "
			"than 65532 sectors.\n"
			"  Better use BGM.\n", i+1 );
		++warnings;
	    }
	    else if (part_table[i].size > 32766) {
		printf( "Note: Partition %d has ID \"GEM\", but has more "
			"than 32766 sectors.\n"
			"  You need TOS 1.04 or higher for this.\n", i+1 );
		++notes;
	    }
	}
    }
    
    /* warn if more than 15 part.s on SCSI/ACSI, 63 on IDE */
    if (partitions > 15 &&
	(!strncmp( disk_device, "/dev/sd", 7 ) ||
	 !strncmp( disk_device, "/dev/ad", 7 ))) {
	printf( "Note: Linux only supports 15 partitions on SCSI/ACSI devices "
		"(you have %d)\n", partitions );
	++notes;
    }
    if (partitions > 63 && (!strncmp( disk_device, "/dev/hd", 7 ))) {
	printf( "Note: Linux only supports 63 partitions on IDE devices "
		"(you have %d)\n", partitions );
	++notes;
    }

    /* warn if empty bad sector list */
    if (bsl_size == 0) {
	printf( "Note: There is no bad sector list. Atari HDX doesn't "
		"like this,\n"
		"  propably you won't be able to process this disk "
		"with HDX.\n" );
	++notes;
    }

    /* check for unallocated sectors */
    header_printed = 0;
    n = free = 0;
    for( sec = 1; sec < hd_size; sec += n ) {
	sec += allocated_sectors_at( sec );
	if (sec >= hd_size || !(n = free_sectors_at( sec )))
	    break;
	if (!header_printed) {
	    printf( "\nUnallocated areas:\n" );
	    printf( "    Begin       End   Sectors MB\n" );
	    header_printed = 1;
	}
	printf( "%9lu %9lu %9lu  %s\n",
		sec, sec+n-1, n, round_mb(n) );
	free += n;
    }
    if (free)
	printf( "\nTotal %lu free sectors.\n", free );
    
    printf( "\n" );
    printf( "%d errors (violations of partition format)\n", errors );
    printf( "%d warnings (things that may or may not work)\n", warnings );
    printf( "%d notes (basically ok, but at least unusual)\n", notes );
}

void write_table(void)
{
    int i, n_prim, n_ext, first_ext, last_ext;
    PARTITION XGM_part;

#ifndef ALPHA_VERSION
    /* run put_boot for alpha versions even if opened read-only; the actual
     * writing will not be avoided later */
    if (type_open == O_RDONLY) {
	printf( "%s is opened read-only; can't write partition table\n",
		disk_device );
	return;
    }
#endif

    /* To be sure, test some conditions. If these weren't fulfilled, we'd
     * write bad tables... */
    table_infos( &n_prim, &n_ext, &first_ext, &last_ext );
    if (n_prim + (n_ext > 0) > MAX_PRIMARY) {
	printf( "Too many primary slots allocated. "
		"This should not happen!\n" );
	return;
    }
    if (!check_ext_start_condition( NONE, 0, 0 )) {
	printf( "The aux. root sector of the first extended partition is "
		"not at\n"
		"the start of the XGM area. This shouldn't happen, probably "
		"it's a bug.\n" );
	return;
    }
    for( i = 0; i < partitions; ++i ) {
	if (IS_EXTENDED(part_table[i]) &&
	    part_table[i].rootsec >= part_table[i].start) {
	    printf( "The aux. root sector of partition %d is not before "
		    "the data sectors!\n"
		    "This shouldn't happen, probably it's a bug.\n", i+1 );
	    return;
	}
    }
    
    if (n_ext > 0) {
	XGM_boundary( &XGM_part.start, &XGM_part.size );
	XGM_part.flag = XGM_flag;
	strcpy( XGM_part.id, "XGM" );
    }
    else
	first_ext = last_ext = -1;

    put_boot( first_ext, last_ext, &XGM_part );
}

/* Local Variables: */
/* tab-width: 8     */
/* End:             */
