/****************************************************************
**
** Attal : Lords of Doom
**
** game.h
** Manages the whole game
**
** Version : $Id: game.h,v 1.17 2004/12/11 13:39:04 lusum Exp $
**
** Author(s) : Pascal Audoux
**
** Date : 17/08/2000
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/


#ifndef GAME_H
#define GAME_H


// include files for QT
#include <qwidget.h>
#include <qlabel.h>
#include <qdialog.h>
#include <qtabwidget.h>

// application specific includes
#include "libClient/player.h"
#include "libClient/cell.h"
#include "libClient/map.h"
#include "libClient/miniMap.h"
#include "libClient/gameControl.h"
#include "libClient/gameDescription.h"
#include "libClient/mapView.h"

#include "libCommon/attalSocket.h"
#include "libCommon/log.h"

class Calendar;
class ChatWidget;
class DisplayLord;
class GameInfo;
class GenericBase;
class LordExchange;
class RessourceBar;

class QPixmap;
class QVBoxLayout;

/*              ------------------------------
 *                         Game
 *              ------------------------------ */


/** comment for the class */
class Game : public QWidget, public GameDescription
{
	Q_OBJECT
public:
	/** Constructor */
	Game( QWidget * parent = 0, const char * name = 0 );
	
	/** Destructor : delete player and lord */
	~Game();

	/** Associate a socket to the Game (for exchange with the server) */
	void setSocket( AttalSocket * sock ) {
		_socket = sock;
	}

	/** Lord 'lord' enters in base 'base' */
	void enter( GenericLord * lord, GenericBase * base );

	/** Lord 'lord' enters in building 'build' */
	void enter( GenericLord * lord, GenericBuilding * building );

	/** Handle socket data */
	void handleSocket();

	/** Begin game with 'nb' players */
	void beginGame( int nb );

	/** End game */
	void endGame();

	/** Begin turn */
	void beginTurn();

	/** End turn */
	void endTurn();

	/** Player active is player 'num' */
	void playerActive( char num );

	/** Activate next lord */
	void nextLord() { _player->nextLord(); }

	/** Return the player associated to the game */
	Player * getGamePlayer() { return _player; }

	/** Sets the name of the player */
	void setPlayerName( QString name );

	enum MapState {
		MS_NOTHING,
		MS_LORD,
		MS_BASE,
		MS_TECHNIC
	};

	void setState( MapState state );

	MapState getState() { return _state; }

	void displayMiniMap( bool state );
	
	void displayTab( bool state );
	
	void displayFullScreen( bool state );
	
	ChatWidget * getChat() {return  (_chat) ? _chat : 0 ; }
	
signals:
	void sig_base( GenericBase * );
	void sig_fight( GenericLord * lord, CLASS_FIGHTER cla );
	void sig_statusMsg( const QString & );
	void sig_endGame();
	void sig_exchange();

public slots:
	/** Slot for displaying lord */
	void slot_displayLord();

	/** Slot for displaying base */
	void slot_displayBase();

	void slot_lordSelected();
	
	void slot_baseSelected();

	/** */
	void slot_message( QString );

private slots:
	void slot_endTurn() { endTurn(); }
	void slot_mouseMoved( Cell *cell );
	void slot_mouseLeftPressed( Cell * cell );
	void slot_mouseRightPressed( Cell * cell );
	void slot_centerMinimap(GenericCell * cell );

private:
	void initWidgets();

	void handleClickNothing( Cell * cell );
	void handleClickLord( Cell * cell );
	void handleClickBase( Cell * cell );
	void handleClickTechnic( Cell * cell );

	/** Manage socket SO_MSG */
	void socketMsg();

	/** Manage socket SO_GAME */
	void socketGame();
	void socketGameLost();
	void socketGameWin();
	void socketGameInfo();

	/** Manage socket SO_TURN */
	void socketTurn();

	/** Manage socket SO_MVT */
	void socketMvt();

	/** Manage socket SO_MODIF */
	void socketModif();
	void socketModifCell();
	void socketModifLord();
	void socketModifBase();
	void socketModifBuilding();
	void socketModifArtefact();
	void socketModifPlayer();
	void socketModifCreature();

	void socketModifEvent();
	void socketNewEvent();

	/** Manage socket SO_CONNECT */
	void socketConnect();

	/** Manage socket SO_EXCH */
	void socketExchange();

	void socketModifLordVisit();
	void socketModifLordNew();
	void socketModifLordUnit();
	void socketModifLordRemove();
	void socketModifLordGarrison();
	void socketModifLordMachine();

	void socketModifBaseNew();
	void socketModifBaseOwner();
	void socketModifBaseName();
	void socketModifBaseBuilding();
	void socketModifBaseUnit();
	void socketModifBasePopulation();

	void socketQR();
	void socketFight();

	void exchangeUnits();
	void exchangeArtefact();	
	void exchangeBaseUnits();

	Player * _player;
	AttalSocket * _socket;
	Map * _map;
	RessourceBar * _resourceBar;
	GameControl * _control;
	GameInfo * _gameInfo;
	DisplayLord * _dispLord;
	LordExchange * _lordExchange;
	MiniMap * _miniMap;
	MapView * _view;
	QTabWidget * _tab;
	ChatWidget * _chat;
	ScrollLord * _scrLord;
	ScrollBase * _scrBase;
	Cell * _currentCell;
	bool _isPlaying;
	QString _msg;

	MapState _state;
};

/** Dialog for displaying message during the game */
class GameMessage : public QDialog
{
public:
	/** Constructor */
	GameMessage( QWidget * parent = 0, const char * name = 0 );

	/** Adds text to the dialog */
	void addText( QString text );

	/** Adds pixmap to the dialog */
	void addPixmap( QPixmap * pixmap );

protected:
	QVBoxLayout * _layout;

};


#endif // GAME_H


