/****************************************************************
**
** Attal : Lords of Doom
**
** scenarioInterface.cpp
** Manages the scenario editor
**
** Version : $Id: scenarioInterface.cpp,v 1.14 2004/07/08 19:25:21 lusum Exp $
**
** Author(s) : Pascal Audoux
**
** Date : 27/08/2000
**
** Licence :    
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/


#include "scenarioInterface.h"
 
// include files for QT
#include <qapplication.h>
#include <qcanvas.h>
#include <qcombobox.h>
#include <qfiledialog.h>
#include <qlayout.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qsignalmapper.h>
#include <qstatusbar.h>

// application specific include files
#include "conf.h"

#include "libCommon/dataTheme.h"

#include "libClient/gui.h"
#include "libClient/imageTheme.h"

#include "scenarioEditor/screen.h"

extern DataTheme DataTheme;
extern ImageTheme ImageTheme;

extern QString IMAGE_PATH;
extern QString DATA_PATH;
extern QString THEME;

//
// ----- ScenarioInterface -----
//

ScenarioInterface::ScenarioInterface()
{
	_filename = "";
	_actions.resize( NB_ACTIONS );

	setCaption( "Scenario editor for 'Attal - Lords of Doom'" );
	initActions();
	initMenuBar();
	initStatusBar();
	activateMenuForScenario( false );
	if( DataTheme.init() && ImageTheme.init() ) {
		_screen = new Screen( this );
		setCentralWidget( _screen );
		resize( 600, 600 );
		connect( _screen, SIGNAL( sig_status( const QString & ) ), this, SLOT( slot_status( const QString & ) ) );
	} else {
		/// XXX: we could manage this better (later :) )
		QMessageBox::critical( this, tr( "Can't load theme" ), tr( "Theme " ) + THEME + tr( " has not been loaded successfully" ) );
	}
}

ScenarioInterface::~ScenarioInterface()
{
	delete _screen;
}

void ScenarioInterface::addAction( 	const QString & text,
					const QString & menuText,
					const QString & key,
					MENU_ACTION num,
					QSignalMapper * sigmap )
{
	QAction * action = new QAction( text, menuText, QKeySequence( key ), this );
	_actions.insert( num, action );
	sigmap->setMapping( action, num );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
}

void ScenarioInterface::initActions()
{
	QSignalMapper * sigmap = new QSignalMapper( this );

	addAction( tr( "New" ), tr( "&New" ), tr( "CTRL+N" ), ACTION_NEW, sigmap );
	addAction( tr( "Open" ), tr( "&Open" ), tr( "CTRL+O" ), ACTION_OPEN, sigmap );
	addAction( tr( "Save" ), tr( "&Save" ), tr( "CTRL+S" ), ACTION_SAVE, sigmap );
	addAction( tr( "Save as" ), tr( "&Save as" ), "", ACTION_SAVEAS, sigmap );
	addAction( tr( "Quit" ), tr( "&Quit" ), tr( "CTRL+Q" ), ACTION_QUIT, sigmap );
	addAction( tr( "Scenario information" ), tr( "&Scenario information" ), tr( "CTRL+I" ),
			ACTION_INFO, sigmap );
	addAction( tr( "Players properties" ), tr( "&Players properties" ), tr( "CTRL+P" ),
			ACTION_PLAYERS, sigmap );
	addAction( tr( "Quests management" ), tr( "Quests management" ), "", ACTION_QUESTS, sigmap );
	addAction( tr( "Clear screen" ), tr( "&Clear screen" ), tr( "CTRL+C" ), ACTION_CLEAR, sigmap );
	addAction( tr( "Fill screen" ), tr( "&Fill screen" ), tr( "CTRL+F" ), ACTION_FILL, sigmap );
	addAction( tr( "Mini map" ), tr( "&Mini map" ), tr( "CTRL+M" ), ACTION_MINIMAP, sigmap );
	_actions[ ACTION_MINIMAP ]->setToggleAction( true );
	_actions[ ACTION_MINIMAP ]->setOn( true );
	addAction( tr( "Help" ), tr( "&Help" ), tr( "F1" ), ACTION_HELP, sigmap );

	connect( sigmap, SIGNAL( mapped( int ) ), SLOT( slot_action( int ) ) );
}

void ScenarioInterface::initStatusBar()
{
	statusBar()->message( "Status Bar", 0 );
}

/*!
  define menus
*/

void ScenarioInterface::initMenuBar()
{
	QPopupMenu * menuFile = new QPopupMenu();
	_actions[ ACTION_NEW ]->addTo( menuFile );
	_actions[ ACTION_OPEN ]->addTo( menuFile );
	_actions[ ACTION_SAVE ]->addTo( menuFile );
	_actions[ ACTION_SAVEAS ]->addTo( menuFile );
	menuFile->insertSeparator();
	_actions[ ACTION_QUIT ]->addTo( menuFile );

	QPopupMenu * menuScen = new QPopupMenu();
	_actions[ ACTION_INFO ]->addTo( menuScen );
	_actions[ ACTION_PLAYERS ]->addTo( menuScen );
	_actions[ ACTION_QUESTS ]->addTo( menuScen );

	QPopupMenu * menuTools = new QPopupMenu();
	_actions[ ACTION_CLEAR ]->addTo( menuTools );
	_actions[ ACTION_FILL ]->addTo( menuTools );
	_actions[ ACTION_MINIMAP ]->addTo( menuTools );

	QPopupMenu * menuHelp = new QPopupMenu();
	_actions[ ACTION_HELP ]->addTo( menuHelp );

	menuBar()->insertItem( tr( "&File" ), menuFile );
	menuBar()->insertItem( tr( "&Scenario" ), menuScen );
	menuBar()->insertItem( tr( "&Tools" ), menuTools );
	menuBar()->insertItem( tr( "&Help" ), menuHelp );
}

void ScenarioInterface::slot_action( int num )
{
	switch( num ) {
	case ACTION_NEW:
		actionNew();
		break;
	case ACTION_OPEN:
		actionOpen();
		break;
	case ACTION_SAVE:
		actionSave();
		break;
	case ACTION_SAVEAS:
		actionSaveAs();
		break;
	case ACTION_QUIT:
		actionQuit();
		break;
	case ACTION_INFO:
		actionInfo();
		break;
	case ACTION_PLAYERS:
		actionPlayers();
		break;
	case ACTION_QUESTS:
		actionQuests();
		break;
	case ACTION_CLEAR:
		actionClear();
		break;
	case ACTION_FILL:
		actionFill();
		break;
	case ACTION_MINIMAP:
		actionMinimap();
		break;
	case ACTION_HELP:
		actionHelp();
		break;
	}
}

void ScenarioInterface::activateMenuForScenario( bool state )
{
	_actions[ ACTION_SAVE ]->setEnabled( state );
	_actions[ ACTION_SAVEAS ]->setEnabled( state );
	_actions[ ACTION_INFO ]->setEnabled( state );
	_actions[ ACTION_PLAYERS ]->setEnabled( state );
	_actions[ ACTION_QUESTS ]->setEnabled( state );
	_actions[ ACTION_CLEAR ]->setEnabled( state );
	_actions[ ACTION_FILL ]->setEnabled( state );
}

void ScenarioInterface::actionNew()
{
	_screen->newScenario();
	_filename = "";
	activateMenuForScenario( true );
}

void ScenarioInterface::actionOpen()
{
	QString filename = QFileDialog::getOpenFileName( "", "*.scn *.gam", this );
	if( !filename.isEmpty() ) {
		if( _screen->load( filename ) ) {
			_filename = filename;
			activateMenuForScenario( true );
		}
	}
}

void ScenarioInterface::actionSave()
{
	if( _filename.isEmpty() ) {
		slot_action( ACTION_SAVEAS );
	} else {
		_screen->save( _filename );
	}
}

void ScenarioInterface::actionSaveAs()
{
	_filename = QFileDialog::getSaveFileName( "", "*.scn *.gam", this );
	if( !_filename.isEmpty() ) {
		_screen->save( _filename );
	}
}

void ScenarioInterface::actionQuit()
{
	qApp->quit();
}

void ScenarioInterface::actionInfo()
{
	_screen->displayInfo();
}

void ScenarioInterface::actionPlayers()
{
	_screen->displayPlayersProperties();
}

void ScenarioInterface::actionQuests()
{
	_screen->manageQuests();
}

void ScenarioInterface::actionClear()
{
	_screen->clearMap();
}

void ScenarioInterface::actionFill()
{
	FillDialog dialog( this );
	if( dialog.exec() ) {
		_screen->fillMap( dialog.getTileNumber(), dialog.getTileDiversification() );
	}
}

void ScenarioInterface::actionMinimap()
{
	_screen->displayMiniMap( _actions[ ACTION_MINIMAP ]->isOn() );
}

void ScenarioInterface::actionHelp()
{
}

void ScenarioInterface::slot_status( const QString & text )
{
	statusBar()->message( text, 0 );
}

//
// ----- FillDialog -----
//

FillDialog::FillDialog( QWidget * parent, const char * name )
	:QDialog( parent, name, true )
{
	setCaption( tr( "Choose tile" ) );

	_comboTile = new QComboBox( this );
	for( uint i = 0; i < DataTheme.tiles.count(); i++ ) {
		QString name = QString::number( i ) + " - " + DataTheme.tiles.at( i )->getName();
		_comboTile->insertItem( name, i );
	}
	FIXEDSIZE( _comboTile );

	_comboDiversification = new QComboBox( this );
	slot_updateDiversification( 0 );

	QHBoxLayout * layH1 = new QHBoxLayout();
	layH1->setMargin( 5 );
	layH1->setSpacing( 5 );
	layH1->addWidget( _comboTile );
	layH1->addWidget( _comboDiversification );
	layH1->addStretch( 1 );

	QPushButton * pb1 = new QPushButton( this );
	pb1->setText( tr( "Ok" ) );
	FIXEDSIZE( pb1 );

	QPushButton * pb2 = new QPushButton( this );
	pb2->setText( tr( "Cancel" ) );
	FIXEDSIZE( pb2 );

	QHBoxLayout * layH2 = new QHBoxLayout();
	layH2->setMargin( 5 );
	layH2->addStretch( 1 );
	layH2->addWidget( pb1 );
	layH2->addStretch( 1 );
	layH2->addWidget( pb2 );
	layH2->addStretch( 1 );

	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->setMargin( 5 );
	layout->setSpacing( 5 );
	layout->addLayout( layH1 );
	layout->addStretch( 1 );
	layout->addLayout( layH2 );
	layout->activate();

	connect( pb1, SIGNAL( clicked() ), SLOT( accept() ) );
	connect( pb2, SIGNAL( clicked() ), SLOT( reject() ) );
	connect( _comboTile, SIGNAL( activated( int ) ), SLOT( slot_updateDiversification( int ) ) );
}

void FillDialog::slot_updateDiversification( int tile )
{
	QString text;
	CellModel * cell;
	cell = DataTheme.tiles.at( tile );

	_comboDiversification->clear();
	_comboDiversification->insertItem( "Random" );
	uint nb = cell->getDiversificationNumber();
	for( uint i = 0; i < nb; i++ ) {
		text.sprintf( tr( "%d (weight=%d)" ), i+1, cell->getDiversification( i ) );
		_comboDiversification->insertItem( text );
	}
	_comboDiversification->setCurrentItem( 0 );
	FIXEDSIZE( _comboDiversification );
}

uint FillDialog::getTileNumber()
{
	return _comboTile->currentItem();
}

uint FillDialog::getTileDiversification()
{
	return _comboDiversification->currentItem();
}
