# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::fork;
use base 'AXP::Command::tree';

use Arch::Util qw(run_tla copy_dir);

sub infoline {
	"create new branch in-place, or by copying to new dir"
}

sub optusage {
	"[options] [archive/]version"
}

sub own_options {
	(
		copy    => { sh => 'c', type => "=s", arg => 'DIR', desc => "don't fork in-place, copy to DIR first" },
	)
}

sub helptext {
	q{
		This command works on the existing tree.  It creates a new
		branch specified by [archive/]version argument from the current
		tree revision.  All possible existing changes are preserved
		(but not committed).

		If copy-DIR is not given (or it is empty or '.') then this
		command works in-place. This effectivelly does "tag -S",
		"join-branch" and "set-tree-version".

		Otherwise, if copy-DIR is given, then the current tree is left
		as is, it is copied to the new DIR together with all precious
		files, and the work (described above) is done on this new DIR.

		Example:
			cd my-arch-projects/archway
			axp fork -c ../archway-mage archway--merge-gui--0
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	my $new_version = shift @ARGV;
	die "No fork version given, exiting\n" unless $new_version;

	die "The --test option is not implemented yet\n" if $opt{test};
	$ENV{DEBUG} = 1 if $opt{verbose};

	my $tree = $self->tree;
	my $dir = $tree->root;
	my $old_version = $tree->get_version;

	if ($opt{copy}) {
		my $copy_dir = $opt{copy};
		die "Directory $copy_dir exists already, exiting\n" if -d $copy_dir;
		copy_dir($dir, $copy_dir);
		die "Can't copy $dir/ to $copy_dir/: exit status $?\n" if $?;
		$dir = $copy_dir;
	}

	print "* tagging from $old_version to $new_version\n"
		unless $opt{quiet};
	run_tla("tag -S", $old_version, $new_version);
	die "Can't tag $old_version as $new_version: exit status $?\n" if $?;

	print "* joining new branch\n"
		unless $opt{quiet};
	run_tla("join-branch", "-d", $dir, $new_version);
	die "Can't join-branch $new_version: exit status $?\n" if $?;

	print "* setting new tree version in $dir\n"
		unless $opt{quiet};
	run_tla("set-tree-version", "-d", $dir, $new_version);
	die "Can't set-tree-version $new_version: exit status $?\n" if $?;

	print "* created new branch $new_version based on $old_version\n"
		unless $opt{quiet};
}

1;
