<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    plugin-defs.php                                         */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing plugin regions for webpages.   */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

// ----------------------------------------------------------------------
/**
* The page section class
* The class takes care of page sections. A page section is any part of a
* webpage. Page sections we define for HTML pages are: head, body and
* foot. Each section has a common set of properties and methods and these
* are defined here.
* @package core
*/
class page_section extends RenderableObject {
  // Public
  /** Page section content */
  var $content = "";

  // Private
  /** Script content for this page section
      @access private */
  var $script;
  /** Scripts which are sourced via a URL
      @access private */
  var $scriptsrc;
  // .....................................................................
  /**
  * Constructor
  * Create a new page section object.
  * @param string $content Initial page content
  */
  function page_section($content="") {
    $this->content = $content;
  } // page_section
  // .....................................................................
  /**
  * Add new content to the page section.
  * @param string $newstuff Initial page content
  */
  function add($newstuff) {
    $this->content .= $newstuff;
    return $this;
  } // add
  // .....................................................................
  /**
  * Clear all content from the page section.
  */
  function clear() {
    $this->content = "";
    $this->script = "";
    $this->scriptsrc = "";
    return $this;
  } // clear
  // .....................................................................
  /** Return content, if not null or newline.
  * @return string The page section content
  */
  function get_trimcontent() {
    $content = trim($this->content);
    if ($content == "\n") $content = "";
    return $content;
  } // get_trimcontent
  // .....................................................................
  /**
  * Add popup window scripting.
  * This method adds Javascript which specifically supports the opening
  * of a popup window from a function call elsewhere in the webpage. There
  * are various options to force the size and position, or leave to be
  * passed as parameters in the javascript function call.
  * NB: This does NOT open the window. It inserts the Javascript for you to
  * be able to call a function to open the window from elsewhere in the page.
  * @param string $popupname The name you want the function to be called
  * @param integer $width Width (px) of window if fixed, else will be a parameter
  * @param integer $height Height (px) of window if fixed, else will be a parameter
  * @param integer $left Left offset (px) of window if fixed, else will be a parameter
  * @param integer $top Left offset (px) of window if fixed, else will be a parameter
  * @param string $features The window features, if you want to override the defaults
  */
  function add_popup_script(
        $popupname,
        $width=false,
        $height=false,
        $left=false,
        $top=false,
        $features="toolbar=no,status=no,scrollbars=no,resizable=yes"
        )
  {
    $js = "var " . $popupname . "Win=null;\n";
    $js .= "function $popupname(theurl,width,height,left,top) {\n";
    if ($width === false) $js .= " if(width) ws='width='+width;else ws='';";
    else $js .= " ws='width='+$width;";
    if ($height === false) $js .= "if(height) hs='height='+height;else hs='';";
    else $js .= "hs='height='+$height;";
    if ($left === false) $js .= "if(left) ls='left='+left;else ls='';";
    else $js .= "ls='left='+$left;";
    if ($top === false) $js .= "if(top) ts='top='+top;else ts='';";
    else $js .= "ts='top='+$top;";
    $js .= "\n var feats='$features';\n";
    $js .= " if(ws!='') feats+=','+ws;";
    $js .= " if(hs!='') feats+=','+hs;";
    $js .= " if(ls!='') feats+=','+ls;";
    $js .= " if(ts!='') feats+=','+ts;\n";
    $js .= " " . $popupname . "Win=window.open(theurl,'$popupname',feats);\n";
    $js .= " if(" . $popupname . "Win != null) " . $popupname . "Win.focus();\n";
    $js .= "}\n";
    $this->add_script($js);
  } // add_popup_script
  // .....................................................................
  /**
  * Add more scripting to this page section.
  * @param string $script   Script content (omit <script> tags)
  * @param string $language The language the script is in
  */
  function add_script($script, $language="javascript") {
    $this->add_named_script($script, "default", $language);
  } // add_script
  // .....................................................................
  /**
  * This adds a specific lump of script to the webpage under a unique name.
  * It allows you to collect multiple additions of script content under
  * a single grouping, and so cause the final rendering to group it all
  * together.
  * @param string $script   Script content (omit <script> tags)
  * @param string $name     Script content grouping identity
  * @param string $language The language the script is in
  */
  function add_named_script($script, $name, $language="javascript") {
    if ($name == "") {
      $name = "default";
    }
    $this->script[$name][$language] .= $script;
  } // add_named_script
  // .....................................................................
  /**
  * Add script reference link
  * Add more scripting to this section in the form of
  * a link to a script source file.
  * @param string $src      URL pointing to the script
  * @param string $language The language the script is in
  */
  function add_scriptsrc($src, $language="javascript") {
    $this->scriptsrc[$language][] = $src;
  } // add_scriptsrc
  // .....................................................................
  /**
  * Returns all of the defined script. Literal script is rendered for each
  * language if more than one language is being used. Each language is
  * bracketed with the usual <script language=foo></script> tags. Any
  * script sources (URL references to scripts) are also rendered.
  * @return string All the defined script content including source references
  */
  // Render script..
  function script() {
    $s = "";
    if (isset($this->scriptsrc) && is_array($this->scriptsrc)) {
      foreach ($this->scriptsrc as $language => $srcs) {
        foreach ($srcs as $src) {
          $s .= "<script type=\"text/$language\" src=\"$src\"></script>\n";
        }
      }
    }
    if (isset($this->script) && is_array($this->script)) {
      foreach ($this->script as $groupname => $scripts) {
        if ($groupname != "default") {
          $s .= "<!-- $groupname -->\n";
        }
        foreach ($scripts as $language => $script) {
          $s .= "<script type=\"text/$language\" language=\"$language\">\n";
          $s .= "<!--\n";
          $s .= $script;
          $s .= "//-->\n";
          $s .= "</script>\n";
        }
      }
    }
    return $s;
  } // script
  // .....................................................................
  /**
  * Replaces multiple occurrences of the given tag (pattern) in the
  * body content with the specified new stuff.
  * @param string $tag       Pattern to replace in content
  * @param string $newstuff  Stuff to replace the tag with
  */
  function replace($tag, $newstuff) {
    $tmp = str_replace($tag, $newstuff, $this->content);
    $this->content = $tmp;
    return $this;
  } // replace
  // ....................................................................
  /**
  * Read in the template for this page section.
  * @param string The full path of the template to get
  * @return string The full content of the template
  */
  function get_template($templatefile="") {
    global $TEMPLATES;
    $template = "";
    if ($templatefile != "") {
      if (isset($TEMPLATES[$templatefile])) {
        // Get cached copy..
        $template = $TEMPLATES[$templatefile];
      }
      else {
        // Read from actual template file..
        $tpfile = new inputfile($templatefile);
        if ($tpfile->opened) {
          $content = $tpfile->readall();
          $tpfile->closefile();
          if ($content !== false) {
            $template = $content;
          }
        }
        $TEMPLATES[$templatefile] = $template;
      }
    }
    return $template;
  } // get_template
  // ....................................................................
  /**
  * This renders the page section as HTML.
  * @return string The page section as HTML.
  */
  function html() {
    return $this->content;
  } // html
  // ....................................................................
  /**
  * This renders the page section as WML.
  * @return string The page section as WML.
  */
  function wml() {
    return $this->content;
  } // wml
} // page_section class

// ----------------------------------------------------------------------
/**
* Pluginset class
* This class manages plugins as a set. It is designed for use by
* webpages, which can have many types of plugin content defined for
* them.
* @package core
*/
class pluginset {
  // Public
  // Private
  /** Array of plugins in this set
      @access private */
  var $plugins = array();
  /** Flag indicating whether plugins exist
      @access private */
  var $hascontent = false;
  // .....................................................................
  /**
  * Constructor
  * Create a new plugin set. This is just a container for plugins.
  */
  function pluginset() {}
  // .....................................................................
  /**
  * Clear all plugins from the set
  */
  function clear() {
    unset($this->plugins);
    $this->hascontent = false;
  } // clear
  // .....................................................................
  /**
  * Add a new plugin to the plugin set. The type of plugin is determined
  * from the content passed as the second paramter. Allowed data-types
  * for content: object (must inherit RenderableObject), a function
  * definition, a file-path, or just literal content.
  * @param string $pluginid ID of this plugin. Used to find plugin location
  * @param mixed $content The plugin content (literal, function, object...)
  */
  function add_plugin($pluginid, $content="") {
    $pluginid = strtoupper($pluginid);
    debugbr("pluginset: defining new plugin '$pluginid'", DBG_DEBUG);
    $this->plugins[$pluginid] = new plugin($pluginid, $content);
    $this->hascontent = true;
    return $this;
  } // add_plugin
  // .....................................................................
  /**
  * Adds content to the given plugin. If the plugin doesn't exist yet,
  * then we create it first. If it already exists, then we append the
  * new content to it.
  * @param string $pluginid ID of this plugin. Used to find plugin location
  * @param mixed $content The plugin content (literal, function, object...)
  */
  function addto($pluginid, $content) {
    $pluginid = strtoupper($pluginid);
    if (!isset($this->plugins[$pluginid])) {
      $this->add_plugin($pluginid, $content);
    }
    else {
      $plugin = $this->plugins[$pluginid];
      $plugin->add_content($content);
      $this->plugins[$pluginid] = $plugin;
    }
    return $this;
  } // addto
  // .....................................................................
  /**
  * This method takes the given template, and renders the plugins on
  * it one by one. Each plugin is applied to the template, replacing
  * all occurences of the appropriate tag based on the pluginid. Then
  * finally the resulting string is returned.
  */
  function render($template="") {
    if ($template != "") {
      $plugged = $template;
      if (isset($this->plugins)) {
        foreach ($this->plugins as $id => $plugin) {
          if (is_object($plugin)) {
            $newstuff = $plugin->render();
            $tag = "<!--" . strtoupper($id) . "-->";
            str_replace($tag, $newstuff, $plugged);
          }
        }
      }
    }
    return $plugged;
  } //render
} // pluginset class

// ----------------------------------------------------------------------
/**
* Plugin class
* A plugin is something which can be used by the system to render
* content to be plugged into a webpage in any specified place.
* The normal plugin just provides a receptacle for content, and
* which will be plugged in when the webpage is built. You can
* also specify a path of a file containing content, an object to
* render() content, or a function to return content.
* We can have multiple types of plugin:
*  - Standard: Literal content as provided
*  - Function: A function call which returns content
*  - Object:   An object which renders content
*  - File:     A file which contains content
* @package core
*/
class plugin extends page_section {
  // Public
  // Private
  /** ID or name of this plugin
      @access private */
  var $pluginid = "";
  /** Array of plugin content objects
      @access private */
  var $plugin_contents;
  // .....................................................................
  /**
  * Constructor
  * Create a new plugin object.
  * @param string $pluginid  ID of this plugin. Used to find plugin location
  * @param string $content   Content to put into this plugin location
  */
  function plugin($pluginid, $content="") {
    $this->page_section();
    $this->pluginid = strtoupper($pluginid);
    $this->add_content($content);
  } // plugin
  // .....................................................................
  /**
  * Allows adding of any type of content to the plugin. This could be
  * literal (string), a file path referencing a file full of content, or
  * and object supporting render(), or the name of a function which is in
  * scope. In the latter case, the function name is saved here, and will
  * only be executed later on when plugins are rendered prior to sending
  * the webpage. This allows 'late' content rendering.
  * @param mixed $content Content to plug in: string, path, object or func name
  */
  function add_content($content="") {
    if (is_object($content)) {
      // An object which returns content via render()..
      $this->plugin_contents[] = new object_plugin_content($content);
    }
    elseif (is_string($content)) {
      $pattern = "/^.+\(.*\)" . chr(36) . "/";
      if (preg_match($pattern, $content)) {
        // A function definition..
        $this->plugin_contents[] = new func_plugin_content($content);
      }
      elseif (realpath($content) && file_exists($content)) {
        // File content..
        $this->plugin_contents[] = new file_plugin_content($content);
      }
      else {
        // Standard string content..
        $this->plugin_contents[] = new plugin_content($content);
      }
    }
  } // add_content
  // .....................................................................
  /**
  * Returns the string which represents all of the content
  * types which have been stored in this plugin. Note that we do not
  * differentiate between HTML or WML etc. since the content from our
  * point of view is all generic at this stage. Hence we override the
  * render() method, and not html() and/or wml().
  * @return string Plugin content.
  */
  function render() {
    $s = "";
    if (isset($this->plugin_contents)) {
      foreach ($this->plugin_contents as $content) {
        $new_content = $content->render();
        // Now render any content block tags. This scans the
        // content string for BLOCKID tags and renders the block
        // content in their place if any are found..
        if (function_exists("render_layouts")) {
          $new_content = render_layouts($new_content);
        }
        $s .= $new_content;
      }
    }
    return $s;
  } // render
} // plugin class

// ----------------------------------------------------------------------
/**
* Plugin content.
* Literal plugin content.
* for a plugin.
* @package core
* @access private
*/
class plugin_content {
  /** The plugin content */
  var $content;
  /** Whether the content is valid */
  var $valid = false;
  /**
  * Constructor. Create a new literal plugin object.
  * @param string $content Literal content to be plugged in
  */
  function plugin_content($content="") {
    $this->content = $content;
    $this->valid = true;
  } // plugin_content
  // .....................................................................
  /**
  * Generate plugin content
  * Return content.
  * @return string Plugin content.
  */
  function render() {
    return $this->content;
  } // render
} // plugin_content class

// ----------------------------------------------------------------------
/**
* File plugin content
* You can specify the name of a file which contains the content
* for a plugin.
* @package core
* @access private
*/
class file_plugin_content extends plugin_content {
  /** The path to the file */
  var $path;
  /**
  * Constructor
  * Create a new file plugin content object.
  * @param string $path Path to a file containing content
  */
  function file_plugin_content($path="") {
    if ($path != "") {
      if (file_exists($path)) {
        $this->path = $path;
        $plugfile = new inputfile($path);
        if ($plugfile->opened) {
          $this->content = trim($plugfile->readall());
          $plugfile->closefile();
          $this->valid = true;
        }
      }
    }
  } // file_plugin_content
} // file_plugin_content class

// ----------------------------------------------------------------------
/**
* Function plugin content
* Contains details of a function to be used to generated content,
* @package core
* @access private
*/
class func_plugin_content extends plugin_content {
  /** The name of the function which returns content */
  var $funcname = "";
  /** The arguments for the function */
  var $funcargs;
  // .....................................................................
  /**
  * Constructor
  * Create a new function plugin object.
  * @param string $funcdef Definition of the function to call for content
  */
  function func_plugin_content($funcdef) {
    $pattern = "/^(.+)\((.*)\)" . chr(36) . "/";
    if (preg_match($pattern, $funcdef, $matches)) {
      $this->funcname = $matches[1];
      $argstr = $matches[2];
      $argstr = str_replace("\"", "", $argstr);
      $argstr = str_replace("'",  "", $argstr);
      $this->funcargs = explode(",", $argstr);
      $this->valid = true;
    }
  } // func_plugin_content
  // .....................................................................
  /**
  * Generate plugin content
  * Call the givenfunction which should return a string, containing
  * the content.
  * @return string Plugin content.
  */
  function render() {
    if (isset($this->funcname)) {
      if (function_exists($this->funcname)) {
        $content = call_user_func_array($this->funcname, $this->funcargs);
        if (is_string($content) && $content != "") {
          $this->content = $content;
        }
      }
    }
    return $this->content;
  } //render
} // func_plugin_content class

// ----------------------------------------------------------------------
/**
* Object plugin content
* Contains details of an object to be used to generated content,
* @package core
* @access private
*/
class object_plugin_content extends plugin_content {
  /** The objects which return content */
  var $obj;
  // .....................................................................
  /**
  * Constructor
  * Create a new object plugin object.
  * @param object $obj Object to use for content
  */
  function object_plugin_content($obj) {
    if (method_exists($obj, "html")
     || method_exists($obj, "wml")
     || method_exists($obj, "wmlup")
    ) {
      $this->obj = $obj;
      $this->valid = true;
    }
  } // object_plugin_content
  // .....................................................................
  /**
  * Generate plugin content
  * Call object render() method. Return the result.
  * @return string Plugin content.
  */
  function render() {
    if (isset($this->obj)) {
      $this->content = $this->obj->render();
    }
    return $this->content;
  } //render
} // object_plugin_content class

// ----------------------------------------------------------------------
/**
* Cm Layout plugin content
* Contains details of a Content-Managed Layout block which is to
* be generated and plugged in. The ID of the layout is provided.
* @package core
* @access private
*/
class cm_plugin_content extends plugin_content {
  /** The text identifier of the CM layout
      which returns content */
  var $layoutid;
  // .....................................................................
  /**
  * Constructor
  * Create a new CM layout plugin object.
  * @param string $layoutid Unique string naming/identifying the layout.
  */
  function cm_plugin_content($layoutid="") {
    if ($layoutid != "") {
      $this->layoutid = $layoutid;
      $this->valid = true;
    }
  } // cm_plugin_content
  // .....................................................................
  /**
  * Generate CM Layout content. All that is required here is the
  * specific format required to be plugged into the webpage.
  * @return string Plugin content.
  */
  function render() {
    if ($this->valid) {
      $this->content = "\"<!--LAYOUTID=\\\"" . $this->layoutid . "\\\"-->\"";
    }
    return $this->content;
  } //render
} // cm_plugin_content class

// ----------------------------------------------------------------------
?>