/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef ITEM_H
#define ITEM_H

#include <qcheckbox.h>
#include <qframe.h>
#include <qclipboard.h>
#include <qtimer.h>
#include <qcursor.h>

#include <qstring.h>
#include <qpixmap.h>
#include <qcolor.h>
#include <kurl.h>
#include <kio/jobclasses.h>
#include <arts/kplayobject.h>
#include <arts/kplayobjectfactory.h>
#include <arts/kartsserver.h>

#include "onclickaction.h"

class SearchData;

class QWidget;
class QHBoxLayout;
class QVBoxLayout;
class QSpacerItem;
class QLabel;
class QColor;
class KURL;
class QDragObject;
class KService;
class QMovie;

class Basket;
class LinkLabel;

/**Provide an vertically alignable QCheckBox.
  *@author Sbastien Laot
  */
class AlignableCheckBox : public QWidget
{
  Q_OBJECT
  public:
	AlignableCheckBox(const QString &text, QWidget *parent, const char *name = 0);
	~AlignableCheckBox();
	inline QCheckBox* checkBox() { return &m_check; }
	void setAlignment(int align);
	virtual void setFocusPolicy(QWidget::FocusPolicy policy);
	void setCheckCursor(const QCursor &cursor);
  private:
	QVBoxLayout *m_layout;
	QSpacerItem *m_spacer;
	QCheckBox    m_check;
};

/**Implement all droped objects.
  *@author Sbastien Laot
  */
class Item : public QFrame
{
  Q_OBJECT
  public:
	/** Return the type of the stored contain */
	enum Type { Text = 1, Html, Image, Animation, Sound, File, Link, Launcher, Color, Unknow };
	inline Type type() { return m_type; } // "enum Type" is needed for a constructor
	QString typeName();
	QString lowerTypeName();
	QString toString();
	QString toHtml(const QString &imageName);
	/** Construtors, initializer and destructor */
	Item(const QString &fileName,    int fontType, const QColor &fontColor,     // Text
	     const QString &annotations, bool checked, Basket *parent);
	Item(const QString &fileName,  //bool showSource,                           // Html
	     const QString &annotations, bool checked, Basket *parent);
	Item(const KURL    &url,         const QString &title, const QString &icon, // Link
	     bool autoTitle, bool autoIcon,
	     const QString &annotations, bool checked, Basket *parent);
	Item(const QString &fileName,    Type type,                                 // Image, Animation, Sound, File, Launcher
	     const QString &annotations, bool checked, Basket *parent);
	Item(const QColor  &color,                                                  // Color
	     const QString &annotations, bool checked, Basket *parent);
	~Item();
  private:
	void initItem(bool checked); // Create and initialize a new item
  public:
	/** Various functions to change content (according to the type of the item) */
	// Text
	QString text();
	int textFontType(); //
	QColor textColor();
	void setText(const QString &text);
	void setText(const QString &text, int type, QColor fontColor);  // TODO: void setTextStyle(TextStyle style);
	void setTextStyle(int type, QColor fontColor);
	// Html
	QString html();
//	bool showSource();
//	void setShowSource(bool show);
	void setHtml(const QString &html);
//	void setHtml(const QString &html, bool showSource);
	// Image
	QPixmap* pixmap();
	void setPixmap(const QPixmap &pixmap);
	// Animation
	QMovie* movie();
	// Sound

	// File
	void setFile(); // Display the file name in a LinkLabel. Also used by Sound items
	// Link
	KURL url();
	QString title();
	QString icon();
	bool autoTitle() { return m_autoTitle; }
	bool autoIcon()  { return m_autoIcon;  }
	void setUrl(const KURL &url, const QString &title, const QString &icon);
	void setAuto(bool autoTitle, bool autoIcon) { m_autoTitle = autoTitle; m_autoIcon = autoIcon; }
	// Launcher
	void setLauncher(const QString &name, const QString &icon); // Only for displaying in the basket: no save
	//void setService(KService *service); // No get method: should open the .desktop file (KService (read only) or KConfig)
	KService* service(); // Convenient method that return a KService you SHOULD delete.
	// Color
	QColor color();
	void setColor(QColor color);
	// Unknow

	/** Common configs */
	QString annotations();
	void setAnnotations(const QString &annotations);
	bool isChecked();
	void setChecked(bool check);
	bool useFile()       { return (m_type == Text      || m_type == Html  || m_type == Image ||
	                               m_type == Animation || m_type == Sound || m_type == File  ||
	                               m_type == Launcher  || m_type == Unknow                     ); }
	bool useLinkLabel()  { return (m_type == Sound || m_type == File || m_type == Link || m_type == Launcher); }
	inline bool isAlternate()          { return m_isAlternate; }
	inline void setAlternate(bool alt) { m_isAlternate = alt;  }
	/** Save work */
	QString fileName();
	QString fullPath();
	bool setFileName(const QString &fileName);
	void fileNameChanged(const QString &fileName);
	bool isAMirror();
	static bool isAMirror(const QString &fileName);
	void unmirror();
  signals:
	/** Pass signals to parent basket */
	void wantDelete(Item *item);
	void wantPaste(QClipboard::Mode mode);
  protected:
	virtual void enterEvent(QEvent*);
	virtual void leaveEvent(QEvent*);
	virtual void mousePressEvent(QMouseEvent *event);
	virtual void mouseReleaseEvent(QMouseEvent *event);
	virtual void mouseMoveEvent(QMouseEvent *event);
	virtual void mouseDoubleClickEvent(QMouseEvent *event);
	virtual void paintEvent(QPaintEvent*);
	virtual void resizeEvent(QResizeEvent*);
  public:
	void dragItem();
	bool isDuplicateOf(Item *item);
	// Propage event because m_title QLabel's LinkLabel doesn't care of color changes
	void setPaletteBackgroundColor(const QColor &color);
	void setPaletteForegroundColor(const QColor &color);
  public slots:
	/** Update display according of new basket properties */	// private or friend Basket
	void updateToolTip();
	void showCheckBoxesChanged(bool show);
	void alignChanged(int hAlign, int vAlign);
	void linkLookChanged();
	void lockedChanged(bool lock);
	/** Save & load work */
	void saveAs(const KURL &dest);
	void saveProperties();
	void loadContent();
	void movieStatus(int status);
	void delayedEnsureVisible();
	/** User interaction */
	void slotEdit(bool editAnnotations = false);
	void slotEditAnnotations(); // Provided to item popupMenu ! Please use slotEdit(true)
	void slotDelete();
	void slotCut();
	void slotCopy(bool toSelection = false, bool cutting = false);
	void slotCopySelection();   // Provided to item popupMenu ! Please use slotCopy(true)
	void slotPaste();
	void slotOpen();
	void slotOpenWith();
	void slotSaveAs();
	void execAction(OnClickAction::Action action);
  private slots:
	void slotChecked();
	void slotResult(KIO::Job *job);
  public:
	int onClickActionPolicy();
	void setSelected(bool selected);
	void setFocused(bool focused);
	/** Search */
	bool match(const SearchData &data);
	bool isSelected();
  public slots:
	void select();     // called when clicked on double click config, or after an hovering and timer in single click conf
	void execAction();
  private:
	bool tiggerableActions(QMouseEvent *event);
  private:
	/* Widgets for the UI */
	QHBoxLayout       *m_layout;
	AlignableCheckBox *m_check;
	QLabel            *m_item;
	LinkLabel         *m_linkLabel;

	QTimer             m_selectTimer;

	/* Type and properties */
	Type     m_type;
	Basket  *m_parentBasket;
	QString  m_annotations;
	QString  m_fileName;

	/* Special variables to handle contents by type */
	int       m_textFontType;
	QColor    m_textColor;
//	bool      m_showSource;
	QColor   *m_color;
	KURL     *m_url;
	QString  *m_title;
	QString  *m_icon;
	bool      m_autoTitle;
	bool      m_autoIcon;

	/* Other variables and flags */
	bool     m_isAlternate;
	bool     m_isSelected;
	bool     m_isFocused;
	bool     m_canDrag;
	QPoint   m_pressPos;
	bool     m_wasPressed;
	KArtsServer            *m_playServer;
	KDE::PlayObject        *m_playObj;
	KDE::PlayObjectFactory *m_playFactory;

	int      m_x;
	int      m_y;

	static const int margins = 2; // Must be at least 1 to show the selection dotted rectangle
  public:
	static QString urlWithoutProtocol(const KURL &url) {
		return url.prettyURL().right(url.prettyURL().length() - url.protocol().length() - 1);
	}

	/** Size managment : */
	int heightForWidth(int w = -1) const;
	int realXWithCheckbox() const;
	int realWidthWithCheckbox() const;
	inline int x()          { return m_x; }
	inline int y()          { return m_y; }
	inline void setX(int x) { m_x = x;    }
	inline void setY(int y) { m_y = y;    }

	inline Basket* parentBasket() { return m_parentBasket; }

	/** Doubly-linked list : */
  private:
	Item *m_previous;
	Item *m_next;
  public:
	inline Item* previous()              { return m_previous; }
	inline Item* next()                  { return m_next;     }
	inline void  setPrevious(Item *item) { m_previous = item; }
	inline void  setNext(Item *item)     { m_next     = item; }
	inline void  plugTo(Item *previous, Item *next) { m_previous = previous; m_next = next; }
};

#endif // ITEM_H
