/* Bluefish HTML Editor
 * parsedtd.h - Parse a DTD file with definition of HTML
 *
 * Copyright (C) 2000 Santiago Capel Torres
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



#ifndef _PARSEDTD_H_
#define _PARSEDTD_H_

#include <glib.h>

/*
  SGML Basic Types.
  www.w3.org/TR/html401/types.html
*/

#define SGML_BT_FIRST          0

#define SGML_BT_CDATA          0
#define SGML_BT_ID             1
#define SGML_BT_NAME           2
#define SGML_BT_IDREF          3
#define SGML_BT_IDREFS         4
#define SGML_BT_NUMBER         5
#define SGML_BT_BOOLEAN        6

#define SGML_BT_LAST           6


/*
  HTML Attrib Types
  www.w3.org/TR/html401/types.html
*/

#define HTML_AT_FIRST         100

#define HTML_AT_TEXT          100
#define HTML_AT_URI           101
#define HTML_AT_COLOR         102
#define HTML_AT_LENGTH        103
#define HTML_AT_PIXELS        104
#define HTML_AT_MULTILENGTH   105
#define HTML_AT_CONTENTYPE    106
#define HTML_AT_CONTENTYPES   107
#define HTML_AT_LANGUAGECODE  108
#define HTML_AT_CHARSET       109
#define HTML_AT_CHARSETS      110
#define HTML_AT_CHARACTER     111
#define HTML_AT_DATETIME      112
#define HTML_AT_LYNKTYPES     113
#define HTML_AT_MEDIADESC     114
#define HTML_AT_SCRIPT        115
#define HTML_AT_STYLESHEET    116
#define HTML_AT_FRAMETARGET   117

#define HTML_AT_LAST          117

/* First ID for the elements */
#define HTML_EL_FIRST           0

/* Properties of the elements */
#define HTML_STARTTAG_OPTIONAL (1 << 0)
#define HTML_ENDTAG_OPTIONAL   (1 << 1)

/* Default values of the attributes */
#define HTML_DEFVALUE_LITERAL  0
#define HTML_DEFVALUE_IMPLIED  1
#define HTML_DEFVALUE_REQUIRED 2
#define HTML_DEFVALUE_FIXED    3


/*
  Structs for the internal representation of the DTDs
*/

/* Representation of an attribute */
typedef struct {
        char *name;                                     /* Name of the attribute */
        int basictype;                          /* SGML basic type (see above) */
        int attrtype;                           /* HTML attribute type (see above) */
        char *othertype;                        /* Other type, if different from basictype and attrtype */
        int defaultvaluetype;           /* Type of default value:  #IMPLIED, #REQUIRED, #FIXED */
        char *defaultvalue;                     /* Default value of the attribute if not of above types */
} SGML_attrinfo;

typedef struct {
        int id;                                         /* Automatic incremental id of the element */
        char *name;                                     /* Name of the element (=HTML tag) */
        int flags;                                      /* Properties of the element */
        char *content;                          /* contents of the element */
        GList *attrs;                           /* List of all attributes (SGML_attrinfo) of this element */
        int clonedattrs;                        /* The list of attrs is cloned from other element */
} SGML_elementinfo;

typedef struct {
        const char *doctype;            /* <!DOCTYPE ...> */
        char *filename;                         /* path of the DTD file */
        int reserved;
        GList *elements;                        /* Parsed DTD */
        int ref;                                        /* reference count to the GList */
} SGML_doctype;

/* Supported Doctype's names */
#define DTD_HTML_L_0  "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML Level 0//EN\""
#define DTD_HTML_L_0S "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML Strict Level 0//EN\""
#define DTD_HTML_L_1  "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML Level 1//EN\""
#define DTD_HTML_L_1S "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML Strict Level 1//EN\""
#define DTD_HTML_2_0  "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\""
#define DTD_HTML_2_0S "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML Strict//EN\""
#define DTD_HTML_2_1E "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.1E//EN\""
#define DTD_HTML_3_0  "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 3.0//EN//\""
#define DTD_HTML_3_0S "!DOCTYPE HTML PUBLIC \"-//W3O//DTD W3 HTML Strict 3.0//EN//\""
#define DTD_HTML_3_2  "!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 3.2//EN\""
#define DTD_HTML_4_0S "!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Strict//EN\""
#define DTD_HTML_4_0F "!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Frameset//EN\""
#define DTD_HTML_4_0T "!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\""
#define DTD_HTML_4_01S "!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Strict//EN\""
#define DTD_HTML_4_01F "!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\""
#define DTD_HTML_4_01T "!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\""
#define DTD_XHTML_1_0 "!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0//EN\""
#define DTD_XHTML_1_0S "!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"DTD/xhtml1-strict.dtd\""
#define DTD_XHTML_1_0F "!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Frameset//EN\"
\"DTD/xhtml1-frameset.dtd\""
#define DTD_XHTML_1_0T "!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"
\"DTD/xhtml1-transitional.dtd\""
#define DTD_XHTML_1_1 "!DOCTYPE html PUBLIC \"-//IETF//DTD XHTML 1.1//EN\""

extern SGML_doctype HTML_doctypes[];
#define DOCTYPE_DEFAULT (15)
#define RET_DTD_LOADED  (0)

SGML_elementinfo *get_element_byname(GList * elements, const char *name);
gint get_doctype_from_text(gchar * text, gchar * doctype, int maxlen);
gint find_dtd_index(const gchar * doctype);
gint find_a_loaded_dtd();
gint load_DTD(int index);
gint unload_DTD(int index);
gint filetoDTD(GList ** elementsptr, const gchar * filename);
gint getDTDTagList(GList * elements, gchar * buffer, gchar * sep,
                   int maxlen);
gint getDTDAttList(GList * elements, gchar * buffer, gchar * sep,
                   int maxlen);
GList *get_attr_othervalues(SGML_attrinfo *ai);

#endif
