//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <iostream>

#include <gtk/gtk.h>
#include <gtkmm.h>
#include <glibmm.h>
#include <gdkmm.h>
#include <pangomm.h>

#include <gtkmm/icontheme.h>

#include "util.hh"
#include "x_play.hh"
#include "x_vfs.hh"

#include "fs.hh"

namespace 
{
  struct NameSorter
  {
    bool operator () (std::string const& a, std::string const& b)
    {
      if (a[0] == '.')
        return true; 

      if (a < b)
        return true; 

      return false;
    }
  };
}

namespace Bmp
{
  void
  FilesystemTree::append_placeholder (Gtk::TreeModel::iterator const& i_node)
  {
    Gtk::TreeModel::iterator i = store->append (i_node->children());
    (*i)[node.name] = "";
    (*i)[node.type] = NODE_PLACEHOLDER; 
    (*i)[node.path] = "";
    (*i)[node.children] = false; 
    (*i)[node.xpanded] = false; 
  }

  void
  FilesystemTree::append_child_nodes (Gtk::TreeModel::iterator const& i_root, std::string const& path, TPathList const& dir)
  {
    using namespace Gtk;

    TPathList paths;
    TPathList files;

    for (TPathList::const_iterator n = dir.begin() ; n != dir.end() ; ++n)
    {
      std::string path_full (Glib::build_filename (path, *n));

      if (Glib::file_test (path_full, Glib::FILE_TEST_IS_DIR))
        paths.push_back (path_full);
      else
        files.push_back (path_full);
    }
    
    std::sort (paths.begin(), paths.end(), NameSorter ());
    std::sort (files.begin(), files.end(), NameSorter ());

    for (TPathList::const_iterator n = paths.begin() ; n != paths.end() ; ++n)
    {
      try {
        Glib::Dir test (*n);
        if (test.read_name().size())
          {
            TreeModel::iterator i_node = store->append (i_root->children());
            (*i_node)[node.name] = Glib::filename_to_utf8 (Glib::path_get_basename (*n));
            (*i_node)[node.type] = NODE_PATH; 
            (*i_node)[node.path] = *n;
            (*i_node)[node.children] = true;
            (*i_node)[node.xpanded] = false; 
            append_placeholder (i_node);
          }
        test.close ();
        }
      catch (Glib::FileError & cxe) { std::cerr << cxe.what() << std::endl; }
      catch (...) {}
    }

    for (TPathList::const_iterator n = files.begin() ; n != files.end() ; ++n)
    { 
      if ((::play->is_audio_file (Glib::filename_to_uri (*n))) || (Util::str_has_suffixes_nocase_on_crack (*n, m_containers)))
        {
          TreeModel::iterator i_node = store->append (i_root->children());
          (*i_node)[node.name] = Glib::filename_to_utf8 (Glib::path_get_basename (*n));
          (*i_node)[node.type] = NODE_FILE; 
          (*i_node)[node.path] = *n;
          (*i_node)[node.children] = false;
        }
    }
  }

  void
  FilesystemTree::on_row_activated (Gtk::TreeModel::Path const& path, Gtk::TreeViewColumn * column)
  {
    on_path_activated_.emit();
  }

  bool
  FilesystemTree::on_test_expand_row (Gtk::TreeModel::iterator const& i_node, Gtk::TreeModel::Path const& path_)
  {
    if ((*i_node)[node.xpanded])
      return false;  

    std::string const& path ((*i_node)[node.path]);  
    store->erase (i_node->children().begin());

    Glib::Dir d (path);
    TPathList dir (d.begin(), d.end());
    d.close ();

    (*i_node)[node.xpanded] = true; 
    append_child_nodes (i_node, path, dir); 

    return false;
  }

  void
  FilesystemTree::set_root (std::string const& path)
  {
    using namespace Gtk;
    using namespace Glib;
    using namespace Bmp;

    if (!Glib::file_test (path, Glib::FILE_TEST_EXISTS))
      throw ErrorPathDoesNotExist();

    if (!Glib::file_test (path, Glib::FILE_TEST_IS_DIR))
      throw ErrorPathIsNotDirectory();

    store->clear ();

    Glib::Dir d (path);
    TPathList dir (d.begin(), d.end());
    d.close ();

    if (dir.empty())
      throw ErrorPathIsEmpty();

    TreeModel::iterator i_root = store->append();
    (*i_root)[node.name] = Glib::filename_to_utf8 (path);
    (*i_root)[node.type] = NODE_PATH; 
    (*i_root)[node.path] = path;
    (*i_root)[node.children] = true; 
    append_placeholder (i_root);

    expand_row (store->get_path (i_root), false);
  }

  void
  FilesystemTree::get_selected_uris (Bmp::VUri & uris)
  {
    using namespace Gtk;

    PathList paths (get_selection()->get_selected_rows());  
    
    for (PathList::const_iterator p = paths.begin() ; p != paths.end() ; ++p)
    {
      TreeModel::iterator i_node (store->get_iter (*p));
      uris.push_back (Glib::filename_to_uri ((*i_node)[node.path]));
    }
  }

  void
  FilesystemTree::cell_data_func_node_icon (Gtk::CellRenderer * cell_, Gtk::TreeModel::iterator const& m_iter)
  {
    Gtk::CellRendererPixbuf * cell = dynamic_cast<Gtk::CellRendererPixbuf *>(cell_);

    if ((*m_iter)[node.type] == NODE_FILE)
      cell->property_pixbuf() = m_icon_file; 
    else
    if ((*m_iter)[node.type] == NODE_PATH)
      cell->property_pixbuf() = m_icon_path; 
    else
      cell->property_pixbuf() = Glib::RefPtr<Gdk::Pixbuf>(0); 
    
  }

  FilesystemTree::FilesystemTree (BaseObjectType                       * obj,
          Glib::RefPtr<Gnome::Glade::Xml> const& xml) : Gtk::TreeView (obj)
  {
    using namespace Gtk;

#if GTK_CHECK_VERSION(2, 10, 0)
    gtk_tree_view_set_rubber_banding (GTK_TREE_VIEW (gobj()), TRUE);
    gtk_tree_view_set_enable_tree_lines (GTK_TREE_VIEW (gobj()), TRUE);
#endif

    set_fixed_height_mode ();
    get_selection()->set_mode (SELECTION_MULTIPLE);
    store = Gtk::TreeStore::create (node);
    set_model (store);

    TreeViewColumn * column = 0; 
    CellRendererPixbuf * cell_p = 0;
    CellRendererText * cell_t = 0;

    column = manage (new TreeViewColumn());

    cell_p = manage (new CellRendererPixbuf());
    column->pack_start (*cell_p, false);
    column->set_cell_data_func (*cell_p, sigc::mem_fun (this, &Bmp::FilesystemTree::cell_data_func_node_icon));
    
    cell_t = manage (new CellRendererText());
    column->pack_start (*cell_t, false);
    column->add_attribute (*cell_t, "text", 0); 
    column->set_sizing (TREE_VIEW_COLUMN_FIXED);

    append_column (*column);
    
    try {
      Glib::RefPtr<Gtk::IconTheme> icon_theme = Gtk::IconTheme::get_default();
      if (icon_theme)
        {
          m_icon_path = icon_theme->load_icon ("gnome-fs-directory", 24, Gtk::IconLookupFlags(0));
          if (!m_icon_path)
            m_icon_path = icon_theme->load_icon (GTK_STOCK_DIRECTORY, 24, Gtk::IconLookupFlags(0));

          m_icon_file = icon_theme->load_icon ("gnome-mime-audio", 24, Gtk::IconLookupFlags(0));
          if (!m_icon_file)
            m_icon_file = icon_theme->load_icon (GTK_STOCK_FILE, 24, Gtk::IconLookupFlags(0));
        }
      else throw 0;
      }
    catch (...)
      {
        m_icon_path = render_icon (Gtk::StockID (GTK_STOCK_DIRECTORY), Gtk::ICON_SIZE_SMALL_TOOLBAR); 
        m_icon_file = render_icon (Gtk::StockID (GTK_STOCK_FILE), Gtk::ICON_SIZE_SMALL_TOOLBAR); 
      }

    // Add container plugin filters
    Bmp::VFS::ExportDataList containers;
    vfs->get_containers (containers);
    for (Bmp::VFS::ExportDataList::const_iterator i = containers.begin () ; i != containers.end() ; ++i)
    {
      m_containers.push_back (i->extension);
    }
  }

  FilesystemTree::~FilesystemTree ()
  {
  }
}
