//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>

extern "C"
{
  #include <cdda_interface.h>
  namespace { const char * _dummy = strerror_tr[0]; }
}

// Musicbrainz
#include <musicbrainz/musicbrainz.h>

#include "main.hh"
#include "paths.hh"
#include "debug.hh"
#include "stock.hh"
#include "util.hh"
#include "util_file.hh"
#include "mbxml.hh"
#include "amazon.hh"

#include "uri++.hh"
#include "ui_toolbox.hh"
#include "network.hh"
#include "audio.hh"

#include "x_library.hh"
#include "x_service_core.hh"

#ifdef HAVE_HAL
#  include "x_hal.hh"
#endif //HAVE_HAL

#include "ui-part-cdda.hh"
#include "taskdialog.hh"

using namespace Gtk;
using namespace Glib;
using namespace Bmp::MusicBrainzXML;
using namespace std;

namespace
{
  typedef std::vector<bool> VBool; 

  const char *unknown_artist = _("Unknown Artist");
  const char *unknown_album  = _("Unknown Album");
  const char *unknown_title  = _("Unknown Title");

  int
  get_duration_from_sectors (int sectors)
  {
    const int bytes_per_sector  = 2352;
    const int bytes_per_seconds = (44100 / 8) / 16 / 2;
    return (sectors * bytes_per_sector / bytes_per_seconds)/1000;
  }

  const char *processor_names[] =
  {
    "MP3 (MPEG 1 Layer III)",
    "Ogg Vorbis",
    "FLAC Lossless Encoding"
  };

  const char *suffixes[] =
  {
    ".mp3",
    ".ogg",
    ".flac"
  };

  static boost::format ftrack ("%02d");
  static boost::format format_uri   ("cdda:///%d");
}

namespace Bmp
{
  namespace UiPart
  {

    guint
    CDDA::add_ui ()
    {
      return 0; 
    };

    CDDA::~CDDA ()
    {
    }

    CDDA::CDDA (RefPtr<Gnome::Glade::Xml> const& xml, Glib::RefPtr<Gtk::UIManager> ui_manager)

         : PlaybackSource (PlaybackSource::CAN_SEEK), Base (xml, ui_manager),
           m_processing_cd (false),
           m_playing (Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_MAIN, "playing.png"))),
           m_album_unknown (Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT))),
           m_anonymous_cd (false)

    {
        m_ref_xml->get_widget ("notebook-cdda", m_notebook_cdda);

        m_ref_xml->get_widget ("cdda-label-current-artist", m_label_artist);
        m_ref_xml->get_widget ("cdda-label-current-album", m_label_album);
        m_ref_xml->get_widget ("cdda-label-current-date", m_label_date);
        m_ref_xml->get_widget ("cdda-image-cover", m_cover_image);

        Audio::Caps audiocaps (Bmp::Audio::get_caps());

        if (!(audiocaps & Audio::CAPS_CDDA))
          {
            dynamic_cast<Image *>(m_ref_xml->get_widget ("i-audiocd-bad-2"))->set
                            (BMP_IMAGE_DIR_CDDA G_DIR_SEPARATOR_S "audiocd-unplayable.png");

            m_notebook_cdda->set_current_page (PAGE_NO_SUPPORT); 
            m_ref_xml->get_widget ("hbox-cdda-controls")->hide ();
            return;
          }
      
        m_ref_xml->get_widget ("b-cdda-rip",          m_b_cdda_rip);
        m_ref_xml->get_widget ("b-cdda-reload",       m_b_cdda_reload);
        m_ref_xml->get_widget ("b-cdda-stop",         m_b_cdda_stop);
        m_ref_xml->get_widget ("cdda-cbox-format",    m_cdda_cbox_format);
        m_ref_xml->get_widget ("cdda-cbox-quality",   m_cdda_cbox_quality);

        m_cdda_cbox_quality->set_active (1);

        m_cdda_cbox_format->clear ();
        CellRendererText *cell = manage (new CellRendererText());
        m_cdda_cbox_format->pack_start (*cell);
        m_cdda_cbox_format->add_attribute (*cell, "text", 0);
        cbox_processors_store = ListStore::create (processors); 
        m_cdda_cbox_format->set_model (cbox_processors_store);
    
        try {
          Audio::ProcessorCDDA_MP3 *p = new Audio::ProcessorCDDA_MP3("/tmp/dummy.mp3", 1, "/dev/cdrom", 0);
          delete p;
          TreeModel::iterator m_iter = cbox_processors_store->append ();
          (*m_iter)[processors.name] = processor_names[CDDA_MP3]; 
          (*m_iter)[processors.type] = CDDA_MP3; 
        } catch (...) {}

        try {
          Audio::ProcessorCDDA_Vorbis *p = new Audio::ProcessorCDDA_Vorbis("/tmp/dummy.ogg", 1, "/dev/cdrom", 0);
          delete p;
          TreeModel::iterator m_iter = cbox_processors_store->append ();
          (*m_iter)[processors.name] = processor_names[CDDA_VORBIS]; 
          (*m_iter)[processors.type] = CDDA_VORBIS; 
        } catch (...) {}

        try {
          Audio::ProcessorCDDA_FLAC *p = new Audio::ProcessorCDDA_FLAC("/tmp/dummy.flac", 1, "/dev/cdrom", 0);
          delete p;
          TreeModel::iterator m_iter = cbox_processors_store->append ();
          (*m_iter)[processors.name] = processor_names[CDDA_FLAC]; 
          (*m_iter)[processors.type] = CDDA_FLAC; 
        } catch (...) {}

        dynamic_cast<Image *>(m_ref_xml->get_widget ("throbber-cdda"))->set
                        (BMP_IMAGE_DIR G_DIR_SEPARATOR_S BMP_THROBBER);

        dynamic_cast<Image *>(m_ref_xml->get_widget ("i-audiocd"))->set
                        (BMP_IMAGE_DIR_CDDA G_DIR_SEPARATOR_S "audiocd-big.png");

        dynamic_cast<Image *>(m_ref_xml->get_widget ("i-audiocd-bad"))->set
                        (BMP_IMAGE_DIR_CDDA G_DIR_SEPARATOR_S "audiocd-unplayable.png");

        const char *headers[] =
        {
             N_("Track"),
             N_("Artist"), 
             N_("Title"), 
             N_("Duration"),
             N_("Progress"),
             N_("State"),
        };

        // --- CDDA View+Store 
        m_ref_xml->get_widget ("cdda-view", cdda_view);
        cdda_view->get_selection()->set_mode (SELECTION_SINGLE);

        TreeViewColumn *column = 0;

        cell_playing = manage (new CellRendererPixbuf());
        cell_playing->property_xalign() = 0.5; 

        cdda_view->append_column ("", *cell_playing);

        column = cdda_view->get_column (0);
        column->set_resizable (false);
        column->set_expand (false);
        column->property_min_width() = 30; 
        column->property_max_width() = 30; 
        column->set_cell_data_func (*cell_playing, sigc::mem_fun (this, &Bmp::UiPart::CDDA::cell_data_func_playing));

        Gtk::Image * image = Gtk::manage ( new Gtk::Image() );
        image->set (Glib::build_filename (BMP_IMAGE_DIR, "blue-speaker.png")); 
        column->set_widget (*image); 
        image->show();

        for (unsigned int n = 0; n < G_N_ELEMENTS(headers); ++n)
        {
          if (n == 3)
            {
              CellRendererText *cell = manage ( new CellRendererText() );

              cdda_view->append_column (_(headers[n]), *cell);

              column = cdda_view->get_column (n+1);
              column->set_resizable (false);
              column->set_expand (false);
              column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::UiPart::CDDA::cell_data_func_numbers));
            }
          else
          if (n == 4)
            {
              CellRendererProgress *cell = manage ( new CellRendererProgress() );

              cdda_view->append_column (_(headers[n]), *cell);

              column = cdda_view->get_column (n+1);
              column->add_attribute (*cell, "value", n);
              column->set_resizable (false);
              column->set_expand (false);
              column->set_fixed_width (100);
              column->set_sizing (TREE_VIEW_COLUMN_FIXED);
            }
          else
            {
              CellRendererText *cell = manage ( new CellRendererText() );
              if (n == 0) cell->property_xalign() = 1.0;

              cdda_view->append_column (_(headers[n]), *cell);
              if (n != 5)
              {
                column = cdda_view->get_column (n+1);
                column->add_attribute (*cell, "markup", n);
                column->set_resizable (false);
                column->set_expand (false);
              }
              else
              {
                column = cdda_view->get_column (n+1);
                column->set_resizable (false);
                column->set_expand (true);
                column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::UiPart::CDDA::cell_data_func_state));
              }
            }
        }

        cdda_view->get_column (5)->set_visible (false);
        cdda_view->get_column (6)->set_visible (false);

        cdda_store = ListStore::create (audiocd);
        cdda_view->set_model (cdda_store);
        cdda_view->get_selection()->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::UiPart::CDDA::selection_changed));

        if (cbox_processors_store->children().size())
        {
          m_cdda_cbox_format->
            set_active (0);

#ifdef HAVE_HAL

          m_b_cdda_reload->hide();

          hal->signal_cdda_inserted().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::on_hal_cdda_inserted)); 
          hal->signal_device_removed().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::on_hal_device_removed));
          hal->signal_ejected().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::on_hal_ejected));

#else

          m_b_cdda_reload->signal_clicked().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::refresh));

#endif //HAVE_HAL

          m_b_cdda_rip->signal_clicked().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::rip_start));
          m_b_cdda_stop->signal_clicked().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::rip_stop));

          m_b_cdda_rip->
            set_sensitive (false);

#ifdef HAVE_HAL

          m_notebook_cdda->
            set_current_page (PAGE_CD_INSERT);

#else
          m_notebook_cdda->
            set_current_page (PAGE_TOC);

#endif //HAVE_HAL

        }
      else
        {
          m_cdda_cbox_format->set_sensitive (false);
          m_cdda_cbox_quality->set_sensitive (false);

#ifdef HAVE_HAL
          m_b_cdda_reload->hide();
          m_b_cdda_reload->set_sensitive (false);
#endif //HAVE_HAL

          m_b_cdda_rip->set_sensitive (false);
          cdda_view->set_sensitive (false);
          m_notebook_cdda->set_current_page (PAGE_TOC);
        }

        cdda_view->signal_row_activated().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::activate_default));

        core->signal_shutdown_request().connect
              (sigc::mem_fun (this, &Bmp::UiPart::CDDA::on_shutdown_request));
    }

    bool
    CDDA::on_shutdown_request ()
    {
      using namespace Gtk;

      if (m_processing_cd)
      {
        MessageDialog dialog ((*(dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget ("main-ui")))),
                                _("<b>Ripping in Progress: Can't Shutdown!</b>\n\nBMPx currently cannot shut down because a "
                                  "CD is being ripped. "
                                  "Please either abort the encoding process, or wait until it is finished to exit BMPx."),
                                  true, MESSAGE_WARNING, BUTTONS_OK, true);
        dialog.set_title (_("Ripping CD: Can't Shutdown - BMP"));
        dialog.set_position (Gtk::WIN_POS_CENTER);
        dialog.run();
        return false;
      }
      else
      {
        return true;
      }
    }

    void
    CDDA::cell_data_func_numbers (Gtk::CellRenderer* cell, Gtk::TreeModel::iterator const& m_iter)
    {
      static boost::format fduration ("%d:%02d");
      Gtk::CellRendererText *cell_text = dynamic_cast<Gtk::CellRendererText *>(cell);
      unsigned int duration = (*m_iter)[audiocd.duration];
      unsigned int minutes = duration / 60;
      unsigned int seconds = duration % 60;
      cell_text->property_text() = (fduration % minutes % seconds).str(); 
    }

    void
    CDDA::activate_default (TreeModel::Path const& path, TreeViewColumn* column)
    {
      dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget ("main-ui"))->activate_default ();
    }

#ifdef HAVE_HAL
    void
    CDDA::on_hal_cdda_inserted (string udi, string devicefile)
    {
      if (!m_processing_cd)
        {
            MusicBrainz o;
            MBReleaseV releases; 
            string release_id;
            string album_id_url;
            list<int> argList;
            int n_tracks = 0;

            m_notebook_cdda->set_current_page (PAGE_LOADING);
            while (gtk_events_pending()) gtk_main_iteration ();

            m_current_udi = udi;
            m_current_device = devicefile;
            mcs->key_set ("audio", "cdrom-device", devicefile);

            // Use CDParanoia to make a map of which tracks are data tracks
            cdrom_drive *d = 0;
            d = cdda_identify (devicefile.c_str(), FALSE, NULL);
            if (!d)
              {
                m_notebook_cdda->set_current_page (PAGE_CD_BAD);
                while (gtk_events_pending()) gtk_main_iteration ();
                return;
              }

            if (cdda_open (d))
              {
                m_notebook_cdda->set_current_page (PAGE_CD_BAD);
                while (gtk_events_pending()) gtk_main_iteration ();
                return;
              }

            VBool audiotracks;
            for (int n = 0; n < d->tracks; ++n)
              {
                audiotracks.push_back (IS_AUDIO(d, n) ? true : false);
              }
            n_tracks = d->tracks;
            cdda_close (d);

            if (!Network::check_host("musicbrainz.org"))
              goto anonymous_cd;

            bool ret;
            o.UseUTF8 (true);
            o.SetDevice (devicefile);

            ret = o.Query(string (MBQ_GetCDInfo));
            if (!ret)
            {
              string error;
              o.GetQueryError (error);
              debug ("cdda", "%s: Query failed: %s", G_STRLOC, error.c_str());
              goto anonymous_cd;
            }

            debug ("cdda", "AudioCD: Got CDInfo");

            // Select first album
            argList.push_back (1);
            if (!o.Select (string (MBS_SelectAlbum), &argList))
              goto anonymous_cd;

            debug ("cdda", "AudioCD: Selected First Album");

            album_id_url = o.Data(string (MBE_AlbumGetAlbumId));

            if (album_id_url.empty ())
            {
              debug ("cdda", "No AlbumIDURL");
              goto anonymous_cd;
            }

            debug ("cdda", "AudioCD: Got Album ID URL");

            o.GetIDFromURL (album_id_url, release_id);
            if (release_id.size() && !release_id.empty())
              mb_releases_by_id (release_id, releases);

            anonymous_cd:

            if (releases.empty())
              {
                ret = o.Query(MBQ_GetCDTOC);
                if (!ret) 
                  {
                    m_notebook_cdda->set_current_page (PAGE_CD_BAD);
                    while (gtk_events_pending()) gtk_main_iteration ();
                    return;
                  }

                ustring _album;
                _album  .append ("<i>")
                        .append (unknown_album)
                        .append ("</i>");

                ustring _artist;
                _artist .append ("<b>")
                        .append (unknown_artist)
                        .append ("</b>");

                m_label_artist->
                  set_markup (_artist);

                m_label_album->
                  set_markup (_album);

                m_label_date->
                  set_text ("");

                m_cover_image->
                  set (m_album_unknown);

                for (int n = 0; n < n_tracks; ++n)
                {
                  TreeModel::iterator m_iter = cdda_store->append ();
                  
                  (*m_iter)[audiocd.tracknumber]  = n+1; 
                  (*m_iter)[audiocd.artist]       = unknown_artist; 
                  (*m_iter)[audiocd.duration]     = get_duration_from_sectors (o.DataInt(MBE_TOCGetTrackNumSectors, n+2));
                  (*m_iter)[audiocd.audio]        = audiotracks[n];

                  if (audiotracks[n])
                      (*m_iter)[audiocd.title] = (boost::format ("Track %02d") % (n+1)).str(); 
                  else
                      (*m_iter)[audiocd.title] = _("[data track]"); 
                }

                m_anonymous_cd = true;

                m_b_cdda_rip->
                  set_sensitive (true);

                m_cdda_cbox_format->
                  set_sensitive (true);

                m_cdda_cbox_quality->
                  set_sensitive (true);
              }
            else
              {
                ret = o.Query(MBQ_GetCDTOC);

                if (!ret) 
                  {
                    m_notebook_cdda->set_current_page (PAGE_CD_BAD);
                    while (gtk_events_pending()) gtk_main_iteration ();
                    return;
                  }

                m_release = releases[0];

                ustring artist ("<b>");
                artist.append (Markup::escape_text (m_release.artist_sort_name?
                                                    m_release.artist_sort_name.get() 
                                                    : m_release.artist_name)); 
                artist.append ("</b>");
                m_label_artist->set_markup (artist);
                
                ustring album ("<i>"); 
                album.append (Markup::escape_text (m_release.release_title));
                album.append ("</i>");
                m_label_album->set_markup (album);

                if (m_release.date)
                    m_label_date->set_text (m_release.date.get());
                else
                    m_label_date->set_text ("");

                if (m_release.asin)
                  {
                    Glib::RefPtr<Gdk::Pixbuf> cover;
                    Amazon::fetch( m_release.asin.get(), cover, false );
                    m_cover_image->set (cover->scale_simple (128, 128, Gdk::INTERP_BILINEAR));
                  }
                else
                  {
                    m_cover_image->set (m_album_unknown);
                  }

                unsigned int i = 1;
                for (MBTrackV::const_iterator n = m_release.tracks.get().begin(); n != m_release.tracks.get().end(); ++n)
                {
                  TreeModel::iterator m_iter = cdda_store->append ();

                  (*m_iter)[audiocd.tracknumber]  = n->tracknumber;
                  (*m_iter)[audiocd.artist]       = Markup::escape_text (n->artist_name);
                  (*m_iter)[audiocd.duration]     = get_duration_from_sectors (o.DataInt(MBE_TOCGetTrackNumSectors, i+1));
                  (*m_iter)[audiocd.percentage]   = 0; 
                  (*m_iter)[audiocd.state]        = ES_UNPROCESSED; 

                  if (audiotracks[i-1])
                    {
                      (*m_iter)[audiocd.title] = Markup::escape_text (n->track_title);
                      (*m_iter)[audiocd.track] = (*n); 
                    }
                  else
                    {
                      (*m_iter)[audiocd.title] = _("[data track]"); 
                    }

                  (*m_iter)[audiocd.audio] = audiotracks[i-1];
                  ++i;
                }

                m_anonymous_cd = false;

                m_b_cdda_rip->
                  set_sensitive (true);

                m_cdda_cbox_format->
                  set_sensitive (true);

                m_cdda_cbox_quality->
                  set_sensitive (true);
              }
          m_notebook_cdda->set_current_page (PAGE_TOC);
        }
    }

    void
    CDDA::on_hal_ejected (string udi)
    {
      std::cerr << "UDI: " << udi << std::endl;
      std::cerr << "Current UDI: " << m_current_udi << std::endl;

      if (udi == m_current_udi)
        {
          if (m_processing_cd)
            rip_stop ();
          else
            s_stop_request_.emit (); 
        }
    }

    void
    CDDA::on_hal_device_removed (string udi)
    {
        if ((!m_processing_cd) && udi == m_current_udi)
          {
            cdda_store->clear ();

            m_label_artist->
              set_text ("");

            m_label_album->
              set_text ("");

            m_label_date->
              set_text ("");

            m_cover_image->
              clear();

            m_b_cdda_rip->
              set_sensitive (false);

            m_notebook_cdda->
              set_current_page (PAGE_CD_INSERT);

            m_cdda_cbox_format->
              set_sensitive (false);
  
            m_cdda_cbox_quality->
              set_sensitive (false);

            cdda_view->get_column (5)->set_visible (false);
            cdda_view->get_column (6)->set_visible (false);
          }
    }
#else
    void
    CDDA::refresh ()
    {
      if (!m_processing_cd)
      {
        MusicBrainz o;
        MBReleaseV releases; 
        string release_id;
        string album_id_url;
        list<int> argList;
        int n_tracks = 0;

        cdda_store->
          clear();

        m_label_artist->
          set_text ("");

        m_label_album->
          set_text ("");

        m_label_date->
          set_text ("");

        m_cover_image->
          clear();

        m_b_cdda_rip->
          set_sensitive (false);

        m_b_cdda_stop-> 
          set_sensitive (false);

        m_cdda_cbox_format->
          set_sensitive (false);

        m_cdda_cbox_quality->
          set_sensitive (false);

        m_notebook_cdda->
          set_current_page (PAGE_LOADING);

        while (gtk_events_pending()) gtk_main_iteration ();

        // Use CDParanoia to make a map of which tracks are data tracks
        cdrom_drive *d = 0;
        d = cdda_identify (mcs->key_get<string>("audio", "cdrom-device").c_str() , 0, 0);
        if (!d)
          {
            m_notebook_cdda->set_current_page (PAGE_TOC);
            while (gtk_events_pending()) gtk_main_iteration ();
            return;
          }

        if (cdda_open (d))
          {
            m_notebook_cdda->set_current_page (PAGE_TOC);
            while (gtk_events_pending()) gtk_main_iteration ();
            return;
          }

        VBool audiotracks;
        for (int n = 0; n < d->tracks; ++n)
          {
            audiotracks.push_back (IS_AUDIO(d, n) ? true : false);
          }
        n_tracks = d->tracks;
        cdda_close (d);

        if (!Network::check_host("musicbrainz.org")) goto anonymous_cd;

        bool ret;
        o.UseUTF8 (true);
        o.SetDevice (mcs->key_get<string>("audio", "cdrom-device"));

        ret = o.Query(string (MBQ_GetCDInfo));
        if (!ret)
        {
          string error;
          o.GetQueryError (error);
          debug ("cdda", "%s: Query failed: %s", G_STRLOC, error.c_str());
          goto anonymous_cd; 
        }

        debug ("cdda", "AudioCD: Got CDInfo");

        // Select first album
        argList.push_back (1);
        if (!o.Select (string (MBS_SelectAlbum), &argList))
        {
          goto anonymous_cd; 
        }

        debug ("cdda", "AudioCD: Selected First Album");

        album_id_url = o.Data(string (MBE_AlbumGetAlbumId));

        if (album_id_url.empty ())
        {
          debug ("cdda", "No AlbumIDURL");
          goto anonymous_cd; 
        }

        debug ("cdda", "AudioCD: Got Album ID URL");

        o.GetIDFromURL (album_id_url, release_id);
        if (release_id.size() && !release_id.empty())
          {
            mb_releases_by_id (release_id, releases);
          }

        anonymous_cd:

        if (releases.empty())
          {
            ret = o.Query(MBQ_GetCDTOC);
            if (!ret) 
              {
                m_notebook_cdda->set_current_page (PAGE_TOC);
                while (gtk_events_pending()) gtk_main_iteration ();
                return;
              }
      
            ustring _album;
            _album  .append ("<i>")
                    .append (unknown_album)
                    .append ("</i>");

            ustring _artist;
            _artist .append ("<b>")
                    .append (unknown_artist)
                    .append ("</b>");

            m_label_artist->
              set_markup (_artist);

            m_label_album->
              set_markup (_album);

            m_label_date->
              set_text ("");

            m_cover_image->
              set (m_album_unknown);

            for (int n = 0; n < n_tracks; ++n)
            {
              TreeModel::iterator m_iter = cdda_store->append ();
              
              (*m_iter)[audiocd.tracknumber]  = n+1; 
              (*m_iter)[audiocd.artist]       = unknown_artist; 
              (*m_iter)[audiocd.duration]     = get_duration_from_sectors (o.DataInt(MBE_TOCGetTrackNumSectors, n+2));
              (*m_iter)[audiocd.audio]        = audiotracks[n];

              if (audiotracks[n])
                  (*m_iter)[audiocd.title] = (boost::format ("Track %02d") % (n+1)).str(); 
              else
                  (*m_iter)[audiocd.title] = _("[data track]"); 
            }

            m_anonymous_cd = true;

            m_b_cdda_rip->
              set_sensitive (false);

            m_cdda_cbox_format->
              set_sensitive (false);

            m_cdda_cbox_quality->
              set_sensitive (false);
          }
        else
          {
            ret = o.Query(MBQ_GetCDTOC);
            if (!ret) 
              {
                m_notebook_cdda->set_current_page (PAGE_CD_BAD);
                while (gtk_events_pending()) gtk_main_iteration ();
                return;
              }

            cdrom_drive *d = 0;
            d = cdda_identify (mcs->key_get<string>("audio", "cdrom-device").c_str() , 0, 0);
            if (!d)
              {
                m_notebook_cdda->set_current_page (PAGE_CD_BAD);
                while (gtk_events_pending()) gtk_main_iteration ();
                return;
              }

            if (cdda_open (d))
              {
                m_notebook_cdda->set_current_page (PAGE_CD_BAD);
                while (gtk_events_pending()) gtk_main_iteration ();
                return;
              }

            VBool audiotracks;
            for (int n = 0; n < d->tracks; ++n)
              {
                audiotracks.push_back (IS_AUDIO(d, n) ? true : false);
              }

            n_tracks = d->tracks;
            cdda_close (d);

            m_release = releases[0];

            ustring artist ("<b>");
            artist.append (Markup::escape_text (m_release.artist_sort_name?
                                                m_release.artist_sort_name.get() 
                                                : m_release.artist_name)); 
            artist.append ("</b>");
            m_label_artist->set_markup (artist);
            
            ustring album ("<i>"); 
            album.append (Markup::escape_text (m_release.release_title));
            album.append ("</i>");
            m_label_album->set_markup (album);

            if (m_release.date)
                m_label_date->set_text (m_release.date.get());
            else
                m_label_date->set_text ("");

            if (m_release.asin)
              {
                Glib::RefPtr<Gdk::Pixbuf> cover;
                Amazon::fetch( m_release.asin.get(), cover, false );
                m_cover_image->set (cover->scale_simple (128, 128, Gdk::INTERP_BILINEAR));
              }
            else
              {
                m_cover_image->set (m_album_unknown);
              }

            unsigned int i = 1;

            for (MBTrackV::const_iterator n = m_release.tracks.get().begin(); n != m_release.tracks.get().end(); ++n)
            {
              TreeModel::iterator m_iter = cdda_store->append ();

              (*m_iter)[audiocd.tracknumber]  = n->tracknumber;
              (*m_iter)[audiocd.artist]       = Markup::escape_text (n->artist_name);
              (*m_iter)[audiocd.duration]     = get_duration_from_sectors (o.DataInt(MBE_TOCGetTrackNumSectors, i+1));
              (*m_iter)[audiocd.percentage]   = 0; 
              (*m_iter)[audiocd.state]        = ES_UNPROCESSED; 

              if (audiotracks[i-1])
                {
                  (*m_iter)[audiocd.title] = Markup::escape_text (n->track_title);
                  (*m_iter)[audiocd.track] = (*n); 
                }
              else
                {
                  (*m_iter)[audiocd.title] = _("[data track]"); 
                }

              (*m_iter)[audiocd.audio] = audiotracks[i-1];
              ++i;
            }

            m_anonymous_cd = false;

            m_b_cdda_rip->
              set_sensitive (true);

            m_cdda_cbox_format->
              set_sensitive (true);

            m_cdda_cbox_quality->
              set_sensitive (true);
          }
        m_notebook_cdda->set_current_page (PAGE_TOC);
      }
   }
#endif //HAVE_HAL

    void
    CDDA::rip_stop ()
    {
      m_mutex_rip.unlock ();
    }

    void
    CDDA::rip_start ()
    {
      if (m_processing_cd) return;

      stop ();

      unsigned int current_rip_track;
      std::string  current_rip_basepath;

      // Disable Controls 
      m_b_cdda_rip->set_sensitive (false);
      m_cdda_cbox_format->set_sensitive (false);
      m_cdda_cbox_quality->set_sensitive (false);
      m_processing_cd = true;

      // Select the Destination Folder
      FileChooserDialog fcdialog (_("Select Destination Folder - BMP"), FILE_CHOOSER_ACTION_SELECT_FOLDER);
      fcdialog.add_button (Stock::CANCEL, GTK_RESPONSE_CANCEL);
      fcdialog.add_button (Stock::OK, GTK_RESPONSE_OK);
      if (fcdialog.run () == GTK_RESPONSE_CANCEL)
        {
          fcdialog.hide ();
          m_b_cdda_rip->set_sensitive (true);
          m_cdda_cbox_format->set_sensitive (true);
          m_cdda_cbox_quality->set_sensitive (true);
          m_processing_cd = false;
          return;
        }

      fcdialog.hide ();
      current_rip_basepath = filename_from_uri (fcdialog.get_current_folder_uri());

      for (TreeNodeChildren::iterator m_iter  = cdda_store->children().begin()  ;
                                      m_iter != cdda_store->children().end()    ; ++m_iter)
        {
          (*m_iter)[audiocd.state] = ES_WAITING; 
        }

      m_mutex_rip.lock ();
      m_b_cdda_stop->set_sensitive (true);

#ifndef HAVE_HAL
      m_b_cdda_reload->set_sensitive (false);
#endif //HAVE_HAL

      cdda_view->get_column (5)->set_visible (true);
      cdda_view->get_column (6)->set_visible (true);

      cdda_view->get_selection()->unselect_all ();
      cdda_view->get_selection()->set_mode (SELECTION_NONE);

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);

      TreeModel::iterator c_iter = m_cdda_cbox_format->get_active ();
      ProcessorType audioformat = (*c_iter)[processors.type]; 

      for (unsigned int n = 0; n < m_release.tracks.get().size(); ++n)
      {
        current_rip_track = n; 

        TreePath path;        
        path.append_index (current_rip_track);

        m_current_iter = cdda_store->get_iter (path);   

        if ((*m_current_iter)[audiocd.state] == ES_DONE)
          continue;

        MBTrack const& track = (*m_current_iter)[audiocd.track];

        ustring filename;
        filename  .append ((ftrack % (n+1)).str())
                  .append (" - ")
                  .append (m_release.artist_name)
                  .append (" - ")
                  .append (m_release.release_title)
                  .append (" - ")
                  .append (track.track_title);

        filename.append (suffixes[audioformat]);

        Audio::ProcessorBase *processor = 0;

        string rip_basename = filename_from_utf8 (filename);
        boost::algorithm::replace_all (rip_basename, "/" , "-");
        string rip_filename = build_filename (current_rip_basepath, rip_basename); 

        ustring rip_uri = filename_to_uri (rip_filename);
        (*m_current_iter)[audiocd.location] = rip_uri; 

        std::string _device;

#ifdef HAVE_HAL
        _device = m_current_device;
#else
        _device = mcs->key_get<string>("audio", "cdrom-device");
#endif //HAVE_HAL

        if (audioformat == CDDA_MP3)
          {
              processor = 
                new Bmp::Audio::ProcessorCDDA_MP3 (rip_filename, current_rip_track+1,
                      _device, 
                      m_cdda_cbox_quality->get_active_row_number());
          }
        else
        if (audioformat == CDDA_VORBIS)
          {
              processor = 
                new Bmp::Audio::ProcessorCDDA_Vorbis (rip_filename, current_rip_track+1,
                                                      _device, 
                                                      m_cdda_cbox_quality->get_active_row_number());
          }
        else
        if (audioformat == CDDA_FLAC)
          {
              processor = 
                new Bmp::Audio::ProcessorCDDA_FLAC (rip_filename, current_rip_track+1,
                                                    _device, 
                                                    m_cdda_cbox_quality->get_active_row_number());
          }

        processor->signal_position().connect
          (sigc::mem_fun (this, &Bmp::UiPart::CDDA::stream_position));
        processor->signal_eos().connect
          (sigc::mem_fun (this, &Bmp::UiPart::CDDA::stream_eos));
        processor->signal_error().connect
          (sigc::mem_fun (this, &Bmp::UiPart::CDDA::stream_error));

        m_stream_eos = false;

        (*m_current_iter)[audiocd.state] = ES_RIPPING; 

        processor->run ();

        while (!m_stream_eos)
          {
            while (gtk_events_pending()) gtk_main_iteration ();
            if (m_mutex_rip.trylock())
              {
                processor->stop ();
                delete processor;

                (*m_current_iter)[audiocd.state] = ES_WAITING; 
                m_mutex_rip.unlock ();

                m_processing_cd = false;

                m_b_cdda_rip->
                  set_sensitive (true);

                m_b_cdda_stop->
                  set_sensitive (false);

#ifndef HAVE_HAL
                m_b_cdda_reload->
                  set_sensitive (true);
#endif //!HAVE_HAL

                cdda_view->get_selection()->set_mode (SELECTION_SINGLE);

                return;
              }
          }
        
        delete processor;

        (*m_current_iter)[audiocd.state] = ES_TAGGING; 

        Library::UTrack update_track; 
        mb_track_to_update_track ((*m_current_iter)[audiocd.track], update_track);
        mb_release_to_update_track (m_release, update_track);

        try {
            library->metadata_set_taglib (rip_uri, update_track);
            (*m_current_iter)[audiocd.percentage] = 100; 
            (*m_current_iter)[audiocd.state] = ES_DONE; 
          }
        catch (Bmp::Library::MetadataWriteError& cxe)
          {
            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "Unable to modify metadata for file %s, Error: %s", rip_filename.c_str(), cxe.what());
            (*m_current_iter)[audiocd.percentage] = 0; 
            (*m_current_iter)[audiocd.state] = ES_ERROR; 
            g_unlink (rip_filename.c_str());
          }
      }

      cdda_view->get_selection()->set_mode (SELECTION_SINGLE);
      m_b_cdda_rip->set_sensitive (true);
      m_b_cdda_stop->set_sensitive (false);

      m_label_artist->set_text ("");
      m_label_album->set_text ("");
      m_label_date->set_text ("");
      m_cover_image->clear();

      m_processing_cd = false;

#ifdef HAVE_HAL

      m_notebook_cdda->set_current_page (PAGE_CD_INSERT);
      m_cdda_cbox_format->set_sensitive (false);
      m_cdda_cbox_quality->set_sensitive (false);

#else

      m_notebook_cdda->set_current_page (PAGE_TOC);
      m_b_cdda_reload->set_sensitive (true);

#endif //HAVE_HAL

      m_mutex_rip.unlock ();

      VUri uri_list;
      int warn = 0;
      for (TreeNodeChildren::iterator m_iter  = cdda_store->children().begin(); 
                                      m_iter != cdda_store->children().end(); ++m_iter)
        {
          if ((*m_iter)[audiocd.state] == ES_DONE)
            {
              uri_list.push_back (ustring((*m_iter)[audiocd.location]));
            }
          else
            {
              warn = 1;
            }
        }

      ustring message = _("You can now insert the new album into the Library if you wish to.");

      if (warn)
          message += _("\n<b>NOTE:</b> Not all files were ripped successfully, the import will be incomplete!");

      TaskDialog dialog ( _("Audio CD Ripping - BMP"),
                          _("Ripping Complete."),
                          Gtk::MESSAGE_INFO, message);

      dialog.add_button ( _("Import Album"),
                          _("This will import the album into your Music Library"),
                          Gtk::Stock::GO_FORWARD,
                          GTK_RESPONSE_OK);

      dialog.add_button ( _("Don't Import"),
                          _("Do not import the Album at this time"),
                          Gtk::Stock::CANCEL,
                          GTK_RESPONSE_CANCEL);

      dialog.set_default_response (GTK_RESPONSE_OK);

      int response = dialog.run ();
      dialog.hide ();

      if (response == GTK_RESPONSE_OK)
        {
          if (m_release.asin) 
              try { Amazon::cache( m_release.asin.get() ); } catch (...) {}
          s_request_import_.emit (uri_list);
        } 

      cdda_store->clear (); 
    }

    void
    CDDA::stream_eos ()
    {
      m_stream_eos = true;
    }

    void
    CDDA::stream_error (Glib::ustring error)
    {
      m_stream_eos = true;
      Gtk::MessageDialog dialog (error, true, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
      dialog.run ();
    }
   
    void
    CDDA::stream_position (int position)
    {
      unsigned int duration = (*m_current_iter)[audiocd.duration]; 
      int percentage = (int(((1.*position)/(1.*duration)) * 100));
      if (!duration || !percentage)
        {
          (*m_current_iter)[audiocd.percentage] = 0; 
        }
      else
        {
          (*m_current_iter)[audiocd.percentage] = percentage; 
        }

      while (gtk_events_pending()) gtk_main_iteration ();

    }

    void
    CDDA::cell_data_func_state (CellRenderer* cell, TreeModel::iterator const& m_iter)
    {
      const char *states[] =
      {
        "Unprocessed",
        "Waiting", 
        "Encoding",      
        "Tagging",      
        "Done",
        "Error!"
      };
      CellRendererText *cell_text = dynamic_cast<CellRendererText *>(cell);
      cell_text->property_text() = states[int((*m_iter)[audiocd.state])];
    }

    void
    CDDA::send_title ()
    {
      TreeModel::iterator const& m_iter = m_current_row.get();

      if (!m_anonymous_cd)
        {
          MBTrack const& track = (*m_iter)[audiocd.track];

          ustring artist, album, title;

          artist  .append (Markup::escape_text (track.artist_name)); 

          album   .append ("<i>")
                  .append (Markup::escape_text (m_release.release_title))
                  .append ("</i>");

          title   .append ("<b>")
                  .append (Markup::escape_text (track.track_title)) 
                  .append ("</b>");

          static boost::format format_title ("%s (%s: %s)");
          ustring t ((format_title % title.c_str() % artist.c_str() % album.c_str()).str()); 

          SimpleTrackInfo sti;

          sti.artist    = track.artist_name;
          sti.title     = track.track_title;
          sti.album     = m_release.release_title;
          sti.asin      = m_release.asin;
          sti.duration  = track.duration/1000;

          if (m_release.asin)
            {
              Amazon::fetch( m_release.asin.get(), sti.image );
            }

          s_track_info_.emit (t, sti);
        }
      else
        {
          Glib::ustring const& title = (*m_iter)[audiocd.title];
          Glib::ustring text = (boost::format ("<b>%s</b> (%s: <i>%s</i>)") % title.c_str() % unknown_artist % unknown_album).str(); 
          SimpleTrackInfo sti;

          sti.artist    = Glib::ustring (unknown_artist);
          sti.album     = Glib::ustring (unknown_album);
          sti.title     = title;
          sti.duration  = ((unsigned int)(*m_iter)[audiocd.duration])/1000;

          s_track_info_.emit (text, sti);
        }
    }

    // Bmp::PlaybackSource
    void
    CDDA::cell_data_func_playing (CellRenderer* cell_, TreeModel::iterator const& m_iter)
    {
      CellRendererPixbuf *cell = dynamic_cast<CellRendererPixbuf *>(cell_);
      if (m_current_row)
        {
          TreeModel::Path path1 (cdda_store->get_path (m_iter)); 
          TreeModel::Path path2 (cdda_store->get_path (m_current_row.get()));
          if (path1 == path2)
            {
              cell->property_pixbuf() = m_playing; 
              return;
            } 
        }
      cell->property_pixbuf() = RefPtr<Gdk::Pixbuf>(0);
    }

    ustring
    CDDA::get_uri ()
    {
      TreeModel::Path path (m_current_row.get());
      ustring uri ((format_uri % (path.get_indices().data()[0] + 1)).str());
      return uri;
    }

    bool
    CDDA::go_next ()
    {
      TreeModel::Path path (m_current_row.get());

      TreeModel::Path path_old (path);
      TreeModel::iterator m_iter_old = cdda_view->get_model()->get_iter (path);

      unsigned int size = cdda_view->get_model()->children().size()-1; 
      unsigned int position = path.get_indices().data()[0];
  
      go_next_loop:

      if ((position + 1) > size) return false;

      path = TreeModel::Path (1, position+1); 
      TreeModel::iterator const& m_iter = cdda_view->get_model()->get_iter (path);
      if (!(*m_iter)[audiocd.audio])
        {
          position++;
          goto go_next_loop;
        }

      m_current_row = m_iter; 

      cdda_view->get_model()->row_changed (path_old, m_iter_old); 
      cdda_view->get_model()->row_changed (path, m_iter); 

      if (position < size)
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if (position > 0) 
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      s_caps_.emit (m_caps);
      send_title ();
      return true;
    }

    bool
    CDDA::go_prev ()
    {
      TreeModel::Path path (m_current_row.get());

      TreeModel::Path path_old (path);
      TreeModel::iterator m_iter_old = cdda_view->get_model()->get_iter (path);

      unsigned int size = cdda_view->get_model()->children().size()-1; 
      unsigned int position = path.get_indices().data()[0];

      go_prev_loop:

      if ((position - 1) < 0) return false;

      path = TreeModel::Path (1, position-1); 
      TreeModel::iterator const& m_iter = cdda_view->get_model()->get_iter (path);
      if (!(*m_iter)[audiocd.audio])
        {
          position--;
          goto go_prev_loop;
        }

      m_current_row = m_iter; 

      cdda_view->get_model()->row_changed (path_old, m_iter_old); 
      cdda_view->get_model()->row_changed (path, m_iter); 

      if (position < size)
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if (position > 0) 
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      s_caps_.emit (m_caps);
      send_title ();
      return true;
    }

    void
    CDDA::stop ()
    {
      if (m_current_row)
        {
          TreeModel::iterator m_iter = m_current_row.get();
          TreeModel::Path path (m_iter);
          m_current_row.reset ();
          cdda_view->get_model()->row_changed (path, m_iter);
        }

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);

      s_caps_.emit (m_caps);

      m_b_cdda_rip->
        set_sensitive (true);

      m_cdda_cbox_format->
        set_sensitive (true);

      m_cdda_cbox_quality->
        set_sensitive (true);
    }

    void
    CDDA::play ()
    {
      m_current_row = cdda_view->get_selection()->get_selected ();
      TreeModel::Path path (m_current_row.get());
      cdda_view->get_model()->row_changed (path, m_current_row.get());

      m_b_cdda_rip->
        set_sensitive (false);

      m_cdda_cbox_format->
        set_sensitive (false);

      m_cdda_cbox_quality->
        set_sensitive (false);

      cdda_view->get_column (5)->set_visible (false);
      cdda_view->get_column (6)->set_visible (false);

      unsigned int position = path.get_indices().data()[0];
      unsigned int size     = cdda_view->get_model()->children().size()-1; 

      if (position < size)
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if (position > 0) 
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);

      if (!m_anonymous_cd) m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);

      s_caps_.emit (m_caps);
      send_title ();
    }

    void
    CDDA::restore_context ()
    {
    }

    GHashTable*
    CDDA::get_metadata ()
    {
      if (m_current_row && !m_anonymous_cd)
        {
          TreeModel::iterator const& m_iter = (m_current_row.get());
          MBTrack const& track = (*m_iter)[audiocd.track];

          GHashTable *table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
          GValue     *value = 0;

          // Time
          value = g_new0 (GValue,1);
          g_value_init (value, G_TYPE_INT);
          g_value_set_int (value, int(track.duration/1000)); 
          g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_TIME).c_str()), value);

          // Artist + ID
          value = g_new0 (GValue,1);
          g_value_init (value, G_TYPE_STRING);
          g_value_set_string (value, track.artist_name.c_str()); 
          g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_ARTIST).c_str()), value);

          value = g_new0 (GValue,1);
          g_value_init (value, G_TYPE_STRING);
          g_value_set_string (value, track.artist_id.c_str()); 
          g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_MB_ARTIST_ID).c_str()), value);

          // Album + ID
          value = g_new0 (GValue,1);
          g_value_init (value, G_TYPE_STRING);
          g_value_set_string (value, m_release.release_title.c_str()); 
          g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_ALBUM).c_str()), value);

          value = g_new0 (GValue,1);
          g_value_init (value, G_TYPE_STRING);
          g_value_set_string (value, m_release.release_id.c_str()); 
          g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_MB_ALBUM_ID).c_str()), value);
      
          // Title, ASIN
          value = g_new0 (GValue,1);
          g_value_init (value, G_TYPE_STRING);
          g_value_set_string (value, track.track_title.c_str()); 
          g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_TITLE).c_str()), value);

          if (m_release.asin)
            {
              value = g_new0 (GValue,1);
              g_value_init (value, G_TYPE_STRING);
              g_value_set_string (value, m_release.asin.get().c_str()); 
              g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_ASIN).c_str()), value);
            }

          return table;
        }
      else
        {
          return NULL;
        }
    }

    void
    CDDA::selection_changed ()
    {
      using namespace Gtk;

      if (cdda_view->get_selection ()->count_selected_rows() == 1)
      {
        TreeModel::iterator const& m_iter = cdda_view->get_selection()->get_selected ();
        if ((*m_iter)[audiocd.audio])
          {
            m_caps = Caps (m_caps |  PlaybackSource::CAN_PLAY);
            s_caps_.emit (m_caps);
            return;
          }
      }

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);
    }
  } // namespace UiPart
} // namespace Bmp
