/*
 * BRLTTY - A background process providing access to the console screen (when in
 *          text mode) for a blind person using a refreshable braille display.
 *
 * Copyright (C) 1995-2008 by The BRLTTY Developers.
 *
 * BRLTTY comes with ABSOLUTELY NO WARRANTY.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * Web Page: http://mielke.cc/brltty/
 *
 * This software is maintained by Dave Mielke <dave@mielke.cc>.
 */

#include "prologue.h"

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>

#ifdef HAVE_LIBICUUC
#include <unicode/uchar.h>
#endif /* HAVE_LIBICUUC */

#if defined(HAVE_PKG_CURSES)
#define USE_CURSES
#include <curses.h>
#elif defined(HAVE_PKG_NCURSES)
#define USE_CURSES
#include <ncurses.h>
#elif defined(HAVE_PKG_NCURSESW)
#define USE_CURSES
#define USE_FUNC_GET_WCH
#include <ncursesw/ncurses.h>
#else
#warning curses package either unspecified or unsupported
#define printw printf
#define clear() printf("\r\n\v")
#define refresh() fflush(stdout)
#define beep() printf("\a")
#endif /* HAVE_PKG_CURSES */

#include "program.h"
#include "options.h"
#include "misc.h"
#include "brl.h"
#include "charset.h"
#include "tbl.h"
#include "tbl_internal.h"

#define BRLAPI_NO_DEPRECATED
#include "brlapi.h"

static char *opt_characterSet;
static char *opt_inputFormat;
static char *opt_outputFormat;
static int opt_translate;
static char *opt_dataDirectory;
static int opt_edit;

BEGIN_OPTION_TABLE(programOptions)
  { .letter = 'D',
    .word = "data-directory",
    .flags = OPT_Hidden,
    .argument = "file",
    .setting.string = &opt_dataDirectory,
    .defaultSetting = DATA_DIRECTORY,
    .description = "Path to directory for configuration files."
  },

  { .letter = 'e',
    .word = "edit",
    .setting.flag = &opt_edit,
    .description = "Edit table."
  },

  { .letter = 't',
    .word = "translate",
    .setting.flag = &opt_translate,
    .description = "Translate."
  },

  { .letter = 'i',
    .word = "input-format",
    .argument = "format",
    .setting.string = &opt_inputFormat,
    .description = "Format of input file."
  },

  { .letter = 'o',
    .word = "output-format",
    .argument = "format",
    .setting.string = &opt_outputFormat,
    .description = "Format of output file."
  },

  { .letter = 'c',
    .word = "character-set",
    .argument = "charset",
    .setting.string = &opt_characterSet,
    .description = "8-bit character set to use."
  },
END_OPTION_TABLE

static const DotsTable dotsInternal = {
  BRL_DOT1, BRL_DOT2, BRL_DOT3, BRL_DOT4,
  BRL_DOT5, BRL_DOT6, BRL_DOT7, BRL_DOT8
};

static const DotsTable dots12345678 = {
  0X01, 0X02, 0X04, 0X08, 0X10, 0X20, 0X40, 0X80
};

static const DotsTable dots14253678 = {
  0X01, 0X04, 0X10, 0X02, 0X08, 0X20, 0X40, 0X80
};

static unsigned char
mapDots (unsigned char input, const DotsTable from, const DotsTable to) {
  unsigned char output = 0;
  {
    int dot;
    for (dot=0; dot<DOTS_TABLE_SIZE; ++dot) {
      if (input & from[dot]) output |= to[dot];
    }
  }
  return output;
}

static int
readTable_Native (const char *path, FILE *file, TranslationTable table, void *data) {
  return tblLoad_Native(path, file, table,
                        TBL_UNDEFINED | TBL_DUPLICATE | TBL_UNUSED);
}

static int
writeTable_Native (const char *path, FILE *file, const TranslationTable table, void *data) {
  int index;

  if (fprintf(file, "# generated by %s\n", programName) == EOF) goto error;

  {
    const char *charset = getCharset();
    if (charset)
      if (fprintf(file, "# charset: %s\n", charset) == EOF)
        goto error;
  }

  for (index=0; index<TRANSLATION_TABLE_SIZE; ++index) {
    unsigned char cell = table[index];
    if (fprintf(file, "\\X%02X (", index) == EOF) goto error;

#define DOT(dot) if (fputs(((cell & BRL_DOT##dot)? #dot: " "), file) == EOF) goto error
    DOT(1);
    DOT(2);
    DOT(3);
    DOT(4);
    DOT(5);
    DOT(6);
    DOT(7);
    DOT(8);
#undef DOT

    if (fprintf(file, ")\n") == EOF) goto error;
  }
  return 1;

error:
  return 0;
}

static int
readTable_Binary (const char *path, FILE *file, TranslationTable table, void *data) {
  {
    int character;
    for (character=0; character<TRANSLATION_TABLE_SIZE; ++character) {
      int cell = fgetc(file);

      if (cell == EOF) {
        if (ferror(file)) {
          LogPrint(LOG_ERR, "input error: %s: %s", path, strerror(errno));
        } else {
          LogPrint(LOG_ERR, "table too short: %s", path);
        }
        return 0;
      }

      if (data) cell = mapDots(cell, data, dotsInternal);
      table[character] = cell;
    }
  }

  return 1;
}

static int
writeTable_Binary (const char *path, FILE *file, const TranslationTable table, void *data) {
  {
    int character;
    for (character=0; character<TRANSLATION_TABLE_SIZE; ++character) {
      unsigned char cell = table[character];
      if (data) cell = mapDots(cell, dotsInternal, data);
      if (fputc(cell, file) == EOF) {
        LogPrint(LOG_ERR, "output error: %s: %s", path, strerror(errno));
        return 0;
      }
    }
  }

  return 1;
}

#ifdef HAVE_ICONV_H
static int
readTable_Gnome (const char *path, FILE *file, TranslationTable table, void *data) {
  return tblLoad_Gnome(path, file, table,
                       TBL_UNDEFINED | TBL_DUPLICATE | TBL_UNUSED);
}

static int
writeTable_Gnome (const char *path, FILE *file, const TranslationTable table, void *data) {
  int i;

  /* TODO UNKNOWN-CHAR %wc all */
  if (fprintf(file, "ENCODING UTF-8\n") == EOF) goto error;
  if (fprintf(file, "# generated by %s\n", programName) == EOF) goto error;

  for (i=0; i<=0XFF; i++) {
    char c = i;
    wchar_t wc;
    wchar_t pattern = BRL_UC_ROW | table[i];

    if ((wc = convertCharToWchar(c)) == WEOF) continue;
    if (isprint(i) && !isspace(i)) {
      Utf8Buffer utf8C, utf8Pattern;
      if (!convertWcharToUtf8(wc, utf8C)) continue;
      if (!convertWcharToUtf8(pattern, utf8Pattern)) continue;
      if (fprintf(file, "UCS-CHAR %s %s\n", utf8C, utf8Pattern) == EOF) goto error;
    } else {
      uint32_t u32 = pattern;
      if (fprintf(file, "UNICODE-CHAR U+%04x U+%04"PRIx32"\n", i, u32) == EOF) goto error;
    }
  }
  return 1;

error:
  return 0;
}
#endif /* HAVE_ICONV_H */

typedef int TableReader (const char *path, FILE *file, TranslationTable table, void *data);
typedef int TableWriter (const char *path, FILE *file, const TranslationTable table, void *data);

typedef struct {
  const char *name;
  TableReader *read;
  TableWriter *write;
  void *data;
} FormatEntry;

static const FormatEntry formatEntries[] = {
  {"tbl", readTable_Native, writeTable_Native, NULL},
  {"a2b", readTable_Binary, writeTable_Binary, &dots12345678},
  {"sbl", readTable_Binary, writeTable_Binary, &dots14253678},

#ifdef HAVE_ICONV_H
  {"gnb", readTable_Gnome, writeTable_Gnome, NULL},
#endif /* HAVE_ICONV_H */

  {NULL}
};

static const FormatEntry *
findFormatEntry (const char *name) {
  const FormatEntry *format = formatEntries;
  while (format->name) {
    if (strcmp(name, format->name) == 0) return format;
    ++format;
  }
  return NULL;
}

static const char *
findFileExtension (const char *path) {
  const char *name = strrchr(path, '/');
  return strrchr((name? name: path), '.');
}

static const FormatEntry *
getFormatEntry (const char *name, const char *path, const char *description) {
  if (!(name && *name)) {
    name = findFileExtension(path);
    if (!(name && *++name)) {
      LogPrint(LOG_ERR, "unspecified %s format.", description);
      exit(2);
    }
  }

  {
    const FormatEntry *format = findFormatEntry(name);
    if (format) return format;
  }

  LogPrint(LOG_ERR, "unknown %s format: %s", description, name);
  exit(2);
}

static FILE *
openTable (const char **file, const char *mode, const char *directory, FILE *stdStream, const char *stdName) {
  if (stdStream) {
    if (strcmp(*file, "-") == 0) {
      *file = stdName;
      return stdStream;
    }
  }

  if (directory) {
    const char *path = makePath(directory, *file);
    if (!path) return NULL;
    *file = path;
  }

  {
    FILE *stream = fopen(*file, mode);
    if (!stream) LogPrint(LOG_ERR, "table open error: %s: %s", *file, strerror(errno));
    return stream;
  }
}

static const char *inputPath;
static const char *outputPath;
static const FormatEntry *inputFormat;
static const FormatEntry *outputFormat;

static int
convertTable (void) {
  int status;

  if (outputFormat != inputFormat) {
    FILE *inputFile = openTable(&inputPath, "r", opt_dataDirectory, stdin, "<standard-input>");

    if (inputFile) {
      TranslationTable table;

      if (inputFormat->read(inputPath, inputFile, table, inputFormat->data)) {
        if (outputPath) {
          FILE *outputFile = openTable(&outputPath, "w", NULL, stdout, "<standard-output>");

          if (outputFile) {
            if (outputFormat->write(outputPath, outputFile, table, outputFormat->data)) {
              status = 0;
            } else {
              status = 6;
            }

            fclose(outputFile);
          } else {
            status = 5;
          }
        } else {
          status = 0;
        }
      } else {
        status = 4;
      }

      fclose(inputFile);
    } else {
      status = 3;
    }
  } else {
    LogPrint(LOG_ERR, "same input and output formats: %s", outputFormat->name);
    status = 2;
  }

  return status;
}

static wchar_t *
makeCharacterDescription (TranslationTable table, unsigned char byte, size_t *length, unsigned char *braille) {
  char buffer[0X100];
  int characterIndex;
  int brailleIndex;
  int descriptionLength;

  wint_t character = convertCharToWchar(byte);
  unsigned char dots;
  wchar_t printableCharacter;
  char printablePrefix;

  if (character == WEOF) character = WC_C('?');
  dots = convertWcharToDots(table, character);

  printableCharacter = character;
  if (iswLatin1(printableCharacter)) {
    printablePrefix = '^';
    if (!(printableCharacter & 0X60)) {
      printableCharacter |= 0X40;
      if (printableCharacter & 0X80) printablePrefix = '~';
    } else if (printableCharacter == 0X7F) {
      printableCharacter ^= 0X40;       
    } else if (printableCharacter != printablePrefix) {
      printablePrefix = ' ';
    }
  } else {
    printablePrefix = ' ';
  }

#define DOT(n) ((dots & BRLAPI_DOT##n)? ((n) + '0'): ' ')
  snprintf(buffer, sizeof(buffer),
           "%02X %3u %c%nx %nx [%c%c%c%c%c%c%c%c]%n",
           byte, byte, printablePrefix, &characterIndex, &brailleIndex,
           DOT(1), DOT(2), DOT(3), DOT(4), DOT(5), DOT(6), DOT(7), DOT(8),
           &descriptionLength);
#undef DOT

#ifdef HAVE_LIBICUUC
  {
    char *name = buffer + descriptionLength + 1;
    int size = buffer + sizeof(buffer) - name;
    UErrorCode error = U_ZERO_ERROR;

    u_charName(character, U_EXTENDED_CHAR_NAME, name, size, &error);
    if (U_SUCCESS(error)) {
      descriptionLength += strlen(name) + 1;
      *--name = ' ';
    }
  }
#endif /* HAVE_LIBICUUC */

  {
    wchar_t *description = calloc(descriptionLength+1, sizeof(*description));
    if (description) {
      int i;
      for (i=0; i<descriptionLength; i+=1) {
        wint_t wc = convertCharToWchar(buffer[i]);
        description[i] = (wc != WEOF)? wc: WC_C(' ');
      }

      description[characterIndex] = printableCharacter;
      description[brailleIndex] = BRL_UC_ROW | dots;
      description[descriptionLength] = 0;

      if (length) *length = descriptionLength;
      if (braille) *braille = dots;
      return description;
    }
  }

  return NULL;
}

static void
printCharacterString (const wchar_t *wcs) {
  while (*wcs) {
    wint_t wc = *wcs++;
    printw("%" PRIwc, wc);
  }
}

typedef struct {
  TranslationTable translationTable;
  unsigned char currentCharacter;

  brlapi_fileDescriptor brlapiFileDescriptor;
  unsigned int displayWidth;
  unsigned int displayHeight;
  const char *brlapiErrorFunction;
  const char *brlapiErrorMessage;
} EditTableData;

static int
haveBrailleDisplay (EditTableData *data) {
  return data->brlapiFileDescriptor != (brlapi_fileDescriptor)-1;
}

static void
setBrlapiError (EditTableData *data, const char *function) {
  if ((data->brlapiErrorFunction = function)) {
    data->brlapiErrorMessage = brlapi_strerror(&brlapi_error);
  } else {
    data->brlapiErrorMessage = NULL;
  }
}

static void
releaseBrailleDisplay (EditTableData *data) {
  brlapi_closeConnection();
  data->brlapiFileDescriptor = (brlapi_fileDescriptor)-1;
}

static int
claimBrailleDisplay (EditTableData *data) {
  data->brlapiFileDescriptor = brlapi_openConnection(NULL, NULL);
  if (haveBrailleDisplay(data)) {
    if (brlapi_getDisplaySize(&data->displayWidth, &data->displayHeight) != -1) {
      if (brlapi_enterTtyMode(BRLAPI_TTY_DEFAULT, NULL) != -1) {
        setBrlapiError(data, NULL);
        return 1;
      } else {
        setBrlapiError(data, "brlapi_enterTtyMode");
      }
    } else {
      setBrlapiError(data, "brlapi_getDisplaySize");
    }

    releaseBrailleDisplay(data);
  } else {
    setBrlapiError(data, "brlapi_openConnection");
  }

  return 0;
}

static int
updateCharacterDescription (EditTableData *data) {
  int ok = 0;

  size_t descriptionLength;
  unsigned char descriptionDots;
  wchar_t *descriptionText = makeCharacterDescription(
    data->translationTable,
    data->currentCharacter,
    &descriptionLength,
    &descriptionDots
  );

  if (descriptionText) {
    ok = 1;
    clear();

#if defined(USE_CURSES)
    printw("F2:Save F8:Exit\n");
    printw("Up:Up1 Dn:Down1 PgUp:Up16 PgDn:Down16 Home:First End:Last\n");
    printw("\n");
#else /* standard input/output */
#endif /* clear screen */

    printCharacterString(descriptionText);
    printw("\n");

#define DOT(n) ((descriptionDots & BRLAPI_DOT##n)? '#': ' ')
    printw(" +---+ \n");
    printw("1|%c %c|4\n", DOT(1), DOT(4));
    printw("2|%c %c|5\n", DOT(2), DOT(5));
    printw("3|%c %c|6\n", DOT(3), DOT(6));
    printw("7|%c %c|8\n", DOT(7), DOT(8));
    printw(" +---+ \n");
#undef DOT

    if (data->brlapiErrorFunction) {
      printw("BrlAPI error: %s: %s\n",
             data->brlapiErrorFunction, data->brlapiErrorMessage);
      setBrlapiError(data, NULL);
    }

    refresh();

    if (haveBrailleDisplay(data)) {
      brlapi_writeArguments_t args = BRLAPI_WRITEARGUMENTS_INITIALIZER;
      wchar_t text[data->displayWidth];
      size_t length = MIN(data->displayWidth, descriptionLength);

      wmemcpy(text, descriptionText, length);
      wmemset(&text[length], WC_C(' '), data->displayWidth-length);

      args.regionBegin = 1;
      args.regionSize = data->displayWidth;
      args.text = (char *)text;
      args.textSize = data->displayWidth * sizeof(text[0]);
      args.charset = "WCHAR_T";

      if (brlapi_write(&args) == -1) {
        setBrlapiError(data, "brlapi_write");
        releaseBrailleDisplay(data);
      }
    }

    free(descriptionText);
  }

  return ok;
}

static int
doKeyboardCommand (EditTableData *data) {
#if defined(USE_CURSES)
#ifdef USE_FUNC_GET_WCH
  wint_t ch;
  int ret = get_wch(&ch);

  if (ret == KEY_CODE_YES)
#else /* USE_FUNC_GET_WCH */
  int ch = getch();

  if (ch >= 0X100)
#endif /* USE_FUNC_GET_WCH */
  {
    switch (ch) {
      case KEY_UP:
        data->currentCharacter -= 1;
        break;

      case KEY_DOWN:
        data->currentCharacter += 1;
        break;

      case KEY_PPAGE:
        data->currentCharacter -= 0X10;
        break;

      case KEY_NPAGE:
        data->currentCharacter += 0X10;
        break;

      case KEY_HOME:
        data->currentCharacter = 0;
        break;

      case KEY_END:
        data->currentCharacter = 0XFF;
        break;

      case KEY_F(2): {
        FILE *outputFile;

        if (!outputPath) outputPath = inputPath;
        if (!outputFormat) outputFormat = inputFormat;
        outputFile = openTable(&outputPath, "w", NULL, stdout, "<standard-output>");

        if (outputFile) {
          outputFormat->write(outputPath, outputFile, data->translationTable, outputFormat->data);
          fclose(outputFile);
        }

        break;
      }

      case KEY_F(8):
        return 0;

      default:
        beep();
        break;
    }
  } else

#ifdef USE_FUNC_GET_WCH
  if (ret == OK)
#endif /* USE_FUNC_GET_WCH */
#else /* standard input/output */
  wint_t ch = fgetwc(stdin);
  if (ch == WEOF) return 0;
#endif /* read character */
  {
    if ((ch >= BRL_UC_ROW) && (ch <= (BRL_UC_ROW|0xFF))) {
      /* Set braille pattern */
      data->translationTable[data->currentCharacter] = ch & 0XFF;
    } else {
      /* Switch to char */
      int c = convertWcharToChar(ch);
      if (c != EOF) {
        data->currentCharacter = c;
      } else {
        beep();
      }
    }
  }

  return 1;
}

static int
doBrailleCommand (EditTableData *data) {
  if (haveBrailleDisplay(data)) {
    brlapi_keyCode_t key;
    int ret = brlapi_readKey(0, &key);

    if (ret == 1) {
      unsigned long code = key & BRLAPI_KEY_CODE_MASK;

      switch (key & BRLAPI_KEY_TYPE_MASK) {
        case BRLAPI_KEY_TYPE_CMD:
          switch (code & BRLAPI_KEY_CMD_BLK_MASK) {
            case 0:
              switch (code) {
                case BRLAPI_KEY_CMD_LNUP:
                  data->currentCharacter -= 1;
                  break;

                case BRLAPI_KEY_CMD_LNDN:
                  data->currentCharacter += 1;
                  break;

                case BRLAPI_KEY_CMD_PRPGRPH:
                  data->currentCharacter -= 0X10;
                  break;

                case BRLAPI_KEY_CMD_NXPGRPH:
                  data->currentCharacter += 0X10;
                  break;

                case BRLAPI_KEY_CMD_TOP_LEFT:
                case BRLAPI_KEY_CMD_TOP:
                  data->currentCharacter = 0;
                  break;

                case BRLAPI_KEY_CMD_BOT_LEFT:
                case BRLAPI_KEY_CMD_BOT:
                  data->currentCharacter = 0XFF;
                  break;

                default:
                  beep();
                  break;
              }
              break;

            case BRLAPI_KEY_CMD_PASSDOTS:
              data->translationTable[data->currentCharacter] = code & BRLAPI_KEY_CMD_ARG_MASK;
              break;

            default:
              beep();
              break;
          }
          break;

        case BRLAPI_KEY_TYPE_SYM: {
          /* latin1 */
          if (code < 0X100) code |= BRLAPI_KEY_SYM_UNICODE;

          if ((code & 0X1f000000) == BRLAPI_KEY_SYM_UNICODE) {
            /* unicode */
            if ((code & 0Xffff00) == BRL_UC_ROW) {
              /* Set braille pattern */
              data->translationTable[data->currentCharacter] = code & 0XFF;
            } else {
              /* Switch to char */
              int c = convertWcharToChar(code & 0XFFFFFF);
              if (c != EOF) {
                data->currentCharacter = c;
              } else {
                beep();
              }
            }
          } else {
            switch (code) {
              case BRLAPI_KEY_SYM_UP:
                data->currentCharacter -= 1;
                break;

              case BRLAPI_KEY_SYM_DOWN:
                data->currentCharacter += 1;
                break;

              case BRLAPI_KEY_SYM_PAGE_UP:
                data->currentCharacter -= 0X10;
                break;

              case BRLAPI_KEY_SYM_PAGE_DOWN:
                data->currentCharacter += 0X10;
                break;

              case BRLAPI_KEY_SYM_HOME:
                data->currentCharacter = 0;
                break;

              case BRLAPI_KEY_SYM_END:
                data->currentCharacter = 0XFF;
                break;

              default:
                beep();
                break;
            }
          }
          break;
        }

        default:
          beep();
          break;
      }
    } else if (ret == -1) {
      setBrlapiError(data, "brlapi_readKey");
      releaseBrailleDisplay(data);
    }
  }

  return 1;
}

static int
editTable (void) {
  int status;
  EditTableData data;

  memset(data.translationTable, 0xFF, sizeof(data.translationTable));

  {
    FILE *inputFile = openTable(&inputPath, "r", opt_dataDirectory, NULL, NULL);

    if (inputFile) {
      if (inputFormat->read(inputPath, inputFile, data.translationTable, inputFormat->data)) {
        status = 0;
      } else {
        status = 4;
      }

      fclose(inputFile);
    } else {
      status = 3;
    }
  }

  if (!status) {
    claimBrailleDisplay(&data);

#if defined(USE_CURSES)
    initscr();
    cbreak();
    noecho();
    nonl();
    intrflush(stdscr, FALSE);
    keypad(stdscr, TRUE);
#else /* standard input/output */
#endif /* initialize keyboard and screen */

    data.currentCharacter = 0;
    while (updateCharacterDescription(&data)) {
      fd_set set;
      FD_ZERO(&set);

      {
        int maximumFileDescriptor = STDIN_FILENO;
        FD_SET(STDIN_FILENO, &set);

        if (haveBrailleDisplay(&data)) {
          FD_SET(data.brlapiFileDescriptor, &set);
          if (data.brlapiFileDescriptor > maximumFileDescriptor)
            maximumFileDescriptor = data.brlapiFileDescriptor;
        }

        select(maximumFileDescriptor+1, &set, NULL, NULL, NULL);
      }

      if (FD_ISSET(STDIN_FILENO, &set))
        if (!doKeyboardCommand(&data))
          break;

      if (haveBrailleDisplay(&data) && FD_ISSET(data.brlapiFileDescriptor, &set))
        if (!doBrailleCommand(&data))
          break;
    }

    clear();
    refresh();

#if defined(USE_CURSES)
    endwin();
#else /* standard input/output */
#endif /* restore keyboard and screen */

    if (haveBrailleDisplay(&data)) releaseBrailleDisplay(&data);
  }

  return status;
}

static int
translateText (void) {
  int status;
  FILE *inputFile = openTable(&inputPath, "r", opt_dataDirectory, NULL, NULL);

  if (inputFile) {
    TranslationTable inputTable;

    if (inputFormat->read(inputPath, inputFile, inputTable, inputFormat->data)) {
      if (outputPath) {
        FILE *outputFile = openTable(&outputPath, "r", opt_dataDirectory, NULL, NULL);

        if (outputFile) {
          TranslationTable outputTable;

          if (outputFormat->read(outputPath, outputFile, outputTable, outputFormat->data)) {
            TranslationTable table;

            {
              TranslationTable unoutputTable;
              int byte;
              reverseTranslationTable(outputTable, unoutputTable);
              memset(&table, 0, sizeof(table));
              for (byte=TRANSLATION_TABLE_SIZE-1; byte>=0; byte--)
                table[byte] = unoutputTable[inputTable[byte]];
            }

            status = 0;
            while (1) {
              int character;

              if ((character = fgetc(stdin)) == EOF) {
                if (ferror(stdin)) {
                  LogPrint(LOG_ERR, "input error: %s", strerror(errno));
                  status = 6;
                }
                break;
              }

              if (fputc(table[character], stdout) == EOF) {
                LogPrint(LOG_ERR, "output error: %s", strerror(errno));
                status = 7;
                break;
              }
            }
          } else {
            status = 4;
          }

          if (fclose(outputFile) == EOF) {
            if (!status) {
              LogPrint(LOG_ERR, "output error: %s", strerror(errno));
              status = 7;
            }
          }
        } else {
          status = 3;
        }
      } else {
        LogPrint(LOG_ERR, "output table not specified.");
        status = 2;
      }
    } else {
      status = 4;
    }

    fclose(inputFile);
  } else {
    status = 3;
  }

  return status;
}

int
main (int argc, char *argv[]) {
  int status;

  {
    static const OptionsDescriptor descriptor = {
      OPTION_TABLE(programOptions),
      .applicationName = "tbltest",
      .argumentsSummary = "input-table [output-table]"
    };
    processOptions(&descriptor, &argc, &argv);
  }

  {
    char **const paths[] = {
      &opt_dataDirectory,
      NULL
    };
    fixInstallPaths(paths);
  }

  if (argc == 0) {
    LogPrint(LOG_ERR, "missing input table.");
    exit(2);
  }
  inputPath = *argv++, argc--;

  if (argc > 0) {
    outputPath = *argv++, argc--;
  } else if (opt_outputFormat && *opt_outputFormat) {
    const char *extension = findFileExtension(inputPath);
    int prefix = extension? (extension - inputPath): strlen(inputPath);
    char buffer[prefix + 1 + strlen(opt_outputFormat) + 1];
    snprintf(buffer, sizeof(buffer), "%.*s.%s", prefix, inputPath, opt_outputFormat);
    outputPath = strdupWrapper(buffer);
  } else {
    outputPath = NULL;
  }

  if (argc > 0) {
    LogPrint(LOG_ERR, "too many parameters.");
    exit(2);
  }

  inputFormat = getFormatEntry(opt_inputFormat, inputPath, "input");
  if (outputPath) {
    outputFormat = getFormatEntry(opt_outputFormat, outputPath, "output");
  } else {
    outputFormat = NULL;
  }

  if (opt_characterSet && !setCharset(opt_characterSet)) {
    LogPrint(LOG_ERR, "can't establish character set: %s", opt_characterSet);
    exit(9);
  }

  if (opt_edit) {
    status = editTable();
  } else if (opt_translate) {
    status = translateText();
  } else {
    status = convertTable();
  }

  return status;
}
