/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#include <wx/notebook.h>
#include <wx/listctrl.h>
#include <wx/filename.h>
#include <wx/settings.h> // for GetColor

#include "ImportContactFrame.hh"
#include "ContactDlg.hh"
#include "PropertiesFrame.hh"
#include "AddCrlDlg.hh"
#include "Common.hh"

#include "../AddressBook.hh"
#include "../DirectoryService.hh"
#include "../Certificate.hh"

#ifndef __WXMSW__ // en attendant
#include "pics/search_contact.xpm"
#include "pics/import.xpm"
#include "pics/config.xpm"
#include "pics/stop.xpm"
#include "pics/exit.xpm"

#endif


using namespace Cryptonit;

BEGIN_EVENT_TABLE(ImportContactFrame, wxFrame)
    EVT_TOOL ( IC_LDAP_SEARCH_BTN_ID , ImportContactFrame::ldapSearch )
    EVT_TOOL ( IC_IMPORT_CONTACT_BTN_ID , ImportContactFrame::importContact )
    EVT_TOOL ( IC_CONFIG_BTN_ID , ImportContactFrame::openConfig )
    EVT_TOOL ( IC_CLOSE_BTN_ID , ImportContactFrame::close )
    EVT_TOOL ( IC_LDAP_STOP_BTN_ID , ImportContactFrame::ldapStop )
    
    EVT_LIST_ITEM_ACTIVATED ( IC_CONTACT_LIST_ID, ImportContactFrame::viewContact )
    EVT_LIST_ITEM_SELECTED ( IC_CONTACT_LIST_ID, ImportContactFrame::selectContact )
    EVT_LIST_ITEM_DESELECTED ( IC_CONTACT_LIST_ID, ImportContactFrame::unselectContact )
    EVT_LIST_COL_CLICK (IC_CONTACT_LIST_ID, ImportContactFrame::onColumnClick)

    EVT_CHECKBOX( IC_ENABLE_SECOND_PARAMETER_ID, ImportContactFrame::enableSecondParameter )
END_EVENT_TABLE();


/* Struct and function used by wxListCtrl::SortItems */
struct compareEntryDatas {
    DirectoryService *dsLdap;
    wxString *column;
    bool ascending;
};

int wxCALLBACK compareEntries(long item1, long item2, long sortData)
{
    struct compareEntryDatas* datas;
    datas = (compareEntryDatas*)sortData;

    DirectoryService *dsLdap = datas->dsLdap;
    std::string col = std::string( wx2std(*(wxString*)datas->column) );
    
    std::string* dn1 = (std::string*)item1;
    std::string* dn2 = (std::string*)item2;

    // Do we have to sort in ascending order
    if( datas->ascending ) {
	return strcasecmp( dsLdap->getEntry(*dn1)->getAttributeFirstValue(col).c_str(),
			   dsLdap->getEntry(*dn2)->getAttributeFirstValue(col).c_str() );
    } else { // or descending order
	return ( -1 * strcasecmp( dsLdap->getEntry(*dn1)->getAttributeFirstValue(col).c_str(),
				  dsLdap->getEntry(*dn2)->getAttributeFirstValue(col).c_str() ));
    }

}


ImportContactFrame::ImportContactFrame( AddressBookFrame* parent, 
					wxWindowID id, 
					const wxString& title, 
					User* u,
					std::vector< KeyStore*> *keySt,
					const wxPoint& pos, 
					const wxSize& size, 
					long style, 
					const wxString& name) :
  wxFrame(parent, id, title, pos, size, style, name)
{
    createToolBar();	
    
    // Create status bar without the "size grip"
    CreateStatusBar(1, 0);
    
    // Copy all entries of ContactInfo::AttributeList info field list, since
    // wxCombobox handle only an array wxString
    for( unsigned int i = 0; i < ContactInfo::AttributeListLength; i++ ) {
				fieldList[i] = std2wx( ContactInfo::AttributeList[i][0] );
		}
    
    comparaisonsOperations[0] = _("contains");
    comparaisonsOperations[1] = _("doesn't contain");
    comparaisonsOperations[2] = _("equals");
    comparaisonsOperations[3] = _("not equal");
    comparaisonsOperations[4] = _("approximates");
  

    user = u;
    keyStore = keySt;
    dsLdap = NULL;
    myParent = parent;
    mainSizer = new wxBoxSizer(wxVERTICAL);
    
    initFieldList();
    createLDAPPage();
    mainSizer->Fit( this );
    
    
#ifdef __WXMSW__
    SetIcon(wxICON(aaa));
#endif
    
    Centre();
    Show();
}
  


ImportContactFrame::~ImportContactFrame()
{
    if( contactList != NULL ) {
	for( int i=0 ; i < contactList->GetItemCount() ; i++ ){
	    std::string *data = (std::string*) contactList->GetItemData( i );
	    if( data ){
		delete data;
	    }
	}
    }

    contactFieldList[0].Clear();
    contactFieldList[1].Clear();
}


void ImportContactFrame::initFieldList()
{
    contactFieldList[0].Clear();
    contactFieldList[1].Clear();
    
    /* Initalize the contact fields to be displayed in the contactList.
     * contactFieldList[0] is the text to be displayed in the contactList header
     * contactFieldList[1] is the real information field in the Contact class.
     */
    contactFieldList[0].Add(_("Distinguished name"));
    contactFieldList[1].Add(_T("dn"));

    std::vector<std::string> fields = user->getInfos("LdapFields");

    // If the user already have preferred fields, use them
    if( fields.size() > 0 ) {
				std::vector<std::string>::iterator itr;
				for( itr = fields.begin(); itr != fields.end(); itr++ ) {
						contactFieldList[0].Add(std2wx(*itr));
						contactFieldList[1].Add(std2wx(*itr));
				}
    } else { // Use default ones, if needed
	contactFieldList[0].Add(_T("SN"));
	contactFieldList[0].Add(_T("CN"));
	contactFieldList[0].Add(_T("Mail"));
	
	contactFieldList[1].Add(_T("sn"));
	contactFieldList[1].Add(_T("cn"));
	contactFieldList[1].Add(_T("mail"));
    }

    /* This array is used for sorting in an ascending/descending way
     * each columns.
     */
	int count = contactFieldList[0].GetCount();
    contactFieldListAscending = new bool[count];
	for(int i = 0; i < count; i++) {
		contactFieldListAscending[i] = false;
	}
}


void ImportContactFrame::createLDAPPage( )
{

    wxPanel *panel = new wxPanel ( this );
    
    wxStaticBox *frame = new wxStaticBox( panel , -1 , _("Search parameters"));   
    wxStaticBoxSizer *parametersSizer = new wxStaticBoxSizer(frame , wxVERTICAL );
    wxFlexGridSizer* searchParameterFlexSizer = new wxFlexGridSizer( 4 );

    wxStaticText* searchContactWhereText = new wxStaticText(panel, -1, _("Search contact where:"));
    searchParameterFlexSizer->Add
	( searchContactWhereText, 0, wxEXPAND | wxALL | 
	  wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5 );
    
    /* first line */
    firstField = new wxComboBox(panel, -1, _T(""), wxDefaultPosition, wxSize(150,20),
 				ContactInfo::AttributeListLength, fieldList );
    firstField->SetSelection(0);
    searchParameterFlexSizer->Add( firstField , 0, wxEXPAND | wxALL, 5);

    /* Comparaison combobox */
    firstComparaison = new wxComboBox(panel, -1, _T(""), wxDefaultPosition, wxSize(100,20),
				      comparaisonsOperationsSize, comparaisonsOperations,
				      wxCB_READONLY);
    firstComparaison->SetSelection(2);
    searchParameterFlexSizer->Add( firstComparaison, 0, wxEXPAND | wxALL, 5);
    
    /* Parameter */
    firstParameter = new wxTextCtrl(panel, -1, _("inetOrgPerson"), wxDefaultPosition, wxSize(150,20));
    searchParameterFlexSizer->Add( firstParameter, 0, wxEXPAND | wxALL, 5);

    /* Second line */
    /* Checkbox Label */
    andCheckBox = new wxCheckBox(panel, IC_ENABLE_SECOND_PARAMETER_ID, _(" and:"));
    andCheckBox->SetValue( false );
    searchParameterFlexSizer->Add( andCheckBox, 0, wxEXPAND | wxALL, 5 );

    
    /* Field combobox */
    secondField = new wxComboBox(panel, -1, _T(""), wxDefaultPosition, wxSize(150,20),
				 ContactInfo::AttributeListLength, fieldList );
    secondField->SetSelection(0);
    secondField->Disable();
    searchParameterFlexSizer->Add( secondField, 0, wxEXPAND | wxALL, 5);
    
    /* Comparaison combobox */
    secondComparaison = new wxComboBox(panel, -1, _T(""), wxDefaultPosition, wxSize(100,20),
				     comparaisonsOperationsSize, comparaisonsOperations,
				     wxCB_READONLY);
    secondComparaison->SetSelection(0);
    secondComparaison->Disable();
    searchParameterFlexSizer->Add( secondComparaison, 0, wxEXPAND | wxALL, 5);
    
    /* Parameter */
    secondParameter = new wxTextCtrl(panel, -1, _T(""), wxDefaultPosition, wxSize(150,20));
    secondParameter->Disable();
    searchParameterFlexSizer->Add( secondParameter, 0, wxEXPAND | wxALL, 5);
    parametersSizer->Add( searchParameterFlexSizer );

    contactList = new wxListCtrl( panel, IC_CONTACT_LIST_ID, wxDefaultPosition, 
				  wxDefaultSize, wxLC_REPORT | wxLC_HRULES );
    
    for( unsigned int i = 0; i < contactFieldList[0].GetCount(); i++ ) {
	contactList->InsertColumn( i, contactFieldList[0][i], wxLIST_FORMAT_LEFT, 150 );
    }
    

    mainSizer->Add( parametersSizer , 1, wxEXPAND | wxALL, 2  );
    mainSizer->Add( contactList , 2, wxEXPAND | wxALL, 2 );
    panel->SetSizer( mainSizer );
}



void ImportContactFrame::ldapSearch(wxCommandEvent &WXUNUSED(event))
{
    wxString expr;

    /** First expression **/
    wxString expr1;
    expr1 = firstField->GetStringSelection();
    
    switch( firstComparaison->GetSelection() ) {
    case 0: /* contains */
	{
	    if( firstParameter->GetValue() == _T("") )
					expr1 += _T("=*");
	    else
					expr1 += _T("=*") + firstParameter->GetValue() + _T("*");
	    break;
	}
    case 1:  /* doesn't contain */ 
	/* expr syntax is incorrect */
	{
	    if( firstParameter->GetValue() == _T("") )
		expr1 += _T("=!*");
	    else
		expr1 += _T("=*") + firstParameter->GetValue() + _T("*");
	    break;
	}
    case 2:  /* equals */
	{	    
	    expr1 += _T("=") + firstParameter->GetValue();
	    break;
	}
    case 3:  /* not equal */ 
	/* expr syntax is incorrect */
	{
      expr1 += _T("= !(") + firstParameter->GetValue() + _T(")");
      break;
	}
    case 4:  /* approximates */
	{
	    expr1 += _T("~=") + firstParameter->GetValue();
	    break;
	}
    }

    expr = _T("(") + expr1 + _T(")");
    
    /** Second expression **/
    if( andCheckBox->IsChecked() ) {
	wxString expr2;
	expr2 = secondField->GetStringSelection();
	
	switch( secondComparaison->GetSelection() ) {
	case 0: /* contains */
	    {
		if( secondParameter->GetValue() == _T("") )
		    expr2 += _T("=*");
		else
	  expr2 += _T("=*") + secondParameter->GetValue() + _T("*");
		break;
	    }
	case 1:  /* doesn't contain */ 
	    /* expr syntax is incorrect */
	    {
		if( secondParameter->GetValue() )
		    expr2 += _T("!*");
		else
		    expr2 += _T("= !(*") + secondParameter->GetValue() + _T("*)");
		break;
	    }
	case 2:  /* equals */
	    {	    
		expr2 += _T("=") + secondParameter->GetValue();
		break;
	    }
	case 3:  /* not equal */ 
	    /* expr syntax is incorrect */
	    {
					expr2 += _T("= !(") + secondParameter->GetValue() + _T(")");
		break;
	    }
	case 4:  /* approximates */
	    {
		expr2 += _T("~=") + secondParameter->GetValue();
		break;
	    }
	}
	
	expr += _T(" & (") + expr2 + _T(")");
    }
    
    

    std::string serverName;
    std::string serverPort;
    std::string userRetainLDAPv2;
    std::string dn;
    std::string login, password;
    
    serverName = user->getInfo("LdapServer") != "" ? user->getInfo("LdapServer") : "localhost";
    serverPort = user->getInfo("LdapPort") != "" ? user->getInfo("LdapPort") : "389";
    userRetainLDAPv2 = user->getInfo("Use LDAPv2") != "" ? user->getInfo("Use LDAPv2") : "0";
    dn = user->getInfo("LdapDn") != "" ? user->getInfo("LdapDn") : "";
    login = user->getInfo("LdapLogin");
    password = user->getInfo("LdapPassword");
    
    contactList->DeleteAllItems();
    
    icToolBar->EnableTool( IC_LDAP_SEARCH_BTN_ID, FALSE );
    icToolBar->EnableTool( IC_LDAP_STOP_BTN_ID, TRUE );

    wxString buffer = _("Connecting to");
		buffer += _T(" ");
		buffer += std2wx(serverName);
		buffer += _T("...");
    SetStatusText( buffer );
    
    // If a session was already in use, delete it.
    if( dsLdap != NULL ) {
	delete dsLdap;
	dsLdap = NULL;
    }
    
    dsLdap = dsLdap->factory("ldapcache");
    
    

    std::string params[] = { serverName, serverPort, userRetainLDAPv2,dn, wx2std(expr), 
			     "LDAP_SCOPE_SUBTREE", "" };
    //   std::string params[] = { serverName->GetValue().c_str(), serverPort->GetValue().c_str(),
    // 			   dn->GetValue().c_str(), expr.c_str(), "LDAP_SCOPE_SUBTREE",
    // 			   "" };
    
    dsLdap->setLogin( login );
    dsLdap->setPassword( password );
    
    if( ! dsLdap->read( params ) ) {
	wxMessageBox( _("Cannot connect to the LDAP server."),
		      _("Error"), wxOK | wxICON_ERROR, this);

	delete dsLdap;
	dsLdap = NULL;
	icToolBar->EnableTool( IC_LDAP_SEARCH_BTN_ID , TRUE );
	return;
    }

    if( dsLdap->getNbEntry() > 0 ) {
	long index;
	DirectoryService::iterator entry = dsLdap->begin();

	while( entry != dsLdap->end() ) {
	    wxListItem item;
	    item.SetId(0);
	    item.SetColumn(0);
	    item.SetMask(wxLIST_MASK_DATA | wxLIST_MASK_TEXT);

	    // Store the contact DN into the item 'data' attribute
	    std::string *data = new std::string( entry.first().c_str() );
	    item.SetData( (void*)data );

	    // Set the item text
	    item.SetText(std2wx(entry.first()));

	    // Add the item into the list and store the index were it was 
	    // effectivly added, since the SetId() parameters seems to not
	    // be repestived on Windows.
	    index = contactList->InsertItem( item );

	    // Fill columns with all other fields
	    for( unsigned int j = 1; j < contactFieldList[1].GetCount(); j++ )
					contactList->SetItem( index, j, std2wx(entry.second()->getAttributeFirstValue( wx2std(contactFieldList[1][j]))));
	    
	    entry++;
	}
    }
    
    // Set number of entries in the status bar.
    if( dsLdap->getNbEntry() > 0 ) {
      wxString buffer= wxString::Format(wxPLURAL("%d entry found.","%d entries found.",dsLdap->getNbEntry()),dsLdap->getNbEntry()); 				SetStatusText( buffer );
    }
    
    if( dsLdap->getNbEntry() == 0 ) {
				SetStatusText( _("No entry found.") );
    }

    // Sort contacts using the first field
    struct compareEntryDatas data;
    data.dsLdap = dsLdap;
    data.column = new wxString(contactFieldList[1][0]);
    
    contactList->SortItems( compareEntries, (long)&data );
    
    delete data.column;

    icToolBar->EnableTool( IC_LDAP_SEARCH_BTN_ID, TRUE );
    icToolBar->EnableTool( IC_LDAP_STOP_BTN_ID, FALSE );
}


void ImportContactFrame::ldapStop(wxCommandEvent &WXUNUSED(event))
{
    if( dsLdap != NULL ) {
	delete dsLdap;
	dsLdap = NULL;
    }
    icToolBar->EnableTool( IC_LDAP_SEARCH_BTN_ID, TRUE );
    icToolBar->EnableTool( IC_LDAP_STOP_BTN_ID, FALSE );
}


void ImportContactFrame::viewContact(wxListEvent& event)
{
    wxListItem info;
    info.m_itemId = event.m_itemIndex;
    info.m_col = 0;
    info.m_mask = wxLIST_MASK_TEXT;
    
    if( contactList->GetItem(info) ) {
	wxString title = _("View contact: ") + info.m_text;
	ContactDlg cd(this, -1, dsLdap->getEntry
								(wx2std(info.m_text)),user, true, title );
	
	if( cd.showModal(wxSize(450,400), _("Import")) == wxID_OK ) {
	    /* Import contact into the address book */
	    long item = -1;
	    
	    /* Get the first item */
	    item = contactList->GetNextItem
		(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	    if ( item == -1 ) return;
	    
	    wxListItem currentSelectedItem;
	    currentSelectedItem.m_itemId = item;
	    currentSelectedItem.m_col = 0;
	    currentSelectedItem.m_mask = wxLIST_MASK_TEXT;
	    currentSelectedItem.m_state = wxLIST_STATE_SELECTED;
	    
	    if( contactList->GetItem(currentSelectedItem) ) {
		wxCommandEvent e;
		importContact(e);
		//if( addContact( currentSelectedItem.m_text.c_str() ) == 0 )
		    /* Refresh the parent AddressBook */
		  //  if( myParent != NULL ) myParent->refreshContactList();*/
	    }
	}
    }
}


void ImportContactFrame::selectContact(wxListEvent& event)
{
    icToolBar->EnableTool( IC_IMPORT_CONTACT_BTN_ID, TRUE );
}



void ImportContactFrame::unselectContact(wxListEvent& event)
{
    icToolBar->EnableTool( IC_IMPORT_CONTACT_BTN_ID, FALSE );
}



void ImportContactFrame::enableSecondParameter(wxCommandEvent &WXUNUSED(event))
{
    if( andCheckBox->IsChecked() ) {
	secondField->Enable();
	secondComparaison->Enable();
	secondParameter->Enable();
    } else {
	secondField->Disable();
	secondComparaison->Disable();
	secondParameter->Disable();
    }
}



void ImportContactFrame::importContact(wxCommandEvent &WXUNUSED(event))
{
    long item = -1;
    while( true ) {
	/* Iterate through all selected items */
	item = contactList->GetNextItem
	    (item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if ( item == -1 ) {
	    break;
	}

	wxListItem currentSelectedItem;
	currentSelectedItem.m_itemId = item;
	currentSelectedItem.m_col = 0;
	currentSelectedItem.m_mask = wxLIST_MASK_TEXT;
	currentSelectedItem.m_state = wxLIST_STATE_SELECTED;
    
	if( contactList->GetItem(currentSelectedItem) ) {

#ifdef DEBUG		
	    std::cout << "Import item[" << item << "] : "
		      << currentSelectedItem.m_text << " [";
	    std::cout << dsLdap->getEntry(wx2std(currentSelectedItem.m_text))
		      << "]" << std::endl;
#endif	    
	    
	    /* Check if the user already exist in the user address book */
	    if( user->addressBook->getContactDSE(wx2std(currentSelectedItem.m_text)) != NULL ) {
	      wxString msg = wxString::Format(_("There alreay exists a contact associated with distinguished name: '%s'. Remove it from the address book if you really want to add it."),currentSelectedItem.m_text.c_str());  
		wxMessageBox(msg, _("Error"), wxOK | wxICON_ERROR, this);
		continue;
	    }

	    /* Check if contact have a certificate */
	    if( dsLdap->getEntry(wx2std(currentSelectedItem.m_text))->getAttributeFirstValue
		("userCertificate;binary") == "" ) {
	      wxString msg = wxString::Format(_("Cannot import contact '%s' because he does not have any certificate."),currentSelectedItem.m_text.c_str());
		wxMessageBox(msg, _("Error"), wxOK | wxICON_ERROR, this);
		continue;
	    }
	    addContact( wx2std(currentSelectedItem.m_text) );
	}
    }
    
    /* Refresh the parent AddressBook */
    if( myParent != NULL ) {
	myParent->refreshContactList();
    }
}


void ImportContactFrame::close(wxCommandEvent &WXUNUSED(event))
{
    Close(TRUE);
}


int ImportContactFrame::addContact( std::string contactName )
{
    /* Save the certificate in a file in DER format 
     * Filename is HOME/CERTIFICATES_PATH/hash.der
     */
    Cryptonit::Certificate* certificate;
    
    certificate= new Certificate
	((void*)dsLdap->getEntry(contactName)->getAttributeFirstValue
	 ("userCertificate;binary").c_str(),
	 dsLdap->getEntry(contactName)->getAttributeFirstValue
	 ("userCertificate;binary").size() );
    
    /* Construct the certificate filename. */
    wxFileName certFilename( std2wx(certificate->getHash()) );
    certFilename.SetExt( _T("der") );
    certFilename=wxMyPrepend(certFilename, std2wx(user->getCertificatesDir()) );
    
    /* Write the certificate on disk. */
    if( certificate->save( wx2std(certFilename.GetFullPath()).c_str() ) != SUCCESS ) {
	wxString msg = _("An error occured while writing the contact's certificate on disk.");
	msg +=  _T(" (") + certFilename.GetFullPath() + _T(")");
	wxMessageBox(msg, _("Error"), wxOK | wxICON_ERROR, this);
	delete certificate;
	return 1;
    }

    std::string cert = certificate->getHash();
    cert += ".der";

    Contact *contact = new Contact( certificate->getSubjectName().getValues
				    (DN_DISPLAY_SHORT | DN_DISPLAY_VALUE, ','),
				    dsLdap->getEntry(contactName) );
  
    /* Remove 'userCertificate;binary' entry and add a 'userCertificate' pointing to
     * the contact certificate filename.
     */
    contact->removeInfo( "userCertificate;binary" );
    contact->setInfo( "userCertificate", cert );

    /* Add the contact into the address book */
    if( ! user->addressBook->addContact( contact ) ) {
      wxString msg = wxString::Format(_("An error occured while adding contact: %s."),std2wx(contactName).c_str());
	wxMessageBox(msg, _("Error"), wxOK | wxICON_ERROR, this);
	delete certificate;
	return 1;
    }
    
  
    // ask the user if he wants to download the new contact crl.
    std::string uri = certificate->extractCrlUri();
    if( uri != "" && Crl::outDated(uri , user , certificate)) {
#ifdef DEBUG	    
	std::cout << certificate->getSubjectName().getValue( "CN" ) << std::endl;
#endif	


	wxString msg = wxString::Format(_("Would you like to download a CRL for the certificate: %s?"),std2wx(certificate->getSubjectName().getValue( "CN" )).c_str());
	
	int answer = wxMessageBox( msg, _("Question") , wxICON_QUESTION | wxYES_NO , this );
	if( answer == wxYES ){
	    wxString newCrlFile = AddCrlDlg::fetchCrl( std2wx(uri) );
	    Crl newCrl;
	    if( newCrlFile != _T("")){
					if( newCrl.load( wx2std(newCrlFile).c_str()) == SUCCESS ){
							AddCrlDlg::saveCrl( std2wx(uri) , &newCrl , *certificate , user ); 
		    
		} else {
		    wxMessageBox( _("The downloaded CRL is not valid! "), _("Error") , wxICON_ERROR | wxOK , this );
		}
	    }
	}
    } else if(uri == ""){
	wxMessageBox( _("This contact certificate have no CRL.\nPlease add a CRL in the Certificate Viewer 'CRL' tab."),
		      _("Info") , wxICON_INFORMATION | wxOK , this );
    }
    
  
    delete certificate;
    
    return 0;
}


void ImportContactFrame::openConfig(wxCommandEvent &WXUNUSED(event))
{
    PropertiesFrame *pf = new  PropertiesFrame(this, -1 ,_("Properties"), user, keyStore);
    pf->displayPage( 5 );
    pf->ShowModal();

    // Refresh the column list
    initFieldList();
    bool needRefresh = false;
    if( contactList->GetItemCount() > 0 ) {
	needRefresh = true;
    }
    contactList->ClearAll();

    for( unsigned int i = 0; i < contactFieldList[0].GetCount(); i++ ) {
	contactList->InsertColumn( i, contactFieldList[0][i], wxLIST_FORMAT_LEFT, 150 );
    }
    
    if( needRefresh ) {
	wxCommandEvent dummy;
	ldapSearch(dummy);
    }
}


void ImportContactFrame::createToolBar()
{
    wxBitmap toolBarBitmaps[5];

    toolBarBitmaps[0] = wxBITMAP(search_contact);
    toolBarBitmaps[1] = wxBITMAP(import);
    toolBarBitmaps[2] = wxBITMAP(stop);
    toolBarBitmaps[3] = wxBITMAP(config);
    toolBarBitmaps[4] = wxBITMAP(exit);


#ifdef __WXMSW__
    icToolBar = CreateToolBar(wxTB_FLAT |wxTB_TEXT |  wxTB_HORIZONTAL | wxNO_BORDER, 500);
    icToolBar->SetToolBitmapSize(wxSize(32, 32));
#else
    icToolBar = new wxToolBar( this, AB_TOOLBAR_ID, wxDefaultPosition, wxDefaultSize, 
			       wxTB_FLAT |wxTB_TEXT |  wxTB_HORIZONTAL | wxNO_BORDER);
#endif
    
    icToolBar->AddTool( IC_IMPORT_CONTACT_BTN_ID, _("Import"), toolBarBitmaps[1],_("Import selected contacts"));

    icToolBar->AddTool( IC_LDAP_SEARCH_BTN_ID, _("Search"), toolBarBitmaps[0],_("Search contact in ldap"));
    icToolBar->AddTool( IC_LDAP_STOP_BTN_ID, _("Stop"), toolBarBitmaps[2],_("Stop the current search"));
    icToolBar->AddSeparator();
    icToolBar->AddTool( IC_CONFIG_BTN_ID, _("Settings"), toolBarBitmaps[3],_("Configure ldap access"));
    icToolBar->AddSeparator();
    icToolBar->AddTool( IC_CLOSE_BTN_ID, _("Close"), toolBarBitmaps[4], _("Close this window"));

    
    icToolBar->EnableTool( IC_IMPORT_CONTACT_BTN_ID , false );
    icToolBar->EnableTool( IC_LDAP_STOP_BTN_ID , false );

#ifndef __WXMSW__
    SetToolBar(icToolBar);
#endif
    icToolBar->Realize();
 
    icToolBar->SetRows(1);
}



void ImportContactFrame::onColumnClick(wxListEvent& event)
{
    struct compareEntryDatas datas;

    /* Give the adresse of the buffer containing 
     * the title of the clicked column.
     */
    wxString *col = new wxString( contactFieldList[1][event.GetColumn()] );

    datas.column = col;
    datas.ascending = contactFieldListAscending[event.GetColumn()];
    datas.dsLdap = dsLdap;

    contactList->SortItems( compareEntries, (long)&datas );

    contactFieldListAscending[event.GetColumn()] = contactFieldListAscending[event.GetColumn()] ? false : true;

    delete col;
}

